"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constants_1 = require("./constants");
const workflow_actions_1 = require("./workflow-actions");
const workflows_1 = require("./workflows");
const DEFAULT_JOB_ID = "build";
/**
 * A GitHub workflow for common build tasks within a project.
 */
class TaskWorkflow extends workflows_1.GithubWorkflow {
    constructor(github, options) {
        super(github, options.name);
        this.jobId = options.jobId ?? DEFAULT_JOB_ID;
        this.github = github;
        this.artifactsDirectory = options.artifactsDirectory;
        if (options.triggers) {
            if (options.triggers.issueComment) {
                // https://docs.github.com/en/actions/learn-github-actions/security-hardening-for-github-actions#potential-impact-of-a-compromised-runner
                throw new Error('Trigger "issueComment" should not be used due to a security concern');
            }
            this.on(options.triggers);
        }
        this.on({
            workflowDispatch: {},
        });
        const preCheckoutSteps = options.preCheckoutSteps ?? [];
        const checkoutWith = {};
        if (options.downloadLfs ?? github.downloadLfs) {
            checkoutWith.lfs = true;
        }
        // 'checkoutWith' can override 'lfs'
        Object.assign(checkoutWith, options.checkoutWith ?? {});
        const preBuildSteps = options.preBuildSteps ?? [];
        const postBuildSteps = options.postBuildSteps ?? [];
        const gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        if (this.artifactsDirectory) {
            postBuildSteps.push({
                name: "Upload artifact",
                uses: "actions/upload-artifact@v3",
                // Setting to always will ensure that this step will run even if
                // the previous ones have failed (e.g. coverage report, internal logs, etc)
                if: "always()",
                with: {
                    name: this.artifactsDirectory,
                    path: this.artifactsDirectory,
                },
            });
        }
        const job = {
            runsOn: options.runsOn ?? ["ubuntu-latest"],
            container: options.container,
            env: options.env,
            permissions: options.permissions,
            if: options.condition,
            outputs: options.outputs,
            steps: [
                ...preCheckoutSteps,
                // check out sources.
                {
                    name: "Checkout",
                    uses: "actions/checkout@v3",
                    ...(Object.keys(checkoutWith).length > 0
                        ? { with: checkoutWith }
                        : {}),
                },
                // sets git identity so we can push later
                ...workflow_actions_1.WorkflowActions.setGitIdentity(gitIdentity),
                ...preBuildSteps,
                // run the main build task
                {
                    name: options.task.name,
                    run: this.github.project.runTaskCommand(options.task),
                },
                ...postBuildSteps,
            ],
        };
        this.addJobs({ [this.jobId]: job });
    }
}
exports.TaskWorkflow = TaskWorkflow;
_a = JSII_RTTI_SYMBOL_1;
TaskWorkflow[_a] = { fqn: "projen.github.TaskWorkflow", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,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