import { GitHubActionsProvider } from "./actions-provider";
import { Dependabot, DependabotOptions } from "./dependabot";
import { GithubCredentials } from "./github-credentials";
import { Mergify, MergifyOptions } from "./mergify";
import { PullRequestTemplate } from "./pr-template";
import { PullRequestLintOptions } from "./pull-request-lint";
import { GithubWorkflow } from "./workflows";
import { Component } from "../component";
import { Project } from "../project";
export interface GitHubOptions {
    /**
     * Whether mergify should be enabled on this repository or not.
     *
     * @default true
     */
    readonly mergify?: boolean;
    /**
     * Options for Mergify.
     *
     * @default - default options
     */
    readonly mergifyOptions?: MergifyOptions;
    /**
     * Enables GitHub workflows. If this is set to `false`, workflows will not be created.
     *
     * @default true
     */
    readonly workflows?: boolean;
    /**
     * Add a workflow that performs basic checks for pull requests, like
     * validating that PRs follow Conventional Commits.
     *
     * @default true
     */
    readonly pullRequestLint?: boolean;
    /**
     * Options for configuring a pull request linter.
     *
     * @default - see defaults in `PullRequestLintOptions`
     */
    readonly pullRequestLintOptions?: PullRequestLintOptions;
    /**
     * Choose a method of providing GitHub API access for projen workflows.
     *
     * @default - use a personal access token named PROJEN_GITHUB_TOKEN
     */
    readonly projenCredentials?: GithubCredentials;
    /**
     * The name of a secret which includes a GitHub Personal Access Token to be
     * used by projen workflows. This token needs to have the `repo`, `workflows`
     * and `packages` scope.
     *
     * @default "PROJEN_GITHUB_TOKEN"
     * @deprecated - use `projenCredentials`
     */
    readonly projenTokenSecret?: string;
    /**
     * Download files in LFS in workflows
     *
     * @default true if the associated project has `lfsPatterns`, `false` otherwise
     */
    readonly downloadLfs?: boolean;
}
export declare class GitHub extends Component {
    /**
     * Returns the `GitHub` component of a project or `undefined` if the project
     * does not have a GitHub component.
     */
    static of(project: Project): GitHub | undefined;
    /**
     * The `Mergify` configured on this repository. This is `undefined` if Mergify
     * was not enabled when creating the repository.
     */
    readonly mergify?: Mergify;
    /**
     * Are workflows enabled?
     */
    readonly workflowsEnabled: boolean;
    /**
     * GitHub API authentication method used by projen workflows.
     */
    readonly projenCredentials: GithubCredentials;
    /**
     *
     */
    readonly actions: GitHubActionsProvider;
    private readonly _downloadLfs?;
    constructor(project: Project, options?: GitHubOptions);
    /**
     * All workflows.
     */
    get workflows(): GithubWorkflow[];
    /**
     * Adds a workflow to the project.
     * @param name Name of the workflow
     * @returns a GithubWorkflow instance
     */
    addWorkflow(name: string): GithubWorkflow;
    addPullRequestTemplate(...content: string[]): PullRequestTemplate;
    addDependabot(options?: DependabotOptions): Dependabot;
    /**
     * Finds a GitHub workflow by name. Returns `undefined` if the workflow cannot be found.
     * @param name The name of the GitHub workflow
     */
    tryFindWorkflow(name: string): undefined | GithubWorkflow;
    /**
     * Whether downloading from LFS is enabled for this GitHub project
     */
    get downloadLfs(): boolean;
}
