"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dependabot = exports.VersioningStrategy = exports.DependabotScheduleInterval = exports.DependabotRegistryType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Each configuration type requires you to provide particular settings.
 * Some types allow more than one way to connect
 * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#configuration-options-for-private-registries
 */
var DependabotRegistryType;
(function (DependabotRegistryType) {
    /**
     * The composer-repository type supports username and password.
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#composer-repository
     */
    DependabotRegistryType["COMPOSER_REGISTRY"] = "composer-registry";
    /**
     * The docker-registry type supports username and password.
     * The docker-registry type can also be used to pull from Amazon ECR using static AWS credentials
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#docker-registry
     */
    DependabotRegistryType["DOCKER_REGISTRY"] = "docker-registry";
    /**
     * The git type supports username and password
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#git
     */
    DependabotRegistryType["GIT"] = "git";
    /**
     * The hex-organization type supports organization and key
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#hex-organization
     */
    DependabotRegistryType["HEX_ORGANIZATION"] = "hex-organization";
    /**
     * The maven-repository type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#maven-repository
     */
    DependabotRegistryType["MAVEN_REPOSITORY"] = "maven-repository";
    /**
     * The npm-registry type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#npm-registry
     */
    DependabotRegistryType["NPM_REGISTRY"] = "npm-registry";
    /**
     * The nuget-feed type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#nuget-feed
     */
    DependabotRegistryType["NUGET_FEED"] = "nuget-feed";
    /**
     * The python-index type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#python-index
     */
    DependabotRegistryType["PYTHON_INDEX"] = "python-index";
    /**
     * The rubygems-server type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#rubygems-server
     */
    DependabotRegistryType["RUBYGEMS_SERVER"] = "rubygems-server";
    /**
     * The terraform-registry type supports a token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#terraform-registry
     */
    DependabotRegistryType["TERRAFORM_REGISTRY"] = "terraform-registry";
})(DependabotRegistryType = exports.DependabotRegistryType || (exports.DependabotRegistryType = {}));
/**
 * How often to check for new versions and raise pull requests for version
 * updates.
 */
var DependabotScheduleInterval;
(function (DependabotScheduleInterval) {
    /**
     * Runs on every weekday, Monday to Friday.
     */
    DependabotScheduleInterval["DAILY"] = "daily";
    /**
     * Runs once each week. By default, this is on Monday.
     */
    DependabotScheduleInterval["WEEKLY"] = "weekly";
    /**
     * Runs once each month. This is on the first day of the month.
     */
    DependabotScheduleInterval["MONTHLY"] = "monthly";
})(DependabotScheduleInterval = exports.DependabotScheduleInterval || (exports.DependabotScheduleInterval = {}));
/**
 * The strategy to use when edits manifest and lock files.
 */
var VersioningStrategy;
(function (VersioningStrategy) {
    /**
     * Only create pull requests to update lockfiles updates. Ignore any new
     * versions that would require package manifest changes.
     */
    VersioningStrategy["LOCKFILE_ONLY"] = "lockfile-only";
    /**
     * - For apps, the version requirements are increased.
     * - For libraries, the range of versions is widened.
     */
    VersioningStrategy["AUTO"] = "auto";
    /**
     * Relax the version requirement to include both the new and old version, when
     * possible.
     */
    VersioningStrategy["WIDEN"] = "widen";
    /**
     * Always increase the version requirement to match the new version.
     */
    VersioningStrategy["INCREASE"] = "increase";
    /**
     * Increase the version requirement only when required by the new version.
     */
    VersioningStrategy["INCREASE_IF_NECESSARY"] = "increase-if-necessary";
})(VersioningStrategy = exports.VersioningStrategy || (exports.VersioningStrategy = {}));
/**
 * Defines dependabot configuration for node projects.
 *
 * Since module versions are managed in projen, the versioning strategy will be
 * configured to "lockfile-only" which means that only updates that can be done
 * on the lockfile itself will be proposed.
 */
class Dependabot extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        const project = github.project;
        this.ignore = [];
        this.ignoresProjen = options.ignoreProjen ?? true;
        const registries = options.registries
            ? util_1.kebabCaseKeys(options.registries)
            : undefined;
        this.config = {
            version: 2,
            registries,
            updates: [
                {
                    "package-ecosystem": "npm",
                    "versioning-strategy": "lockfile-only",
                    directory: "/",
                    schedule: {
                        interval: options.scheduleInterval ?? DependabotScheduleInterval.DAILY,
                    },
                    ignore: () => (this.ignore.length > 0 ? this.ignore : undefined),
                    labels: options.labels ? options.labels : undefined,
                    registries: registries ? Object.keys(registries) : undefined,
                },
            ],
        };
        new yaml_1.YamlFile(project, ".github/dependabot.yml", {
            obj: this.config,
            committed: true,
        });
        for (const i of options.ignore ?? []) {
            this.addIgnore(i.dependencyName, ...(i.versions ?? []));
        }
        if (this.ignoresProjen) {
            this.addIgnore("projen");
        }
    }
    /**
     * Ignores a dependency from automatic updates.
     *
     * @param dependencyName Use to ignore updates for dependencies with matching
     * names, optionally using `*` to match zero or more characters.
     * @param versions Use to ignore specific versions or ranges of versions. If
     * you want to define a range, use the standard pattern for the package
     * manager (for example: `^1.0.0` for npm, or `~> 2.0` for Bundler).
     */
    addIgnore(dependencyName, ...versions) {
        this.ignore.push({
            "dependency-name": dependencyName,
            versions: () => (versions.length > 0 ? versions : undefined),
        });
    }
}
exports.Dependabot = Dependabot;
_a = JSII_RTTI_SYMBOL_1;
Dependabot[_a] = { fqn: "projen.github.Dependabot", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwZW5kYWJvdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9naXRodWIvZGVwZW5kYWJvdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUNBLDRDQUF5QztBQUN6QyxrQ0FBd0M7QUFDeEMsa0NBQW1DO0FBeUduQzs7OztHQUlHO0FBQ0gsSUFBWSxzQkE2RFg7QUE3REQsV0FBWSxzQkFBc0I7SUFDaEM7OztPQUdHO0lBQ0gsaUVBQXVDLENBQUE7SUFFdkM7Ozs7T0FJRztJQUNILDZEQUFtQyxDQUFBO0lBRW5DOzs7T0FHRztJQUNILHFDQUFXLENBQUE7SUFFWDs7O09BR0c7SUFDSCwrREFBcUMsQ0FBQTtJQUVyQzs7O09BR0c7SUFDSCwrREFBcUMsQ0FBQTtJQUVyQzs7O09BR0c7SUFDSCx1REFBNkIsQ0FBQTtJQUU3Qjs7O09BR0c7SUFDSCxtREFBeUIsQ0FBQTtJQUV6Qjs7O09BR0c7SUFDSCx1REFBNkIsQ0FBQTtJQUU3Qjs7O09BR0c7SUFDSCw2REFBbUMsQ0FBQTtJQUVuQzs7O09BR0c7SUFDSCxtRUFBeUMsQ0FBQTtBQUMzQyxDQUFDLEVBN0RXLHNCQUFzQixHQUF0Qiw4QkFBc0IsS0FBdEIsOEJBQXNCLFFBNkRqQztBQXdCRDs7O0dBR0c7QUFDSCxJQUFZLDBCQWVYO0FBZkQsV0FBWSwwQkFBMEI7SUFDcEM7O09BRUc7SUFDSCw2Q0FBZSxDQUFBO0lBRWY7O09BRUc7SUFDSCwrQ0FBaUIsQ0FBQTtJQUVqQjs7T0FFRztJQUNILGlEQUFtQixDQUFBO0FBQ3JCLENBQUMsRUFmVywwQkFBMEIsR0FBMUIsa0NBQTBCLEtBQTFCLGtDQUEwQixRQWVyQztBQUVEOztHQUVHO0FBQ0gsSUFBWSxrQkE0Qlg7QUE1QkQsV0FBWSxrQkFBa0I7SUFDNUI7OztPQUdHO0lBQ0gscURBQStCLENBQUE7SUFFL0I7OztPQUdHO0lBQ0gsbUNBQWEsQ0FBQTtJQUViOzs7T0FHRztJQUNILHFDQUFlLENBQUE7SUFFZjs7T0FFRztJQUNILDJDQUFxQixDQUFBO0lBRXJCOztPQUVHO0lBQ0gscUVBQStDLENBQUE7QUFDakQsQ0FBQyxFQTVCVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQTRCN0I7QUFFRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLFVBQVcsU0FBUSxxQkFBUztJQWN2QyxZQUFZLE1BQWMsRUFBRSxVQUE2QixFQUFFO1FBQ3pELEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFdEIsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQztRQUUvQixJQUFJLENBQUMsTUFBTSxHQUFHLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsYUFBYSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDO1FBRWxELE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVO1lBQ25DLENBQUMsQ0FBQyxvQkFBYSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7WUFDbkMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLElBQUksQ0FBQyxNQUFNLEdBQUc7WUFDWixPQUFPLEVBQUUsQ0FBQztZQUNWLFVBQVU7WUFDVixPQUFPLEVBQUU7Z0JBQ1A7b0JBQ0UsbUJBQW1CLEVBQUUsS0FBSztvQkFDMUIscUJBQXFCLEVBQUUsZUFBZTtvQkFDdEMsU0FBUyxFQUFFLEdBQUc7b0JBQ2QsUUFBUSxFQUFFO3dCQUNSLFFBQVEsRUFDTixPQUFPLENBQUMsZ0JBQWdCLElBQUksMEJBQTBCLENBQUMsS0FBSztxQkFDL0Q7b0JBQ0QsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ2hFLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTO29CQUNuRCxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2lCQUM3RDthQUNGO1NBQ0YsQ0FBQztRQUVGLElBQUksZUFBUSxDQUFDLE9BQU8sRUFBRSx3QkFBd0IsRUFBRTtZQUM5QyxHQUFHLEVBQUUsSUFBSSxDQUFDLE1BQU07WUFDaEIsU0FBUyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsS0FBSyxNQUFNLENBQUMsSUFBSSxPQUFPLENBQUMsTUFBTSxJQUFJLEVBQUUsRUFBRTtZQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxjQUFjLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxRQUFRLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztTQUN6RDtRQUVELElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUN0QixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzFCO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksU0FBUyxDQUFDLGNBQXNCLEVBQUUsR0FBRyxRQUFrQjtRQUM1RCxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztZQUNmLGlCQUFpQixFQUFFLGNBQWM7WUFDakMsUUFBUSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1NBQzdELENBQUMsQ0FBQztJQUNMLENBQUM7O0FBekVILGdDQTBFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEdpdEh1YiB9IGZyb20gXCIuL2dpdGh1YlwiO1xuaW1wb3J0IHsgQ29tcG9uZW50IH0gZnJvbSBcIi4uL2NvbXBvbmVudFwiO1xuaW1wb3J0IHsga2ViYWJDYXNlS2V5cyB9IGZyb20gXCIuLi91dGlsXCI7XG5pbXBvcnQgeyBZYW1sRmlsZSB9IGZyb20gXCIuLi95YW1sXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGVwZW5kYWJvdE9wdGlvbnMge1xuICAvKipcbiAgICogSG93IG9mdGVuIHRvIGNoZWNrIGZvciBuZXcgdmVyc2lvbnMgYW5kIHJhaXNlIHB1bGwgcmVxdWVzdHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFNjaGVkdWxlSW50ZXJ2YWwuREFJTFlcbiAgICovXG4gIHJlYWRvbmx5IHNjaGVkdWxlSW50ZXJ2YWw/OiBEZXBlbmRhYm90U2NoZWR1bGVJbnRlcnZhbDtcblxuICAvKipcbiAgICogVGhlIHN0cmF0ZWd5IHRvIHVzZSB3aGVuIGVkaXRzIG1hbmlmZXN0IGFuZCBsb2NrIGZpbGVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBWZXJzaW9uaW5nU3RyYXRlZ3kuTE9DS0ZJTEVfT05MWSBUaGUgZGVmYXVsdCBpcyB0byBvbmx5IHVwZGF0ZSB0aGVcbiAgICogbG9jayBmaWxlIGJlY2F1c2UgcGFja2FnZS5qc29uIGlzIGNvbnRyb2xsZWQgYnkgcHJvamVuIGFuZCBhbnkgb3V0c2lkZVxuICAgKiB1cGRhdGVzIHdpbGwgZmFpbCB0aGUgYnVpbGQuXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uaW5nU3RyYXRlZ3k/OiBWZXJzaW9uaW5nU3RyYXRlZ3k7XG5cbiAgLyoqXG4gICAqIFlvdSBjYW4gdXNlIHRoZSBgaWdub3JlYCBvcHRpb24gdG8gY3VzdG9taXplIHdoaWNoIGRlcGVuZGVuY2llcyBhcmUgdXBkYXRlZC5cbiAgICogVGhlIGlnbm9yZSBvcHRpb24gc3VwcG9ydHMgdGhlIGZvbGxvd2luZyBvcHRpb25zLlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgaWdub3JlPzogRGVwZW5kYWJvdElnbm9yZVtdO1xuXG4gIC8qKlxuICAgKiBJZ25vcmVzIHVwZGF0ZXMgdG8gYHByb2plbmAuXG4gICAqXG4gICAqIFRoaXMgaXMgcmVxdWlyZWQgc2luY2UgcHJvamVuIHVwZGF0ZXMgbWF5IGNhdXNlIGNoYW5nZXMgaW4gY29tbWl0dGVkIGZpbGVzXG4gICAqIGFuZCBhbnRpLXRhbXBlciBjaGVja3Mgd2lsbCBmYWlsLlxuICAgKlxuICAgKiBQcm9qZW4gdXBncmFkZXMgYXJlIGNvdmVyZWQgdGhyb3VnaCB0aGUgYFByb2plblVwZ3JhZGVgIGNsYXNzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBpZ25vcmVQcm9qZW4/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGxhYmVscyB0byBhcHBseSB0byB0aGUgY3JlYXRlZCBQUidzLlxuICAgKi9cbiAgcmVhZG9ubHkgbGFiZWxzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIE1hcCBvZiBwYWNrYWdlIHJlZ2lzdHJpZXMgdG8gdXNlXG4gICAqIEBkZWZhdWx0IC0gdXNlIHB1YmxpYyByZWdpc3RyaWVzXG4gICAqL1xuICByZWFkb25seSByZWdpc3RyaWVzPzogeyBbbmFtZTogc3RyaW5nXTogRGVwZW5kYWJvdFJlZ2lzdHJ5IH07XG59XG5cbi8qKlxuICogVXNlIHRvIGFkZCBwcml2YXRlIHJlZ2lzdHJ5IHN1cHBvcnQgZm9yIGRlcGVuZGFib3RcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vY29kZS1zZWN1cml0eS9zdXBwbHktY2hhaW4tc2VjdXJpdHkva2VlcGluZy15b3VyLWRlcGVuZGVuY2llcy11cGRhdGVkLWF1dG9tYXRpY2FsbHkvY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1kZXBlbmRlbmN5LXVwZGF0ZXMjY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1wcml2YXRlLXJlZ2lzdHJpZXNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZXBlbmRhYm90UmVnaXN0cnkge1xuICAvKipcbiAgICogUmVnaXN0cnkgdHlwZSBlLmcuICducG0tcmVnaXN0cnknIG9yICdkb2NrZXItcmVnaXN0cnknXG4gICAqL1xuICByZWFkb25seSB0eXBlOiBEZXBlbmRhYm90UmVnaXN0cnlUeXBlO1xuXG4gIC8qKlxuICAgKiBVcmwgZm9yIHRoZSByZWdpc3RyeSBlLmcuICdodHRwczovL25wbS5wa2cuZ2l0aHViLmNvbScgb3IgJ3JlZ2lzdHJ5Lmh1Yi5kb2NrZXIuY29tJ1xuICAgKi9cbiAgcmVhZG9ubHkgdXJsOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB1c2VybmFtZSB0aGF0IERlcGVuZGFib3QgdXNlcyB0byBhY2Nlc3MgdGhlIHJlZ2lzdHJ5XG4gICAqIEBkZWZhdWx0IC0gZG8gbm90IGF1dGhlbnRpY2F0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdXNlcm5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgcmVmZXJlbmNlIHRvIGEgRGVwZW5kYWJvdCBzZWNyZXQgY29udGFpbmluZyB0aGUgcGFzc3dvcmQgZm9yIHRoZSBzcGVjaWZpZWQgdXNlclxuICAgKiBAZGVmYXVsdCB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IHBhc3N3b3JkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIHJlZmVyZW5jZSB0byBhIERlcGVuZGFib3Qgc2VjcmV0IGNvbnRhaW5pbmcgYW4gYWNjZXNzIGtleSBmb3IgdGhpcyByZWdpc3RyeVxuICAgKiBAZGVmYXVsdCB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IGtleT86IHN0cmluZztcblxuICAvKipcbiAgICogU2VjcmV0IHRva2VuIGZvciBkZXBlbmRhYm90IGFjY2VzcyBlLmcuICcke3sgc2VjcmV0cy5ERVBFTkRBQk9UX1BBQ0tBR0VfVE9LRU4gfX0nXG4gICAqIEBkZWZhdWx0IHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgdG9rZW4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZvciByZWdpc3RyaWVzIHdpdGggdHlwZTogcHl0aG9uLWluZGV4LCBpZiB0aGUgYm9vbGVhbiB2YWx1ZSBpcyB0cnVlLCBwaXBcbiAgICogZXNvbHZlcyBkZXBlbmRlbmNpZXMgYnkgdXNpbmcgdGhlIHNwZWNpZmllZCBVUkwgcmF0aGVyIHRoYW4gdGhlIGJhc2UgVVJMIG9mXG4gICAqIHRoZSBQeXRob24gUGFja2FnZSBJbmRleCAoYnkgZGVmYXVsdCBodHRwczovL3B5cGkub3JnL3NpbXBsZSlcbiAgICogQGRlZmF1bHQgdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSByZXBsYWNlc0Jhc2U/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBVc2VkIHdpdGggdGhlIGhleC1vcmdhbml6YXRpb24gcmVnaXN0cnkgdHlwZS5cbiAgICogQHNlZSBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9jb2RlLXNlY3VyaXR5L3N1cHBseS1jaGFpbi1zZWN1cml0eS9rZWVwaW5nLXlvdXItZGVwZW5kZW5jaWVzLXVwZGF0ZWQtYXV0b21hdGljYWxseS9jb25maWd1cmF0aW9uLW9wdGlvbnMtZm9yLWRlcGVuZGVuY3ktdXBkYXRlcyNoZXgtb3JnYW5pemF0aW9uXG4gICAqIEBkZWZhdWx0IHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgb3JnYW5pemF0aW9uPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEVhY2ggY29uZmlndXJhdGlvbiB0eXBlIHJlcXVpcmVzIHlvdSB0byBwcm92aWRlIHBhcnRpY3VsYXIgc2V0dGluZ3MuXG4gKiBTb21lIHR5cGVzIGFsbG93IG1vcmUgdGhhbiBvbmUgd2F5IHRvIGNvbm5lY3RcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vY29kZS1zZWN1cml0eS9zdXBwbHktY2hhaW4tc2VjdXJpdHkva2VlcGluZy15b3VyLWRlcGVuZGVuY2llcy11cGRhdGVkLWF1dG9tYXRpY2FsbHkvY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1kZXBlbmRlbmN5LXVwZGF0ZXMjY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1wcml2YXRlLXJlZ2lzdHJpZXNcbiAqL1xuZXhwb3J0IGVudW0gRGVwZW5kYWJvdFJlZ2lzdHJ5VHlwZSB7XG4gIC8qKlxuICAgKiBUaGUgY29tcG9zZXItcmVwb3NpdG9yeSB0eXBlIHN1cHBvcnRzIHVzZXJuYW1lIGFuZCBwYXNzd29yZC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9jb2RlLXNlY3VyaXR5L3N1cHBseS1jaGFpbi1zZWN1cml0eS9rZWVwaW5nLXlvdXItZGVwZW5kZW5jaWVzLXVwZGF0ZWQtYXV0b21hdGljYWxseS9jb25maWd1cmF0aW9uLW9wdGlvbnMtZm9yLWRlcGVuZGVuY3ktdXBkYXRlcyNjb21wb3Nlci1yZXBvc2l0b3J5XG4gICAqL1xuICBDT01QT1NFUl9SRUdJU1RSWSA9IFwiY29tcG9zZXItcmVnaXN0cnlcIixcblxuICAvKipcbiAgICogVGhlIGRvY2tlci1yZWdpc3RyeSB0eXBlIHN1cHBvcnRzIHVzZXJuYW1lIGFuZCBwYXNzd29yZC5cbiAgICogVGhlIGRvY2tlci1yZWdpc3RyeSB0eXBlIGNhbiBhbHNvIGJlIHVzZWQgdG8gcHVsbCBmcm9tIEFtYXpvbiBFQ1IgdXNpbmcgc3RhdGljIEFXUyBjcmVkZW50aWFsc1xuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRodWIuY29tL2VuL2NvZGUtc2VjdXJpdHkvc3VwcGx5LWNoYWluLXNlY3VyaXR5L2tlZXBpbmcteW91ci1kZXBlbmRlbmNpZXMtdXBkYXRlZC1hdXRvbWF0aWNhbGx5L2NvbmZpZ3VyYXRpb24tb3B0aW9ucy1mb3ItZGVwZW5kZW5jeS11cGRhdGVzI2RvY2tlci1yZWdpc3RyeVxuICAgKi9cbiAgRE9DS0VSX1JFR0lTVFJZID0gXCJkb2NrZXItcmVnaXN0cnlcIixcblxuICAvKipcbiAgICogVGhlIGdpdCB0eXBlIHN1cHBvcnRzIHVzZXJuYW1lIGFuZCBwYXNzd29yZFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRodWIuY29tL2VuL2NvZGUtc2VjdXJpdHkvc3VwcGx5LWNoYWluLXNlY3VyaXR5L2tlZXBpbmcteW91ci1kZXBlbmRlbmNpZXMtdXBkYXRlZC1hdXRvbWF0aWNhbGx5L2NvbmZpZ3VyYXRpb24tb3B0aW9ucy1mb3ItZGVwZW5kZW5jeS11cGRhdGVzI2dpdFxuICAgKi9cbiAgR0lUID0gXCJnaXRcIixcblxuICAvKipcbiAgICogVGhlIGhleC1vcmdhbml6YXRpb24gdHlwZSBzdXBwb3J0cyBvcmdhbml6YXRpb24gYW5kIGtleVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRodWIuY29tL2VuL2NvZGUtc2VjdXJpdHkvc3VwcGx5LWNoYWluLXNlY3VyaXR5L2tlZXBpbmcteW91ci1kZXBlbmRlbmNpZXMtdXBkYXRlZC1hdXRvbWF0aWNhbGx5L2NvbmZpZ3VyYXRpb24tb3B0aW9ucy1mb3ItZGVwZW5kZW5jeS11cGRhdGVzI2hleC1vcmdhbml6YXRpb25cbiAgICovXG4gIEhFWF9PUkdBTklaQVRJT04gPSBcImhleC1vcmdhbml6YXRpb25cIixcblxuICAvKipcbiAgICogVGhlIG1hdmVuLXJlcG9zaXRvcnkgdHlwZSBzdXBwb3J0cyB1c2VybmFtZSBhbmQgcGFzc3dvcmQsIG9yIHRva2VuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vY29kZS1zZWN1cml0eS9zdXBwbHktY2hhaW4tc2VjdXJpdHkva2VlcGluZy15b3VyLWRlcGVuZGVuY2llcy11cGRhdGVkLWF1dG9tYXRpY2FsbHkvY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1kZXBlbmRlbmN5LXVwZGF0ZXMjbWF2ZW4tcmVwb3NpdG9yeVxuICAgKi9cbiAgTUFWRU5fUkVQT1NJVE9SWSA9IFwibWF2ZW4tcmVwb3NpdG9yeVwiLFxuXG4gIC8qKlxuICAgKiBUaGUgbnBtLXJlZ2lzdHJ5IHR5cGUgc3VwcG9ydHMgdXNlcm5hbWUgYW5kIHBhc3N3b3JkLCBvciB0b2tlblxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRodWIuY29tL2VuL2NvZGUtc2VjdXJpdHkvc3VwcGx5LWNoYWluLXNlY3VyaXR5L2tlZXBpbmcteW91ci1kZXBlbmRlbmNpZXMtdXBkYXRlZC1hdXRvbWF0aWNhbGx5L2NvbmZpZ3VyYXRpb24tb3B0aW9ucy1mb3ItZGVwZW5kZW5jeS11cGRhdGVzI25wbS1yZWdpc3RyeVxuICAgKi9cbiAgTlBNX1JFR0lTVFJZID0gXCJucG0tcmVnaXN0cnlcIixcblxuICAvKipcbiAgICogVGhlIG51Z2V0LWZlZWQgdHlwZSBzdXBwb3J0cyB1c2VybmFtZSBhbmQgcGFzc3dvcmQsIG9yIHRva2VuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vY29kZS1zZWN1cml0eS9zdXBwbHktY2hhaW4tc2VjdXJpdHkva2VlcGluZy15b3VyLWRlcGVuZGVuY2llcy11cGRhdGVkLWF1dG9tYXRpY2FsbHkvY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1kZXBlbmRlbmN5LXVwZGF0ZXMjbnVnZXQtZmVlZFxuICAgKi9cbiAgTlVHRVRfRkVFRCA9IFwibnVnZXQtZmVlZFwiLFxuXG4gIC8qKlxuICAgKiBUaGUgcHl0aG9uLWluZGV4IHR5cGUgc3VwcG9ydHMgdXNlcm5hbWUgYW5kIHBhc3N3b3JkLCBvciB0b2tlblxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRodWIuY29tL2VuL2NvZGUtc2VjdXJpdHkvc3VwcGx5LWNoYWluLXNlY3VyaXR5L2tlZXBpbmcteW91ci1kZXBlbmRlbmNpZXMtdXBkYXRlZC1hdXRvbWF0aWNhbGx5L2NvbmZpZ3VyYXRpb24tb3B0aW9ucy1mb3ItZGVwZW5kZW5jeS11cGRhdGVzI3B5dGhvbi1pbmRleFxuICAgKi9cbiAgUFlUSE9OX0lOREVYID0gXCJweXRob24taW5kZXhcIixcblxuICAvKipcbiAgICogVGhlIHJ1YnlnZW1zLXNlcnZlciB0eXBlIHN1cHBvcnRzIHVzZXJuYW1lIGFuZCBwYXNzd29yZCwgb3IgdG9rZW5cbiAgICogQHNlZSBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9jb2RlLXNlY3VyaXR5L3N1cHBseS1jaGFpbi1zZWN1cml0eS9rZWVwaW5nLXlvdXItZGVwZW5kZW5jaWVzLXVwZGF0ZWQtYXV0b21hdGljYWxseS9jb25maWd1cmF0aW9uLW9wdGlvbnMtZm9yLWRlcGVuZGVuY3ktdXBkYXRlcyNydWJ5Z2Vtcy1zZXJ2ZXJcbiAgICovXG4gIFJVQllHRU1TX1NFUlZFUiA9IFwicnVieWdlbXMtc2VydmVyXCIsXG5cbiAgLyoqXG4gICAqIFRoZSB0ZXJyYWZvcm0tcmVnaXN0cnkgdHlwZSBzdXBwb3J0cyBhIHRva2VuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vY29kZS1zZWN1cml0eS9zdXBwbHktY2hhaW4tc2VjdXJpdHkva2VlcGluZy15b3VyLWRlcGVuZGVuY2llcy11cGRhdGVkLWF1dG9tYXRpY2FsbHkvY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1kZXBlbmRlbmN5LXVwZGF0ZXMjdGVycmFmb3JtLXJlZ2lzdHJ5XG4gICAqL1xuICBURVJSQUZPUk1fUkVHSVNUUlkgPSBcInRlcnJhZm9ybS1yZWdpc3RyeVwiLFxufVxuXG4vKipcbiAqIFlvdSBjYW4gdXNlIHRoZSBgaWdub3JlYCBvcHRpb24gdG8gY3VzdG9taXplIHdoaWNoIGRlcGVuZGVuY2llcyBhcmUgdXBkYXRlZC5cbiAqIFRoZSBpZ25vcmUgb3B0aW9uIHN1cHBvcnRzIHRoZSBmb2xsb3dpbmcgb3B0aW9ucy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZXBlbmRhYm90SWdub3JlIHtcbiAgLyoqXG4gICAqIFVzZSB0byBpZ25vcmUgdXBkYXRlcyBmb3IgZGVwZW5kZW5jaWVzIHdpdGggbWF0Y2hpbmcgbmFtZXMsIG9wdGlvbmFsbHlcbiAgICogdXNpbmcgYCpgIHRvIG1hdGNoIHplcm8gb3IgbW9yZSBjaGFyYWN0ZXJzLlxuICAgKlxuICAgKiBGb3IgSmF2YSBkZXBlbmRlbmNpZXMsIHRoZSBmb3JtYXQgb2YgdGhlIGRlcGVuZGVuY3ktbmFtZSBhdHRyaWJ1dGUgaXM6XG4gICAqIGBncm91cElkOmFydGlmYWN0SWRgLCBmb3IgZXhhbXBsZTogYG9yZy5rb2hzdWtlOmdpdGh1Yi1hcGlgLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVwZW5kZW5jeU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVXNlIHRvIGlnbm9yZSBzcGVjaWZpYyB2ZXJzaW9ucyBvciByYW5nZXMgb2YgdmVyc2lvbnMuIElmIHlvdSB3YW50IHRvXG4gICAqIGRlZmluZSBhIHJhbmdlLCB1c2UgdGhlIHN0YW5kYXJkIHBhdHRlcm4gZm9yIHRoZSBwYWNrYWdlIG1hbmFnZXIgKGZvclxuICAgKiBleGFtcGxlOiBgXjEuMC4wYCBmb3IgbnBtLCBvciBgfj4gMi4wYCBmb3IgQnVuZGxlcikuXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9ucz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIEhvdyBvZnRlbiB0byBjaGVjayBmb3IgbmV3IHZlcnNpb25zIGFuZCByYWlzZSBwdWxsIHJlcXVlc3RzIGZvciB2ZXJzaW9uXG4gKiB1cGRhdGVzLlxuICovXG5leHBvcnQgZW51bSBEZXBlbmRhYm90U2NoZWR1bGVJbnRlcnZhbCB7XG4gIC8qKlxuICAgKiBSdW5zIG9uIGV2ZXJ5IHdlZWtkYXksIE1vbmRheSB0byBGcmlkYXkuXG4gICAqL1xuICBEQUlMWSA9IFwiZGFpbHlcIixcblxuICAvKipcbiAgICogUnVucyBvbmNlIGVhY2ggd2Vlay4gQnkgZGVmYXVsdCwgdGhpcyBpcyBvbiBNb25kYXkuXG4gICAqL1xuICBXRUVLTFkgPSBcIndlZWtseVwiLFxuXG4gIC8qKlxuICAgKiBSdW5zIG9uY2UgZWFjaCBtb250aC4gVGhpcyBpcyBvbiB0aGUgZmlyc3QgZGF5IG9mIHRoZSBtb250aC5cbiAgICovXG4gIE1PTlRITFkgPSBcIm1vbnRobHlcIixcbn1cblxuLyoqXG4gKiBUaGUgc3RyYXRlZ3kgdG8gdXNlIHdoZW4gZWRpdHMgbWFuaWZlc3QgYW5kIGxvY2sgZmlsZXMuXG4gKi9cbmV4cG9ydCBlbnVtIFZlcnNpb25pbmdTdHJhdGVneSB7XG4gIC8qKlxuICAgKiBPbmx5IGNyZWF0ZSBwdWxsIHJlcXVlc3RzIHRvIHVwZGF0ZSBsb2NrZmlsZXMgdXBkYXRlcy4gSWdub3JlIGFueSBuZXdcbiAgICogdmVyc2lvbnMgdGhhdCB3b3VsZCByZXF1aXJlIHBhY2thZ2UgbWFuaWZlc3QgY2hhbmdlcy5cbiAgICovXG4gIExPQ0tGSUxFX09OTFkgPSBcImxvY2tmaWxlLW9ubHlcIixcblxuICAvKipcbiAgICogLSBGb3IgYXBwcywgdGhlIHZlcnNpb24gcmVxdWlyZW1lbnRzIGFyZSBpbmNyZWFzZWQuXG4gICAqIC0gRm9yIGxpYnJhcmllcywgdGhlIHJhbmdlIG9mIHZlcnNpb25zIGlzIHdpZGVuZWQuXG4gICAqL1xuICBBVVRPID0gXCJhdXRvXCIsXG5cbiAgLyoqXG4gICAqIFJlbGF4IHRoZSB2ZXJzaW9uIHJlcXVpcmVtZW50IHRvIGluY2x1ZGUgYm90aCB0aGUgbmV3IGFuZCBvbGQgdmVyc2lvbiwgd2hlblxuICAgKiBwb3NzaWJsZS5cbiAgICovXG4gIFdJREVOID0gXCJ3aWRlblwiLFxuXG4gIC8qKlxuICAgKiBBbHdheXMgaW5jcmVhc2UgdGhlIHZlcnNpb24gcmVxdWlyZW1lbnQgdG8gbWF0Y2ggdGhlIG5ldyB2ZXJzaW9uLlxuICAgKi9cbiAgSU5DUkVBU0UgPSBcImluY3JlYXNlXCIsXG5cbiAgLyoqXG4gICAqIEluY3JlYXNlIHRoZSB2ZXJzaW9uIHJlcXVpcmVtZW50IG9ubHkgd2hlbiByZXF1aXJlZCBieSB0aGUgbmV3IHZlcnNpb24uXG4gICAqL1xuICBJTkNSRUFTRV9JRl9ORUNFU1NBUlkgPSBcImluY3JlYXNlLWlmLW5lY2Vzc2FyeVwiLFxufVxuXG4vKipcbiAqIERlZmluZXMgZGVwZW5kYWJvdCBjb25maWd1cmF0aW9uIGZvciBub2RlIHByb2plY3RzLlxuICpcbiAqIFNpbmNlIG1vZHVsZSB2ZXJzaW9ucyBhcmUgbWFuYWdlZCBpbiBwcm9qZW4sIHRoZSB2ZXJzaW9uaW5nIHN0cmF0ZWd5IHdpbGwgYmVcbiAqIGNvbmZpZ3VyZWQgdG8gXCJsb2NrZmlsZS1vbmx5XCIgd2hpY2ggbWVhbnMgdGhhdCBvbmx5IHVwZGF0ZXMgdGhhdCBjYW4gYmUgZG9uZVxuICogb24gdGhlIGxvY2tmaWxlIGl0c2VsZiB3aWxsIGJlIHByb3Bvc2VkLlxuICovXG5leHBvcnQgY2xhc3MgRGVwZW5kYWJvdCBleHRlbmRzIENvbXBvbmVudCB7XG4gIC8qKlxuICAgKiBUaGUgcmF3IGRlcGVuZGFib3QgY29uZmlndXJhdGlvbi5cbiAgICogQHNlZSBodHRwczovL2RvY3MuZ2l0aHViLmNvbS9lbi9naXRodWIvYWRtaW5pc3RlcmluZy1hLXJlcG9zaXRvcnkvY29uZmlndXJhdGlvbi1vcHRpb25zLWZvci1kZXBlbmRlbmN5LXVwZGF0ZXNcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25maWc6IGFueTtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgcHJvamVuIGlzIGFsc28gdXBncmFkZWQgaW4gdGhpcyBjb25maWcsXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaWdub3Jlc1Byb2plbjogYm9vbGVhbjtcblxuICBwcml2YXRlIHJlYWRvbmx5IGlnbm9yZTogYW55W107XG5cbiAgY29uc3RydWN0b3IoZ2l0aHViOiBHaXRIdWIsIG9wdGlvbnM6IERlcGVuZGFib3RPcHRpb25zID0ge30pIHtcbiAgICBzdXBlcihnaXRodWIucHJvamVjdCk7XG5cbiAgICBjb25zdCBwcm9qZWN0ID0gZ2l0aHViLnByb2plY3Q7XG5cbiAgICB0aGlzLmlnbm9yZSA9IFtdO1xuICAgIHRoaXMuaWdub3Jlc1Byb2plbiA9IG9wdGlvbnMuaWdub3JlUHJvamVuID8/IHRydWU7XG5cbiAgICBjb25zdCByZWdpc3RyaWVzID0gb3B0aW9ucy5yZWdpc3RyaWVzXG4gICAgICA/IGtlYmFiQ2FzZUtleXMob3B0aW9ucy5yZWdpc3RyaWVzKVxuICAgICAgOiB1bmRlZmluZWQ7XG5cbiAgICB0aGlzLmNvbmZpZyA9IHtcbiAgICAgIHZlcnNpb246IDIsXG4gICAgICByZWdpc3RyaWVzLFxuICAgICAgdXBkYXRlczogW1xuICAgICAgICB7XG4gICAgICAgICAgXCJwYWNrYWdlLWVjb3N5c3RlbVwiOiBcIm5wbVwiLFxuICAgICAgICAgIFwidmVyc2lvbmluZy1zdHJhdGVneVwiOiBcImxvY2tmaWxlLW9ubHlcIixcbiAgICAgICAgICBkaXJlY3Rvcnk6IFwiL1wiLFxuICAgICAgICAgIHNjaGVkdWxlOiB7XG4gICAgICAgICAgICBpbnRlcnZhbDpcbiAgICAgICAgICAgICAgb3B0aW9ucy5zY2hlZHVsZUludGVydmFsID8/IERlcGVuZGFib3RTY2hlZHVsZUludGVydmFsLkRBSUxZLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgaWdub3JlOiAoKSA9PiAodGhpcy5pZ25vcmUubGVuZ3RoID4gMCA/IHRoaXMuaWdub3JlIDogdW5kZWZpbmVkKSxcbiAgICAgICAgICBsYWJlbHM6IG9wdGlvbnMubGFiZWxzID8gb3B0aW9ucy5sYWJlbHMgOiB1bmRlZmluZWQsXG4gICAgICAgICAgcmVnaXN0cmllczogcmVnaXN0cmllcyA/IE9iamVjdC5rZXlzKHJlZ2lzdHJpZXMpIDogdW5kZWZpbmVkLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9O1xuXG4gICAgbmV3IFlhbWxGaWxlKHByb2plY3QsIFwiLmdpdGh1Yi9kZXBlbmRhYm90LnltbFwiLCB7XG4gICAgICBvYmo6IHRoaXMuY29uZmlnLFxuICAgICAgY29tbWl0dGVkOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgZm9yIChjb25zdCBpIG9mIG9wdGlvbnMuaWdub3JlID8/IFtdKSB7XG4gICAgICB0aGlzLmFkZElnbm9yZShpLmRlcGVuZGVuY3lOYW1lLCAuLi4oaS52ZXJzaW9ucyA/PyBbXSkpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmlnbm9yZXNQcm9qZW4pIHtcbiAgICAgIHRoaXMuYWRkSWdub3JlKFwicHJvamVuXCIpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBJZ25vcmVzIGEgZGVwZW5kZW5jeSBmcm9tIGF1dG9tYXRpYyB1cGRhdGVzLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwZW5kZW5jeU5hbWUgVXNlIHRvIGlnbm9yZSB1cGRhdGVzIGZvciBkZXBlbmRlbmNpZXMgd2l0aCBtYXRjaGluZ1xuICAgKiBuYW1lcywgb3B0aW9uYWxseSB1c2luZyBgKmAgdG8gbWF0Y2ggemVybyBvciBtb3JlIGNoYXJhY3RlcnMuXG4gICAqIEBwYXJhbSB2ZXJzaW9ucyBVc2UgdG8gaWdub3JlIHNwZWNpZmljIHZlcnNpb25zIG9yIHJhbmdlcyBvZiB2ZXJzaW9ucy4gSWZcbiAgICogeW91IHdhbnQgdG8gZGVmaW5lIGEgcmFuZ2UsIHVzZSB0aGUgc3RhbmRhcmQgcGF0dGVybiBmb3IgdGhlIHBhY2thZ2VcbiAgICogbWFuYWdlciAoZm9yIGV4YW1wbGU6IGBeMS4wLjBgIGZvciBucG0sIG9yIGB+PiAyLjBgIGZvciBCdW5kbGVyKS5cbiAgICovXG4gIHB1YmxpYyBhZGRJZ25vcmUoZGVwZW5kZW5jeU5hbWU6IHN0cmluZywgLi4udmVyc2lvbnM6IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5pZ25vcmUucHVzaCh7XG4gICAgICBcImRlcGVuZGVuY3ktbmFtZVwiOiBkZXBlbmRlbmN5TmFtZSxcbiAgICAgIHZlcnNpb25zOiAoKSA9PiAodmVyc2lvbnMubGVuZ3RoID4gMCA/IHZlcnNpb25zIDogdW5kZWZpbmVkKSxcbiAgICB9KTtcbiAgfVxufVxuIl19