"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const _resolve_1 = require("./_resolve");
const common_1 = require("./common");
const component_1 = require("./component");
const projenrc_1 = require("./projenrc");
const util_1 = require("./util");
class FileBase extends component_1.Component {
    constructor(project, filePath, options = {}) {
        super(project);
        this.readonly = !project.ejected && (options.readonly ?? true);
        this.executable = options.executable ?? false;
        this.path = filePath;
        // `marker` is empty if project is being ejected or if explicitly disabled
        const projenrc = projenrc_1.ProjenrcFile.of(project)?.filePath ?? common_1.PROJEN_RC;
        this.marker =
            project.ejected || options.marker === false
                ? undefined
                : `${common_1.PROJEN_MARKER}. To modify, edit ${projenrc} and run "npx projen".`;
        const globPattern = `/${this.path}`;
        const committed = options.committed ?? project.commitGenerated ?? true;
        if (committed && filePath !== ".gitattributes") {
            project.root.annotateGenerated(`/${filePath}`);
        }
        this.absolutePath = path.resolve(project.outdir, filePath);
        // verify file path is unique within project tree
        const existing = project.root.tryFindFile(this.absolutePath);
        if (existing && existing !== this) {
            throw new Error(`there is already a file under ${path.relative(project.root.outdir, this.absolutePath)}`);
        }
        const editGitignore = options.editGitignore ?? true;
        if (editGitignore) {
            this.project.addGitIgnore(`${committed ? "!" : ""}${globPattern}`);
        }
        else {
            if (options.committed != null) {
                throw new Error('"gitignore" is disabled, so it does not make sense to specify "committed"');
            }
        }
    }
    /**
     * Writes the file to the project's output directory
     */
    synthesize() {
        const outdir = this.project.outdir;
        const filePath = path.join(outdir, this.path);
        const resolver = {
            resolve: (obj, options) => _resolve_1.resolve(obj, options),
        };
        const content = this.synthesizeContent(resolver);
        if (content === undefined) {
            // remove file (if exists) and skip rest of synthesis
            fs_1.rmSync(filePath, { force: true, recursive: true });
            return;
        }
        // check if the file was changed.
        const prev = util_1.tryReadFileSync(filePath);
        const prevReadonly = !util_1.isWritable(filePath);
        const prevExecutable = util_1.isExecutable(filePath);
        if (prev !== undefined &&
            content === prev &&
            prevReadonly === this.readonly &&
            prevExecutable === this.executable) {
            this.project.logger.debug(`no change in ${filePath}`);
            this._changed = false;
            return;
        }
        util_1.writeFile(filePath, content, {
            readonly: this.readonly,
            executable: this.executable,
        });
        this.checkForProjenMarker();
        this._changed = true;
    }
    /**
     * For debugging, check whether a file was incorrectly generated with
     * or without the projen marker. The projen marker does not *need* to be
     * included on projen-generated files, but it's recommended since it signals
     * that it probably should not be edited directly.
     */
    checkForProjenMarker() {
        const filePath = path.join(this.project.outdir, this.path);
        const contents = util_1.tryReadFileSync(filePath);
        const containsMarker = contents?.includes(common_1.PROJEN_MARKER);
        if (this.marker && !containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to contain marker but found none.`);
        }
        else if (!this.marker && containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to not contain marker but found one anyway.`);
        }
    }
    /**
     * Indicates if the file has been changed during synthesis. This property is
     * only available in `postSynthesize()` hooks. If this is `undefined`, the
     * file has not been synthesized yet.
     */
    get changed() {
        return this._changed;
    }
}
exports.FileBase = FileBase;
_a = JSII_RTTI_SYMBOL_1;
FileBase[_a] = { fqn: "projen.FileBase", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmlsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9maWxlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkJBQTRCO0FBQzVCLDZCQUE2QjtBQUM3Qix5Q0FBcUM7QUFDckMscUNBQW9EO0FBQ3BELDJDQUF3QztBQUV4Qyx5Q0FBMEM7QUFDMUMsaUNBQThFO0FBd0M5RSxNQUFzQixRQUFTLFNBQVEscUJBQVM7SUE4QjlDLFlBQ0UsT0FBZ0IsRUFDaEIsUUFBZ0IsRUFDaEIsVUFBMkIsRUFBRTtRQUU3QixLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFZixJQUFJLENBQUMsUUFBUSxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUM5QyxJQUFJLENBQUMsSUFBSSxHQUFHLFFBQVEsQ0FBQztRQUVyQiwwRUFBMEU7UUFDMUUsTUFBTSxRQUFRLEdBQUcsdUJBQVksQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLEVBQUUsUUFBUSxJQUFJLGtCQUFTLENBQUM7UUFDakUsSUFBSSxDQUFDLE1BQU07WUFDVCxPQUFPLENBQUMsT0FBTyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssS0FBSztnQkFDekMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ1gsQ0FBQyxDQUFDLEdBQUcsc0JBQWEscUJBQXFCLFFBQVEsd0JBQXdCLENBQUM7UUFFNUUsTUFBTSxXQUFXLEdBQUcsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDcEMsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsSUFBSSxPQUFPLENBQUMsZUFBZSxJQUFJLElBQUksQ0FBQztRQUN2RSxJQUFJLFNBQVMsSUFBSSxRQUFRLEtBQUssZ0JBQWdCLEVBQUU7WUFDOUMsT0FBTyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDaEQ7UUFFRCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUUzRCxpREFBaUQ7UUFDakQsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQzdELElBQUksUUFBUSxJQUFJLFFBQVEsS0FBSyxJQUFJLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FDYixpQ0FBaUMsSUFBSSxDQUFDLFFBQVEsQ0FDNUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQ25CLElBQUksQ0FBQyxZQUFZLENBQ2xCLEVBQUUsQ0FDSixDQUFDO1NBQ0g7UUFFRCxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQztRQUNwRCxJQUFJLGFBQWEsRUFBRTtZQUNqQixJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUNwRTthQUFNO1lBQ0wsSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLElBQUksRUFBRTtnQkFDN0IsTUFBTSxJQUFJLEtBQUssQ0FDYiwyRUFBMkUsQ0FDNUUsQ0FBQzthQUNIO1NBQ0Y7SUFDSCxDQUFDO0lBV0Q7O09BRUc7SUFDSSxVQUFVO1FBQ2YsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDbkMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzlDLE1BQU0sUUFBUSxHQUFjO1lBQzFCLE9BQU8sRUFBRSxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsRUFBRSxDQUFDLGtCQUFPLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQztTQUNqRCxDQUFDO1FBQ0YsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2pELElBQUksT0FBTyxLQUFLLFNBQVMsRUFBRTtZQUN6QixxREFBcUQ7WUFDckQsV0FBTSxDQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7WUFDbkQsT0FBTztTQUNSO1FBRUQsaUNBQWlDO1FBQ2pDLE1BQU0sSUFBSSxHQUFHLHNCQUFlLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDdkMsTUFBTSxZQUFZLEdBQUcsQ0FBQyxpQkFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLE1BQU0sY0FBYyxHQUFHLG1CQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDOUMsSUFDRSxJQUFJLEtBQUssU0FBUztZQUNsQixPQUFPLEtBQUssSUFBSTtZQUNoQixZQUFZLEtBQUssSUFBSSxDQUFDLFFBQVE7WUFDOUIsY0FBYyxLQUFLLElBQUksQ0FBQyxVQUFVLEVBQ2xDO1lBQ0EsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLGdCQUFnQixRQUFRLEVBQUUsQ0FBQyxDQUFDO1lBQ3RELElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3RCLE9BQU87U0FDUjtRQUVELGdCQUFTLENBQUMsUUFBUSxFQUFFLE9BQU8sRUFBRTtZQUMzQixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO1NBQzVCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1FBRTVCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDO0lBQ3ZCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLG9CQUFvQjtRQUMxQixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzRCxNQUFNLFFBQVEsR0FBRyxzQkFBZSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLE1BQU0sY0FBYyxHQUFHLFFBQVEsRUFBRSxRQUFRLENBQUMsc0JBQWEsQ0FBQyxDQUFDO1FBQ3pELElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUNsQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQ3ZCLGtCQUFrQixJQUFJLENBQUMsSUFBSSxvQ0FBb0MsQ0FDaEUsQ0FBQztTQUNIO2FBQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLElBQUksY0FBYyxFQUFFO1lBQ3pDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FDdkIsa0JBQWtCLElBQUksQ0FBQyxJQUFJLDhDQUE4QyxDQUMxRSxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDdkIsQ0FBQzs7QUE3SkgsNEJBOEpDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgcm1TeW5jIH0gZnJvbSBcImZzXCI7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyByZXNvbHZlIH0gZnJvbSBcIi4vX3Jlc29sdmVcIjtcbmltcG9ydCB7IFBST0pFTl9NQVJLRVIsIFBST0pFTl9SQyB9IGZyb20gXCIuL2NvbW1vblwiO1xuaW1wb3J0IHsgQ29tcG9uZW50IH0gZnJvbSBcIi4vY29tcG9uZW50XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4vcHJvamVjdFwiO1xuaW1wb3J0IHsgUHJvamVucmNGaWxlIH0gZnJvbSBcIi4vcHJvamVucmNcIjtcbmltcG9ydCB7IGlzRXhlY3V0YWJsZSwgaXNXcml0YWJsZSwgdHJ5UmVhZEZpbGVTeW5jLCB3cml0ZUZpbGUgfSBmcm9tIFwiLi91dGlsXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRmlsZUJhc2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoaXMgZmlsZSBzaG91bGQgYmUgY29tbWl0dGVkIHRvIGdpdCBvciBpZ25vcmVkLiBCeVxuICAgKiBkZWZhdWx0LCBhbGwgZ2VuZXJhdGVkIGZpbGVzIGFyZSBjb21taXR0ZWQgYW5kIGFudGktdGFtcGVyIGlzIHVzZWQgdG9cbiAgICogcHJvdGVjdCBhZ2FpbnN0IG1hbnVhbCBtb2RpZmljYXRpb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBjb21taXR0ZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIHByb2plY3QncyAuZ2l0aWdub3JlIGZpbGVcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZWRpdEdpdGlnbm9yZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIGdlbmVyYXRlZCBmaWxlIHNob3VsZCBiZSByZWFkb25seS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVhZG9ubHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBnZW5lcmF0ZWQgZmlsZSBzaG91bGQgYmUgbWFya2VkIGFzIGV4ZWN1dGFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBleGVjdXRhYmxlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWRkcyB0aGUgcHJvamVuIG1hcmtlciB0byB0aGUgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBtYXJrZXIgd2lsbCBiZSBpbmNsdWRlZCBhcyBsb25nIGFzIHRoZSBwcm9qZWN0IGlzIG5vdCBlamVjdGVkXG4gICAqL1xuICByZWFkb25seSBtYXJrZXI/OiBib29sZWFuO1xufVxuXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgRmlsZUJhc2UgZXh0ZW5kcyBDb21wb25lbnQge1xuICAvKipcbiAgICogVGhlIGZpbGUgcGF0aCwgcmVsYXRpdmUgdG8gdGhlIHByb2plY3Qgcm9vdC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiB0aGUgZmlsZSBzaG91bGQgYmUgcmVhZC1vbmx5IG9yIHJlYWQtd3JpdGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHk6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiB0aGUgZmlsZSBzaG91bGQgYmUgbWFya2VkIGFzIGV4ZWN1dGFibGVcbiAgICovXG4gIHB1YmxpYyBleGVjdXRhYmxlOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJvamVuIG1hcmtlciwgdXNlZCB0byBpZGVudGlmeSBmaWxlcyBhcyBwcm9qZW4tZ2VuZXJhdGVkLlxuICAgKlxuICAgKiBWYWx1ZSBpcyB1bmRlZmluZWQgaWYgdGhlIHByb2plY3QgaXMgYmVpbmcgZWplY3RlZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtYXJrZXI6IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICAvKipcbiAgICogVGhlIGFic29sdXRlIHBhdGggb2YgdGhpcyBmaWxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFic29sdXRlUGF0aDogc3RyaW5nO1xuXG4gIHByaXZhdGUgX2NoYW5nZWQ/OiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHByb2plY3Q6IFByb2plY3QsXG4gICAgZmlsZVBhdGg6IHN0cmluZyxcbiAgICBvcHRpb25zOiBGaWxlQmFzZU9wdGlvbnMgPSB7fVxuICApIHtcbiAgICBzdXBlcihwcm9qZWN0KTtcblxuICAgIHRoaXMucmVhZG9ubHkgPSAhcHJvamVjdC5lamVjdGVkICYmIChvcHRpb25zLnJlYWRvbmx5ID8/IHRydWUpO1xuICAgIHRoaXMuZXhlY3V0YWJsZSA9IG9wdGlvbnMuZXhlY3V0YWJsZSA/PyBmYWxzZTtcbiAgICB0aGlzLnBhdGggPSBmaWxlUGF0aDtcblxuICAgIC8vIGBtYXJrZXJgIGlzIGVtcHR5IGlmIHByb2plY3QgaXMgYmVpbmcgZWplY3RlZCBvciBpZiBleHBsaWNpdGx5IGRpc2FibGVkXG4gICAgY29uc3QgcHJvamVucmMgPSBQcm9qZW5yY0ZpbGUub2YocHJvamVjdCk/LmZpbGVQYXRoID8/IFBST0pFTl9SQztcbiAgICB0aGlzLm1hcmtlciA9XG4gICAgICBwcm9qZWN0LmVqZWN0ZWQgfHwgb3B0aW9ucy5tYXJrZXIgPT09IGZhbHNlXG4gICAgICAgID8gdW5kZWZpbmVkXG4gICAgICAgIDogYCR7UFJPSkVOX01BUktFUn0uIFRvIG1vZGlmeSwgZWRpdCAke3Byb2plbnJjfSBhbmQgcnVuIFwibnB4IHByb2plblwiLmA7XG5cbiAgICBjb25zdCBnbG9iUGF0dGVybiA9IGAvJHt0aGlzLnBhdGh9YDtcbiAgICBjb25zdCBjb21taXR0ZWQgPSBvcHRpb25zLmNvbW1pdHRlZCA/PyBwcm9qZWN0LmNvbW1pdEdlbmVyYXRlZCA/PyB0cnVlO1xuICAgIGlmIChjb21taXR0ZWQgJiYgZmlsZVBhdGggIT09IFwiLmdpdGF0dHJpYnV0ZXNcIikge1xuICAgICAgcHJvamVjdC5yb290LmFubm90YXRlR2VuZXJhdGVkKGAvJHtmaWxlUGF0aH1gKTtcbiAgICB9XG5cbiAgICB0aGlzLmFic29sdXRlUGF0aCA9IHBhdGgucmVzb2x2ZShwcm9qZWN0Lm91dGRpciwgZmlsZVBhdGgpO1xuXG4gICAgLy8gdmVyaWZ5IGZpbGUgcGF0aCBpcyB1bmlxdWUgd2l0aGluIHByb2plY3QgdHJlZVxuICAgIGNvbnN0IGV4aXN0aW5nID0gcHJvamVjdC5yb290LnRyeUZpbmRGaWxlKHRoaXMuYWJzb2x1dGVQYXRoKTtcbiAgICBpZiAoZXhpc3RpbmcgJiYgZXhpc3RpbmcgIT09IHRoaXMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYHRoZXJlIGlzIGFscmVhZHkgYSBmaWxlIHVuZGVyICR7cGF0aC5yZWxhdGl2ZShcbiAgICAgICAgICBwcm9qZWN0LnJvb3Qub3V0ZGlyLFxuICAgICAgICAgIHRoaXMuYWJzb2x1dGVQYXRoXG4gICAgICAgICl9YFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBlZGl0R2l0aWdub3JlID0gb3B0aW9ucy5lZGl0R2l0aWdub3JlID8/IHRydWU7XG4gICAgaWYgKGVkaXRHaXRpZ25vcmUpIHtcbiAgICAgIHRoaXMucHJvamVjdC5hZGRHaXRJZ25vcmUoYCR7Y29tbWl0dGVkID8gXCIhXCIgOiBcIlwifSR7Z2xvYlBhdHRlcm59YCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChvcHRpb25zLmNvbW1pdHRlZCAhPSBudWxsKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnXCJnaXRpZ25vcmVcIiBpcyBkaXNhYmxlZCwgc28gaXQgZG9lcyBub3QgbWFrZSBzZW5zZSB0byBzcGVjaWZ5IFwiY29tbWl0dGVkXCInXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEltcGxlbWVudGVkIGJ5IGRlcml2ZWQgY2xhc3NlcyBhbmQgcmV0dXJucyB0aGUgY29udGVudHMgb2YgdGhlIGZpbGUgdG9cbiAgICogZW1pdC5cbiAgICogQHBhcmFtIHJlc29sdmVyIENhbGwgYHJlc29sdmVyLnJlc29sdmUob2JqKWAgb24gYW55IG9iamVjdHMgaW4gb3JkZXIgdG9cbiAgICogcmVzb2x2ZSB0b2tlbiBmdW5jdGlvbnMuXG4gICAqIEByZXR1cm5zIHRoZSBjb250ZW50IHRvIHN5bnRoZXNpemUgb3IgdW5kZWZpbmVkIHRvIHNraXAgdGhlIGZpbGVcbiAgICovXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBzeW50aGVzaXplQ29udGVudChyZXNvbHZlcjogSVJlc29sdmVyKTogc3RyaW5nIHwgdW5kZWZpbmVkO1xuXG4gIC8qKlxuICAgKiBXcml0ZXMgdGhlIGZpbGUgdG8gdGhlIHByb2plY3QncyBvdXRwdXQgZGlyZWN0b3J5XG4gICAqL1xuICBwdWJsaWMgc3ludGhlc2l6ZSgpIHtcbiAgICBjb25zdCBvdXRkaXIgPSB0aGlzLnByb2plY3Qub3V0ZGlyO1xuICAgIGNvbnN0IGZpbGVQYXRoID0gcGF0aC5qb2luKG91dGRpciwgdGhpcy5wYXRoKTtcbiAgICBjb25zdCByZXNvbHZlcjogSVJlc29sdmVyID0ge1xuICAgICAgcmVzb2x2ZTogKG9iaiwgb3B0aW9ucykgPT4gcmVzb2x2ZShvYmosIG9wdGlvbnMpLFxuICAgIH07XG4gICAgY29uc3QgY29udGVudCA9IHRoaXMuc3ludGhlc2l6ZUNvbnRlbnQocmVzb2x2ZXIpO1xuICAgIGlmIChjb250ZW50ID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIHJlbW92ZSBmaWxlIChpZiBleGlzdHMpIGFuZCBza2lwIHJlc3Qgb2Ygc3ludGhlc2lzXG4gICAgICBybVN5bmMoZmlsZVBhdGgsIHsgZm9yY2U6IHRydWUsIHJlY3Vyc2l2ZTogdHJ1ZSB9KTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICAvLyBjaGVjayBpZiB0aGUgZmlsZSB3YXMgY2hhbmdlZC5cbiAgICBjb25zdCBwcmV2ID0gdHJ5UmVhZEZpbGVTeW5jKGZpbGVQYXRoKTtcbiAgICBjb25zdCBwcmV2UmVhZG9ubHkgPSAhaXNXcml0YWJsZShmaWxlUGF0aCk7XG4gICAgY29uc3QgcHJldkV4ZWN1dGFibGUgPSBpc0V4ZWN1dGFibGUoZmlsZVBhdGgpO1xuICAgIGlmIChcbiAgICAgIHByZXYgIT09IHVuZGVmaW5lZCAmJlxuICAgICAgY29udGVudCA9PT0gcHJldiAmJlxuICAgICAgcHJldlJlYWRvbmx5ID09PSB0aGlzLnJlYWRvbmx5ICYmXG4gICAgICBwcmV2RXhlY3V0YWJsZSA9PT0gdGhpcy5leGVjdXRhYmxlXG4gICAgKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLmRlYnVnKGBubyBjaGFuZ2UgaW4gJHtmaWxlUGF0aH1gKTtcbiAgICAgIHRoaXMuX2NoYW5nZWQgPSBmYWxzZTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB3cml0ZUZpbGUoZmlsZVBhdGgsIGNvbnRlbnQsIHtcbiAgICAgIHJlYWRvbmx5OiB0aGlzLnJlYWRvbmx5LFxuICAgICAgZXhlY3V0YWJsZTogdGhpcy5leGVjdXRhYmxlLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jaGVja0ZvclByb2plbk1hcmtlcigpO1xuXG4gICAgdGhpcy5fY2hhbmdlZCA9IHRydWU7XG4gIH1cblxuICAvKipcbiAgICogRm9yIGRlYnVnZ2luZywgY2hlY2sgd2hldGhlciBhIGZpbGUgd2FzIGluY29ycmVjdGx5IGdlbmVyYXRlZCB3aXRoXG4gICAqIG9yIHdpdGhvdXQgdGhlIHByb2plbiBtYXJrZXIuIFRoZSBwcm9qZW4gbWFya2VyIGRvZXMgbm90ICpuZWVkKiB0byBiZVxuICAgKiBpbmNsdWRlZCBvbiBwcm9qZW4tZ2VuZXJhdGVkIGZpbGVzLCBidXQgaXQncyByZWNvbW1lbmRlZCBzaW5jZSBpdCBzaWduYWxzXG4gICAqIHRoYXQgaXQgcHJvYmFibHkgc2hvdWxkIG5vdCBiZSBlZGl0ZWQgZGlyZWN0bHkuXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRm9yUHJvamVuTWFya2VyKCkge1xuICAgIGNvbnN0IGZpbGVQYXRoID0gcGF0aC5qb2luKHRoaXMucHJvamVjdC5vdXRkaXIsIHRoaXMucGF0aCk7XG4gICAgY29uc3QgY29udGVudHMgPSB0cnlSZWFkRmlsZVN5bmMoZmlsZVBhdGgpO1xuICAgIGNvbnN0IGNvbnRhaW5zTWFya2VyID0gY29udGVudHM/LmluY2x1ZGVzKFBST0pFTl9NQVJLRVIpO1xuICAgIGlmICh0aGlzLm1hcmtlciAmJiAhY29udGFpbnNNYXJrZXIpIHtcbiAgICAgIHRoaXMucHJvamVjdC5sb2dnZXIuZGVidWcoXG4gICAgICAgIGBub3RlOiBleHBlY3RlZCAke3RoaXMucGF0aH0gdG8gY29udGFpbiBtYXJrZXIgYnV0IGZvdW5kIG5vbmUuYFxuICAgICAgKTtcbiAgICB9IGVsc2UgaWYgKCF0aGlzLm1hcmtlciAmJiBjb250YWluc01hcmtlcikge1xuICAgICAgdGhpcy5wcm9qZWN0LmxvZ2dlci5kZWJ1ZyhcbiAgICAgICAgYG5vdGU6IGV4cGVjdGVkICR7dGhpcy5wYXRofSB0byBub3QgY29udGFpbiBtYXJrZXIgYnV0IGZvdW5kIG9uZSBhbnl3YXkuYFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIHRoZSBmaWxlIGhhcyBiZWVuIGNoYW5nZWQgZHVyaW5nIHN5bnRoZXNpcy4gVGhpcyBwcm9wZXJ0eSBpc1xuICAgKiBvbmx5IGF2YWlsYWJsZSBpbiBgcG9zdFN5bnRoZXNpemUoKWAgaG9va3MuIElmIHRoaXMgaXMgYHVuZGVmaW5lZGAsIHRoZVxuICAgKiBmaWxlIGhhcyBub3QgYmVlbiBzeW50aGVzaXplZCB5ZXQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGNoYW5nZWQoKTogYm9vbGVhbiB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuX2NoYW5nZWQ7XG4gIH1cbn1cblxuLyoqXG4gKiBBUEkgZm9yIHJlc29sdmluZyB0b2tlbnMgd2hlbiBzeW50aGVzaXppbmcgZmlsZSBjb250ZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvbHZlciB7XG4gIC8qKlxuICAgKiBHaXZlbiBhIHZhbHVlIChvYmplY3Qvc3RyaW5nL2FycmF5L3doYXRldmVyLCBsb29rcyB1cCBhbnkgZnVuY3Rpb25zIGluc2lkZVxuICAgKiB0aGUgb2JqZWN0IGFuZCByZXR1cm5zIGFuIG9iamVjdCB3aGVyZSBhbGwgZnVuY3Rpb25zIGFyZSBjYWxsZWQuXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUgdG8gcmVzb2x2ZVxuICAgKiBAcGFja2FnZSBvcHRpb25zIFJlc29sdmUgb3B0aW9uc1xuICAgKi9cbiAgcmVzb2x2ZSh2YWx1ZTogYW55LCBvcHRpb25zPzogUmVzb2x2ZU9wdGlvbnMpOiBhbnk7XG59XG5cbi8qKlxuICogUmVzb2x2ZSBvcHRpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlc29sdmVPcHRpb25zIHtcbiAgLyoqXG4gICAqIE9taXRzIGVtcHR5IGFycmF5cyBhbmQgb2JqZWN0cy5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9taXRFbXB0eT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIENvbnRleHQgYXJndW1lbnRzLlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgYXJncz86IGFueVtdO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvbHZhYmxlIHtcbiAgLyoqXG4gICAqIFJlc29sdmVzIGFuZCByZXR1cm5zIGNvbnRlbnQuXG4gICAqL1xuICB0b0pTT04oKTogYW55O1xufVxuIl19