"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderDockerComposeFile = void 0;
const util_1 = require("../util");
function renderDockerComposeFile(serviceDescriptions, version) {
    // Record volume configuration
    const volumeConfig = {};
    const volumeInfo = {
        addVolumeConfiguration(volumeName, configuration) {
            if (!volumeConfig[volumeName]) {
                // First volume configuration takes precedence.
                volumeConfig[volumeName] = configuration;
            }
        },
    };
    // Record network configuration
    const networkConfig = {};
    const networkInfo = {
        addNetworkConfiguration(networkName, configuration) {
            if (!networkConfig[networkName]) {
                // First network configuration takes precedence.
                networkConfig[networkName] = configuration;
            }
        },
    };
    // Render service configuration
    const services = {};
    for (const [serviceName, serviceDescription] of Object.entries(serviceDescriptions ?? {})) {
        // Resolve the names of each dependency and check that they exist.
        // Note: They may not exist if the user made a mistake when referencing a
        // service by name via `DockerCompose.serviceName()`.
        // @see DockerCompose.serviceName
        const dependsOn = Array();
        for (const dependsOnServiceName of serviceDescription.dependsOn ?? []) {
            const resolvedServiceName = dependsOnServiceName.serviceName;
            if (resolvedServiceName === serviceName) {
                throw new Error(`Service ${serviceName} cannot depend on itself`);
            }
            if (!serviceDescriptions[resolvedServiceName]) {
                throw new Error(`Unable to resolve service named ${resolvedServiceName} for ${serviceName}`);
            }
            dependsOn.push(resolvedServiceName);
        }
        // Give each volume binding a chance to bind any necessary volume
        // configuration and provide volume mount information for the service.
        const volumes = [];
        for (const volumeBinding of serviceDescription.volumes ?? []) {
            volumes.push(volumeBinding.bind(volumeInfo));
        }
        // Give each network binding a chance to bind any necessary network
        // configuration and provide network mount information for the service.
        const networks = [];
        for (const networkBinding of serviceDescription.networks ?? []) {
            networks.push(networkBinding.bind(networkInfo));
        }
        // Create and store the service configuration, taking care not to create
        // object members with undefined values.
        services[serviceName] = {
            ...getObjectWithKeyAndValueIfValueIsDefined("image", serviceDescription.image),
            ...getObjectWithKeyAndValueIfValueIsDefined("build", serviceDescription.imageBuild),
            ...getObjectWithKeyAndValueIfValueIsDefined("entrypoint", serviceDescription.entrypoint),
            ...getObjectWithKeyAndValueIfValueIsDefined("command", serviceDescription.command),
            ...(Object.keys(serviceDescription.environment).length > 0
                ? { environment: serviceDescription.environment }
                : {}),
            ...(serviceDescription.ports.length > 0
                ? { ports: serviceDescription.ports }
                : {}),
            ...(Object.keys(serviceDescription.labels).length > 0
                ? { labels: serviceDescription.labels }
                : {}),
            ...(dependsOn.length > 0 ? { dependsOn } : {}),
            ...(volumes.length > 0 ? { volumes } : {}),
            ...(networks.length > 0 ? { networks } : {}),
        };
    }
    // Explicit with the type here because the decamelize step after this wipes
    // out types.
    const input = {
        version,
        services,
        ...(Object.keys(volumeConfig).length > 0 ? { volumes: volumeConfig } : {}),
        ...(Object.keys(networkConfig).length > 0
            ? { networks: networkConfig }
            : {}),
    };
    // Change most keys to snake case.
    return util_1.decamelizeKeysRecursively(input, {
        shouldDecamelize: shouldDecamelizeDockerComposeKey,
        separator: "_",
    });
}
exports.renderDockerComposeFile = renderDockerComposeFile;
/**
 * Returns `{ [key]: value }` if `value` is defined, otherwise returns `{}` so
 * that object spreading can be used to generate a peculiar interface.
 * @param key
 * @param value
 */
function getObjectWithKeyAndValueIfValueIsDefined(key, value) {
    return value !== undefined ? { [key]: value } : {};
}
/**
 * Determines whether the key at the given path should be decamelized.
 * Largely, all keys should be snake cased. But, there are some
 * exceptions for user-provided names for services, volumes, and
 * environment variables.
 *
 * @param path
 */
function shouldDecamelizeDockerComposeKey(path) {
    const poundPath = path.join("#");
    // Does not decamelize user's names.
    // services.namehere:
    // volumes.namehere:
    // networks.namehere:
    if (/^(services|volumes|networks)#[^#]+$/.test(poundPath)) {
        return false;
    }
    // Does not decamelize environment variables and labels
    // services.namehere.environment.*
    // services.namehere.labels.*
    if (/^services#[^#]+#(environment|labels)#/.test(poundPath)) {
        return false;
    }
    // Does not decamelize build arguments
    // services.namehere.build.args.*
    if (/^services#[^#]+#build#args#/.test(poundPath)) {
        return false;
    }
    // Otherwise, let it all decamelize.
    return true;
}
//# sourceMappingURL=data:application/json;base64,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