"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderInstallCommand = exports.installPackage = void 0;
const fs = require("fs");
const path = require("path");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Installs the npm module (through `npm install`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. `foo@^1.2` or `foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz`)
 * @returns The installed package name (e.g. `@foo/bar`)
 */
function installPackage(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, "package.json");
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    if (!packageJsonExisted) {
        // Make sure we have a package.json to read from later
        util_1.exec("npm init --yes", { cwd: baseDir });
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(renderInstallCommand(baseDir, spec), { cwd: baseDir });
    // Get the true installed package name
    const packageJson = JSON.parse(fs.readFileSync(packageJsonPath, "utf-8"));
    const packageName = Object.keys(packageJson.devDependencies).find((name) => name !== "projen");
    if (!packageName) {
        throw new Error(`Unable to resolve package name from spec ${spec}`);
    }
    // if package.json did not exist before calling `npm install`, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.rmSync(packageJsonPath, { force: true, recursive: true });
    }
    return packageName;
}
exports.installPackage = installPackage;
/**
 * Render a command to install an npm package.
 *
 * Engine checks are ignorred at this point so that the module can be installed
 * regardless of the environment. This was needed to unblock the upgrade of the
 * minimum node version of projen, but also okay generally because engine checks
 * will be performed later and for all eternety.
 *
 * @param dir Base directory
 * @param module The module to install (e.g. foo@^1.2)
 * @returns The string that includes the install command ("npm install ...")
 */
function renderInstallCommand(dir, module) {
    return `npm install --save-dev -f --no-package-lock --prefix="${dir}" ${module}`;
}
exports.renderInstallCommand = renderInstallCommand;
//# sourceMappingURL=data:application/json;base64,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