"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synth = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const common_1 = require("../common");
const logging = require("../logging");
const project_1 = require("../project");
const projenModule = path.dirname(require.resolve("../../package.json"));
async function synth(runtime, options) {
    const workdir = runtime.workdir;
    const rcfile = path.resolve(workdir, options.rcfile ?? common_1.PROJEN_RC); // TODO: support non javascript projenrc (e.g. java projects)
    // if --rc points to .projenrc.js, then behave as if it wasn't specified.
    if (rcfile === path.resolve(workdir, common_1.PROJEN_RC)) {
        delete options.rcfile;
    }
    // if there are no tasks, we assume this is not a projen project (modern
    // projects must at least have the "default" task).
    if (runtime.tasks.length === 0 && !fs.existsSync(rcfile)) {
        logging.error('Unable to find projen project. Use "projen new" to create a new project.');
        process.exit(1);
    }
    // run synth once
    const success = await trySynth();
    if (options.watch) {
        // if we are in watch mode, start the watch loop
        watchLoop();
    }
    else if (!success) {
        // make sure exit code is non-zero if we are not in watch mode
        process.exit(1);
    }
    async function trySynth() {
        // determine if post synthesis tasks should be executed (e.g. "yarn install").
        process.env.PROJEN_DISABLE_POST = (!options.post).toString();
        try {
            const defaultTask = runtime.tasks.find((t) => t.name === project_1.Project.DEFAULT_TASK);
            // if "--rc" is specified, ignore the default task
            if (defaultTask) {
                if (!options.rcfile) {
                    runtime.runTask(defaultTask.name);
                    return true;
                }
                else {
                    logging.warn("Default task skipped. Trying legacy synthesis since --rc is specified");
                }
            }
            // for backwards compatibility, if there is a .projenrc.js file, default to "node .projenrc.js"
            if (tryLegacySynth()) {
                return true;
            }
            throw new Error('Unable to find a task named "default"');
        }
        catch (e) {
            logging.error(`Synthesis failed: ${e.message}`);
            return false;
        }
    }
    function watchLoop() {
        logging.info(`Watching for changes in ${workdir}...`);
        const watch = fs.watch(workdir, { recursive: true });
        watch.on("change", (event) => {
            // we only care about "change" events
            if (event !== "change") {
                return;
            }
            process.stdout.write("\x1Bc"); // clear screen
            watch.close();
            trySynth()
                .then(() => watchLoop())
                .catch(() => watchLoop());
        });
    }
    function tryLegacySynth() {
        const rcdir = path.dirname(rcfile);
        if (!fs.existsSync(rcfile)) {
            return false;
        }
        // if node_modules/projen is not a directory or does not exist, create a
        // temporary symlink to the projen that we are currently running in order to
        // allow .projenrc.js to `require()` it.
        const nodeModules = path.resolve(rcdir, "node_modules");
        const projenModulePath = path.resolve(nodeModules, "projen");
        if (!fs.existsSync(path.join(projenModulePath, "package.json")) ||
            !fs.statSync(projenModulePath).isDirectory()) {
            fs.rmSync(projenModulePath, { force: true, recursive: true });
            fs.mkdirSync(nodeModules, { recursive: true });
            fs.symlinkSync(projenModule, projenModulePath, os.platform() === "win32" ? "junction" : null);
        }
        const ret = child_process_1.spawnSync(process.execPath, [rcfile], {
            stdio: ["inherit", "inherit", "pipe"],
        });
        if (ret.error) {
            throw new Error(`Synthesis failed: ${ret.error}`);
        }
        else if (ret.status !== 0) {
            logging.error(ret.stderr.toString());
            throw new Error(`Synthesis failed: calling "${process.execPath} ${rcfile}" exited with status=${ret.status}`);
        }
        return true;
    }
}
exports.synth = synth;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3ludGguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvY2xpL3N5bnRoLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLGlEQUEwQztBQUMxQyx5QkFBeUI7QUFDekIseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QixzQ0FBc0M7QUFDdEMsc0NBQXNDO0FBQ3RDLHdDQUFxQztBQUdyQyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDO0FBc0JsRSxLQUFLLFVBQVUsS0FBSyxDQUFDLE9BQW9CLEVBQUUsT0FBcUI7SUFDckUsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQztJQUNoQyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsTUFBTSxJQUFJLGtCQUFTLENBQUMsQ0FBQyxDQUFDLDZEQUE2RDtJQUVoSSx5RUFBeUU7SUFDekUsSUFBSSxNQUFNLEtBQUssSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsa0JBQVMsQ0FBQyxFQUFFO1FBQy9DLE9BQVEsT0FBZSxDQUFDLE1BQU0sQ0FBQztLQUNoQztJQUVELHdFQUF3RTtJQUN4RSxtREFBbUQ7SUFDbkQsSUFBSSxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1FBQ3hELE9BQU8sQ0FBQyxLQUFLLENBQ1gsMEVBQTBFLENBQzNFLENBQUM7UUFDRixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO0tBQ2pCO0lBRUQsaUJBQWlCO0lBQ2pCLE1BQU0sT0FBTyxHQUFHLE1BQU0sUUFBUSxFQUFFLENBQUM7SUFFakMsSUFBSSxPQUFPLENBQUMsS0FBSyxFQUFFO1FBQ2pCLGdEQUFnRDtRQUNoRCxTQUFTLEVBQUUsQ0FBQztLQUNiO1NBQU0sSUFBSSxDQUFDLE9BQU8sRUFBRTtRQUNuQiw4REFBOEQ7UUFDOUQsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztLQUNqQjtJQUVELEtBQUssVUFBVSxRQUFRO1FBQ3JCLDhFQUE4RTtRQUM5RSxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQixHQUFHLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDN0QsSUFBSTtZQUNGLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUNwQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxpQkFBTyxDQUFDLFlBQVksQ0FDdkMsQ0FBQztZQUVGLGtEQUFrRDtZQUNsRCxJQUFJLFdBQVcsRUFBRTtnQkFDZixJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRTtvQkFDbkIsT0FBTyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7b0JBQ2xDLE9BQU8sSUFBSSxDQUFDO2lCQUNiO3FCQUFNO29CQUNMLE9BQU8sQ0FBQyxJQUFJLENBQ1YsdUVBQXVFLENBQ3hFLENBQUM7aUJBQ0g7YUFDRjtZQUVELCtGQUErRjtZQUMvRixJQUFJLGNBQWMsRUFBRSxFQUFFO2dCQUNwQixPQUFPLElBQUksQ0FBQzthQUNiO1lBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1NBQzFEO1FBQUMsT0FBTyxDQUFDLEVBQUU7WUFDVixPQUFPLENBQUMsS0FBSyxDQUFDLHFCQUFzQixDQUFTLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztZQUN6RCxPQUFPLEtBQUssQ0FBQztTQUNkO0lBQ0gsQ0FBQztJQUVELFNBQVMsU0FBUztRQUNoQixPQUFPLENBQUMsSUFBSSxDQUFDLDJCQUEyQixPQUFPLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE1BQU0sS0FBSyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDckQsS0FBSyxDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRTtZQUMzQixxQ0FBcUM7WUFDckMsSUFBSSxLQUFLLEtBQUssUUFBUSxFQUFFO2dCQUN0QixPQUFPO2FBQ1I7WUFFRCxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLGVBQWU7WUFDOUMsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQ2QsUUFBUSxFQUFFO2lCQUNQLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxTQUFTLEVBQUUsQ0FBQztpQkFDdkIsS0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDOUIsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsU0FBUyxjQUFjO1FBQ3JCLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFbkMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDMUIsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUVELHdFQUF3RTtRQUN4RSw0RUFBNEU7UUFDNUUsd0NBQXdDO1FBQ3hDLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3hELE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDN0QsSUFDRSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsQ0FBQztZQUMzRCxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFDNUM7WUFDQSxFQUFFLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUM5RCxFQUFFLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQy9DLEVBQUUsQ0FBQyxXQUFXLENBQ1osWUFBWSxFQUNaLGdCQUFnQixFQUNoQixFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FDOUMsQ0FBQztTQUNIO1FBRUQsTUFBTSxHQUFHLEdBQUcseUJBQVMsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDaEQsS0FBSyxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxNQUFNLENBQUM7U0FDdEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFO1lBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUM7U0FDbkQ7YUFBTSxJQUFJLEdBQUcsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNCLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQ2IsOEJBQThCLE9BQU8sQ0FBQyxRQUFRLElBQUksTUFBTSx3QkFBd0IsR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUM3RixDQUFDO1NBQ0g7UUFFRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7QUFDSCxDQUFDO0FBckhELHNCQXFIQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHNwYXduU3luYyB9IGZyb20gXCJjaGlsZF9wcm9jZXNzXCI7XG5pbXBvcnQgKiBhcyBmcyBmcm9tIFwiZnNcIjtcbmltcG9ydCAqIGFzIG9zIGZyb20gXCJvc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgUFJPSkVOX1JDIH0gZnJvbSBcIi4uL2NvbW1vblwiO1xuaW1wb3J0ICogYXMgbG9nZ2luZyBmcm9tIFwiLi4vbG9nZ2luZ1wiO1xuaW1wb3J0IHsgUHJvamVjdCB9IGZyb20gXCIuLi9wcm9qZWN0XCI7XG5pbXBvcnQgeyBUYXNrUnVudGltZSB9IGZyb20gXCIuLi90YXNrLXJ1bnRpbWVcIjtcblxuY29uc3QgcHJvamVuTW9kdWxlID0gcGF0aC5kaXJuYW1lKHJlcXVpcmUucmVzb2x2ZShcIi4uLy4uL3BhY2thZ2UuanNvblwiKSk7XG5cbmV4cG9ydCBpbnRlcmZhY2UgU3ludGhPcHRpb25zIHtcbiAgLyoqXG4gICAqIEV4ZWN1dGUgcG9zdCBzeW50aGVzaXMgY29tbWFuZHMuXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHBvc3Q/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTdGFydCB3YXRjaGluZyAucHJvamVucmMuanMgYW5kIHJlLXN5bnRoIHdoZW4gY2hhbmdlZC5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHdhdGNoPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIC5wcm9qZW5yYy5qcyBmaWxlICB0byB1c2UgaW5zdGVhZCBvZiB0aGUgZGVmYXVsdC5cbiAgICogQGRlZmF1bHQgXCIucHJvamVucmMuanNcIlxuICAgKi9cbiAgcmVhZG9ubHkgcmNmaWxlPzogc3RyaW5nO1xufVxuXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gc3ludGgocnVudGltZTogVGFza1J1bnRpbWUsIG9wdGlvbnM6IFN5bnRoT3B0aW9ucykge1xuICBjb25zdCB3b3JrZGlyID0gcnVudGltZS53b3JrZGlyO1xuICBjb25zdCByY2ZpbGUgPSBwYXRoLnJlc29sdmUod29ya2Rpciwgb3B0aW9ucy5yY2ZpbGUgPz8gUFJPSkVOX1JDKTsgLy8gVE9ETzogc3VwcG9ydCBub24gamF2YXNjcmlwdCBwcm9qZW5yYyAoZS5nLiBqYXZhIHByb2plY3RzKVxuXG4gIC8vIGlmIC0tcmMgcG9pbnRzIHRvIC5wcm9qZW5yYy5qcywgdGhlbiBiZWhhdmUgYXMgaWYgaXQgd2Fzbid0IHNwZWNpZmllZC5cbiAgaWYgKHJjZmlsZSA9PT0gcGF0aC5yZXNvbHZlKHdvcmtkaXIsIFBST0pFTl9SQykpIHtcbiAgICBkZWxldGUgKG9wdGlvbnMgYXMgYW55KS5yY2ZpbGU7XG4gIH1cblxuICAvLyBpZiB0aGVyZSBhcmUgbm8gdGFza3MsIHdlIGFzc3VtZSB0aGlzIGlzIG5vdCBhIHByb2plbiBwcm9qZWN0IChtb2Rlcm5cbiAgLy8gcHJvamVjdHMgbXVzdCBhdCBsZWFzdCBoYXZlIHRoZSBcImRlZmF1bHRcIiB0YXNrKS5cbiAgaWYgKHJ1bnRpbWUudGFza3MubGVuZ3RoID09PSAwICYmICFmcy5leGlzdHNTeW5jKHJjZmlsZSkpIHtcbiAgICBsb2dnaW5nLmVycm9yKFxuICAgICAgJ1VuYWJsZSB0byBmaW5kIHByb2plbiBwcm9qZWN0LiBVc2UgXCJwcm9qZW4gbmV3XCIgdG8gY3JlYXRlIGEgbmV3IHByb2plY3QuJ1xuICAgICk7XG4gICAgcHJvY2Vzcy5leGl0KDEpO1xuICB9XG5cbiAgLy8gcnVuIHN5bnRoIG9uY2VcbiAgY29uc3Qgc3VjY2VzcyA9IGF3YWl0IHRyeVN5bnRoKCk7XG5cbiAgaWYgKG9wdGlvbnMud2F0Y2gpIHtcbiAgICAvLyBpZiB3ZSBhcmUgaW4gd2F0Y2ggbW9kZSwgc3RhcnQgdGhlIHdhdGNoIGxvb3BcbiAgICB3YXRjaExvb3AoKTtcbiAgfSBlbHNlIGlmICghc3VjY2Vzcykge1xuICAgIC8vIG1ha2Ugc3VyZSBleGl0IGNvZGUgaXMgbm9uLXplcm8gaWYgd2UgYXJlIG5vdCBpbiB3YXRjaCBtb2RlXG4gICAgcHJvY2Vzcy5leGl0KDEpO1xuICB9XG5cbiAgYXN5bmMgZnVuY3Rpb24gdHJ5U3ludGgoKSB7XG4gICAgLy8gZGV0ZXJtaW5lIGlmIHBvc3Qgc3ludGhlc2lzIHRhc2tzIHNob3VsZCBiZSBleGVjdXRlZCAoZS5nLiBcInlhcm4gaW5zdGFsbFwiKS5cbiAgICBwcm9jZXNzLmVudi5QUk9KRU5fRElTQUJMRV9QT1NUID0gKCFvcHRpb25zLnBvc3QpLnRvU3RyaW5nKCk7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IGRlZmF1bHRUYXNrID0gcnVudGltZS50YXNrcy5maW5kKFxuICAgICAgICAodCkgPT4gdC5uYW1lID09PSBQcm9qZWN0LkRFRkFVTFRfVEFTS1xuICAgICAgKTtcblxuICAgICAgLy8gaWYgXCItLXJjXCIgaXMgc3BlY2lmaWVkLCBpZ25vcmUgdGhlIGRlZmF1bHQgdGFza1xuICAgICAgaWYgKGRlZmF1bHRUYXNrKSB7XG4gICAgICAgIGlmICghb3B0aW9ucy5yY2ZpbGUpIHtcbiAgICAgICAgICBydW50aW1lLnJ1blRhc2soZGVmYXVsdFRhc2submFtZSk7XG4gICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgbG9nZ2luZy53YXJuKFxuICAgICAgICAgICAgXCJEZWZhdWx0IHRhc2sgc2tpcHBlZC4gVHJ5aW5nIGxlZ2FjeSBzeW50aGVzaXMgc2luY2UgLS1yYyBpcyBzcGVjaWZpZWRcIlxuICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgLy8gZm9yIGJhY2t3YXJkcyBjb21wYXRpYmlsaXR5LCBpZiB0aGVyZSBpcyBhIC5wcm9qZW5yYy5qcyBmaWxlLCBkZWZhdWx0IHRvIFwibm9kZSAucHJvamVucmMuanNcIlxuICAgICAgaWYgKHRyeUxlZ2FjeVN5bnRoKCkpIHtcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9XG5cbiAgICAgIHRocm93IG5ldyBFcnJvcignVW5hYmxlIHRvIGZpbmQgYSB0YXNrIG5hbWVkIFwiZGVmYXVsdFwiJyk7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgbG9nZ2luZy5lcnJvcihgU3ludGhlc2lzIGZhaWxlZDogJHsoZSBhcyBhbnkpLm1lc3NhZ2V9YCk7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICB9XG5cbiAgZnVuY3Rpb24gd2F0Y2hMb29wKCkge1xuICAgIGxvZ2dpbmcuaW5mbyhgV2F0Y2hpbmcgZm9yIGNoYW5nZXMgaW4gJHt3b3JrZGlyfS4uLmApO1xuICAgIGNvbnN0IHdhdGNoID0gZnMud2F0Y2god29ya2RpciwgeyByZWN1cnNpdmU6IHRydWUgfSk7XG4gICAgd2F0Y2gub24oXCJjaGFuZ2VcIiwgKGV2ZW50KSA9PiB7XG4gICAgICAvLyB3ZSBvbmx5IGNhcmUgYWJvdXQgXCJjaGFuZ2VcIiBldmVudHNcbiAgICAgIGlmIChldmVudCAhPT0gXCJjaGFuZ2VcIikge1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIHByb2Nlc3Muc3Rkb3V0LndyaXRlKFwiXFx4MUJjXCIpOyAvLyBjbGVhciBzY3JlZW5cbiAgICAgIHdhdGNoLmNsb3NlKCk7XG4gICAgICB0cnlTeW50aCgpXG4gICAgICAgIC50aGVuKCgpID0+IHdhdGNoTG9vcCgpKVxuICAgICAgICAuY2F0Y2goKCkgPT4gd2F0Y2hMb29wKCkpO1xuICAgIH0pO1xuICB9XG5cbiAgZnVuY3Rpb24gdHJ5TGVnYWN5U3ludGgoKSB7XG4gICAgY29uc3QgcmNkaXIgPSBwYXRoLmRpcm5hbWUocmNmaWxlKTtcblxuICAgIGlmICghZnMuZXhpc3RzU3luYyhyY2ZpbGUpKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgLy8gaWYgbm9kZV9tb2R1bGVzL3Byb2plbiBpcyBub3QgYSBkaXJlY3Rvcnkgb3IgZG9lcyBub3QgZXhpc3QsIGNyZWF0ZSBhXG4gICAgLy8gdGVtcG9yYXJ5IHN5bWxpbmsgdG8gdGhlIHByb2plbiB0aGF0IHdlIGFyZSBjdXJyZW50bHkgcnVubmluZyBpbiBvcmRlciB0b1xuICAgIC8vIGFsbG93IC5wcm9qZW5yYy5qcyB0byBgcmVxdWlyZSgpYCBpdC5cbiAgICBjb25zdCBub2RlTW9kdWxlcyA9IHBhdGgucmVzb2x2ZShyY2RpciwgXCJub2RlX21vZHVsZXNcIik7XG4gICAgY29uc3QgcHJvamVuTW9kdWxlUGF0aCA9IHBhdGgucmVzb2x2ZShub2RlTW9kdWxlcywgXCJwcm9qZW5cIik7XG4gICAgaWYgKFxuICAgICAgIWZzLmV4aXN0c1N5bmMocGF0aC5qb2luKHByb2plbk1vZHVsZVBhdGgsIFwicGFja2FnZS5qc29uXCIpKSB8fFxuICAgICAgIWZzLnN0YXRTeW5jKHByb2plbk1vZHVsZVBhdGgpLmlzRGlyZWN0b3J5KClcbiAgICApIHtcbiAgICAgIGZzLnJtU3luYyhwcm9qZW5Nb2R1bGVQYXRoLCB7IGZvcmNlOiB0cnVlLCByZWN1cnNpdmU6IHRydWUgfSk7XG4gICAgICBmcy5ta2RpclN5bmMobm9kZU1vZHVsZXMsIHsgcmVjdXJzaXZlOiB0cnVlIH0pO1xuICAgICAgZnMuc3ltbGlua1N5bmMoXG4gICAgICAgIHByb2plbk1vZHVsZSxcbiAgICAgICAgcHJvamVuTW9kdWxlUGF0aCxcbiAgICAgICAgb3MucGxhdGZvcm0oKSA9PT0gXCJ3aW4zMlwiID8gXCJqdW5jdGlvblwiIDogbnVsbFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXQgPSBzcGF3blN5bmMocHJvY2Vzcy5leGVjUGF0aCwgW3JjZmlsZV0sIHtcbiAgICAgIHN0ZGlvOiBbXCJpbmhlcml0XCIsIFwiaW5oZXJpdFwiLCBcInBpcGVcIl0sXG4gICAgfSk7XG4gICAgaWYgKHJldC5lcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBTeW50aGVzaXMgZmFpbGVkOiAke3JldC5lcnJvcn1gKTtcbiAgICB9IGVsc2UgaWYgKHJldC5zdGF0dXMgIT09IDApIHtcbiAgICAgIGxvZ2dpbmcuZXJyb3IocmV0LnN0ZGVyci50b1N0cmluZygpKTtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYFN5bnRoZXNpcyBmYWlsZWQ6IGNhbGxpbmcgXCIke3Byb2Nlc3MuZXhlY1BhdGh9ICR7cmNmaWxlfVwiIGV4aXRlZCB3aXRoIHN0YXR1cz0ke3JldC5zdGF0dXN9YFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxufVxuIl19