"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const path_1 = require("path");
const yargs = require("yargs");
const new_1 = require("./cmds/new");
const synth_1 = require("./synth");
const tasks_1 = require("./tasks");
const common_1 = require("../common");
const logging = require("../logging");
const task_runtime_1 = require("../task-runtime");
const util_1 = require("../util");
const DEFAULT_RC = path_1.resolve(common_1.PROJEN_RC);
async function main() {
    const ya = yargs;
    ya.command(new_1.default);
    const pathToProjenDir = findProjenDir(process.cwd());
    const runtime = new task_runtime_1.TaskRuntime(pathToProjenDir ?? ".");
    tasks_1.discoverTaskCommands(runtime, ya);
    ya.recommendCommands();
    ya.strictCommands();
    ya.showHelpOnFail(false);
    ya.wrap(yargs.terminalWidth());
    ya.option("post", {
        type: "boolean",
        default: true,
        desc: "Run post-synthesis steps such as installing dependencies. Use --no-post to skip",
    });
    ya.option("watch", {
        type: "boolean",
        default: false,
        desc: "Keep running and resynthesize when projenrc changes",
        alias: "w",
    });
    ya.options("debug", { type: "boolean", default: false, desc: "Debug logs" });
    ya.options("rc", {
        deprecated: true,
        desc: "path to .projenrc.js file",
        default: DEFAULT_RC,
        type: "string",
    });
    ya.completion();
    ya.help();
    // do not use the default yargs '--version' implementation since it is
    // global by default (it appears on all subcommands)
    ya.version(false);
    ya.option("version", {
        type: "boolean",
        description: "Show version number",
        global: false,
    });
    const args = ya.argv;
    if (args.debug) {
        process.env.DEBUG = "true";
    }
    const nodeVersion = util_1.getNodeMajorVersion();
    if (nodeVersion && nodeVersion < 16) {
        logging.warn(`WARNING: You are using Node v${nodeVersion}, which reaches end of life on April 30, 2023. Support for EOL Node releases may be dropped by projen in the future. Please consider upgrading to Node >= 16 as soon as possible.`);
    }
    // no command means synthesize
    if (args._.length === 0) {
        if (args.version) {
            console.log(common_1.PROJEN_VERSION);
            process.exit(0);
        }
        await synth_1.synth(runtime, {
            post: args.post,
            watch: args.watch,
            rcfile: args.rc,
        });
    }
}
main().catch((e) => {
    console.error(e.stack);
    process.exit(1);
});
/**
 * Run up project tree to find .projen directory
 *
 * @param cwd current working directory
 * @returns path to .projen directory or undefined if not found
 */
function findProjenDir(cwd) {
    if (fs_1.existsSync(path_1.resolve(cwd, common_1.PROJEN_DIR))) {
        return cwd;
    }
    if (cwd === "/") {
        return undefined;
    }
    return findProjenDir(path_1.resolve(cwd, ".."));
}
//# sourceMappingURL=data:application/json;base64,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