"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanup = exports.FILE_MANIFEST = void 0;
const fs = require("fs");
const path = require("path");
const glob = require("glob");
const common_1 = require("./common");
const logging = require("./logging");
exports.FILE_MANIFEST = `${common_1.PROJEN_DIR}/files.json`;
function cleanup(dir, newFiles, exclude) {
    try {
        const manifestFiles = getFilesFromManifest(dir);
        if (manifestFiles.length > 0) {
            // Use `FILE_MANIFEST` to remove files that are no longer managed by projen
            removeFiles(findOrphanedFiles(dir, manifestFiles, newFiles));
        }
        else {
            // Remove all files managed by projen with legacy logic
            removeFiles(findGeneratedFiles(dir, exclude));
        }
    }
    catch (e) {
        logging.warn(`warning: failed to clean up generated files: ${e.stack}`);
    }
}
exports.cleanup = cleanup;
function removeFiles(files) {
    for (const file of files) {
        fs.rmSync(file, { force: true, recursive: true });
    }
}
function findGeneratedFiles(dir, exclude) {
    const ignore = [
        ...readGitIgnore(dir),
        "node_modules/**",
        ...exclude,
        ".git/**",
    ];
    const files = glob.sync("**", {
        ignore,
        cwd: dir,
        dot: true,
        nodir: true,
        absolute: true,
    });
    const generated = new Array();
    for (const file of files) {
        const contents = fs.readFileSync(file, "utf-8");
        if (contents.includes(common_1.PROJEN_MARKER)) {
            generated.push(file);
        }
    }
    return generated;
}
function findOrphanedFiles(dir, oldFiles, newFiles) {
    return oldFiles
        .filter((old) => !newFiles.includes(old))
        .map((f) => path.resolve(dir, f));
}
function getFilesFromManifest(dir) {
    try {
        const fileManifestPath = path.resolve(dir, exports.FILE_MANIFEST);
        if (fs.existsSync(fileManifestPath)) {
            const fileManifest = JSON.parse(fs.readFileSync(fileManifestPath, "utf-8"));
            if (fileManifest.files) {
                return fileManifest.files;
            }
        }
    }
    catch (e) {
        logging.warn(`warning: unable to get files to clean from file manifest: ${e.stack}`);
    }
    return [];
}
function readGitIgnore(dir) {
    const filepath = path.join(dir, ".gitignore");
    if (!fs.existsSync(filepath)) {
        return [];
    }
    return fs
        .readFileSync(filepath, "utf-8")
        .split("\n")
        .filter((x) => x?.trim() !== "")
        .filter((x) => !x.startsWith("#") && !x.startsWith("!"))
        .map((x) => x.replace(/^\//, "")) // remove "/" prefix
        .map((x) => `${x}\n${x}/**`)
        .join("\n")
        .split("\n");
}
//# sourceMappingURL=data:application/json;base64,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