"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver_1 = require("semver");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const javascript_1 = require("../javascript");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability = exports.Stability || (exports.Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        // True if jsii version 1.x is compatible with the requested version range.
        const usesLegacyJsii = options.jsiiVersion == null ||
            (options.jsiiVersion !== "*" &&
                new semver_1.Range(options.jsiiVersion).intersects(new semver_1.Range("1.x")));
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
            jestOptions: usesLegacyJsii ? { jestVersion: "^27" } : undefined,
        };
        const forcedOptions = {
            releaseToNpm: false,
            disableTsconfig: true,
            docgen: false,
        };
        const mergedOptions = util_1.deepMerge([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        const compressAssembly = options.compressAssembly ?? false;
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"];
        if (compressAssembly) {
            jsiiFlags.push("--compress-assembly");
        }
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(["jsii", ...jsiiFlags].join(" "));
        this.watchTask.reset(["jsii", "-w", ...jsiiFlags].join(" "));
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        const prepareRepoForCI = [
            `rsync -a . .repo --exclude .git --exclude node_modules`,
            `rm -rf ${this.artifactsDirectory}`,
            `mv .repo ${this.artifactsDirectory}`,
        ].join(" && ");
        // when running inside CI we just prepare the repo for packaging, which
        // takes place in separate tasks.
        // outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset(`if [ ! -z \${CI} ]; then ${prepareRepoForCI}; else ${this.runTaskCommand(this.packageAllTask)}; fi`);
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        this.release?.publisher.addGitHubPrePublishingSteps({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Collect GitHub Metadata",
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        const extraJobOptions = options.workflowRunsOn
            ? { runsOn: options.workflowRunsOn }
            : {};
        if (options.releaseToNpm != false) {
            const task = this.addPackagingTask("js");
            this.release?.publisher.publishToNpm({
                ...this.pacmakForLanguage("js", task),
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                codeArtifactOptions: options.codeArtifactOptions,
            });
            this.addPackagingTarget("js", task, extraJobOptions);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        if (options.publishToMaven) {
            targets.java = {
                package: options.publishToMaven.javaPackage,
                maven: {
                    groupId: options.publishToMaven.mavenGroupId,
                    artifactId: options.publishToMaven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.release?.publisher.publishToMaven({
                ...this.pacmakForLanguage("java", task),
                ...options.publishToMaven,
            });
            this.addPackagingTarget("java", task, extraJobOptions);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.release?.publisher.publishToPyPi({
                ...this.pacmakForLanguage("python", task),
                ...pypi,
            });
            this.addPackagingTarget("python", task, extraJobOptions);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
                iconUrl: nuget.iconUrl,
            };
            const task = this.addPackagingTask("dotnet");
            this.release?.publisher.publishToNuget({
                ...this.pacmakForLanguage("dotnet", task),
                ...nuget,
            });
            this.addPackagingTarget("dotnet", task, extraJobOptions);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
                packageName: golang.packageName,
            };
            const task = this.addPackagingTask("go");
            this.release?.publisher.publishToGo({
                ...this.pacmakForLanguage("go", task),
                ...golang,
            });
            this.addPackagingTarget("go", task, extraJobOptions);
        }
        const jsiiSuffix = options.jsiiVersion === "*"
            ? // If jsiiVersion is "*", don't specify anything so the user can manage.
                ""
            : // Otherwise, use `jsiiVersion` or fall back to `1.x`.
                `@${options.jsiiVersion ?? "1.x"}`;
        this.addDevDeps(`jsii${jsiiSuffix}`, "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            new jsii_docgen_1.JsiiDocgen(this, { filePath: options.docgenFilePath });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
        // When using jsii@1,x, we need to add some resolutions to avoid including
        // TypeScript-3.9-incompatble dependencies that break the compiler.
        if (usesLegacyJsii) {
            // https://github.com/projen/projen/issues/2165
            this.package.addPackageResolutions("@types/prettier@2.6.0");
            // https://github.com/projen/projen/issues/2264
            this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        }
    }
    /**
     * Adds a target language to the build workflow and creates a package task.
     * @param language
     * @returns
     */
    addPackagingTarget(language, packTask, extraJobOptions) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            runsOn: ["ubuntu-latest"],
            permissions: {},
            tools: {
                node: { version: this.nodeVersion ?? "16.x" },
                ...pacmak.publishTools,
            },
            steps: pacmak.prePublishSteps ?? [],
            ...extraJobOptions,
        });
    }
    addPackagingTask(language) {
        const packageTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        packageTask.exec(`jsii-pacmak -v --target ${language}`);
        this.packageAllTask.spawn(packageTask);
        return packageTask;
    }
    pacmakForLanguage(target, packTask) {
        // at this stage, `artifactsDirectory` contains the prebuilt repository.
        // for the publishing to work seamlessely, that directory needs to contain the actual artifact.
        // so we move the repo, create the artifact, and put it in the expected place.
        const prePublishSteps = [];
        prePublishSteps.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            prePublishSteps.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.2.4",
                with: { version: this.package.pnpmVersion },
            });
        }
        prePublishSteps.push({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Install Dependencies",
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
        }, {
            name: `Create ${target} artifact`,
            run: `cd ${REPO_TEMP_DIRECTORY} && npx projen ${packTask.name}`,
        }, {
            name: `Collect ${target} Artifact`,
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            prePublishSteps,
        };
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.69.3" };
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,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