"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const PULL_REQUEST_REF = "${{ github.event.pull_request.head.ref }}";
const PULL_REQUEST_REPOSITORY = "${{ github.event.pull_request.head.repo.full_name }}";
const BUILD_JOBID = "build";
const SELF_MUTATION_STEP = "self_mutation";
const SELF_MUTATION_HAPPENED_OUTPUT = "self_mutation_happened";
const IS_FORK = "github.event.pull_request.head.repo.full_name != github.repository";
const NOT_FORK = `!(${IS_FORK})`;
const SELF_MUTATION_CONDITION = `needs.${BUILD_JOBID}.outputs.${SELF_MUTATION_HAPPENED_OUTPUT}`;
class BuildWorkflow extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.defaultRunners = ["ubuntu-latest"];
        this._postBuildJobs = [];
        const github = github_1.GitHub.of(project);
        if (!github) {
            throw new Error("BuildWorkflow is currently only supported for GitHub projects");
        }
        this.github = github;
        this.preBuildSteps = options.preBuildSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory = options.artifactsDirectory;
        const mutableBuilds = options.mutableBuild ?? true;
        this.workflow = new github_1.GithubWorkflow(github, "build");
        this.workflow.on(options.workflowTriggers ?? {
            pullRequest: {},
            workflowDispatch: {},
        });
        this.addBuildJob(options);
        if (mutableBuilds) {
            this.addSelfMutationJob(options);
        }
        if (project instanceof javascript_1.NodeProject) {
            project.addPackageIgnore(constants_1.PERMISSION_BACKUP_FILE);
        }
    }
    addBuildJob(options) {
        this.workflow.addJob(BUILD_JOBID, {
            runsOn: options.runsOn ?? this.defaultRunners,
            container: options.containerImage
                ? { image: options.containerImage }
                : undefined,
            env: {
                CI: "true",
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
                ...options.permissions,
            },
            steps: (() => this.renderBuildSteps()),
            outputs: {
                [SELF_MUTATION_HAPPENED_OUTPUT]: {
                    stepId: SELF_MUTATION_STEP,
                    outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                },
            },
        });
    }
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds() {
        return [BUILD_JOBID, ...this._postBuildJobs];
    }
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id, job) {
        const steps = [];
        if (this.artifactsDirectory) {
            steps.push({
                name: "Download build artifacts",
                uses: "actions/download-artifact@v3",
                with: {
                    name: constants_1.BUILD_ARTIFACT_NAME,
                    path: this.artifactsDirectory,
                },
            }, {
                name: "Restore build artifact permissions",
                continueOnError: true,
                run: [
                    `cd ${this.artifactsDirectory} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                ].join("\n"),
            });
        }
        steps.push(...(job.steps ?? []));
        this.workflow.addJob(id, {
            needs: [BUILD_JOBID],
            // only run if build did not self-mutate
            if: `! ${SELF_MUTATION_CONDITION}`,
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._postBuildJobs.push(id);
    }
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task, options = {}) {
        this.addPostBuildJobCommands(`post-build-${task.name}`, [`${this.project.projenCommand} ${task.name}`], {
            checkoutRepo: true,
            installDeps: true,
            tools: options.tools,
            runsOn: options.runsOn,
        });
    }
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id, commands, options) {
        const steps = [];
        if (options?.checkoutRepo) {
            steps.push({
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                    ...(this.github.downloadLfs ? { lfs: true } : {}),
                },
            });
        }
        if (options?.checkoutRepo &&
            options?.installDeps &&
            this.project instanceof javascript_1.NodeProject) {
            steps.push({
                name: "Install dependencies",
                run: `${this.project.package.installCommand}`,
            });
        }
        steps.push({ run: commands.join("\n") });
        this.addPostBuildJob(id, {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: options?.tools,
            runsOn: options?.runsOn ?? this.defaultRunners,
            steps,
        });
    }
    addSelfMutationJob(options) {
        this.workflow.addJob("self-mutation", {
            runsOn: options.runsOn ?? this.defaultRunners,
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            needs: [BUILD_JOBID],
            if: `always() && ${SELF_MUTATION_CONDITION} && ${NOT_FORK}`,
            steps: [
                ...this.workflow.projenCredentials.setupSteps,
                ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                    // we need to use a PAT so that our push will trigger the build workflow
                    token: this.workflow.projenCredentials.tokenRef,
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                    lfs: this.github.downloadLfs,
                }),
                ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
                {
                    name: "Push changes",
                    run: [
                        "  git add .",
                        '  git commit -s -m "chore: self mutation"',
                        `  git push origin HEAD:${PULL_REQUEST_REF}`,
                    ].join("\n"),
                },
            ],
        });
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderBuildSteps() {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                    ...(this.github.downloadLfs ? { lfs: true } : {}),
                },
            },
            ...this.preBuildSteps,
            {
                name: this.buildTask.name,
                run: this.github.project.runTaskCommand(this.buildTask),
            },
            ...this.postBuildSteps,
            // check for mutations and upload a git patch file as an artifact
            ...workflow_actions_1.WorkflowActions.createUploadGitPatch({
                stepId: SELF_MUTATION_STEP,
                outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                mutationError: "Files were changed during build (see build log). If this was triggered from a fork, you will need to update your branch.",
            }),
            // upload the build artifact only if we have post-build jobs (otherwise, there's no point)
            ...(this._postBuildJobs.length == 0
                ? []
                : [
                    {
                        name: "Backup artifact permissions",
                        continueOnError: true,
                        run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
                    },
                    {
                        name: "Upload artifact",
                        uses: "actions/upload-artifact@v3",
                        with: {
                            name: constants_1.BUILD_ARTIFACT_NAME,
                            path: this.artifactsDirectory,
                        },
                    },
                ]),
        ];
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.69.3" };
//# sourceMappingURL=data:application/json;base64,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