"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const internal_1 = require("./internal");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_16_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path.posix.join(path.dirname(entrypoint), path.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? case_1.pascal(path.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: runtime.defaultExternals,
            ...options.bundlingOptions,
            tsconfigPath: project?.tsconfigDev?.fileName,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path.join(project.outdir, bundle.outfile);
        const constructAbs = path.join(project.outdir, constructFile);
        const relativeOutfile = path.relative(path.dirname(constructAbs), path.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${internal_1.convertToPosixPath(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${internal_1.convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${internal_1.convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.69.3" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget, 
    /**
     * Options for this runtime.
     */
    options) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
        this.defaultExternals = options?.defaultExternals ?? ["@aws-sdk/*"];
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.69.3" };
/**
 * Node.js 10.x
 * @deprecated NodeJS10 has been deprecated
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 12.x
 * @deprecated NodeJS12 has been deprecated
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 14.x
 * @deprecated NodeJS14 has been deprecated
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 16.x
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 18.x
 */
LambdaRuntime.NODEJS_18_X = new LambdaRuntime("nodejs18.x", "node18");
//# sourceMappingURL=data:application/json;base64,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