var VueUI = (function (exports) {
  'use strict';

  class PywbI18N {
    static #locale = ''; // private (can only be set here)
    static getLocale() { // get via public static method
      return PywbI18N.#locale;
    }
    static firstDayOfWeek = 1;
    static init = (locale, config) => {
      if (PywbI18N.instance) {
        throw new Error('cannot instantiate PywbI18N twice');
      }
      PywbI18N.#locale = locale;
      PywbI18N.instance = new PywbI18N(config);
      let intlLocale = new Intl.Locale(PywbI18N.getLocale());
      if ('weekInfo' in intlLocale) PywbI18N.firstDayOfWeek = intlLocale.weekInfo.firstDay % 7;
    }

    // PywbI18N expects from the i18n string source to receive months SHORT and LONG names in the config like this:
    // config.jan_short, config.jan_long, ...., config.<mmm>_short, config.<mmm>_long
    static monthIdPrefix = {1:"jan", 2:"feb",3:"mar",4:"apr",5:"may",6:"jun",7:"jul",8:"aug",9:"sep",10:"oct",11:"nov",12:"dec"};

    /**
     *
     * @type {PywbI18N|null}
     */
    static instance = null;

    constructor(config) {
      this.config = {...config}; // make a copy of config
    }

    // can get long (default) or short month string
    getMonth(id, type='long') {
      return decodeURIComponent(this.config[PywbI18N.monthIdPrefix[id]+'_'+type]);
    }
    // can get long (default) or short day string or intial
    // PywbI18N expects to receive day's initials like:
    // config.mon_short, config.tue_long, ...., config.<mmm>_short, config.<mmm>_long
    getWeekDay(id, type='long') {
      return decodeURIComponent(this.config[id+'_'+type])
    }
    getWeekDays(type='long') {
      let weekDays = ['sun', 'mon', 'tue', 'wed', 'thu', 'fri', 'sat'];
      return weekDays.concat(weekDays).slice(PywbI18N.firstDayOfWeek, PywbI18N.firstDayOfWeek + 7).map(d => this.getWeekDay(d, type));
    }
    getText(id, embeddedVariableStrings=null) {
      const translated = decodeURIComponent(this.config[id] || id);
      if (embeddedVariableStrings && id.indexOf('{') >= 0 && id.indexOf('}') >= 0 ) {
        return translated.replace(/{(\w+)}/g, (match, stringId) => embeddedVariableStrings[stringId]);
      }
      return translated
    }
    _(id, embeddedVariableStrings=null) {
      return this.getText(id, embeddedVariableStrings);
    }
  }

  const PywbPeriodIdDelimiter = '-';
  function PywbData(rawSnaps) {
    const allTimePeriod = new PywbPeriod({type: PywbPeriod.Type.all, id: "all"});
    const snapshots = [];
    let lastSingle = null;
    let lastYear, lastMonth, lastDay, lastHour;
    rawSnaps.forEach((rawSnap, i) => {
      const snap = new PywbSnapshot(rawSnap, i);
      let year, month, day, hour, single;

      // Year Period
      //  if year did not exist in "all time", create it
      if (!(year = allTimePeriod.getChildById(snap.year))) {
        if (lastYear) lastYear.checkIfSingleSnapshotOnly(); // check last year for containing single snapshot
        lastYear = year = new PywbPeriod({type: PywbPeriod.Type.year, id: snap.year});
        allTimePeriod.addChild(year);
      }

      // Month Period
      //  if month did not exist in "year" period, create it
      if (!(month = year.getChildById(snap.month))) {
        if (lastMonth) lastMonth.checkIfSingleSnapshotOnly();// check last month for containing single snapshot
        lastMonth = month = new PywbPeriod({type: PywbPeriod.Type.month, id: snap.month});
        year.addChild(month);
      }

      // Day Period
      //  if day did not exist in "month" period, create it
      if (!(day = month.getChildById(snap.day))) {
        if (lastDay) lastDay.checkIfSingleSnapshotOnly(); // check last day for containing single snapshot
        lastDay = day = new PywbPeriod({type: PywbPeriod.Type.day, id: snap.day});
        month.addChild(day);
      }

      // Hour Period
      const hourValue = Math.ceil((snap.hour + .0001) / (24/8)); // divide day in 4 six-hour periods (aka quarters)

      //  if hour did not exist in "day" period, create it
      if (!(hour = day.getChildById(hourValue))) {
        if (lastHour) lastHour.checkIfSingleSnapshotOnly(); // check last hour for containing single snapshot
        lastHour = hour = new PywbPeriod({type: PywbPeriod.Type.hour, id: hourValue});
        day.addChild(hour);
      }
      if (!(single = hour.getChildById(snap.id))) {
        single = new PywbPeriod({type: PywbPeriod.Type.snapshot, id: snap.id});
        hour.addChild(single);
      }

      // De-duplicate single snapshots (sometimes there are multiple snapshots
      //   of the same timestamp with different HTTP status; ignore all
      //   duplicates and take the first entry regardless of status)
      if (!lastSingle || lastSingle.id !== single.id) {
        single.setSnapshot(snap);
        if (lastSingle) {
          lastSingle.setNextSnapshotPeriod(single);
          single.setPreviousSnapshotPeriod(lastSingle);
        }
        lastSingle = single;

        snapshots.push(snap);
      }

      // At end of snapshot loop, check period of each type: year/month/day/hour
      //  as all snapshots are now added to the period hierarchy
      if (i === rawSnaps.length - 1) { // is last snapshot
        year.checkIfSingleSnapshotOnly();
        month.checkIfSingleSnapshotOnly();
        day.checkIfSingleSnapshotOnly();
        hour.checkIfSingleSnapshotOnly();
      }
    });

    this.timeline = allTimePeriod;
    this.snapshots = snapshots;
    this.getSnapshot = function(index) {
      if (index < 0 || index >= this.snapshots.length) {
        return null;
      }
      return this.snapshots[index];
    };
    this.getPreviousSnapshot = function(snapshot) {
      const index = snapshot.index;
      return this.getSnapshot(index-1);
    };
    this.getNextSnapshot = function(snapshot) {
      const index = snapshot.index;
      return this.getSnapshot(index+1);
    };
  }
  /* ---------------- SNAP SHOT object ----------------- */
  class PywbSnapshot {
    constructor(init, index) {
      this.index = index;
      this.year = parseInt(init.timestamp.substr(0, 4));
      this.month = parseInt(init.timestamp.substr(4, 2));
      this.day = parseInt(init.timestamp.substr(6, 2));
      this.hour = parseInt(init.timestamp.substr(8, 2));
      this.minute = parseInt(init.timestamp.substr(10, 2));
      this.second = parseInt(init.timestamp.substr(12, 2));
      this.id = parseInt(init.timestamp);

      this.urlkey = init.urlkey;
      this.url = init.url;
      this.mime = init.mime;
      this.status = init.status;
      this.digest = init.digest;
      this.redirect = init.redirect;
      this.robotflags = init.robotflags;
      this.length = init.length;
      this.offset = init.offset;
      this.filename = init.filename;
      this.load_url = init.load_url;
      this["source-col"] = init["source-col"];
      this.access = init.access;
    }

    getTimeDateFormatted() {
      return new Date(this.year, this.month-1, this.day, this.hour, this.minute, this.second).toLocaleString(PywbI18N.getLocale()).toLowerCase();
    }

    getDateFormatted() {
      return new Date(this.year, this.month-1, this.day).toLocaleDateString(PywbI18N.getLocale()).toLowerCase();
    }

    getTimeFormatted() {
      return new Date(2000, 0, 1, this.hour, this.minute, this.second).toLocaleTimeString(PywbI18N.getLocale()).toLowerCase();
    }

    getParentIds() {
      return [this.year, this.month, this.day, Math.ceil((this.hour + .0001) / (24/8))];
    }

    getFullId() {
      return [this.year, this.month, this.day, Math.ceil((this.hour + .0001) / (24/8)), this.id].join(PywbPeriodIdDelimiter);
    }
  }

  /* ---------------- PERIOD object ----------------- */
  function PywbPeriod(init) {
    this.type = init.type;
    this.id = init.id;
    this.fullId = Math.floor(1000*1000+Math.random()*9*1000*1000).toString(16); // full-id property that include string id of parents and self with a delimitor

    this.childrenIds = {}; // allow for query by ID
    this.children = []; // allow for sequentiality / order

    this.maxGrandchildSnapshotCount = 0;
    this.snapshotCount = 0;
  }
  PywbPeriod.Type = {all: 0,year: 1,month: 2,day: 3,hour: 4,snapshot:5};
  PywbPeriod.TypeLabel = ["timeline","year","month","day","hour","snapshot"];

  PywbPeriod.prototype.getTypeLabel = function() {
    return PywbPeriod.TypeLabel[this.type];
  };
  PywbPeriod.GetTypeLabel = function(type) {
    return PywbPeriod.TypeLabel[type] ? PywbPeriod.TypeLabel[type] : "";
  };

  PywbPeriod.prototype.getChildById = function(id) {
    return this.children[this.childrenIds[id]];
  };

  // previous period (ONLY SET at the period level/type: snapshot)
  PywbPeriod.prototype.getPreviousSnapshotPeriod = () => {};
  PywbPeriod.prototype.setPreviousSnapshotPeriod = function(period) {
    this.getPreviousSnapshotPeriod = () => period;
  };
  // next period (ONLY SET at the period level/type: snapshot)
  PywbPeriod.prototype.getNextSnapshotPeriod = () => {};
  PywbPeriod.prototype.setNextSnapshotPeriod = function(period) {
    this.getNextSnapshotPeriod = () => period;
  };

  PywbPeriod.prototype.getFirstSnapshotPeriod = function() {
    return this.getFirstLastSnapshotPeriod_("first");
  };
  PywbPeriod.prototype.getLastSnapshotPeriod = function() {
    return this.getFirstLastSnapshotPeriod_("last");
  };
  PywbPeriod.prototype.getFirstLastSnapshotPeriod_ = function(direction) {
    let period = this;
    let iFailSafe = 100; // in case a parser has a bug and the snapshotCount is not correct; avoid infinite-loop
    while (period.snapshotCount && period.type !== PywbPeriod.Type.snapshot) {
      let i = 0;
      for(i=0; i < period.children.length; i++) {
        const ii = direction === "first" ? i : (period.children.length - 1 - i);
        if (period.children[ii].snapshotCount) {
          period = period.children[ii];
          break;
        }
      }
      if (iFailSafe-- < 0) {
        break;
      }
    }
    if (period.type === PywbPeriod.Type.snapshot && period.snapshot) {
      return period;
    }
    return null;
  };

  PywbPeriod.prototype.getPrevious = function() {
    const firstSnapshotPeriod = this.getFirstSnapshotPeriod();
    if (!firstSnapshotPeriod) {
      return null;
    }
    const previousSnapshotPeriod = firstSnapshotPeriod.getPreviousSnapshotPeriod();
    if (!previousSnapshotPeriod) {
      return null;
    }
    if (this.type === PywbPeriod.Type.snapshot) {
      return previousSnapshotPeriod;
    }
    let parent = previousSnapshotPeriod.parent;
    while(parent) {
      if (parent.type === this.type) {
        break;
      }
      parent = parent.parent;
    }
    return parent;
  };
  PywbPeriod.prototype.getNext = function() {
    const lastSnapshotPeriod = this.getLastSnapshotPeriod();
    if (!lastSnapshotPeriod) {
      return null;
    }
    const nextSnapshotPeriod = lastSnapshotPeriod.getNextSnapshotPeriod();
    if (!nextSnapshotPeriod) {
      return null;
    }
    if (this.type === PywbPeriod.Type.snapshot) {
      return nextSnapshotPeriod;
    }
    let parent = nextSnapshotPeriod.parent;
    while(parent) {
      if (parent.type === this.type) {
        break;
      }
      parent = parent.parent;
    }
    return parent;
  };

  PywbPeriod.prototype.parent = null;
  PywbPeriod.prototype.addChild = function(period) {
    if (this.getChildById(period.id)) {
      return false;
    }
    period.parent = this;
    this.childrenIds[period.id] = this.children.length;
    this.children.push(period);
    period.initFullId();
    return true;
  };

  PywbPeriod.prototype.getChildrenRange = function() {
    switch (this.type) {
    case PywbPeriod.Type.all:
      // year range: first to last year available
      return [this.children[0].id, this.children[this.children.length-1].id];
    case PywbPeriod.Type.year:
      // month is simple: 1 to 12
      return [1,12];
    case PywbPeriod.Type.month: {
      // days in month: 1 to last day in month
      const y = this.parent.id; const m = this.id;
      const lastDateInMonth = (new Date((new Date(y, m, 1)).getTime() - 1000)).getDate(); // 1 sec earlier
      return [1, lastDateInMonth];
    }
    case PywbPeriod.Type.day:
      // hours: 0 to 23
      // return [1,4];
      return [1,8];
    }
    return null;
  };
  PywbPeriod.prototype.fillEmptyGrandChildPeriods = function() {
    if (this.hasFilledEmptyGrandchildPeriods) {
      return;
    }
    this.children.forEach(c => {
      c.fillEmptyChildPeriods();
    });
    this.hasFilledEmptyGrandchildPeriods = true;
  };

  PywbPeriod.prototype.fillEmptyChildPeriods = function(isFillEmptyGrandChildrenPeriods=false) {
    if (this.type > PywbPeriod.Type.day) {
      return;
    }

    const idRange = this.getChildrenRange();
    if (!idRange) {
      return;
    }

    let i = 0;
    for (let newId = idRange[0]; newId <= idRange[1]; newId++) {
      if (i < this.children.length) {
        // if existing and new id match, skip, item already in place
        // else
        if (this.children[i].id !== newId) {
          const empty = new PywbPeriod({type: this.type + 1, id: newId});
          if (newId < this.children[i].id) {
            // insert new before existing
            this.children.splice(i, 0, empty);
          } else {
            // insert new after existing
            this.children.splice(i+1, 0, empty);
          }
          empty.parent = this;
          empty.initFullId();
        }
        i++;
      } else {
        const empty = new PywbPeriod({type: this.type + 1, id: newId});
        this.addChild(empty);
      }
    }

    // re-calculate indexes
    for(let i=0;i<this.children.length;i++) {
      this.childrenIds[this.children[i].id] = i;
    }

    if (isFillEmptyGrandChildrenPeriods) {
      this.fillEmptyGrandChildPeriods();
    }

    return idRange;
  };

  PywbPeriod.prototype.getParents = function(skipAllTime=false) {
    let parents = [];
    let parent = this.parent;
    while(parent) {
      parents.push(parent);
      parent = parent.parent;
    }
    parents = parents.reverse();
    if (skipAllTime) {
      parents.shift(); // skip first "all-time"
    }
    return parents;
  };

  PywbPeriod.prototype.contains = function(periodOrSnapshot) {
    if (this.type === 0) {
      return true; // all-time contains everything
    }
    if (periodOrSnapshot instanceof PywbPeriod) {
      return periodOrSnapshot.getParents(true).slice(0,this.type).map(p => p.id).join(PywbPeriodIdDelimiter) === this.fullId;
    }
    if (periodOrSnapshot instanceof PywbSnapshot) {
      if (this.type === PywbPeriod.Type.snapshot) {
        return periodOrSnapshot.getFullId() === this.fullId;
      } else {
        return periodOrSnapshot.getParentIds(true).slice(0,this.type).join(PywbPeriodIdDelimiter) === this.fullId;
      }
    }
    return false;
  };

  PywbPeriod.prototype.snapshot = null;
  PywbPeriod.prototype.snapshotPeriod = null;

  PywbPeriod.prototype.checkIfSingleSnapshotOnly = function() {
    if (this.snapshotCount === 1) {
      let snapshotPeriod = this;
      let failSafe = PywbPeriod.Type.snapshot;
      while(!snapshotPeriod.snapshot) {
        if (--failSafe <=0) break;
        snapshotPeriod = snapshotPeriod.children[0];
      }
      this.snapshot = snapshotPeriod.snapshot;
      this.snapshotPeriod = snapshotPeriod;
    }
  };

  PywbPeriod.prototype.setSnapshot = function(snap) {
    this.snapshot = snap;
    this.snapshotCount++;
    let parent = this.parent;
    let child = this;
    while (parent) {
      parent.snapshotCount++;

      let grandParent = parent.parent;
      if (grandParent) { // grandparent
        grandParent.maxGrandchildSnapshotCount = Math.max(grandParent.maxGrandchildSnapshotCount, child.snapshotCount);
      }
      child = parent;
      parent = parent.parent;
    }
  };


  PywbPeriod.prototype.getSnapshotPeriodsFlat = function(flatArray=false) {
    if (!flatArray) {
      flatArray = [];
    }
    if (!this.snapshotCount) {
      return flatArray;
    }

    if (this.snapshotCount === 1) {
      flatArray.push(this.snapshotPeriod || this);
      return flatArray;
    }

    this.children.forEach(child => {
      child.getSnapshotPeriodsFlat(flatArray);
    });
    return flatArray;
  };

  /**
   * Return the "full" id, which includes all parents ID and self ID, delimited by a ${PywbPeriodIdDelimiter}
   * @returns {string}
   */
  PywbPeriod.prototype.initFullId = function() {
    const ids = this.getParents(true).map(p => p.id);
    ids.push(this.id);
    this.fullId = ids.join(PywbPeriodIdDelimiter);
  };

  /**
   * Find a period by its full ID (of all ancestors and self, delimited by a hyphen). Start by locating the great-grand-parent (aka timeline), then looping on all IDs and finding the period in loop
   * @param {string} fullId
   * @returns {boolean}
   */
  PywbPeriod.prototype.findByFullId = function(fullId) {
    let parent = this;
    if (this.type !== PywbPeriod.Type.all) {
      parent = this.getParents()[0];
    }
    const ids = fullId.split(PywbPeriodIdDelimiter);

    let found = false;
    for(let i=0; i<ids.length; i++) {
      parent = parent.getChildById(ids[i]);
      if (parent) {
        // if last chunk of ID in loop, the period is found
        if (i === ids.length - 1) {
          found = parent;
        }
      } else {
        // if no parent is found with ID chunk, abort "mission"
        break;
      }
    }
    return found;
  };
  PywbPeriod.prototype.getFullReadableId = function() {
    // remove "all-time" from parents (getParents(true) when printing readable id (of all parents and currrent
    switch (this.type) {
      case PywbPeriod.Type.all:
        return "";
      case PywbPeriod.Type.year:
        return this.id;
      case PywbPeriod.Type.month:
        return this.getReadableId() + ' ' + this.parent.id;
      case PywbPeriod.Type.day: {
        return new Date(this.parent.parent.id, this.parent.id, this.getReadableId()).toLocaleDateString(PywbI18N.getLocale());
      }
      case PywbPeriod.Type.hour:
        const hourRange = this.getReadableId({hourRange: true});
        return this.parent.getFullReadableId() + ' ' + PywbI18N.instance._('from {hour1} to {hour2}', {
          hour1: hourRange[0],
          hour2: hourRange[1]
        });
      case PywbPeriod.Type.snapshot:
        return this.snapshot.getTimeDateFormatted();
    }
  };
  PywbPeriod.prototype.getReadableId = function(opts={hourRange:null}) {
    switch (this.type) {
    case PywbPeriod.Type.all:
      return PywbI18N.instance._("All-time");
    case PywbPeriod.Type.year:
      return this.id;
    case PywbPeriod.Type.month:
      return PywbI18N.instance.getMonth(this.id, 'short');
    case PywbPeriod.Type.day: {
      let suffix = "";
      // DISABLING cardinal suffix for now, as it is complicated to replicate in multiple locales with 1 simple function
      // TODO: add cardinal suffix handling later IF REQUESTED!
      // if (cardinalSuffix) {
      //   const singleDigit = this.id % 10;
      //   const isTens = Math.floor(this.id / 10) === 1;
      //   const suffixes = {1:"st", 2:"nd",3:"rd"};
      //   suffix = (isTens || !suffixes[singleDigit]) ? "th" : suffixes[singleDigit];
      // }
      return this.id + suffix;
    }
      case PywbPeriod.Type.hour:
      // use browser's locale setting to get time string and remove seconds, and lower-case it (in case AM-PM)
      const hours = [0, 3, 6, 9, 12, 15, 18, 21].map(hour => new Date(2000, 0, 1, hour, 0, 0).toLocaleTimeString(PywbI18N.getLocale()).replace(/^(\d{1,2}:\d\d):\d\d/, (m, m1)=> m1).toLowerCase());
      if (opts.hourRange) {
        return [hours[this.id-1], hours[this.id % hours.length]];
      }
      return hours[this.id-1];
      //return ({1:'midnight', 2: '6 am', 3: 'noon', 4: '6 pm'})[this.id];
      //return (this.id < 13 ? this.id : this.id % 12) + ' ' + (this.id < 12 ? 'am':'pm');
    case PywbPeriod.Type.snapshot:
      return this.snapshot.getTimeFormatted();
    }
  };

  PywbPeriod.prototype.getYear = function() { this.get(PywbPeriod.Type.year); };
  PywbPeriod.prototype.getMonth = function() { this.get(PywbPeriod.Type.month); };
  PywbPeriod.prototype.getDay = function() { this.get(PywbPeriod.Type.day); };
  PywbPeriod.prototype.getHour = function() { this.get(PywbPeriod.Type.hour); };
  PywbPeriod.prototype.get = function(type) {
    if (this.type === type) {
      return this;
    } else if (this.type > type) {
      return this.getParents()[type];
    }
  };

  //

  var script$6 = {
    props: {
      period: { required: true },
      currentSnapshot: { required: false, default: null},
      highlight: { required: false, default: false},
      stayWithinPeriod: { required: false, default: false},
      maxZoomLevel: { required: false, default: PywbPeriod.Type.snapshot}
    },
    data: function() {
      return {
        highlightPeriod: null,
        previousPeriod: null,
        nextPeriod: null,
        isScrollZero: true,
        isScrollMax: true,
        tooltipPeriod: null,
        tooltipPeriodPos: {x:0,y:0}
      };
    },
    created: function() {
      this.addEmptySubPeriods();
    },
    mounted: function() {
      this.$refs.periods._computedStyle = window.getComputedStyle(this.$refs.periods);
      this.$refs.periodScroll._computedStyle = window.getComputedStyle(this.$refs.periodScroll);
      this.$watch("period", this.onPeriodChanged);

      this.$refs.periodScroll.addEventListener("scroll", this.updateScrollArrows);
      window.addEventListener("resize", this.updateScrollArrows);
      this.updateScrollArrows();
    },
    computed: {
      // this determins which the last zoom level is before we go straight to showing snapshot
      canZoom() {
        return this.period.type < this.maxZoomLevel;
      },
      isTooltipPeriodDayOrHour() {
        return this.tooltipPeriod.type >= PywbPeriod.Type.day;
      },
      iContainCurrentSnapshot() {
        return this.currentSnapshot && this.period.contains(this.currentSnapshot);
      }
    },
    updated() {
      // do something on update
    },
    methods: {
      containsCurrentSnapshot(period) {
        return this.iContainCurrentSnapshot && period.contains(this.currentSnapshot);
      },
      addEmptySubPeriods() {
        this.period.fillEmptyChildPeriods(true);
      },
      updateScrollArrows() {
        this.period.scroll = this.$refs.periodScroll.scrollLeft;
        const maxScroll = parseInt(this.$refs.periods._computedStyle.width) - parseInt(this.$refs.periodScroll._computedStyle.width);
        this.isScrollZero = !this.period.scroll; // if 0, then true (we are at scroll zero)
        this.isScrollMax = Math.abs(maxScroll - this.period.scroll) < 5;
      },
      restoreScroll() {
        this.$refs.periodScroll.scrollLeft = this.period.scroll;
      },
      scrollNext: function () {
        if (this.isScrollMax) {
          if (this.nextPeriod) {
            this.$emit("goto-period", this.nextPeriod, true /* onlyZoomToPeriod */);
          }
        } else {
          this.$refs.periodScroll.scrollLeft += 30;
        }
      },
      scrollPrev: function () {
        if (this.isScrollZero) {
          if (this.previousPeriod) {
            this.$emit("goto-period", this.previousPeriod, true /* onlyZoomToPeriod */);
          }
        } else {
          this.$refs.periodScroll.scrollLeft -= 30;
        }
      },
      getTimeFormatted: function(date) {
        return (date.hour < 13 ? date.hour : (date.hour % 12)) + ":" + ((date.minute < 10 ? "0":"")+date.minute) + " " + (date.hour < 12 ? "am":"pm");
      },
      getHistoLineHeight: function(value) {
        const percent = Math.ceil((value/this.period.maxGrandchildSnapshotCount) * 100);
        return (percent ? (5 + Math.ceil(percent*.95)) : 0) + "%";
        // return percent + '%';
      },
      changePeriod(period, $event) {
        // if not empty
        if (period.snapshotCount) {
          let periodToChangeTo = null;
          // if contains a single snapshot only, navigate to snapshot (load snapshot in FRAME, do not ZOOM IN)
          if (period.snapshot) {
            // if period is at level "snapshot" (no more children), send period, else send the child period, a reference to which is stored (by data/model layer) in the current period; App event needs a period to be passed (cannot pass in snapshot object itself)
            if (period.type === PywbPeriod.Type.snapshot) {
              periodToChangeTo = period;
            } else if (period.snapshotPeriod) {
              periodToChangeTo = period.snapshotPeriod;
            }
          } else {
          // if contains mulitple snapshots,
            // zoom if ZOOM level is day or less, OR if period contain TOO MANY (>10)
            if (this.canZoom) {
              periodToChangeTo = period;
            }
          }

          // if we selected a period to go to, emit event
          if (periodToChangeTo) {
            this.$emit("goto-period", periodToChangeTo);
          }
        }
        $event.stopPropagation();
        return false;
      },
      onPeriodChanged(newPeriod, oldPeriod) {
        this.addEmptySubPeriods();
        const previousPeriod = this.period.getPrevious();
        const nextPeriod = this.period.getNext();
        if (!this.stayWithinPeriod || this.stayWithinPeriod.contains(previousPeriod)) {
          this.previousPeriod = previousPeriod;
        }
        if (!this.stayWithinPeriod || this.stayWithinPeriod.contains(nextPeriod)) {
          this.nextPeriod = nextPeriod;
        }

        // detect if going up level of period (new period type should be in old period parents)
        if (oldPeriod && oldPeriod.type - newPeriod.type > 0) {
          let highlightPeriod = oldPeriod;
          for (let i=oldPeriod.type - newPeriod.type; i > 1; i--) {
            highlightPeriod = highlightPeriod.parent;
          }
          this.highlightPeriod = highlightPeriod;
          setTimeout((function() {
            this.highlightPeriod = null;
          }).bind(this), 2000);
        }
        setTimeout((function() {
          this.restoreScroll();
          this.updateScrollArrows();
        }).bind(this), 1);
      },
      setTooltipPeriod(period, event) {
        if (!period || !period.snapshotCount) {
          this.tooltipPeriod = null;
          return;
        }
        this.tooltipPeriod = period;

        this.$nextTick(function() {
          const tooltipContentsEl = document.querySelector('.period-tooltip div');
          if (!tooltipContentsEl) {
            return;
          }

          const periodTooltipStyle = window.getComputedStyle(tooltipContentsEl);
          const tooltipWidth = parseInt(periodTooltipStyle.width);
          const tooltipHeight = parseInt(periodTooltipStyle.height);
          const spacing = 10;
          if (window.innerWidth < event.x + (spacing*2) + tooltipWidth) {
            this.tooltipPeriodPos.x = event.x - (tooltipWidth + spacing);
          } else {
            this.tooltipPeriodPos.x = event.x + spacing;
          }
          this.tooltipPeriodPos.y = event.y - (spacing + tooltipHeight);
        });
        event.stopPropagation();
        return false;
      }
    }
  };

  function normalizeComponent(template, style, script, scopeId, isFunctionalTemplate, moduleIdentifier /* server only */, shadowMode, createInjector, createInjectorSSR, createInjectorShadow) {
      if (typeof shadowMode !== 'boolean') {
          createInjectorSSR = createInjector;
          createInjector = shadowMode;
          shadowMode = false;
      }
      // Vue.extend constructor export interop.
      const options = typeof script === 'function' ? script.options : script;
      // render functions
      if (template && template.render) {
          options.render = template.render;
          options.staticRenderFns = template.staticRenderFns;
          options._compiled = true;
          // functional template
          if (isFunctionalTemplate) {
              options.functional = true;
          }
      }
      // scopedId
      if (scopeId) {
          options._scopeId = scopeId;
      }
      let hook;
      if (moduleIdentifier) {
          // server build
          hook = function (context) {
              // 2.3 injection
              context =
                  context || // cached call
                      (this.$vnode && this.$vnode.ssrContext) || // stateful
                      (this.parent && this.parent.$vnode && this.parent.$vnode.ssrContext); // functional
              // 2.2 with runInNewContext: true
              if (!context && typeof __VUE_SSR_CONTEXT__ !== 'undefined') {
                  context = __VUE_SSR_CONTEXT__;
              }
              // inject component styles
              if (style) {
                  style.call(this, createInjectorSSR(context));
              }
              // register component module identifier for async chunk inference
              if (context && context._registeredComponents) {
                  context._registeredComponents.add(moduleIdentifier);
              }
          };
          // used by ssr in case component is cached and beforeCreate
          // never gets called
          options._ssrRegister = hook;
      }
      else if (style) {
          hook = shadowMode
              ? function (context) {
                  style.call(this, createInjectorShadow(context, this.$root.$options.shadowRoot));
              }
              : function (context) {
                  style.call(this, createInjector(context));
              };
      }
      if (hook) {
          if (options.functional) {
              // register for functional component in vue file
              const originalRender = options.render;
              options.render = function renderWithStyleInjection(h, context) {
                  hook.call(context);
                  return originalRender(h, context);
              };
          }
          else {
              // inject component registration as beforeCreate hook
              const existing = options.beforeCreate;
              options.beforeCreate = existing ? [].concat(existing, hook) : [hook];
          }
      }
      return script;
  }

  const isOldIE = typeof navigator !== 'undefined' &&
      /msie [6-9]\\b/.test(navigator.userAgent.toLowerCase());
  function createInjector(context) {
      return (id, style) => addStyle(id, style);
  }
  let HEAD;
  const styles = {};
  function addStyle(id, css) {
      const group = isOldIE ? css.media || 'default' : id;
      const style = styles[group] || (styles[group] = { ids: new Set(), styles: [] });
      if (!style.ids.has(id)) {
          style.ids.add(id);
          let code = css.source;
          if (css.map) {
              // https://developer.chrome.com/devtools/docs/javascript-debugging
              // this makes source maps inside style tags work properly in Chrome
              code += '\n/*# sourceURL=' + css.map.sources[0] + ' */';
              // http://stackoverflow.com/a/26603875
              code +=
                  '\n/*# sourceMappingURL=data:application/json;base64,' +
                      btoa(unescape(encodeURIComponent(JSON.stringify(css.map)))) +
                      ' */';
          }
          if (!style.element) {
              style.element = document.createElement('style');
              style.element.type = 'text/css';
              if (css.media)
                  style.element.setAttribute('media', css.media);
              if (HEAD === undefined) {
                  HEAD = document.head || document.getElementsByTagName('head')[0];
              }
              HEAD.appendChild(style.element);
          }
          if ('styleSheet' in style.element) {
              style.styles.push(code);
              style.element.styleSheet.cssText = style.styles
                  .filter(Boolean)
                  .join('\n');
          }
          else {
              const index = style.ids.size - 1;
              const textNode = document.createTextNode(code);
              const nodes = style.element.childNodes;
              if (nodes[index])
                  style.element.removeChild(nodes[index]);
              if (nodes.length)
                  style.element.insertBefore(textNode, nodes[index]);
              else
                  style.element.appendChild(textNode);
          }
      }
  }

  /* script */
  const __vue_script__$6 = script$6;

  /* template */
  var __vue_render__$6 = function() {
    var _obj;
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c("div", { staticClass: "timeline" }, [
      _c(
        "div",
        {
          directives: [
            {
              name: "show",
              rawName: "v-show",
              value: _vm.tooltipPeriod,
              expression: "tooltipPeriod"
            }
          ],
          staticClass: "period-tooltip",
          style: {
            left: _vm.tooltipPeriodPos.x + "px",
            top: _vm.tooltipPeriodPos.y + "px"
          }
        },
        [
          _vm.tooltipPeriod
            ? [
                _vm.tooltipPeriod.snapshot
                  ? _c("div", [
                      _vm._v(
                        "\n            " +
                          _vm._s(
                            _vm.$root._("View capture on {date}", {
                              date: _vm.tooltipPeriod.snapshot.getTimeDateFormatted()
                            })
                          ) +
                          "\n          "
                      )
                    ])
                  : _vm.tooltipPeriod.snapshotPeriod
                  ? _c("div", [
                      _vm._v(
                        "\n            " +
                          _vm._s(
                            _vm.$root._("View capture on {date}", {
                              date: _vm.tooltipPeriod.snapshotPeriod.snapshot.getTimeDateFormatted()
                            })
                          ) +
                          "\n          "
                      )
                    ])
                  : _vm.tooltipPeriod.snapshotCount
                  ? _c("div", [
                      _vm._v(
                        "\n            " +
                          _vm._s(
                            _vm.$root._(
                              _vm.isTooltipPeriodDayOrHour
                                ? "{capture_text} on {date}"
                                : "{capture_text} in {month}", // TODO: split translation into "in {year}" and "in {month}"
                              ((_obj = {
                                capture_text: _vm.$root._(
                                  _vm.tooltipPeriod.snapshotCount !== 1
                                    ? "{count} captures"
                                    : "{count} capture",
                                  { count: _vm.tooltipPeriod.snapshotCount }
                                )
                              }),
                              (_obj[
                                _vm.isTooltipPeriodDayOrHour ? "date" : "month"
                              ] = _vm.tooltipPeriod.getFullReadableId()),
                              _obj)
                            )
                          ) +
                          "\n          "
                      )
                    ])
                  : _vm._e()
              ]
            : _vm._e()
        ],
        2
      ),
      _vm._v(" "),
      _c("div", {
        staticClass: "arrow previous",
        class: { disabled: _vm.isScrollZero && !_vm.previousPeriod },
        attrs: { tabindex: "0" },
        domProps: { innerHTML: _vm._s("&#x25C0;") },
        on: {
          click: _vm.scrollPrev,
          keyup: function($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
            ) {
              return null
            }
            return _vm.scrollPrev.apply(null, arguments)
          },
          dblclick: function($event) {
            $event.stopPropagation();
            $event.preventDefault();
          }
        }
      }),
      _vm._v(" "),
      _c(
        "div",
        {
          ref: "periodScroll",
          staticClass: "scroll",
          class: { highlight: _vm.highlight }
        },
        [
          _c(
            "div",
            { ref: "periods", staticClass: "periods" },
            _vm._l(_vm.period.children, function(subPeriod) {
              return _c(
                "div",
                {
                  key: subPeriod.fullId,
                  staticClass: "period",
                  class: {
                    empty: !subPeriod.snapshotCount,
                    highlight: _vm.highlightPeriod === subPeriod,
                    "last-level": !_vm.canZoom,
                    "contains-current-snapshot": _vm.containsCurrentSnapshot(
                      subPeriod
                    )
                  }
                },
                [
                  _c(
                    "div",
                    { staticClass: "histo" },
                    _vm._l(subPeriod.children, function(histoPeriod) {
                      return _c("div", {
                        key: histoPeriod.fullId,
                        staticClass: "line",
                        class: {
                          "has-single-snapshot": histoPeriod.snapshotCount === 1,
                          "contains-current-snapshot": _vm.containsCurrentSnapshot(
                            histoPeriod
                          )
                        },
                        style: {
                          height: _vm.getHistoLineHeight(
                            histoPeriod.snapshotCount
                          )
                        },
                        attrs: { tabindex: "0" },
                        on: {
                          click: function($event) {
                            return _vm.changePeriod(histoPeriod, $event)
                          },
                          keyup: function($event) {
                            if (
                              !$event.type.indexOf("key") &&
                              _vm._k(
                                $event.keyCode,
                                "enter",
                                13,
                                $event.key,
                                "Enter"
                              )
                            ) {
                              return null
                            }
                            return _vm.changePeriod(histoPeriod, $event)
                          },
                          mouseover: function($event) {
                            return _vm.setTooltipPeriod(histoPeriod, $event)
                          },
                          mouseout: function($event) {
                            return _vm.setTooltipPeriod(null, $event)
                          }
                        }
                      })
                    }),
                    0
                  ),
                  _vm._v(" "),
                  _c(
                    "div",
                    {
                      staticClass: "inner",
                      class: {
                        "has-single-snapshot": subPeriod.snapshotCount === 1
                      },
                      attrs: { tabindex: "0" },
                      on: {
                        click: function($event) {
                          return _vm.changePeriod(subPeriod, $event)
                        },
                        keyup: function($event) {
                          if (
                            !$event.type.indexOf("key") &&
                            _vm._k(
                              $event.keyCode,
                              "enter",
                              13,
                              $event.key,
                              "Enter"
                            )
                          ) {
                            return null
                          }
                          return _vm.changePeriod(_vm.histoPeriod, $event)
                        },
                        mouseover: function($event) {
                          return _vm.setTooltipPeriod(subPeriod, $event)
                        },
                        mouseout: function($event) {
                          return _vm.setTooltipPeriod(null, $event)
                        }
                      }
                    },
                    [
                      _c("div", { staticClass: "label" }, [
                        _vm._v(
                          "\n                      " +
                            _vm._s(subPeriod.getReadableId()) +
                            "\n                    "
                        )
                      ])
                    ]
                  )
                ]
              )
            }),
            0
          )
        ]
      ),
      _vm._v(" "),
      _c("div", {
        staticClass: "arrow next",
        class: { disabled: _vm.isScrollMax && !_vm.nextPeriod },
        attrs: { tabindex: "0" },
        domProps: { innerHTML: _vm._s("&#x25B6;") },
        on: {
          click: _vm.scrollNext,
          keyup: function($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
            ) {
              return null
            }
            return _vm.scrollNext.apply(null, arguments)
          },
          dblclick: function($event) {
            $event.stopPropagation();
            $event.preventDefault();
          }
        }
      })
    ])
  };
  var __vue_staticRenderFns__$6 = [];
  __vue_render__$6._withStripped = true;

    /* style */
    const __vue_inject_styles__$6 = function (inject) {
      if (!inject) return
      inject("data-v-7627cc2c_0", { source: "\n.timeline {\n    position: relative;\n    display: flex;\n    width: auto;\n    height: 60px;\n    margin: 5px;\n    justify-content: left;\n}\n.timeline .id {\n    display: inline-block;\n    font-size: 30px;\n}\n.timeline .arrow {\n    display: inline-block;\n    width: 20px;\n    font-size: 20px; /* font-size = width of arrow, as it UTF char */\n    line-height: 60px;\n    vertical-align: top;\n    cursor: pointer;\n}\n.timeline .arrow.previous {\n}\n.timeline .arrow.next {\n}\n.timeline .arrow.disabled, .timeline .arrow.disabled:hover {\n    /*color: lightgray;*/\n    background-color: transparent;\n    /*cursor: not-allowed;*/\n    visibility: hidden;\n}\n.timeline .arrow:hover {\n    background-color: antiquewhite;\n    color: firebrick;\n}\n.timeline .scroll {\n    position: relative;\n    display: inline-block;\n    width: 100%; /* */\n    height: 100%;\n\n    /* maker scrollable horizontally */\n    overflow-x: scroll;\n    overflow-y: hidden;\n    white-space: nowrap;\n    scroll-behavior: smooth;\n\n    text-align: center;\n\n    transition: background-color 500ms ease-in;\n}\n/* hide scroll bar */\n.timeline .scroll::-webkit-scrollbar {\n    display: none;\n}\n/* highlight the scroll period: usually triggered from root app */\n.timeline .scroll.highlight {\n    background-color: #fff7ce;\n}\n.timeline .scroll .periods {\n    display: flex;\n    justify-content: space-between;\n    height: 100%;\n    width: 100%;\n    min-width: 600px;\n}\n.timeline .period {\n    flex-grow: 1;\n    position: relative;\n    display: inline-block;\n    height: 100%;\n    /* line-height: 80px; /* use to center middle vertically */\n    white-space: normal;\n    vertical-align: top;\n    text-align: center;\n    background-color: transparent;\n\n    transition: background-color 500ms ease-in-out;\n}\n/* 1st period period child el */\n.timeline .period:nth-child(1) {\n    /*border-left: 1px solid white; !* has left border; all other periods have right border *!*/\n}\n.timeline .period:hover {\n    background-color: #eeeeee;\n}\n.timeline .period.contains-current-snapshot, .timeline .period.contains-current-snapshot:hover {\n    background-color: #f7def4;\n}\n\n/* empty period */\n.timeline .period.empty {\n    color: #aaa;\n    /*background-color: transparent;*/\n}\n/* highlighted period */\n.timeline .period.highlight {\n    background-color: cyan;\n}\n.timeline .period .inner {\n    display: block;\n    position: absolute;\n    bottom: 0;\n    left: 0;\n    width: 100%;\n    height: 20px;\n    background-color: white;\n    border-top: 1px solid gray;\n    white-space: nowrap;\n    cursor: zoom-in;\n}\n.timeline .period .inner.has-single-snapshot {\n  cursor: pointer;\n}\n.timeline .period.last-level .inner, .timeline .period.empty .inner {\n  cursor: default;\n}\n.timeline .period .label {\n  width: 100%;\n  font-weight: bold;\n  font-size: 14px;\n  transition: background-color 500ms ease-in;\n}\n.timeline .period:hover .label {\n  position: absolute;\n  z-index: 20;\n  background-color: lightgrey;\n}\n.timeline .period .histo {\n    display: flex;\n    position: absolute;\n    top: 1px;\n    left: 0;\n    width: 100%;\n    height: 39px;\n    align-items: flex-end;\n    justify-content: space-between;\n    text-align: left;\n}\n.timeline .period .histo .line {\n    position: relative;\n    flex-grow: 1;\n    display: inline-block;\n    background-color: #a6cdf5;\n    margin: 0;\n    padding: 0;\n    cursor: zoom-in;\n}\n.timeline .period .histo .line.has-single-snapshot {\n  cursor: pointer;\n}\n\n/* Last level period histogram spaces things evenly */\n.timeline .period.last-level .histo {\n    justify-content: space-around;\n}\n\n/* Last level period histogram lines do not grow, but are fixed width/margin */\n.timeline .period.last-level .histo .line {\n    flex-grow: unset;\n    width: 5px;\n    margin-left: 2px;\n}\n\n    /* update line color on hover*/\n.timeline .period .histo .line:hover {\n        background-color: #f5a6eb;\n}\n.timeline .period .histo .line.contains-current-snapshot {\n        background-color: red;\n}\n\n    /* Period that contains ONE snapshot only will show snapshot info*/\n.timeline .period-tooltip {\n        position: fixed;\n        z-index: 100;\n        /*left or right set programmatically*/\n        display: block;\n        background-color: white;\n        border: 1px solid gray;\n        padding: 2px;\n        white-space: nowrap; /*no wrapping allowed*/\n}\n        /*show on hover*/\n.timeline .period-tooltip.show {\n            display: block;\n}\n\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/components/Timeline.vue"],"names":[],"mappings":";AAyPA;IACA,kBAAA;IACA,aAAA;IACA,WAAA;IACA,YAAA;IACA,WAAA;IACA,qBAAA;AACA;AAEA;IACA,qBAAA;IACA,eAAA;AACA;AACA;IACA,qBAAA;IACA,WAAA;IACA,eAAA,EAAA,+CAAA;IACA,iBAAA;IACA,mBAAA;IACA,eAAA;AACA;AACA;AACA;AACA;AACA;AACA;IACA,oBAAA;IACA,6BAAA;IACA,uBAAA;IACA,kBAAA;AACA;AACA;IACA,8BAAA;IACA,gBAAA;AACA;AAEA;IACA,kBAAA;IACA,qBAAA;IACA,WAAA,EAAA,IAAA;IACA,YAAA;;IAEA,kCAAA;IACA,kBAAA;IACA,kBAAA;IACA,mBAAA;IACA,uBAAA;;IAEA,kBAAA;;IAEA,0CAAA;AACA;AACA,oBAAA;AACA;IACA,aAAA;AACA;AACA,iEAAA;AACA;IACA,yBAAA;AACA;AACA;IACA,aAAA;IACA,8BAAA;IACA,YAAA;IACA,WAAA;IACA,gBAAA;AACA;AAGA;IACA,YAAA;IACA,kBAAA;IACA,qBAAA;IACA,YAAA;IACA,0DAAA;IACA,mBAAA;IACA,mBAAA;IACA,kBAAA;IACA,6BAAA;;IAEA,8CAAA;AACA;AACA,+BAAA;AACA;IACA,2FAAA;AACA;AAEA;IACA,yBAAA;AACA;AACA;IACA,yBAAA;AACA;;AAEA,iBAAA;AACA;IACA,WAAA;IACA,iCAAA;AACA;AACA,uBAAA;AACA;IACA,sBAAA;AACA;AAEA;IACA,cAAA;IACA,kBAAA;IACA,SAAA;IACA,OAAA;IACA,WAAA;IACA,YAAA;IACA,uBAAA;IACA,0BAAA;IACA,mBAAA;IACA,eAAA;AACA;AACA;EACA,eAAA;AACA;AACA;EACA,eAAA;AACA;AAEA;EACA,WAAA;EACA,iBAAA;EACA,eAAA;EACA,0CAAA;AACA;AACA;EACA,kBAAA;EACA,WAAA;EACA,2BAAA;AACA;AAEA;IACA,aAAA;IACA,kBAAA;IACA,QAAA;IACA,OAAA;IACA,WAAA;IACA,YAAA;IACA,qBAAA;IACA,8BAAA;IACA,gBAAA;AACA;AAEA;IACA,kBAAA;IACA,YAAA;IACA,qBAAA;IACA,yBAAA;IACA,SAAA;IACA,UAAA;IACA,eAAA;AACA;AACA;EACA,eAAA;AACA;;AAEA,qDAAA;AACA;IACA,6BAAA;AACA;;AAEA,8EAAA;AACA;IACA,gBAAA;IACA,UAAA;IACA,gBAAA;AACA;;IAEA,8BAAA;AACA;QACA,yBAAA;AACA;AAEA;QACA,qBAAA;AACA;;IAEA,kEAAA;AACA;QACA,eAAA;QACA,YAAA;QACA,qCAAA;QACA,cAAA;QACA,uBAAA;QACA,sBAAA;QACA,YAAA;QACA,mBAAA,EAAA,sBAAA;AACA;QACA,gBAAA;AACA;YACA,cAAA;AACA","file":"Timeline.vue","sourcesContent":["<template>\n    <div class=\"timeline\">\n        <div class=\"period-tooltip\" v-show=\"tooltipPeriod\" :style=\"{left: tooltipPeriodPos.x+'px', top: tooltipPeriodPos.y+'px'}\">\n            <template v-if=\"tooltipPeriod\">\n              <div v-if=\"tooltipPeriod.snapshot\">\n                {{ $root._('View capture on {date}', {date: tooltipPeriod.snapshot.getTimeDateFormatted()}) }}\n              </div>\n              <div v-else-if=\"tooltipPeriod.snapshotPeriod\">\n                {{ $root._('View capture on {date}', {date: tooltipPeriod.snapshotPeriod.snapshot.getTimeDateFormatted()}) }}\n              </div>\n              <div v-else-if=\"tooltipPeriod.snapshotCount\">\n                {{ $root._(\n                  isTooltipPeriodDayOrHour ? '{capture_text} on {date}':'{capture_text} in {month}', // TODO: split translation into \"in {year}\" and \"in {month}\"\n                  { capture_text: $root._(tooltipPeriod.snapshotCount !== 1 ? '{count} captures' : '{count} capture', {count: tooltipPeriod.snapshotCount}), [isTooltipPeriodDayOrHour ? 'date':'month']: tooltipPeriod.getFullReadableId() } )\n                }}\n              </div>\n            </template>\n        </div>\n        <div v-html=\"'&#x25C0;'\"\n          class=\"arrow previous\"\n          :class=\"{disabled: isScrollZero && !previousPeriod}\"\n          @click=\"scrollPrev\"\n          @keyup.enter=\"scrollPrev\"\n          @dblclick.stop.prevent tabindex=\"0\"></div>\n        <div class=\"scroll\" ref=\"periodScroll\" :class=\"{highlight: highlight}\">\n            <div class=\"periods\" ref=\"periods\">\n                <div v-for=\"subPeriod in period.children\"\n                     :key=\"subPeriod.fullId\"\n                     class=\"period\"\n                     :class=\"{empty: !subPeriod.snapshotCount, highlight: highlightPeriod === subPeriod, 'last-level': !canZoom, 'contains-current-snapshot': containsCurrentSnapshot(subPeriod) }\"\n                >\n                    <div class=\"histo\">\n                        <div class=\"line\"\n                             v-for=\"histoPeriod in subPeriod.children\"\n                             :key=\"histoPeriod.fullId\"\n                             :style=\"{height: getHistoLineHeight(histoPeriod.snapshotCount)}\"\n                             :class=\"{'has-single-snapshot': histoPeriod.snapshotCount === 1, 'contains-current-snapshot': containsCurrentSnapshot(histoPeriod)}\"\n                             @click=\"changePeriod(histoPeriod, $event)\"\n                             @keyup.enter=\"changePeriod(histoPeriod, $event)\"\n                             @mouseover=\"setTooltipPeriod(histoPeriod, $event)\"\n                             @mouseout=\"setTooltipPeriod(null, $event)\"\n                             tabindex=\"0\"\n                        >\n                        </div>\n                    </div>\n                    <div class=\"inner\"\n                         :class=\"{'has-single-snapshot': subPeriod.snapshotCount === 1}\"\n                         @click=\"changePeriod(subPeriod, $event)\"\n                         @keyup.enter=\"changePeriod(histoPeriod, $event)\"\n                         @mouseover=\"setTooltipPeriod(subPeriod, $event)\"\n                         @mouseout=\"setTooltipPeriod(null, $event)\"\n                         tabindex=\"0\"\n                    >\n                        <div class=\"label\">\n                          {{subPeriod.getReadableId()}}\n                        </div>\n                    </div>\n                </div>\n            </div>\n        </div>\n        <div\n          v-html=\"'&#x25B6;'\"\n          class=\"arrow next\"\n          :class=\"{disabled: isScrollMax && !nextPeriod}\"\n          @click=\"scrollNext\"\n          @keyup.enter=\"scrollNext\"\n          @dblclick.stop.prevent tabindex=\"0\"></div>\n    </div>\n</template>\n\n<script>\nimport { PywbPeriod } from \"../model.js\";\n\nexport default{\n  props: {\n    period: { required: true },\n    currentSnapshot: { required: false, default: null},\n    highlight: { required: false, default: false},\n    stayWithinPeriod: { required: false, default: false},\n    maxZoomLevel: { required: false, default: PywbPeriod.Type.snapshot}\n  },\n  data: function() {\n    return {\n      highlightPeriod: null,\n      previousPeriod: null,\n      nextPeriod: null,\n      isScrollZero: true,\n      isScrollMax: true,\n      tooltipPeriod: null,\n      tooltipPeriodPos: {x:0,y:0}\n    };\n  },\n  created: function() {\n    this.addEmptySubPeriods();\n  },\n  mounted: function() {\n    this.$refs.periods._computedStyle = window.getComputedStyle(this.$refs.periods);\n    this.$refs.periodScroll._computedStyle = window.getComputedStyle(this.$refs.periodScroll);\n    this.$watch(\"period\", this.onPeriodChanged);\n\n    this.$refs.periodScroll.addEventListener(\"scroll\", this.updateScrollArrows);\n    window.addEventListener(\"resize\", this.updateScrollArrows);\n    this.updateScrollArrows();\n  },\n  computed: {\n    // this determins which the last zoom level is before we go straight to showing snapshot\n    canZoom() {\n      return this.period.type < this.maxZoomLevel;\n    },\n    isTooltipPeriodDayOrHour() {\n      return this.tooltipPeriod.type >= PywbPeriod.Type.day;\n    },\n    iContainCurrentSnapshot() {\n      return this.currentSnapshot && this.period.contains(this.currentSnapshot);\n    }\n  },\n  updated() {\n    // do something on update\n  },\n  methods: {\n    containsCurrentSnapshot(period) {\n      return this.iContainCurrentSnapshot && period.contains(this.currentSnapshot);\n    },\n    addEmptySubPeriods() {\n      this.period.fillEmptyChildPeriods(true);\n    },\n    updateScrollArrows() {\n      this.period.scroll = this.$refs.periodScroll.scrollLeft;\n      const maxScroll = parseInt(this.$refs.periods._computedStyle.width) - parseInt(this.$refs.periodScroll._computedStyle.width);\n      this.isScrollZero = !this.period.scroll; // if 0, then true (we are at scroll zero)\n      this.isScrollMax = Math.abs(maxScroll - this.period.scroll) < 5;\n    },\n    restoreScroll() {\n      this.$refs.periodScroll.scrollLeft = this.period.scroll;\n    },\n    scrollNext: function () {\n      if (this.isScrollMax) {\n        if (this.nextPeriod) {\n          this.$emit(\"goto-period\", this.nextPeriod, true /* onlyZoomToPeriod */);\n        }\n      } else {\n        this.$refs.periodScroll.scrollLeft += 30;\n      }\n    },\n    scrollPrev: function () {\n      if (this.isScrollZero) {\n        if (this.previousPeriod) {\n          this.$emit(\"goto-period\", this.previousPeriod, true /* onlyZoomToPeriod */);\n        }\n      } else {\n        this.$refs.periodScroll.scrollLeft -= 30;\n      }\n    },\n    getTimeFormatted: function(date) {\n      return (date.hour < 13 ? date.hour : (date.hour % 12)) + \":\" + ((date.minute < 10 ? \"0\":\"\")+date.minute) + \" \" + (date.hour < 12 ? \"am\":\"pm\");\n    },\n    getHistoLineHeight: function(value) {\n      const percent = Math.ceil((value/this.period.maxGrandchildSnapshotCount) * 100);\n      return (percent ? (5 + Math.ceil(percent*.95)) : 0) + \"%\";\n      // return percent + '%';\n    },\n    changePeriod(period, $event) {\n      // if not empty\n      if (period.snapshotCount) {\n        let periodToChangeTo = null;\n        // if contains a single snapshot only, navigate to snapshot (load snapshot in FRAME, do not ZOOM IN)\n        if (period.snapshot) {\n          // if period is at level \"snapshot\" (no more children), send period, else send the child period, a reference to which is stored (by data/model layer) in the current period; App event needs a period to be passed (cannot pass in snapshot object itself)\n          if (period.type === PywbPeriod.Type.snapshot) {\n            periodToChangeTo = period;\n          } else if (period.snapshotPeriod) {\n            periodToChangeTo = period.snapshotPeriod;\n          }\n        } else {\n        // if contains mulitple snapshots,\n          // zoom if ZOOM level is day or less, OR if period contain TOO MANY (>10)\n          if (this.canZoom) {\n            periodToChangeTo = period;\n          }\n        }\n\n        // if we selected a period to go to, emit event\n        if (periodToChangeTo) {\n          this.$emit(\"goto-period\", periodToChangeTo);\n        }\n      }\n      $event.stopPropagation();\n      return false;\n    },\n    onPeriodChanged(newPeriod, oldPeriod) {\n      this.addEmptySubPeriods();\n      const previousPeriod = this.period.getPrevious();\n      const nextPeriod = this.period.getNext();\n      if (!this.stayWithinPeriod || this.stayWithinPeriod.contains(previousPeriod)) {\n        this.previousPeriod = previousPeriod;\n      }\n      if (!this.stayWithinPeriod || this.stayWithinPeriod.contains(nextPeriod)) {\n        this.nextPeriod = nextPeriod;\n      }\n\n      // detect if going up level of period (new period type should be in old period parents)\n      if (oldPeriod && oldPeriod.type - newPeriod.type > 0) {\n        let highlightPeriod = oldPeriod;\n        for (let i=oldPeriod.type - newPeriod.type; i > 1; i--) {\n          highlightPeriod = highlightPeriod.parent;\n        }\n        this.highlightPeriod = highlightPeriod;\n        setTimeout((function() {\n          this.highlightPeriod = null;\n        }).bind(this), 2000);\n      }\n      setTimeout((function() {\n        this.restoreScroll();\n        this.updateScrollArrows();\n      }).bind(this), 1);\n    },\n    setTooltipPeriod(period, event) {\n      if (!period || !period.snapshotCount) {\n        this.tooltipPeriod = null;\n        return;\n      }\n      this.tooltipPeriod = period;\n\n      this.$nextTick(function() {\n        const tooltipContentsEl = document.querySelector('.period-tooltip div');\n        if (!tooltipContentsEl) {\n          return;\n        }\n\n        const periodTooltipStyle = window.getComputedStyle(tooltipContentsEl);\n        const tooltipWidth = parseInt(periodTooltipStyle.width);\n        const tooltipHeight = parseInt(periodTooltipStyle.height);\n        const spacing = 10;\n        if (window.innerWidth < event.x + (spacing*2) + tooltipWidth) {\n          this.tooltipPeriodPos.x = event.x - (tooltipWidth + spacing);\n        } else {\n          this.tooltipPeriodPos.x = event.x + spacing;\n        }\n        this.tooltipPeriodPos.y = event.y - (spacing + tooltipHeight);\n      });\n      event.stopPropagation();\n      return false;\n    }\n  }\n};\n</script>\n\n\n<style>\n    .timeline {\n        position: relative;\n        display: flex;\n        width: auto;\n        height: 60px;\n        margin: 5px;\n        justify-content: left;\n    }\n\n    .timeline .id {\n        display: inline-block;\n        font-size: 30px;\n    }\n    .timeline .arrow {\n        display: inline-block;\n        width: 20px;\n        font-size: 20px; /* font-size = width of arrow, as it UTF char */\n        line-height: 60px;\n        vertical-align: top;\n        cursor: pointer;\n    }\n    .timeline .arrow.previous {\n    }\n    .timeline .arrow.next {\n    }\n    .timeline .arrow.disabled, .timeline .arrow.disabled:hover {\n        /*color: lightgray;*/\n        background-color: transparent;\n        /*cursor: not-allowed;*/\n        visibility: hidden;\n    }\n    .timeline .arrow:hover {\n        background-color: antiquewhite;\n        color: firebrick;\n    }\n\n    .timeline .scroll {\n        position: relative;\n        display: inline-block;\n        width: 100%; /* */\n        height: 100%;\n\n        /* maker scrollable horizontally */\n        overflow-x: scroll;\n        overflow-y: hidden;\n        white-space: nowrap;\n        scroll-behavior: smooth;\n\n        text-align: center;\n\n        transition: background-color 500ms ease-in;\n    }\n    /* hide scroll bar */\n    .timeline .scroll::-webkit-scrollbar {\n        display: none;\n    }\n    /* highlight the scroll period: usually triggered from root app */\n    .timeline .scroll.highlight {\n        background-color: #fff7ce;\n    }\n    .timeline .scroll .periods {\n        display: flex;\n        justify-content: space-between;\n        height: 100%;\n        width: 100%;\n        min-width: 600px;\n    }\n\n\n    .timeline .period {\n        flex-grow: 1;\n        position: relative;\n        display: inline-block;\n        height: 100%;\n        /* line-height: 80px; /* use to center middle vertically */\n        white-space: normal;\n        vertical-align: top;\n        text-align: center;\n        background-color: transparent;\n\n        transition: background-color 500ms ease-in-out;\n    }\n    /* 1st period period child el */\n    .timeline .period:nth-child(1) {\n        /*border-left: 1px solid white; !* has left border; all other periods have right border *!*/\n    }\n\n    .timeline .period:hover {\n        background-color: #eeeeee;\n    }\n    .timeline .period.contains-current-snapshot, .timeline .period.contains-current-snapshot:hover {\n        background-color: #f7def4;\n    }\n\n    /* empty period */\n    .timeline .period.empty {\n        color: #aaa;\n        /*background-color: transparent;*/\n    }\n    /* highlighted period */\n    .timeline .period.highlight {\n        background-color: cyan;\n    }\n\n    .timeline .period .inner {\n        display: block;\n        position: absolute;\n        bottom: 0;\n        left: 0;\n        width: 100%;\n        height: 20px;\n        background-color: white;\n        border-top: 1px solid gray;\n        white-space: nowrap;\n        cursor: zoom-in;\n    }\n    .timeline .period .inner.has-single-snapshot {\n      cursor: pointer;\n    }\n    .timeline .period.last-level .inner, .timeline .period.empty .inner {\n      cursor: default;\n    }\n\n    .timeline .period .label {\n      width: 100%;\n      font-weight: bold;\n      font-size: 14px;\n      transition: background-color 500ms ease-in;\n    }\n    .timeline .period:hover .label {\n      position: absolute;\n      z-index: 20;\n      background-color: lightgrey;\n    }\n\n    .timeline .period .histo {\n        display: flex;\n        position: absolute;\n        top: 1px;\n        left: 0;\n        width: 100%;\n        height: 39px;\n        align-items: flex-end;\n        justify-content: space-between;\n        text-align: left;\n    }\n\n    .timeline .period .histo .line {\n        position: relative;\n        flex-grow: 1;\n        display: inline-block;\n        background-color: #a6cdf5;\n        margin: 0;\n        padding: 0;\n        cursor: zoom-in;\n    }\n    .timeline .period .histo .line.has-single-snapshot {\n      cursor: pointer;\n    }\n\n    /* Last level period histogram spaces things evenly */\n    .timeline .period.last-level .histo {\n        justify-content: space-around;\n    }\n\n    /* Last level period histogram lines do not grow, but are fixed width/margin */\n    .timeline .period.last-level .histo .line {\n        flex-grow: unset;\n        width: 5px;\n        margin-left: 2px;\n    }\n\n        /* update line color on hover*/\n        .timeline .period .histo .line:hover {\n            background-color: #f5a6eb;\n        }\n\n        .timeline .period .histo .line.contains-current-snapshot {\n            background-color: red;\n        }\n\n        /* Period that contains ONE snapshot only will show snapshot info*/\n        .timeline .period-tooltip {\n            position: fixed;\n            z-index: 100;\n            /*left or right set programmatically*/\n            display: block;\n            background-color: white;\n            border: 1px solid gray;\n            padding: 2px;\n            white-space: nowrap; /*no wrapping allowed*/\n        }\n            /*show on hover*/\n            .timeline .period-tooltip.show {\n                display: block;\n            }\n\n</style>\n"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__$6 = undefined;
    /* module identifier */
    const __vue_module_identifier__$6 = undefined;
    /* functional template */
    const __vue_is_functional_template__$6 = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__$6 = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__$6, staticRenderFns: __vue_staticRenderFns__$6 },
      __vue_inject_styles__$6,
      __vue_script__$6,
      __vue_scope_id__$6,
      __vue_is_functional_template__$6,
      __vue_module_identifier__$6,
      false,
      createInjector,
      undefined,
      undefined
    );

  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //
  //

  var script$5 = {
    props: {
      period: {
        required: true
      }
    },
    computed: {
      parents: function() {
        return this.period.getParents();
      }
    },
    methods: {
      getPeriodZoomOutText(period) {
        return 'Zoom out to '+period.getReadableId(true)+ ' ('+period.snapshotCount+' captures)';
      },
      changePeriod(period) {
        if (period.snapshotCount) {
          this.$emit("goto-period", period);
        }
      },
    }
  };

  /* script */
  const __vue_script__$5 = script$5;

  /* template */
  var __vue_render__$5 = function() {
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c(
      "div",
      { staticClass: "breadcrumbs" },
      [
        _vm.parents.length
          ? [
              _c("span", { staticClass: "item" }, [
                _c(
                  "span",
                  {
                    staticClass: "goto",
                    attrs: {
                      title: _vm.getPeriodZoomOutText(_vm.parents[0]),
                      tabindex: "1"
                    },
                    on: {
                      click: function($event) {
                        return _vm.changePeriod(_vm.parents[0])
                      },
                      keyup: function($event) {
                        if (
                          !$event.type.indexOf("key") &&
                          _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
                        ) {
                          return null
                        }
                        return _vm.changePeriod(_vm.parents[0])
                      }
                    }
                  },
                  [
                    _c("img", {
                      attrs: { src: "/static/zoom-out-icon-333316.png" }
                    }),
                    _vm._v(
                      " " +
                        _vm._s(_vm.parents[0].getReadableId(true)) +
                        "\n            "
                    )
                  ]
                )
              ]),
              _vm._v("\n        >\n        "),
              _vm._l(_vm.parents, function(parent, i) {
                return i > 0
                  ? _c("span", { key: parent.id, staticClass: "item" }, [
                      _c(
                        "span",
                        {
                          staticClass: "goto",
                          attrs: {
                            title: _vm.getPeriodZoomOutText(parent),
                            tabindex: "1"
                          },
                          on: {
                            click: function($event) {
                              return _vm.changePeriod(parent)
                            },
                            keyup: function($event) {
                              if (
                                !$event.type.indexOf("key") &&
                                _vm._k(
                                  $event.keyCode,
                                  "enter",
                                  13,
                                  $event.key,
                                  "Enter"
                                )
                              ) {
                                return null
                              }
                              return _vm.changePeriod(parent)
                            }
                          }
                        },
                        [
                          _vm._v(
                            "\n              " +
                              _vm._s(parent.getReadableId(true)) +
                              "\n            "
                          )
                        ]
                      )
                    ])
                  : _vm._e()
              })
            ]
          : _vm._e(),
        _vm._v(" "),
        _c("span", { staticClass: "item" }, [
          _c("span", { staticClass: "current" }, [
            _vm._v(_vm._s(_vm.period.getReadableId(true)))
          ]),
          _vm._v(" "),
          _c("span", { staticClass: "count" }, [
            _vm._v(
              "(" +
                _vm._s(
                  _vm.$root._(
                    _vm.period.snapshotCount !== 1
                      ? "{count} captures"
                      : "{count} capture",
                    { count: _vm.period.snapshotCount }
                  )
                ) +
                ")"
            )
          ])
        ])
      ],
      2
    )
  };
  var __vue_staticRenderFns__$5 = [];
  __vue_render__$5._withStripped = true;

    /* style */
    const __vue_inject_styles__$5 = function (inject) {
      if (!inject) return
      inject("data-v-7ed1ae1d_0", { source: "\n.breadcrumbs {\n  text-align: center;\n}\n.breadcrumbs .item {\n    position: relative;\n    display: inline;\n    margin: 0 2px 0 0;\n    font-size: inherit;\n}\n.breadcrumbs .count {\n    /*vertical-align: middle;*/\n    font-size: inherit;\n}\n.breadcrumbs .item .goto {\n    display: inline-block;\n    margin: 1px;\n    padding: 1px;\n    cursor: zoom-out;\n    border-radius: 5px;\n    background-color: #eeeeee;\n}\n.breadcrumbs .item .goto:hover {\n    background-color: #a6cdf5;\n}\n.breadcrumbs .item .goto img {\n  height: 15px;\n}\n.breadcrumbs .item.snapshot {\n    display: block;\n}\n\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/components/TimelineBreadcrumbs.vue"],"names":[],"mappings":";AA0DA;EACA,kBAAA;AACA;AACA;IACA,kBAAA;IACA,eAAA;IACA,iBAAA;IACA,kBAAA;AACA;AACA;IACA,0BAAA;IACA,kBAAA;AACA;AAEA;IACA,qBAAA;IACA,WAAA;IACA,YAAA;IACA,gBAAA;IACA,kBAAA;IACA,yBAAA;AACA;AACA;IACA,yBAAA;AACA;AACA;EACA,YAAA;AACA;AAEA;IACA,cAAA;AACA","file":"TimelineBreadcrumbs.vue","sourcesContent":["<template>\n    <div class=\"breadcrumbs\">\n        <template v-if=\"parents.length\">\n            <span class=\"item\">\n                <span\n                    class=\"goto\"\n                    @click=\"changePeriod(parents[0])\"\n                    @keyup.enter=\"changePeriod(parents[0])\"\n                    :title=\"getPeriodZoomOutText(parents[0])\"\n                    tabindex=\"1\">\n                  <img src=\"/static/zoom-out-icon-333316.png\" /> {{parents[0].getReadableId(true)}}\n                </span>\n            </span>\n            &gt;\n            <span v-for=\"(parent,i) in parents\" :key=\"parent.id\" class=\"item\" v-if=\"i > 0\">\n                <span\n                    class=\"goto\"\n                    @click=\"changePeriod(parent)\"\n                    @keyup.enter=\"changePeriod(parent)\"\n                    :title=\"getPeriodZoomOutText(parent)\"\n                    tabindex=\"1\">\n                  {{parent.getReadableId(true)}}\n                </span>\n            </span>\n        </template>\n        <span class=\"item\">\n            <span class=\"current\">{{period.getReadableId(true)}}</span>\n            <span class=\"count\">({{ $root._(period.snapshotCount !== 1 ? '{count} captures':'{count} capture', {count: period.snapshotCount}) }})</span>\n        </span>\n    </div>\n</template>\n\n<script>\nexport default {\n  props: {\n    period: {\n      required: true\n    }\n  },\n  computed: {\n    parents: function() {\n      return this.period.getParents();\n    }\n  },\n  methods: {\n    getPeriodZoomOutText(period) {\n      return 'Zoom out to '+period.getReadableId(true)+ ' ('+period.snapshotCount+' captures)';\n    },\n    changePeriod(period) {\n      if (period.snapshotCount) {\n        this.$emit(\"goto-period\", period);\n      }\n    },\n  }\n};\n</script>\n\n<style>\n    .breadcrumbs {\n      text-align: center;\n    }\n    .breadcrumbs .item {\n        position: relative;\n        display: inline;\n        margin: 0 2px 0 0;\n        font-size: inherit;\n    }\n    .breadcrumbs .count {\n        /*vertical-align: middle;*/\n        font-size: inherit;\n    }\n\n    .breadcrumbs .item .goto {\n        display: inline-block;\n        margin: 1px;\n        padding: 1px;\n        cursor: zoom-out;\n        border-radius: 5px;\n        background-color: #eeeeee;\n     }\n    .breadcrumbs .item .goto:hover {\n        background-color: #a6cdf5;\n    }\n    .breadcrumbs .item .goto img {\n      height: 15px;\n    }\n\n    .breadcrumbs .item.snapshot {\n        display: block;\n    }\n\n</style>\n"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__$5 = undefined;
    /* module identifier */
    const __vue_module_identifier__$5 = undefined;
    /* functional template */
    const __vue_is_functional_template__$5 = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__$5 = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__$5, staticRenderFns: __vue_staticRenderFns__$5 },
      __vue_inject_styles__$5,
      __vue_script__$5,
      __vue_scope_id__$5,
      __vue_is_functional_template__$5,
      __vue_module_identifier__$5,
      false,
      createInjector,
      undefined,
      undefined
    );

  //

  var script$4 = {
    props: ["month", "year", "isCurrent", "yearContainsCurrentSnapshot", "currentSnapshot"],
    data: function() {
      return {
        maxInDay: 0,
        daySize: 30,
      };
    },
    computed: {
      dayInitials() {
        return PywbI18N.instance.getWeekDays().map(d => d.substr(0,1));
      },
      dayStyle() {
        const s = this.daySize;
        return `height: ${s}px; width: ${s}px; line-height: ${s}px`;
      },
      days() {
        if (!this.month) {
          return [];
        }
        const days = [];
        // Get days in month, and days in the complete weeks before first day and after last day
        const [firstDay, lastDay] = this.month.getChildrenRange();
        const daysBeforeFirst = (7 + (new Date(this.year.id, this.month.id-1, firstDay)).getDay() - PywbI18N.firstDayOfWeek) % 7;
        const daysAfterLastDay = (6 - (new Date(this.year.id, this.month.id-1, lastDay)).getDay() + PywbI18N.firstDayOfWeek) % 7;
        for(let i=0; i<daysBeforeFirst; i++) {
          days.push(null);
        }
        const hasChildren = !!this.month.children.length;
        for(let i=0; i<lastDay; i++) {
          days.push(hasChildren ? this.month.children[i] : null);
        }
        for(let i=0; i<daysAfterLastDay; i++) {
          days.push(null);
        }
        return days;
      },
      containsCurrentSnapshot() {
        return this.currentSnapshot &&
            this.month.contains(this.currentSnapshot);
      }
    },
    methods: {
      _(id, embeddedVariableStrings=null) {
        return PywbI18N.instance.getText(id, embeddedVariableStrings);
      },
      getLongMonthName(id) {
        return PywbI18N.instance.getMonth(id);
      },
      gotoDay(day, event) {
        if (!day || !day.snapshotCount) {
          return;
        }
        // upon doing to day, tell timeline to highlight itself
        // this.$root.timelineHighlight = true;
        this.$emit("show-day-timeline", day, event);
      },
      getDayCountCircleStyle(snapshotCount) {
        const size = Math.ceil((snapshotCount/this.year.maxGrandchildSnapshotCount) * this.daySize);
        const scaledSize = size ? (this.daySize*.3 + Math.ceil(size*.7)) : 0;
        const margin = (this.daySize-scaledSize)/2;

        // TEMPORARILY DISABLE AUTO-HUE calculation as it is contributing to better understand of data
        // color hue should go form blue (240deg) to red (360deg)
        // const colorHue = Math.ceil((snapshotCount/this.year.maxGrandchildSnapshotCount) * (360-240));
        // const scaledColorHue = size ? (240 + colorHue) : 240;
        // background-color: hsl(${scaledColorHue}, 100%, 50%, .2)

        return `width: ${scaledSize}px; height: ${scaledSize}px; top: ${margin}px; left: ${margin}px; border-radius: ${scaledSize/2}px;`;
      },
      dayContainsCurrentSnapshot(day) {
        return !!day && day.snapshotCount > 0 && this.containsCurrentSnapshot && day.contains(this.currentSnapshot);
      }
    }
  };

  /* script */
  const __vue_script__$4 = script$4;

  /* template */
  var __vue_render__$4 = function() {
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c(
      "div",
      {
        staticClass: "calendar-month",
        class: {
          current: _vm.isCurrent,
          "contains-current-snapshot": _vm.containsCurrentSnapshot
        }
      },
      [
        _c("h3", [
          _vm._v(_vm._s(_vm.getLongMonthName(_vm.month.id)) + " "),
          _vm.month.snapshotCount
            ? _c("span", [_vm._v("(" + _vm._s(_vm.month.snapshotCount) + ")")])
            : _vm._e()
        ]),
        _vm._v(" "),
        _vm._l(_vm.dayInitials, function(dayInitial) {
          return _c("span", { staticClass: "day", style: _vm.dayStyle }, [
            _vm._v(_vm._s(dayInitial))
          ])
        }),
        _c("br"),
        _vm._v(" "),
        _vm._l(_vm.days, function(day, i) {
          return _c("span", [
            i && i % 7 === 0 ? _c("br") : _vm._e(),
            _c(
              "span",
              {
                staticClass: "day",
                class: {
                  empty: !day || !day.snapshotCount,
                  "contains-current-snapshot": _vm.dayContainsCurrentSnapshot(day)
                },
                style: _vm.dayStyle,
                on: {
                  click: function($event) {
                    return _vm.gotoDay(day, $event)
                  },
                  keyup: function($event) {
                    if (!$event.type.indexOf("key") && $event.keyCode !== 13) {
                      return null
                    }
                    return _vm.gotoDay(day, $event)
                  }
                }
              },
              [
                day
                  ? [
                      day.snapshotCount
                        ? _c("span", {
                            staticClass: "size",
                            style: _vm.getDayCountCircleStyle(day.snapshotCount),
                            attrs: { tabindex: "0" }
                          })
                        : _vm._e(),
                      _c(
                        "span",
                        {
                          staticClass: "day-id",
                          class: { empty: !day.snapshotCount }
                        },
                        [_vm._v(_vm._s(day.id))]
                      ),
                      day.snapshotCount
                        ? _c("span", { staticClass: "count" }, [
                            _vm._v(
                              _vm._s(
                                _vm.$root._(
                                  day.snapshotCount !== 1
                                    ? "{count} captures"
                                    : "{count} capture",
                                  { count: day.snapshotCount }
                                )
                              )
                            )
                          ])
                        : _vm._e()
                    ]
                  : void 0
              ],
              2
            )
          ])
        })
      ],
      2
    )
  };
  var __vue_staticRenderFns__$4 = [];
  __vue_render__$4._withStripped = true;

    /* style */
    const __vue_inject_styles__$4 = function (inject) {
      if (!inject) return
      inject("data-v-d0d0e1ec_0", { source: "\n.calendar-month {\n    position: relative;\n    display: inline-block;\n    padding: 5px;\n    margin: 0;\n    height: 260px;\n    width: 220px;\n    text-align: center;\n    vertical-align: top;\n    box-sizing: content-box;\n    border-radius: 10px;\n}\n.calendar-month:hover {\n    background-color: #F5F5F5;\n}\n.calendar-month.current {\n    background-color: #fff7ce;\n}\n.calendar-month.contains-current-snapshot {\n    border: solid 1px red;\n}\n.calendar-month > h3 {\n    margin: 0;\n    font-size: 16px;\n}\n.calendar-month > .empty {\n    position: absolute;\n    top: 45%;\n    width: 100%;\n    color: #454545;\n}\n.calendar-month .day {\n    position: relative;\n    display: inline-block;\n    margin: 0;\n    text-align: center;\n}\n.calendar-month .day.empty {\n    color: #454545;\n}\n.calendar-month .day .count {\n    display: none;\n    position: absolute;\n    bottom: 80%;\n    left: 80%;\n    line-height: 1; /* reset to normal */\n    padding: 3px;\n    border-radius: 10px;\n    border-bottom-left-radius: 0;\n    border: 1px solid gray;\n    background-color: white;\n    z-index: 30;\n    white-space: nowrap;\n}\n.calendar-month .day:hover .count {\n    display: block;\n}\n.calendar-month .day .size {\n    position: absolute;\n    box-sizing: border-box;\n    background-color: rgba(166, 205, 245, .85);\n    z-index: 10;\n}\n.calendar-month .day.contains-current-snapshot .size {\n  background-color: rgba(255, 100, 100, .85);\n}\n.calendar-month .day .day-id {\n    position: absolute;\n    top: 0;\n    left: 0;\n    z-index: 11;\n\n    display: inline-block;\n    width: 100%;\n    text-align: center;\n\n    color: black;\n}\n.calendar-month .day .day-id.empty {\n  color: #454545 !important;\n}\n.calendar-month .day:hover .size {\n    border: 1px solid black;\n}\n.calendar-month .day:hover {\n    cursor: zoom-in;\n}\n.calendar-month .day.empty:hover {\n    cursor: not-allowed;\n}\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/components/CalendarMonth.vue"],"names":[],"mappings":";AACA;IACA,kBAAA;IACA,qBAAA;IACA,YAAA;IACA,SAAA;IACA,aAAA;IACA,YAAA;IACA,kBAAA;IACA,mBAAA;IACA,uBAAA;IACA,mBAAA;AACA;AACA;IACA,yBAAA;AACA;AACA;IACA,yBAAA;AACA;AACA;IACA,qBAAA;AACA;AACA;IACA,SAAA;IACA,eAAA;AACA;AACA;IACA,kBAAA;IACA,QAAA;IACA,WAAA;IACA,cAAA;AACA;AACA;IACA,kBAAA;IACA,qBAAA;IACA,SAAA;IACA,kBAAA;AACA;AACA;IACA,cAAA;AACA;AACA;IACA,aAAA;IACA,kBAAA;IACA,WAAA;IACA,SAAA;IACA,cAAA,EAAA,oBAAA;IACA,YAAA;IACA,mBAAA;IACA,4BAAA;IACA,sBAAA;IACA,uBAAA;IACA,WAAA;IACA,mBAAA;AACA;AACA;IACA,cAAA;AACA;AACA;IACA,kBAAA;IACA,sBAAA;IACA,0CAAA;IACA,WAAA;AACA;AACA;EACA,0CAAA;AACA;AAEA;IACA,kBAAA;IACA,MAAA;IACA,OAAA;IACA,WAAA;;IAEA,qBAAA;IACA,WAAA;IACA,kBAAA;;IAEA,YAAA;AACA;AACA;EACA,yBAAA;AACA;AACA;IACA,uBAAA;AACA;AACA;IACA,eAAA;AACA;AACA;IACA,mBAAA;AACA","file":"CalendarMonth.vue","sourcesContent":["<style>\n    .calendar-month {\n        position: relative;\n        display: inline-block;\n        padding: 5px;\n        margin: 0;\n        height: 260px;\n        width: 220px;\n        text-align: center;\n        vertical-align: top;\n        box-sizing: content-box;\n        border-radius: 10px;\n    }\n    .calendar-month:hover {\n        background-color: #F5F5F5;\n    }\n    .calendar-month.current {\n        background-color: #fff7ce;\n    }\n    .calendar-month.contains-current-snapshot {\n        border: solid 1px red;\n    }\n    .calendar-month > h3 {\n        margin: 0;\n        font-size: 16px;\n    }\n    .calendar-month > .empty {\n        position: absolute;\n        top: 45%;\n        width: 100%;\n        color: #454545;\n    }\n    .calendar-month .day {\n        position: relative;\n        display: inline-block;\n        margin: 0;\n        text-align: center;\n    }\n    .calendar-month .day.empty {\n        color: #454545;\n    }\n    .calendar-month .day .count {\n        display: none;\n        position: absolute;\n        bottom: 80%;\n        left: 80%;\n        line-height: 1; /* reset to normal */\n        padding: 3px;\n        border-radius: 10px;\n        border-bottom-left-radius: 0;\n        border: 1px solid gray;\n        background-color: white;\n        z-index: 30;\n        white-space: nowrap;\n    }\n    .calendar-month .day:hover .count {\n        display: block;\n    }\n    .calendar-month .day .size {\n        position: absolute;\n        box-sizing: border-box;\n        background-color: rgba(166, 205, 245, .85);\n        z-index: 10;\n    }\n    .calendar-month .day.contains-current-snapshot .size {\n      background-color: rgba(255, 100, 100, .85);\n    }\n\n    .calendar-month .day .day-id {\n        position: absolute;\n        top: 0;\n        left: 0;\n        z-index: 11;\n\n        display: inline-block;\n        width: 100%;\n        text-align: center;\n\n        color: black;\n    }\n    .calendar-month .day .day-id.empty {\n      color: #454545 !important;\n    }\n    .calendar-month .day:hover .size {\n        border: 1px solid black;\n    }\n    .calendar-month .day:hover {\n        cursor: zoom-in;\n    }\n    .calendar-month .day.empty:hover {\n        cursor: not-allowed;\n    }\n</style>\n\n<template>\n  <div class=\"calendar-month\" :class=\"{current: isCurrent, 'contains-current-snapshot': containsCurrentSnapshot}\">\n    <h3>{{getLongMonthName(month.id)}} <span v-if=\"month.snapshotCount\">({{ month.snapshotCount }})</span></h3>\n    <span v-for=\"(dayInitial) in dayInitials\" class=\"day\" :style=\"dayStyle\">{{dayInitial}}</span><br/>\n    <span v-for=\"(day,i) in days\"><br v-if=\"i && i % 7===0\"/><span class=\"day\" :class=\"{empty: !day || !day.snapshotCount, 'contains-current-snapshot':dayContainsCurrentSnapshot(day)}\" :style=\"dayStyle\"  @click=\"gotoDay(day, $event)\" @keyup.13=\"gotoDay(day, $event)\"><template v-if=\"day\"><span class=\"size\" v-if=\"day.snapshotCount\" :style=\"getDayCountCircleStyle(day.snapshotCount)\" tabindex=\"0\"> </span><span class=\"day-id\" :class=\"{empty: !day.snapshotCount}\">{{day.id}}</span><span v-if=\"day.snapshotCount\" class=\"count\">{{ $root._(day.snapshotCount !== 1 ? '{count} captures':'{count} capture', {count: day.snapshotCount}) }}</span></template><template v-else v-html=\"'&nbsp;'\"></template></span></span>\n  </div>\n</template>\n\n<script>\nimport {PywbI18N} from \"../i18n\";\n\nexport default {\n  props: [\"month\", \"year\", \"isCurrent\", \"yearContainsCurrentSnapshot\", \"currentSnapshot\"],\n  data: function() {\n    return {\n      maxInDay: 0,\n      daySize: 30,\n    };\n  },\n  computed: {\n    dayInitials() {\n      return PywbI18N.instance.getWeekDays().map(d => d.substr(0,1));\n    },\n    dayStyle() {\n      const s = this.daySize;\n      return `height: ${s}px; width: ${s}px; line-height: ${s}px`;\n    },\n    days() {\n      if (!this.month) {\n        return [];\n      }\n      const days = [];\n      // Get days in month, and days in the complete weeks before first day and after last day\n      const [firstDay, lastDay] = this.month.getChildrenRange();\n      const daysBeforeFirst = (7 + (new Date(this.year.id, this.month.id-1, firstDay)).getDay() - PywbI18N.firstDayOfWeek) % 7;\n      const daysAfterLastDay = (6 - (new Date(this.year.id, this.month.id-1, lastDay)).getDay() + PywbI18N.firstDayOfWeek) % 7;\n      for(let i=0; i<daysBeforeFirst; i++) {\n        days.push(null);\n      }\n      const hasChildren = !!this.month.children.length;\n      for(let i=0; i<lastDay; i++) {\n        days.push(hasChildren ? this.month.children[i] : null);\n      }\n      for(let i=0; i<daysAfterLastDay; i++) {\n        days.push(null);\n      }\n      return days;\n    },\n    containsCurrentSnapshot() {\n      return this.currentSnapshot &&\n          this.month.contains(this.currentSnapshot);\n    }\n  },\n  methods: {\n    _(id, embeddedVariableStrings=null) {\n      return PywbI18N.instance.getText(id, embeddedVariableStrings);\n    },\n    getLongMonthName(id) {\n      return PywbI18N.instance.getMonth(id);\n    },\n    gotoDay(day, event) {\n      if (!day || !day.snapshotCount) {\n        return;\n      }\n      // upon doing to day, tell timeline to highlight itself\n      // this.$root.timelineHighlight = true;\n      this.$emit(\"show-day-timeline\", day, event);\n    },\n    getDayCountCircleStyle(snapshotCount) {\n      const size = Math.ceil((snapshotCount/this.year.maxGrandchildSnapshotCount) * this.daySize);\n      const scaledSize = size ? (this.daySize*.3 + Math.ceil(size*.7)) : 0;\n      const margin = (this.daySize-scaledSize)/2;\n\n      // TEMPORARILY DISABLE AUTO-HUE calculation as it is contributing to better understand of data\n      // color hue should go form blue (240deg) to red (360deg)\n      // const colorHue = Math.ceil((snapshotCount/this.year.maxGrandchildSnapshotCount) * (360-240));\n      // const scaledColorHue = size ? (240 + colorHue) : 240;\n      // background-color: hsl(${scaledColorHue}, 100%, 50%, .2)\n\n      return `width: ${scaledSize}px; height: ${scaledSize}px; top: ${margin}px; left: ${margin}px; border-radius: ${scaledSize/2}px;`;\n    },\n    dayContainsCurrentSnapshot(day) {\n      return !!day && day.snapshotCount > 0 && this.containsCurrentSnapshot && day.contains(this.currentSnapshot);\n    }\n  }\n};\n</script>\n\n"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__$4 = undefined;
    /* module identifier */
    const __vue_module_identifier__$4 = undefined;
    /* functional template */
    const __vue_is_functional_template__$4 = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__$4 = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__$4, staticRenderFns: __vue_staticRenderFns__$4 },
      __vue_inject_styles__$4,
      __vue_script__$4,
      __vue_scope_id__$4,
      __vue_is_functional_template__$4,
      __vue_module_identifier__$4,
      false,
      createInjector,
      undefined,
      undefined
    );

  //

  var script$3 = {
    name: "TimelineLinear",
    props: ['period', 'currentSnapshot'],
    computed: {
      snapshotPeriods() {
        return this.period.getSnapshotPeriodsFlat();
      },
      containsCurrentSnapshot() {
        return this.currentSnapshot &&
            this.period.contains(this.currentSnapshot);
      },
      displayDate() {
        // replace '-' in date string with '/' so firefox and chrome will return same result
        let dateStringNoSlashes = this.period.fullId.replace(/-/g,'/');
        return new Date(dateStringNoSlashes).toLocaleDateString(PywbI18N.getLocale());
      }
    },
    methods: {
      isCurrentSnapshot(period) {
        if (!!this.currentSnapshot && !!period.snapshot) {
          return this.currentSnapshot && this.currentSnapshot.id === period.snapshot.id;
        }
        return false;
      },
      changePeriod(period) {
        this.$emit("goto-period", period);
      }
    }
  };

  /* script */
  const __vue_script__$3 = script$3;

  /* template */
  var __vue_render__$3 = function() {
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c("div", { staticClass: "timeline-linear" }, [
      _c("div", { staticClass: "title" }, [
        _c("div", [_vm._v(_vm._s(_vm.displayDate))]),
        _vm._v(" "),
        _c("div", [
          _vm._v(
            _vm._s(
              _vm.$root._(
                _vm.period.snapshotCount !== 1
                  ? "{count} captures"
                  : "{count} capture",
                { count: _vm.period.snapshotCount }
              )
            )
          )
        ])
      ]),
      _vm._v(" "),
      _c(
        "div",
        { staticClass: "list" },
        _vm._l(_vm.snapshotPeriods, function(snapshotPeriod) {
          return _c("div", [
            _c(
              "span",
              {
                staticClass: "link",
                attrs: { tabindex: "1" },
                on: {
                  click: function($event) {
                    return _vm.changePeriod(snapshotPeriod)
                  },
                  keyup: function($event) {
                    if (
                      !$event.type.indexOf("key") &&
                      _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
                    ) {
                      return null
                    }
                    return _vm.changePeriod(snapshotPeriod)
                  }
                }
              },
              [
                _vm._v(
                  "\n        " +
                    _vm._s(snapshotPeriod.snapshot.getTimeFormatted()) +
                    "\n      "
                )
              ]
            ),
            _vm._v(" "),
            _vm.isCurrentSnapshot(_vm.period)
              ? _c("span", { staticClass: "current" }, [
                  _vm._v(_vm._s(_vm.$root._("current")))
                ])
              : _vm._e()
          ])
        }),
        0
      )
    ])
  };
  var __vue_staticRenderFns__$3 = [];
  __vue_render__$3._withStripped = true;

    /* style */
    const __vue_inject_styles__$3 = function (inject) {
      if (!inject) return
      inject("data-v-3fb9ef40_0", { source: "\n.timeline-linear[data-v-3fb9ef40] {\n  width: auto;\n  padding: 5px;\n  background-color: white;\n  border: 1px solid gray;\n  border-radius: 5px;\n  z-index: 1100;\n}\n.timeline-linear .list[data-v-3fb9ef40] {\n  max-height: 80vh;\n  min-height: 50px;\n  overflow-x: hidden;\n  overflow-y: auto;\n}\n.timeline-linear .title[data-v-3fb9ef40] {\n  border-bottom: 1px solid black;\n  font-weight: bold;\n  font-family: Arial, sans-serif;\n}\n.timeline-linear .link[data-v-3fb9ef40] {\n  text-decoration: underline;\n  color: darkblue;\n}\n.timeline-linear .link[data-v-3fb9ef40]:hover {\n  color: lightseagreen;\n  cursor: pointer;\n}\n.timeline-linear .current[data-v-3fb9ef40] {\n  background-color: deeppink;\n  color: white;\n  border-radius: 5px;\n}\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/components/TimelineLinear.vue"],"names":[],"mappings":";AAyDA;EACA,WAAA;EACA,YAAA;EACA,uBAAA;EACA,sBAAA;EACA,kBAAA;EACA,aAAA;AACA;AACA;EACA,gBAAA;EACA,gBAAA;EACA,kBAAA;EACA,gBAAA;AACA;AACA;EACA,8BAAA;EACA,iBAAA;EACA,8BAAA;AACA;AACA;EACA,0BAAA;EACA,eAAA;AACA;AACA;EACA,oBAAA;EACA,eAAA;AACA;AACA;EACA,0BAAA;EACA,YAAA;EACA,kBAAA;AACA","file":"TimelineLinear.vue","sourcesContent":["<template>\n  <div class=\"timeline-linear\">\n    <div class=\"title\">\n      <div>{{ displayDate }}</div>\n      <div>{{ $root._(period.snapshotCount !== 1 ? '{count} captures':'{count} capture', {count: period.snapshotCount}) }}</div>\n    </div>\n\n    <div class=\"list\">\n      <div v-for=\"snapshotPeriod in snapshotPeriods\">\n        <span\n          @click=\"changePeriod(snapshotPeriod)\"\n          @keyup.enter=\"changePeriod(snapshotPeriod)\"\n          class=\"link\"\n          tabindex=\"1\">\n          {{ snapshotPeriod.snapshot.getTimeFormatted() }}\n        </span>\n        <span v-if=\"isCurrentSnapshot(period)\" class=\"current\">{{$root._('current')}}</span>\n      </div>\n    </div>\n  </div>\n</template>\n\n<script>\nimport { PywbI18N } from \"../i18n\";\n\nexport default {\n  name: \"TimelineLinear\",\n  props: ['period', 'currentSnapshot'],\n  computed: {\n    snapshotPeriods() {\n      return this.period.getSnapshotPeriodsFlat();\n    },\n    containsCurrentSnapshot() {\n      return this.currentSnapshot &&\n          this.period.contains(this.currentSnapshot);\n    },\n    displayDate() {\n      // replace '-' in date string with '/' so firefox and chrome will return same result\n      let dateStringNoSlashes = this.period.fullId.replace(/-/g,'/');\n      return new Date(dateStringNoSlashes).toLocaleDateString(PywbI18N.getLocale());\n    }\n  },\n  methods: {\n    isCurrentSnapshot(period) {\n      if (!!this.currentSnapshot && !!period.snapshot) {\n        return this.currentSnapshot && this.currentSnapshot.id === period.snapshot.id;\n      }\n      return false;\n    },\n    changePeriod(period) {\n      this.$emit(\"goto-period\", period);\n    }\n  }\n}\n</script>\n\n<style scoped>\n.timeline-linear {\n  width: auto;\n  padding: 5px;\n  background-color: white;\n  border: 1px solid gray;\n  border-radius: 5px;\n  z-index: 1100;\n}\n.timeline-linear .list {\n  max-height: 80vh;\n  min-height: 50px;\n  overflow-x: hidden;\n  overflow-y: auto;\n}\n.timeline-linear .title {\n  border-bottom: 1px solid black;\n  font-weight: bold;\n  font-family: Arial, sans-serif;\n}\n.timeline-linear .link {\n  text-decoration: underline;\n  color: darkblue;\n}\n.timeline-linear .link:hover {\n  color: lightseagreen;\n  cursor: pointer;\n}\n.timeline-linear .current {\n  background-color: deeppink;\n  color: white;\n  border-radius: 5px;\n}\n</style>\n"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__$3 = "data-v-3fb9ef40";
    /* module identifier */
    const __vue_module_identifier__$3 = undefined;
    /* functional template */
    const __vue_is_functional_template__$3 = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__$3 = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__$3, staticRenderFns: __vue_staticRenderFns__$3 },
      __vue_inject_styles__$3,
      __vue_script__$3,
      __vue_scope_id__$3,
      __vue_is_functional_template__$3,
      __vue_module_identifier__$3,
      false,
      createInjector,
      undefined,
      undefined
    );

  //
  var script$2 = {
    name: "Tooltip",
    props: ['position'],
    mounted() {
      this.$watch('position', this.updatePosition);
      this.updatePosition();
    },
    methods: {
      updatePosition() {
        this.$el.style.top = 0;
        this.$el.style.left = 0;
        this.$el.style.maxHeight = 'auto';

        const style = window.getComputedStyle(this.$el);
        const width = parseInt(style.width);
        const height = parseInt(style.height);
        const spacing = 10;
        const [initX, initY] = this.position.split(',').map(s => parseInt(s));
        if (window.innerWidth < initX + (spacing*2) + width) {
          this.$el.style.left = (initX - (width + spacing)) + 'px';
        } else {
          this.$el.style.left = (initX + spacing) + 'px';
        }
        if ((window.innerHeight < initY + (spacing*2) + height) && (initY - (spacing*2) - height < 0) ) {
          if (initY > window.innerHeight / 2) {
            this.$el.style.top = (window.innerHeight - (height + (spacing*2))) + 'px';
          } else {
            this.$el.style.top = (spacing*2) + 'px';
          }
        } else if (window.innerHeight < initY + (spacing*2) + height) {
          this.$el.style.top = (initY - (spacing + height)) + 'px';
        } else {
          this.$el.style.top = initY + 'px';
        }
      }
    }
  };

  /* script */
  const __vue_script__$2 = script$2;

  /* template */
  var __vue_render__$2 = function() {
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c("div", { staticClass: "pywb-tooltip" }, [_vm._t("default")], 2)
  };
  var __vue_staticRenderFns__$2 = [];
  __vue_render__$2._withStripped = true;

    /* style */
    const __vue_inject_styles__$2 = function (inject) {
      if (!inject) return
      inject("data-v-6f307cbc_0", { source: "\n.pywb-tooltip[data-v-6f307cbc] {\n  position: fixed;\n  top: 0;\n  left: 0;\n  z-index: 100;\n  background-color: white;\n  border: 1px solid grey;\n  border-radius: 5px;\n}\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/components/Tooltip.vue"],"names":[],"mappings":";AAgDA;EACA,eAAA;EACA,MAAA;EACA,OAAA;EACA,YAAA;EACA,uBAAA;EACA,sBAAA;EACA,kBAAA;AACA","file":"Tooltip.vue","sourcesContent":["<template>\n  <div class=\"pywb-tooltip\">\n    <slot></slot>\n  </div>\n</template>\n\n<script>\nlet elStyle = null;\nexport default {\n  name: \"Tooltip\",\n  props: ['position'],\n  mounted() {\n    this.$watch('position', this.updatePosition);\n    this.updatePosition();\n  },\n  methods: {\n    updatePosition() {\n      this.$el.style.top = 0;\n      this.$el.style.left = 0;\n      this.$el.style.maxHeight = 'auto';\n\n      const style = window.getComputedStyle(this.$el);\n      const width = parseInt(style.width);\n      const height = parseInt(style.height);\n      const spacing = 10;\n      const [initX, initY] = this.position.split(',').map(s => parseInt(s));\n      if (window.innerWidth < initX + (spacing*2) + width) {\n        this.$el.style.left = (initX - (width + spacing)) + 'px';\n      } else {\n        this.$el.style.left = (initX + spacing) + 'px';\n      }\n      if ((window.innerHeight < initY + (spacing*2) + height) && (initY - (spacing*2) - height < 0) ) {\n        if (initY > window.innerHeight / 2) {\n          this.$el.style.top = (window.innerHeight - (height + (spacing*2))) + 'px';\n        } else {\n          this.$el.style.top = (spacing*2) + 'px';\n        }\n      } else if (window.innerHeight < initY + (spacing*2) + height) {\n        this.$el.style.top = (initY - (spacing + height)) + 'px';\n      } else {\n        this.$el.style.top = initY + 'px';\n      }\n    }\n  }\n}\n</script>\n\n<style scoped>\n.pywb-tooltip {\n  position: fixed;\n  top: 0;\n  left: 0;\n  z-index: 100;\n  background-color: white;\n  border: 1px solid grey;\n  border-radius: 5px;\n}\n</style>"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__$2 = "data-v-6f307cbc";
    /* module identifier */
    const __vue_module_identifier__$2 = undefined;
    /* functional template */
    const __vue_is_functional_template__$2 = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__$2 = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__$2, staticRenderFns: __vue_staticRenderFns__$2 },
      __vue_inject_styles__$2,
      __vue_script__$2,
      __vue_scope_id__$2,
      __vue_is_functional_template__$2,
      __vue_module_identifier__$2,
      false,
      createInjector,
      undefined,
      undefined
    );

  //

  var script$1 = {
    components: {CalendarMonth: __vue_component__$4, TimelineLinear: __vue_component__$3, Tooltip: __vue_component__$2},
    props: ["period", "currentSnapshot"],
    data: function() {
      return {
        firstZoomLevel: PywbPeriod.Type.day,
        currentTimelinePeriod: null,
        currentTimelinePos: '0,0'
      };
    },
    mounted() {
      document.querySelector('body').addEventListener('click', this.resetCurrentTimeline);
    },
    computed: {
      year() { // the year that the timeline period is in
        let year = null;
        // if timeline is showing all year
        if (this.period.type === PywbPeriod.Type.all) {
          // if no current snapshot => pick the LAST YEAR
          if (!this.currentSnapshot) {
            year = this.period.children[this.period.children.length-1];
          } else {
            year = this.period.findByFullId(String(this.currentSnapshot.year));
          }
        } else if (this.period.type === PywbPeriod.Type.year) {
          year = this.period;
        } else {
          year = this.period.getParents().filter(p => p.type === PywbPeriod.Type.year)[0];
        }
        if (year) {
          year.fillEmptyChildPeriods(true);
        }
        return year;
      },
      currentYearIndex() {
        if (this.year.parent) {
          return this.year.parent.children.findIndex(year => year.fullId === this.year.fullId);
        }
      },
      previousYear() {
        return this.year.getPrevious();
      },
      nextYear() {
        return this.year.getNext();
      },
      currentMonth() { // the month that the timeline period is in
        let month = null;
        if (this.period.type === PywbPeriod.Type.month) {
          month = this.period;
        } else {
          month = this.period.getParents().filter(p => p.type === PywbPeriod.Type.month)[0];
        }
        return month;
      },
      containsCurrentSnapshot() {
        return this.currentSnapshot &&
          this.year.contains(this.currentSnapshot);
      }
    },
    methods: {
      gotoPreviousYear() {
        this.gotoPeriod(this.previousYear, true /* changeYearOnly */);
      },
      gotoNextYear() {
        this.gotoPeriod(this.nextYear, true /* changeYearOnly */);
      },
      resetCurrentTimeline(event) {
        if (event && this.$refs.timelineLinearTooltip) {
          let el = event.target;
          let clickWithinTooltip = false;
          while(el.parentElement) {
            if (el === this.$refs.timelineLinearTooltip.$el) {
              clickWithinTooltip = true;
              break;
            }
            el = el.parentElement;
          }
          if (!clickWithinTooltip) {
            this.currentTimelinePeriod = null;
          }
        }
      },
      setCurrentTimeline(day, event) {
        this.currentTimelinePeriod = day;
        if (!day) {
          return;
        }
        if (event.code === "Enter") {
          let middleXPos = (window.innerWidth / 2) - 60;
          this.currentTimelinePos = `${middleXPos},200`;
        } else {
          this.currentTimelinePos = `${event.x},${event.y}`;
        }
        
        event.stopPropagation();
        event.preventDefault();
      },
      gotoPeriod(period, changeYearOnly=false) {
        if (period.snapshot || period.snapshotPeriod || changeYearOnly) {
          this.$emit('goto-period', period);
        } else {
          this.currentTimelinePeriod = period;
        }
      },

    }
  };

  /* script */
  const __vue_script__$1 = script$1;

  /* template */
  var __vue_render__$1 = function() {
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c(
      "div",
      {
        staticClass:
          "full-view border-top-0 border-left-0 border-right-0 border-bottom border-dark shadow"
      },
      [
        _c("h2", [
          _vm.previousYear
            ? _c("i", {
                staticClass: "fas fa-arrow-left year-arrow",
                attrs: { tabindex: "0" },
                on: {
                  click: _vm.gotoPreviousYear,
                  keyup: function($event) {
                    if (
                      !$event.type.indexOf("key") &&
                      _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
                    ) {
                      return null
                    }
                    return _vm.gotoPreviousYear.apply(null, arguments)
                  }
                }
              })
            : _vm._e(),
          _vm._v(" "),
          _c("span", { staticClass: "mx-1" }, [
            _vm._v(
              "\n    " +
                _vm._s(_vm.year.id) +
                " (" +
                _vm._s(
                  _vm.$root._(
                    _vm.year.snapshotCount !== 1
                      ? "{count} captures"
                      : "{count} capture",
                    { count: _vm.year.snapshotCount }
                  )
                ) +
                ")\n    "
            )
          ]),
          _vm._v(" "),
          _vm.nextYear
            ? _c("i", {
                staticClass: "fas fa-arrow-right year-arrow",
                attrs: { tabindex: "0" },
                on: {
                  click: _vm.gotoNextYear,
                  keyup: function($event) {
                    if (
                      !$event.type.indexOf("key") &&
                      _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
                    ) {
                      return null
                    }
                    return _vm.gotoNextYear.apply(null, arguments)
                  }
                }
              })
            : _vm._e()
        ]),
        _vm._v(" "),
        _c(
          "div",
          { staticClass: "months" },
          _vm._l(_vm.year.children, function(month) {
            return _c("CalendarMonth", {
              key: month.id,
              attrs: {
                month: month,
                year: _vm.year,
                "current-snapshot": _vm.containsCurrentSnapshot
                  ? _vm.currentSnapshot
                  : null,
                "is-current": month === _vm.currentMonth
              },
              on: {
                "goto-period": function($event) {
                  return _vm.$emit("goto-period", $event)
                },
                "show-day-timeline": _vm.setCurrentTimeline
              }
            })
          }),
          1
        ),
        _vm._v(" "),
        _vm.currentTimelinePeriod
          ? _c(
              "Tooltip",
              {
                ref: "timelineLinearTooltip",
                attrs: { position: _vm.currentTimelinePos }
              },
              [
                _c("TimelineLinear", {
                  attrs: {
                    period: _vm.currentTimelinePeriod,
                    "current-snapshot": _vm.containsCurrentSnapshot
                      ? _vm.currentSnapshot
                      : null
                  },
                  on: { "goto-period": _vm.gotoPeriod }
                })
              ],
              1
            )
          : _vm._e()
      ],
      1
    )
  };
  var __vue_staticRenderFns__$1 = [];
  __vue_render__$1._withStripped = true;

    /* style */
    const __vue_inject_styles__$1 = function (inject) {
      if (!inject) return
      inject("data-v-33b44335_0", { source: "\n.full-view[data-v-33b44335] {\n  display: flex;\n  flex: 1;\n  flex-wrap: wrap;\n  z-index: 10;\n  overflow-y: auto;\n  width: 100%;\n  background-color: white;\n  padding-bottom: 1em;\n  justify-content: center;\n}\n.full-view .months[data-v-33b44335] {\n  display: flex;\n  justify-content: center;\n  flex-wrap: wrap;\n  align-items: flex-start;\n}\n.full-view h2[data-v-33b44335] {\n  margin: 10px 0;\n  font-size: 20px;\n  text-align: center;\n}\n.year-arrow[data-v-33b44335]:hover {\n  cursor: pointer;\n}\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/components/CalendarYear.vue"],"names":[],"mappings":";AAgKA;EACA,aAAA;EACA,OAAA;EACA,eAAA;EACA,WAAA;EACA,gBAAA;EACA,WAAA;EACA,uBAAA;EACA,mBAAA;EACA,uBAAA;AACA;AACA;EACA,aAAA;EACA,uBAAA;EACA,eAAA;EACA,uBAAA;AACA;AACA;EACA,cAAA;EACA,eAAA;EACA,kBAAA;AACA;AACA;EACA,eAAA;AACA","file":"CalendarYear.vue","sourcesContent":["<template>\n  <div class=\"full-view border-top-0 border-left-0 border-right-0 border-bottom border-dark shadow\">\n    <h2>\n      <i\n        class=\"fas fa-arrow-left year-arrow\"\n        @click=\"gotoPreviousYear\"\n        @keyup.enter=\"gotoPreviousYear\"\n        v-if=\"previousYear\"\n        tabindex=\"0\"></i>\n      <span class=\"mx-1\">\n      {{year.id}} ({{ $root._(year.snapshotCount !== 1 ? '{count} captures':'{count} capture', {count: year.snapshotCount}) }})\n      </span>\n      <i\n        class=\"fas fa-arrow-right year-arrow\"\n        @click=\"gotoNextYear\"\n        @keyup.enter=\"gotoNextYear\"\n        v-if=\"nextYear\"\n        tabindex=\"0\"></i>\n    </h2>\n    <div class=\"months\">\n      <CalendarMonth\n        v-for=\"month in year.children\"\n        :key=\"month.id\"\n        :month=\"month\"\n        :year=\"year\"\n        :current-snapshot=\"containsCurrentSnapshot ? currentSnapshot : null\"\n        :is-current=\"month === currentMonth\"\n        @goto-period=\"$emit('goto-period', $event)\"\n        @show-day-timeline=\"setCurrentTimeline\"\n      ></CalendarMonth>\n    </div>\n    <Tooltip\n      :position=\"currentTimelinePos\"\n      v-if=\"currentTimelinePeriod\"\n      ref=\"timelineLinearTooltip\">\n      <TimelineLinear\n        :period=\"currentTimelinePeriod\"\n        :current-snapshot=\"containsCurrentSnapshot ? currentSnapshot : null\"\n        @goto-period=\"gotoPeriod\"\n      ></TimelineLinear>\n    </Tooltip>\n  </div>\n</template>\n\n<script>\nimport CalendarMonth from \"./CalendarMonth.vue\";\nimport TimelineLinear from \"./TimelineLinear.vue\";\nimport Tooltip from \"./Tooltip.vue\";\nimport { PywbPeriod } from \"../model.js\";\n\nexport default {\n  components: {CalendarMonth, TimelineLinear, Tooltip},\n  props: [\"period\", \"currentSnapshot\"],\n  data: function() {\n    return {\n      firstZoomLevel: PywbPeriod.Type.day,\n      currentTimelinePeriod: null,\n      currentTimelinePos: '0,0'\n    };\n  },\n  mounted() {\n    document.querySelector('body').addEventListener('click', this.resetCurrentTimeline);\n  },\n  computed: {\n    year() { // the year that the timeline period is in\n      let year = null;\n      // if timeline is showing all year\n      if (this.period.type === PywbPeriod.Type.all) {\n        // if no current snapshot => pick the LAST YEAR\n        if (!this.currentSnapshot) {\n          year = this.period.children[this.period.children.length-1];\n        } else {\n          year = this.period.findByFullId(String(this.currentSnapshot.year));\n        }\n      } else if (this.period.type === PywbPeriod.Type.year) {\n        year = this.period;\n      } else {\n        year = this.period.getParents().filter(p => p.type === PywbPeriod.Type.year)[0];\n      }\n      if (year) {\n        year.fillEmptyChildPeriods(true);\n      }\n      return year;\n    },\n    currentYearIndex() {\n      if (this.year.parent) {\n        return this.year.parent.children.findIndex(year => year.fullId === this.year.fullId);\n      }\n    },\n    previousYear() {\n      return this.year.getPrevious();\n    },\n    nextYear() {\n      return this.year.getNext();\n    },\n    currentMonth() { // the month that the timeline period is in\n      let month = null;\n      if (this.period.type === PywbPeriod.Type.month) {\n        month = this.period;\n      } else {\n        month = this.period.getParents().filter(p => p.type === PywbPeriod.Type.month)[0];\n      }\n      return month;\n    },\n    containsCurrentSnapshot() {\n      return this.currentSnapshot &&\n        this.year.contains(this.currentSnapshot);\n    }\n  },\n  methods: {\n    gotoPreviousYear() {\n      this.gotoPeriod(this.previousYear, true /* changeYearOnly */);\n    },\n    gotoNextYear() {\n      this.gotoPeriod(this.nextYear, true /* changeYearOnly */);\n    },\n    resetCurrentTimeline(event) {\n      if (event && this.$refs.timelineLinearTooltip) {\n        let el = event.target;\n        let clickWithinTooltip = false;\n        while(el.parentElement) {\n          if (el === this.$refs.timelineLinearTooltip.$el) {\n            clickWithinTooltip = true;\n            break;\n          }\n          el = el.parentElement;\n        }\n        if (!clickWithinTooltip) {\n          this.currentTimelinePeriod = null;\n        }\n      }\n    },\n    setCurrentTimeline(day, event) {\n      this.currentTimelinePeriod = day;\n      if (!day) {\n        return;\n      }\n      if (event.code === \"Enter\") {\n        let middleXPos = (window.innerWidth / 2) - 60;\n        this.currentTimelinePos = `${middleXPos},200`;\n      } else {\n        this.currentTimelinePos = `${event.x},${event.y}`;\n      }\n      \n      event.stopPropagation();\n      event.preventDefault();\n    },\n    gotoPeriod(period, changeYearOnly=false) {\n      if (period.snapshot || period.snapshotPeriod || changeYearOnly) {\n        this.$emit('goto-period', period);\n      } else {\n        this.currentTimelinePeriod = period;\n      }\n    },\n\n  }\n};\n</script>\n\n<style scoped>\n  .full-view {\n    display: flex;\n    flex: 1;\n    flex-wrap: wrap;\n    z-index: 10;\n    overflow-y: auto;\n    width: 100%;\n    background-color: white;\n    padding-bottom: 1em;\n    justify-content: center;\n  }\n  .full-view .months {\n    display: flex;\n    justify-content: center;\n    flex-wrap: wrap;\n    align-items: flex-start;\n  }\n  .full-view h2 {\n    margin: 10px 0;\n    font-size: 20px;\n    text-align: center;\n  }\n  .year-arrow:hover {\n    cursor: pointer;\n  }\n</style>\n"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__$1 = "data-v-33b44335";
    /* module identifier */
    const __vue_module_identifier__$1 = undefined;
    /* functional template */
    const __vue_is_functional_template__$1 = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__$1 = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__$1, staticRenderFns: __vue_staticRenderFns__$1 },
      __vue_inject_styles__$1,
      __vue_script__$1,
      __vue_scope_id__$1,
      __vue_is_functional_template__$1,
      __vue_module_identifier__$1,
      false,
      createInjector,
      undefined,
      undefined
    );

  //

  var script = {
    name: "PywbReplayApp",
    //el: '[data-app="webrecorder-replay-app"]',
    data: function() {
      return {
        snapshots: [],
        currentPeriod: null,
        currentSnapshot: null,
        currentSnapshotIndex: null,
        msgs: [],
        showFullView: false,
        showTimelineView: false,
        maxTimelineZoomLevel: PywbPeriod.Type.day,
        config: {
          title: "",
          initialView: {},
          allLocales: {}
        },
        timelineHighlight: false,
        locales: [],
      };
    },
    components: {Timeline: __vue_component__$6, TimelineBreadcrumbs: __vue_component__$5, CalendarYear: __vue_component__$1},
    mounted: function() {
      // add empty unload event listener to make this page bfcache ineligible.
      // bfcache otherwises prevent the query template from reloading as expected
      // when the user navigates there via browser back/forward buttons
      addEventListener('unload', (event) => { });
    },
    updated: function() {
      // set top frame title equal to value pulled from replay frame
      document.title = this._("Archived Page: ") + this.config.title;
    },
    computed: {
      sessionStorageUrlKey() {
        // remove http(s), www and trailing slash
        return 'zoom__' + this.config.url.replace(/^https?:\/\/(www\.)?/, '').replace(/\/$/, '');
      },
      localesAreSet() {
        return Object.entries(this.config.allLocales).length > 0;
      },
      navbarStyle() {
        return {
          '--navbar-background': `#${this.config.navbarBackground}`,
          '--navbar-color': `#${this.config.navbarColor}`
        }
      },
      lightButtons() {
        return !!this.config.navbarLightButtons;
      },
      previousSnapshot() {
        if (!this.currentSnapshotIndex) {
          return null;
        }
        if (this.currentSnapshotIndex > 0) {
          return this.snapshots[this.currentSnapshotIndex - 1];
        }
        return null;
      },
      nextSnapshot() {
        if (this.currentSnapshotIndex == null) {
          return null;
        }
        if (
          (this.currentSnapshotIndex >= 0)
          && (this.currentSnapshotIndex !== this.snapshots.length - 1)) {
          return this.snapshots[this.currentSnapshotIndex + 1];
        }
        return null;
      }
    },
    methods: {
      _(id, embeddedVariableStrings=null) {
        return PywbI18N.instance.getText(id, embeddedVariableStrings);
      },
      gotoPeriod: function(newPeriod, onlyZoomToPeriod) {
        if (this.timelineHighlight) {
          setTimeout((() => {
            this.timelineHighlight=false;
          }).bind(this), 3000);
        }
        // only go to snapshot if caller did not request to zoom only
        if (newPeriod.snapshot && !onlyZoomToPeriod) {
          this.gotoSnapshot(newPeriod.snapshot, newPeriod, true /* reloadIFrame */);
        } else {
          // save current period (aka zoom)
          // use sessionStorage (not localStorage), as we want this to be a very temporary memory for current page tab/window and no longer; NOTE: it serves when navigating from an "*" query to a specific capture and subsequent reloads
          if (window.sessionStorage) {
            window.sessionStorage.setItem(this.sessionStorageUrlKey, newPeriod.fullId);
          }
          // If new period goes beyond allowed max level
          if (newPeriod.type > this.maxTimelineZoomLevel) {
            this.currentPeriod = newPeriod.get(this.maxTimelineZoomLevel);
          } else {
            this.currentPeriod = newPeriod;
          }
        }
      },
      gotoSnapshot(snapshot, fromPeriod, reloadIFrame=false) {
        this.currentSnapshot = snapshot;

        const isCurrentSnapshot = (snapshotInArray) => snapshotInArray.id == snapshot.id && snapshotInArray.url == snapshot.url;
        this.currentSnapshotIndex = this.snapshots.findIndex(isCurrentSnapshot);

        // if the current period doesn't match the current snapshot, update it
        if (!this.currentPeriod || (fromPeriod && !this.currentPeriod.contains(fromPeriod))) {
          const fromPeriodAtMaxZoomLevel = fromPeriod.get(this.maxTimelineZoomLevel);
          if (!this.currentPeriod || fromPeriodAtMaxZoomLevel !== this.currentPeriod) {
            this.currentPeriod = fromPeriodAtMaxZoomLevel;
          }
        }

        // update iframe only if the snapshot was selected from the calendar/timeline.
        // if the change originated from a user clicking a link in the iframe, emitting
        // snow-shapshot will only cause a flash of content
        if (reloadIFrame !== false) {
          this.$emit("show-snapshot", snapshot);
        }
        this.initBannerState(true);
      },
      gotoPreviousSnapshot() {
        let periodToChangeTo = this.currentPeriod.findByFullId(this.previousSnapshot.getFullId());
        this.gotoPeriod(periodToChangeTo, false /* onlyZoomToPeriod */);
      },
      gotoNextSnapshot() {
        let periodToChangeTo = this.currentPeriod.findByFullId(this.nextSnapshot.getFullId());
        this.gotoPeriod(periodToChangeTo, false /* onlyZoomToPeriod */);
      },
      gotoUrl(event) {
        event.preventDefault();
        const newUrl = document.querySelector("#theurl").value;
        if (newUrl !== this.config.url) {
          const ts = this.config.timestamp === undefined ? "*" : this.config.timestamp;
          window.location.href = this.config.prefix + ts + (ts ? "/" : "") + newUrl;
        }
      },
      toggleTimelineView() {
        this.showTimelineView = !this.showTimelineView;
        window.localStorage.setItem("showTimelineView", this.showTimelineView ? "1" : "0");
      },
      setData(/** @type {PywbData} data */ data) {

        // data-set will usually happen at App INIT (from parent caller)
        this.$set(this, "snapshots", data.snapshots);
        this.$set(this, "currentPeriod", data.timeline);

        // get last-saved current period from previous page/app refresh (if there was such)
        if (window.sessionStorage) {
          const currentPeriodId = window.sessionStorage.getItem(this.sessionStorageUrlKey);
          if (currentPeriodId) {
            const newCurrentPeriodFromStorage = this.currentPeriod.findByFullId(currentPeriodId);
            if (newCurrentPeriodFromStorage) {
              this.currentPeriod = newCurrentPeriodFromStorage;
            }
          }
        }

        // signal app is DONE setting and rendering data; ON NEXT TICK
        this.$nextTick(function isDone() {
          this.$emit('data-set-and-render-completed');
        }.bind(this));
      },
      setSnapshot(view) {
        if (!this.currentPeriod) {
          return false;
        }

        // turn off calendar (aka full) view
        this.showFullView = false;

        // convert to snapshot object to support proper rendering of time/date
        const snapshot = new PywbSnapshot(view, 0);

        this.config.url = view.url;

        let periodToChangeTo = this.currentPeriod.findByFullId(snapshot.getFullId());
        if (periodToChangeTo) {
          this.gotoPeriod(periodToChangeTo, false /* onlyZoomToPeriod */);
          return true;
        }
        return false;
      },
      initBannerState(isReplay) {
        // if not replay, always show both
        if (!isReplay) {
          this.showFullView = true;
          this.showTimelineView = true;
        } else {
          this.showFullView = false;
          this.showTimelineView = window.localStorage.getItem("showTimelineView") === "1";
        }
      },
      updateTitle(title) {
        this.config.title = title;
      }
    }
  };

  /* script */
  const __vue_script__ = script;

  /* template */
  var __vue_render__ = function() {
    var _vm = this;
    var _h = _vm.$createElement;
    var _c = _vm._self._c || _h;
    return _c(
      "div",
      {
        staticClass: "app",
        class: { expanded: _vm.showTimelineView || _vm.showFullView },
        attrs: { "data-app": "webrecorder-replay-app" }
      },
      [
        _c(
          "nav",
          {
            staticClass:
              "navbar navbar-light navbar-expand-lg fixed-top top-navbar justify-content-center",
            style: _vm.navbarStyle
          },
          [
            _vm.config.logoHomeUrl
              ? _c(
                  "a",
                  {
                    staticClass: "navbar-brand flex-grow-1 my-1",
                    attrs: { href: _vm.config.logoHomeUrl }
                  },
                  [
                    _c("img", {
                      attrs: {
                        src: _vm.config.logoImg,
                        id: "logo-img",
                        alt: "_('pywb logo')"
                      }
                    })
                  ]
                )
              : _c("div", { staticClass: "navbar-brand flex-grow-1 my-1" }, [
                  _c("img", {
                    attrs: {
                      src: _vm.config.logoImg,
                      id: "logo-img",
                      alt: "_('pywb logo')"
                    }
                  })
                ]),
            _vm._v(" "),
            _c(
              "div",
              { staticClass: "flex-grow-1 d-flex", attrs: { id: "searchdiv" } },
              [
                _c(
                  "form",
                  {
                    staticClass: "form-inline my-2 my-md-0 mx-lg-auto",
                    attrs: { role: "search" },
                    on: { submit: _vm.gotoUrl }
                  },
                  [
                    _c("input", {
                      attrs: {
                        id: "theurl",
                        type: "text",
                        height: "31",
                        "aria-label": "_('Search for archival capture of URL')",
                        title: "_('Search for archival capture of URL')"
                      },
                      domProps: { value: _vm.config.url }
                    })
                  ]
                )
              ]
            ),
            _vm._v(" "),
            _vm._m(0),
            _vm._v(" "),
            _c(
              "div",
              {
                staticClass: "collapse navbar-collapse ml-auto",
                attrs: { id: "navbarCollapse" }
              },
              [
                _c(
                  "ul",
                  { staticClass: "navbar-nav ml-3", attrs: { id: "toggles" } },
                  [
                    _c("li", { staticClass: "nav-item" }, [
                      _vm.previousSnapshot
                        ? _c(
                            "button",
                            {
                              staticClass: "btn btn-sm",
                              class: {
                                active: _vm.showFullView,
                                "btn-outline-light": _vm.lightButtons,
                                "btn-outline-dark": !_vm.lightButtons
                              },
                              attrs: { title: _vm._("Previous capture") },
                              on: { click: _vm.gotoPreviousSnapshot }
                            },
                            [
                              _c("i", {
                                staticClass: "fas fa-arrow-left",
                                attrs: { title: _vm._("Previous capture") }
                              })
                            ]
                          )
                        : _vm._e()
                    ]),
                    _vm._v(" "),
                    _c("li", { staticClass: "nav-item" }, [
                      _vm.nextSnapshot
                        ? _c(
                            "button",
                            {
                              staticClass: "btn btn-sm",
                              class: {
                                active: _vm.showFullView,
                                "btn-outline-light": _vm.lightButtons,
                                "btn-outline-dark": !_vm.lightButtons
                              },
                              attrs: { title: _vm._("Next capture") },
                              on: { click: _vm.gotoNextSnapshot }
                            },
                            [
                              _c("i", {
                                staticClass: "fas fa-arrow-right",
                                attrs: { title: _vm._("Next capture") }
                              })
                            ]
                          )
                        : _vm._e()
                    ]),
                    _vm._v(" "),
                    _c("li", { staticClass: "nav-item active" }, [
                      _c(
                        "button",
                        {
                          staticClass: "btn btn-sm",
                          class: {
                            active: _vm.showFullView,
                            "btn-outline-light": _vm.lightButtons,
                            "btn-outline-dark": !_vm.lightButtons
                          },
                          attrs: {
                            "aria-pressed": _vm.showFullView ? true : false,
                            title: _vm.showFullView
                              ? _vm._("Hide calendar")
                              : _vm._("Show calendar")
                          },
                          on: {
                            click: function($event) {
                              _vm.showFullView = !_vm.showFullView;
                            }
                          }
                        },
                        [_c("i", { staticClass: "far fa-calendar-alt" })]
                      )
                    ]),
                    _vm._v(" "),
                    _c("li", { staticClass: "nav-item" }, [
                      _c(
                        "button",
                        {
                          staticClass: "btn btn-sm",
                          class: {
                            active: _vm.showTimelineView,
                            "btn-outline-light": _vm.lightButtons,
                            "btn-outline-dark": !_vm.lightButtons
                          },
                          attrs: {
                            "aria-pressed": _vm.showTimelineView,
                            title: _vm.showTimelineView
                              ? _vm._("Hide timeline")
                              : _vm._("Show timeline")
                          },
                          on: { click: _vm.toggleTimelineView }
                        },
                        [_c("i", { staticClass: "far fa-chart-bar" })]
                      )
                    ]),
                    _vm._v(" "),
                    _vm.localesAreSet
                      ? _c("li", { staticClass: "nav-item dropdown" }, [
                          _c(
                            "button",
                            {
                              staticClass: "btn btn-sm dropdown-toggle",
                              class: {
                                "btn-outline-light": _vm.lightButtons,
                                "btn-outline-dark": !_vm.lightButtons
                              },
                              attrs: {
                                type: "button",
                                id: "locale-dropdown",
                                "data-toggle": "dropdown",
                                "aria-haspopup": "true",
                                "aria-expanded": "false",
                                title: _vm._("Select language")
                              }
                            },
                            [
                              _c("i", {
                                staticClass: "fas fa-globe-africa",
                                attrs: { title: _vm._("Language") }
                              })
                            ]
                          ),
                          _vm._v(" "),
                          _c(
                            "div",
                            {
                              staticClass: "dropdown-menu dropdown-menu-right",
                              attrs: { "aria-labelledby": "locale-dropdown" }
                            },
                            _vm._l(_vm.config.allLocales, function(locPath, key) {
                              return _c(
                                "a",
                                {
                                  key: key,
                                  staticClass: "dropdown-item",
                                  attrs: {
                                    href:
                                      locPath +
                                      (_vm.currentSnapshot
                                        ? _vm.currentSnapshot.id
                                        : "*") +
                                      "/" +
                                      _vm.config.url
                                  }
                                },
                                [
                                  _vm._v(
                                    "\n              " +
                                      _vm._s(key) +
                                      "\n            "
                                  )
                                ]
                              )
                            }),
                            0
                          )
                        ])
                      : _vm._e()
                  ]
                )
              ]
            )
          ]
        ),
        _vm._v(" "),
        _c(
          "nav",
          {
            staticClass:
              "navbar navbar-light justify-content-center title-nav fixed-top",
            style: _vm.navbarStyle,
            attrs: { id: "second-navbar" }
          },
          [
            !_vm.currentSnapshot
              ? _c("span", { staticClass: "hidden" }, [_vm._v(" ")])
              : _vm._e(),
            _vm._v(" "),
            _vm.currentSnapshot
              ? _c("span", [
                  _c("span", { staticClass: "strong mr-1" }, [
                    _vm._v(
                      "\n        " +
                        _vm._s(_vm._("Current Capture")) +
                        ": \n        "
                    ),
                    _vm.config.title
                      ? _c("span", { staticClass: "ml-1" }, [
                          _vm._v(
                            "\n          " +
                              _vm._s(_vm.config.title) +
                              "\n        "
                          )
                        ])
                      : _vm._e()
                  ]),
                  _vm._v(" "),
                  _vm.config.title
                    ? _c("span", { staticClass: "mr-1" }, [_vm._v("|")])
                    : _vm._e(),
                  _vm._v(
                    "\n      " +
                      _vm._s(_vm.currentSnapshot.getTimeDateFormatted()) +
                      "\n    "
                  )
                ])
              : _vm._e()
          ]
        ),
        _vm._v(" "),
        _c(
          "div",
          {
            staticClass:
              "card border-top-0 border-left-0 border-right-0 timeline-wrap"
          },
          [
            _vm.currentPeriod && _vm.showTimelineView
              ? _c("div", { staticClass: "card-body" }, [
                  _c("div", { staticClass: "row" }, [
                    _c(
                      "div",
                      { staticClass: "col col-12" },
                      [
                        _c("TimelineBreadcrumbs", {
                          attrs: { period: _vm.currentPeriod },
                          on: { "goto-period": _vm.gotoPeriod }
                        })
                      ],
                      1
                    ),
                    _vm._v(" "),
                    _c(
                      "div",
                      { staticClass: "col col-12 mt-2" },
                      [
                        _c("Timeline", {
                          attrs: {
                            period: _vm.currentPeriod,
                            highlight: _vm.timelineHighlight,
                            "current-snapshot": _vm.currentSnapshot,
                            "max-zoom-level": _vm.maxTimelineZoomLevel
                          },
                          on: { "goto-period": _vm.gotoPeriod }
                        })
                      ],
                      1
                    )
                  ])
                ])
              : _vm._e()
          ]
        ),
        _vm._v(" "),
        _vm.currentPeriod && _vm.showFullView && _vm.currentPeriod.children.length
          ? _c("div", { staticClass: "card", attrs: { id: "calendar-card" } }, [
              _c(
                "div",
                { staticClass: "card-body", attrs: { id: "calendar-card-body" } },
                [
                  _c("CalendarYear", {
                    attrs: {
                      period: _vm.currentPeriod,
                      "current-snapshot": _vm.currentSnapshot
                    },
                    on: { "goto-period": _vm.gotoPeriod }
                  })
                ],
                1
              )
            ])
          : _vm._e()
      ]
    )
  };
  var __vue_staticRenderFns__ = [
    function() {
      var _vm = this;
      var _h = _vm.$createElement;
      var _c = _vm._self._c || _h;
      return _c(
        "button",
        {
          staticClass: "navbar-toggler btn btn-sm",
          attrs: {
            id: "collapse-button",
            type: "button",
            "data-toggle": "collapse",
            "data-target": "#navbarCollapse",
            "aria-controls": "navbarCollapse",
            "aria-expanded": "false",
            "aria-label": "_('Toggle navigation')"
          }
        },
        [_c("span", { staticClass: "navbar-toggler-icon" })]
      )
    }
  ];
  __vue_render__._withStripped = true;

    /* style */
    const __vue_inject_styles__ = function (inject) {
      if (!inject) return
      inject("data-v-74a508dc_0", { source: "\nbody {\n  padding-top: 89px !important;\n}\n.app {\n  font-family: Calibri, Arial, sans-serif;\n  /*border-bottom: 1px solid lightcoral;*/\n  width: 100%;\n}\n.app.expanded {\n  /*height: 130px;*/\n  max-height: calc(100vh - 90px);\n  display: flex;\n  flex-direction: column;\n}\n.full-view {\n  /*position: fixed;*/\n  /*top: 150px;*/\n  left: 0;\n}\n.navbar {\n  background-color: var(--navbar-background);\n  color:  var(--navbar-color);\n}\n.top-navbar {\n  z-index: 90;\n  padding: 2px 16px 0 16px;\n}\n.top-navbar span.navbar-toggler-icon {\n  margin: .25rem !important;\n}\n#logo-img {\n  max-height: 40px;\n}\n.title-nav {\n  margin-top: 50px;\n  z-index: 80;\n}\n#secondNavbar {\n  height: 24px !important;\n}\n#navbarCollapse {\n  justify-content: right;\n}\n#navbarCollapse ul#toggles {\n  display: flex;\n  align-content: center;\n}\n#navbarCollapse:not(.show) ul#toggles li:not(:first-child) {\n  margin-left: .25rem;\n}\n#navbarCollapse.show {\n  padding-bottom: 1em;\n}\n#navbarCollapse.show ul#toggles li {\n  margin-top: 5px;\n}\n#navbarCollapse.show ul#toggles li {\n  margin-left: 0px;\n}\n.iframe iframe {\n  width: 100%;\n  height: 80vh;\n}\n#searchdiv {\n  height: 31px;\n}\n#theurl {\n  width: 250px;\n}\n@media (min-width: 576px) {\n#theurl {\n    width: 350px;\n}\n}\n@media (min-width: 768px) {\n#theurl {\n    width: 500px;\n}\n}\n@media (min-width: 992px) {\n#theurl {\n    width: 600px;\n}\n}\n@media (min-width: 1200px) {\n#theurl {\n    width: 900px;\n}\n}\n#toggles {\n  align-items: center;\n}\n.breadcrumb-row {\n  display: flex;\n  align-items: center;\n  justify-content: center;\n}\ndiv.timeline-wrap div.card {\n  margin-top: 55px;\n}\n#calendar-card {\n  overflow-y: auto;\n  max-height: 100%;\n}\ndiv.timeline-wrap div.card-body {\n  display: flex;\n  align-items: center;\n  justify-content: center;\n}\ndiv.timeline-wrap div.card-body div.row {\n  width: 100%;\n  align-items: center;\n  justify-content: center;\n}\n#calendar-card-body {\n  padding: 0;\n}\n.strong {\n  font-weight: bold;\n}\n.hidden {\n  color: var(--navbar-background);\n}\n", map: {"version":3,"sources":["/Users/tessa/dev/pywb/pywb/vueui/src/App.vue"],"names":[],"mappings":";AAgXA;EACA,4BAAA;AACA;AACA;EACA,uCAAA;EACA,uCAAA;EACA,WAAA;AACA;AACA;EACA,iBAAA;EACA,8BAAA;EACA,aAAA;EACA,sBAAA;AACA;AACA;EACA,mBAAA;EACA,cAAA;EACA,OAAA;AACA;AACA;EACA,0CAAA;EACA,2BAAA;AACA;AACA;EACA,WAAA;EACA,wBAAA;AACA;AACA;EACA,yBAAA;AACA;AACA;EACA,gBAAA;AACA;AACA;EACA,gBAAA;EACA,WAAA;AACA;AACA;EACA,uBAAA;AACA;AACA;EACA,sBAAA;AACA;AACA;EACA,aAAA;EACA,qBAAA;AACA;AACA;EACA,mBAAA;AACA;AACA;EACA,mBAAA;AACA;AACA;EACA,eAAA;AACA;AACA;EACA,gBAAA;AACA;AACA;EACA,WAAA;EACA,YAAA;AACA;AACA;EACA,YAAA;AACA;AACA;EACA,YAAA;AACA;AACA;AACA;IACA,YAAA;AACA;AACA;AACA;AACA;IACA,YAAA;AACA;AACA;AACA;AACA;IACA,YAAA;AACA;AACA;AACA;AACA;IACA,YAAA;AACA;AACA;AACA;EACA,mBAAA;AACA;AACA;EACA,aAAA;EACA,mBAAA;EACA,uBAAA;AACA;AACA;EACA,gBAAA;AACA;AACA;EACA,gBAAA;EACA,gBAAA;AACA;AACA;EACA,aAAA;EACA,mBAAA;EACA,uBAAA;AACA;AACA;EACA,WAAA;EACA,mBAAA;EACA,uBAAA;AACA;AAEA;EACA,UAAA;AACA;AACA;EACA,iBAAA;AACA;AACA;EACA,+BAAA;AACA","file":"App.vue","sourcesContent":["<template>\n  <div class=\"app\" :class=\"{expanded: showTimelineView || showFullView }\" data-app=\"webrecorder-replay-app\">\n    <!-- Top navbar -->\n    <nav\n      class=\"navbar navbar-light navbar-expand-lg fixed-top top-navbar justify-content-center\"\n      :style=\"navbarStyle\">\n      <a class=\"navbar-brand flex-grow-1 my-1\" :href=\"config.logoHomeUrl\" v-if=\"config.logoHomeUrl\">\n        <img :src=\"config.logoImg\" id=\"logo-img\" alt=\"_('pywb logo')\">\n      </a>\n      <div class=\"navbar-brand flex-grow-1 my-1\" v-else>\n        <img :src=\"config.logoImg\" id=\"logo-img\" alt=\"_('pywb logo')\">\n      </div>\n      <div class=\"flex-grow-1 d-flex\" id=\"searchdiv\">\n        <form\n          class=\"form-inline my-2 my-md-0 mx-lg-auto\"\n          role=\"search\"\n          @submit=\"gotoUrl\">\n          <input\n            id=\"theurl\"\n            type=\"text\"\n            :value=\"config.url\"\n            height=\"31\"\n            aria-label=\"_('Search for archival capture of URL')\"\n            title=\"_('Search for archival capture of URL')\"></input>\n        </form>\n      </div>\n      <button\n        class=\"navbar-toggler btn btn-sm\"\n        id=\"collapse-button\"\n        type=\"button\"\n        data-toggle=\"collapse\"\n        data-target=\"#navbarCollapse\"\n        aria-controls=\"navbarCollapse\"\n        aria-expanded=\"false\"\n        aria-label=\"_('Toggle navigation')\">\n        <span class=\"navbar-toggler-icon\"></span>\n      </button>\n      <div class=\"collapse navbar-collapse ml-auto\" id=\"navbarCollapse\">\n        <ul class=\"navbar-nav ml-3\" id=\"toggles\">\n          <li class=\"nav-item\">\n            <button\n              class=\"btn btn-sm\"\n              :class=\"{active: showFullView, 'btn-outline-light': lightButtons, 'btn-outline-dark': !lightButtons}\"\n              :title=\"_('Previous capture')\"\n              v-if=\"previousSnapshot\"\n              @click=\"gotoPreviousSnapshot\">\n              <i class=\"fas fa-arrow-left\" :title=\"_('Previous capture')\"></i>\n            </button>\n          </li>\n          <li class=\"nav-item\">\n            <button\n              class=\"btn btn-sm\"\n              :class=\"{active: showFullView, 'btn-outline-light': lightButtons, 'btn-outline-dark': !lightButtons}\"\n              :title=\"_('Next capture')\"\n              v-if=\"nextSnapshot\"\n              @click=\"gotoNextSnapshot\">\n              <i class=\"fas fa-arrow-right\" :title=\"_('Next capture')\"></i>\n            </button>\n          </li>\n          <li class=\"nav-item active\">\n            <button\n              class=\"btn btn-sm\"\n              :class=\"{active: showFullView, 'btn-outline-light': lightButtons, 'btn-outline-dark': !lightButtons}\"\n              :aria-pressed=\"(showFullView ? true : false)\"\n              @click=\"showFullView = !showFullView\"\n              :title=\"(showFullView ? _('Hide calendar') : _('Show calendar'))\">\n              <i class=\"far fa-calendar-alt\"></i>\n            </button>\n          </li>\n          <li class=\"nav-item\">\n            <button\n              class=\"btn btn-sm\"\n              :class=\"{active: showTimelineView, 'btn-outline-light': lightButtons, 'btn-outline-dark': !lightButtons}\"\n              :aria-pressed=\"showTimelineView\"\n              @click=\"toggleTimelineView\"\n              :title=\"(showTimelineView ? _('Hide timeline') : _('Show timeline'))\">\n              <i class=\"far fa-chart-bar\"></i>\n            </button>\n          </li>\n          <li class=\"nav-item dropdown\" v-if=\"localesAreSet\">\n            <button\n              class=\"btn btn-sm dropdown-toggle\"\n              :class=\"{'btn-outline-light': lightButtons, 'btn-outline-dark': !lightButtons}\"\n              type=\"button\"\n              id=\"locale-dropdown\"\n              data-toggle=\"dropdown\"\n              aria-haspopup=\"true\"\n              aria-expanded=\"false\"\n              :title=\"_('Select language')\">\n              <i class=\"fas fa-globe-africa\" :title=\"_('Language')\"></i>\n            </button>\n            <div class=\"dropdown-menu dropdown-menu-right\" aria-labelledby=\"locale-dropdown\">\n              <a\n                class=\"dropdown-item\"\n                v-for=\"(locPath, key) in config.allLocales\"\n                :key=\"key\"\n                :href=\"locPath + (currentSnapshot ? currentSnapshot.id : '*') + '/' + config.url\">\n                {{ key }}\n              </a>\n            </div>\n          </li>\n        </ul>\n      </div>\n    </nav>\n\n    <!-- Capture title and date -->\n    <nav\n      class=\"navbar navbar-light justify-content-center title-nav fixed-top\"\n      id=\"second-navbar\"\n      :style=\"navbarStyle\">\n      <span class=\"hidden\" v-if=\"!currentSnapshot\">&nbsp;</span>\n      <span v-if=\"currentSnapshot\">\n        <span class=\"strong mr-1\">\n          {{_('Current Capture')}}: \n          <span class=\"ml-1\" v-if=\"config.title\">\n            {{ config.title }}\n          </span>\n        </span>\n        <span class=\"mr-1\" v-if=\"config.title\">|</span>\n        {{currentSnapshot.getTimeDateFormatted()}}\n      </span>\n    </nav>\n\n    <!-- Timeline -->\n    <div class=\"card border-top-0 border-left-0 border-right-0 timeline-wrap\">\n      <div class=\"card-body\" v-if=\"currentPeriod && showTimelineView\">\n        <div class=\"row\">\n          <div class=\"col col-12\">\n            <TimelineBreadcrumbs\n              :period=\"currentPeriod\"\n              @goto-period=\"gotoPeriod\"\n            ></TimelineBreadcrumbs>\n          </div>\n          <div class=\"col col-12 mt-2\">\n            <Timeline\n              :period=\"currentPeriod\"\n              :highlight=\"timelineHighlight\"\n              :current-snapshot=\"currentSnapshot\"\n              :max-zoom-level=\"maxTimelineZoomLevel\"\n              @goto-period=\"gotoPeriod\"\n            ></Timeline>\n          </div>\n        </div>\n      </div>    \n    </div>\n\n    <!-- Calendar -->\n    <div class=\"card\" id=\"calendar-card\" v-if=\"currentPeriod && showFullView && currentPeriod.children.length\">\n      <div class=\"card-body\" id=\"calendar-card-body\">\n        <CalendarYear\n          :period=\"currentPeriod\"\n          :current-snapshot=\"currentSnapshot\"\n           @goto-period=\"gotoPeriod\">\n        </CalendarYear>\n      </div>\n    </div>\n    \n  </div>\n</template>\n\n<script>\nimport Timeline from \"./components/Timeline.vue\";\nimport TimelineBreadcrumbs from \"./components/TimelineBreadcrumbs.vue\";\nimport CalendarYear from \"./components/CalendarYear.vue\";\n\nimport { PywbSnapshot, PywbPeriod } from \"./model.js\";\nimport {PywbI18N} from \"./i18n\";\n\nexport default {\n  name: \"PywbReplayApp\",\n  //el: '[data-app=\"webrecorder-replay-app\"]',\n  data: function() {\n    return {\n      snapshots: [],\n      currentPeriod: null,\n      currentSnapshot: null,\n      currentSnapshotIndex: null,\n      msgs: [],\n      showFullView: false,\n      showTimelineView: false,\n      maxTimelineZoomLevel: PywbPeriod.Type.day,\n      config: {\n        title: \"\",\n        initialView: {},\n        allLocales: {}\n      },\n      timelineHighlight: false,\n      locales: [],\n    };\n  },\n  components: {Timeline, TimelineBreadcrumbs, CalendarYear},\n  mounted: function() {\n    // add empty unload event listener to make this page bfcache ineligible.\n    // bfcache otherwises prevent the query template from reloading as expected\n    // when the user navigates there via browser back/forward buttons\n    addEventListener('unload', (event) => { });\n  },\n  updated: function() {\n    // set top frame title equal to value pulled from replay frame\n    document.title = this._(\"Archived Page: \") + this.config.title;\n  },\n  computed: {\n    sessionStorageUrlKey() {\n      // remove http(s), www and trailing slash\n      return 'zoom__' + this.config.url.replace(/^https?:\\/\\/(www\\.)?/, '').replace(/\\/$/, '');\n    },\n    localesAreSet() {\n      return Object.entries(this.config.allLocales).length > 0;\n    },\n    navbarStyle() {\n      return {\n        '--navbar-background': `#${this.config.navbarBackground}`,\n        '--navbar-color': `#${this.config.navbarColor}`\n      }\n    },\n    lightButtons() {\n      return !!this.config.navbarLightButtons;\n    },\n    previousSnapshot() {\n      if (!this.currentSnapshotIndex) {\n        return null;\n      }\n      if (this.currentSnapshotIndex > 0) {\n        return this.snapshots[this.currentSnapshotIndex - 1];\n      }\n      return null;\n    },\n    nextSnapshot() {\n      if (this.currentSnapshotIndex == null) {\n        return null;\n      }\n      if (\n        (this.currentSnapshotIndex >= 0)\n        && (this.currentSnapshotIndex !== this.snapshots.length - 1)) {\n        return this.snapshots[this.currentSnapshotIndex + 1];\n      }\n      return null;\n    }\n  },\n  methods: {\n    _(id, embeddedVariableStrings=null) {\n      return PywbI18N.instance.getText(id, embeddedVariableStrings);\n    },\n    gotoPeriod: function(newPeriod, onlyZoomToPeriod) {\n      if (this.timelineHighlight) {\n        setTimeout((() => {\n          this.timelineHighlight=false;\n        }).bind(this), 3000);\n      }\n      // only go to snapshot if caller did not request to zoom only\n      if (newPeriod.snapshot && !onlyZoomToPeriod) {\n        this.gotoSnapshot(newPeriod.snapshot, newPeriod, true /* reloadIFrame */);\n      } else {\n        // save current period (aka zoom)\n        // use sessionStorage (not localStorage), as we want this to be a very temporary memory for current page tab/window and no longer; NOTE: it serves when navigating from an \"*\" query to a specific capture and subsequent reloads\n        if (window.sessionStorage) {\n          window.sessionStorage.setItem(this.sessionStorageUrlKey, newPeriod.fullId);\n        }\n        // If new period goes beyond allowed max level\n        if (newPeriod.type > this.maxTimelineZoomLevel) {\n          this.currentPeriod = newPeriod.get(this.maxTimelineZoomLevel);\n        } else {\n          this.currentPeriod = newPeriod;\n        }\n      }\n    },\n    gotoSnapshot(snapshot, fromPeriod, reloadIFrame=false) {\n      this.currentSnapshot = snapshot;\n\n      const isCurrentSnapshot = (snapshotInArray) => snapshotInArray.id == snapshot.id && snapshotInArray.url == snapshot.url;\n      this.currentSnapshotIndex = this.snapshots.findIndex(isCurrentSnapshot);\n\n      // if the current period doesn't match the current snapshot, update it\n      if (!this.currentPeriod || (fromPeriod && !this.currentPeriod.contains(fromPeriod))) {\n        const fromPeriodAtMaxZoomLevel = fromPeriod.get(this.maxTimelineZoomLevel);\n        if (!this.currentPeriod || fromPeriodAtMaxZoomLevel !== this.currentPeriod) {\n          this.currentPeriod = fromPeriodAtMaxZoomLevel;\n        }\n      }\n\n      // update iframe only if the snapshot was selected from the calendar/timeline.\n      // if the change originated from a user clicking a link in the iframe, emitting\n      // snow-shapshot will only cause a flash of content\n      if (reloadIFrame !== false) {\n        this.$emit(\"show-snapshot\", snapshot);\n      }\n      this.initBannerState(true);\n    },\n    gotoPreviousSnapshot() {\n      let periodToChangeTo = this.currentPeriod.findByFullId(this.previousSnapshot.getFullId());\n      this.gotoPeriod(periodToChangeTo, false /* onlyZoomToPeriod */);\n    },\n    gotoNextSnapshot() {\n      let periodToChangeTo = this.currentPeriod.findByFullId(this.nextSnapshot.getFullId());\n      this.gotoPeriod(periodToChangeTo, false /* onlyZoomToPeriod */);\n    },\n    gotoUrl(event) {\n      event.preventDefault();\n      const newUrl = document.querySelector(\"#theurl\").value;\n      if (newUrl !== this.config.url) {\n        const ts = this.config.timestamp === undefined ? \"*\" : this.config.timestamp;\n        window.location.href = this.config.prefix + ts + (ts ? \"/\" : \"\") + newUrl;\n      }\n    },\n    toggleTimelineView() {\n      this.showTimelineView = !this.showTimelineView;\n      window.localStorage.setItem(\"showTimelineView\", this.showTimelineView ? \"1\" : \"0\");\n    },\n    setData(/** @type {PywbData} data */ data) {\n\n      // data-set will usually happen at App INIT (from parent caller)\n      this.$set(this, \"snapshots\", data.snapshots);\n      this.$set(this, \"currentPeriod\", data.timeline);\n\n      // get last-saved current period from previous page/app refresh (if there was such)\n      if (window.sessionStorage) {\n        const currentPeriodId = window.sessionStorage.getItem(this.sessionStorageUrlKey);\n        if (currentPeriodId) {\n          const newCurrentPeriodFromStorage = this.currentPeriod.findByFullId(currentPeriodId);\n          if (newCurrentPeriodFromStorage) {\n            this.currentPeriod = newCurrentPeriodFromStorage;\n          }\n        }\n      }\n\n      // signal app is DONE setting and rendering data; ON NEXT TICK\n      this.$nextTick(function isDone() {\n        this.$emit('data-set-and-render-completed');\n      }.bind(this));\n    },\n    setSnapshot(view) {\n      if (!this.currentPeriod) {\n        return false;\n      }\n\n      // turn off calendar (aka full) view\n      this.showFullView = false;\n\n      // convert to snapshot object to support proper rendering of time/date\n      const snapshot = new PywbSnapshot(view, 0);\n\n      this.config.url = view.url;\n\n      let periodToChangeTo = this.currentPeriod.findByFullId(snapshot.getFullId());\n      if (periodToChangeTo) {\n        this.gotoPeriod(periodToChangeTo, false /* onlyZoomToPeriod */);\n        return true;\n      }\n      return false;\n    },\n    initBannerState(isReplay) {\n      // if not replay, always show both\n      if (!isReplay) {\n        this.showFullView = true;\n        this.showTimelineView = true;\n      } else {\n        this.showFullView = false;\n        this.showTimelineView = window.localStorage.getItem(\"showTimelineView\") === \"1\";\n      }\n    },\n    updateTitle(title) {\n      this.config.title = title;\n    }\n  }\n};\n</script>\n\n<style>\n  body {\n    padding-top: 89px !important;\n  }\n  .app {\n    font-family: Calibri, Arial, sans-serif;\n    /*border-bottom: 1px solid lightcoral;*/\n    width: 100%;\n  }\n  .app.expanded {\n    /*height: 130px;*/\n    max-height: calc(100vh - 90px);\n    display: flex;\n    flex-direction: column;\n  }\n  .full-view {\n    /*position: fixed;*/\n    /*top: 150px;*/\n    left: 0;\n  }\n  .navbar {\n    background-color: var(--navbar-background);\n    color:  var(--navbar-color);\n  }\n  .top-navbar {\n    z-index: 90;\n    padding: 2px 16px 0 16px;\n  }\n  .top-navbar span.navbar-toggler-icon {\n    margin: .25rem !important;\n  }\n  #logo-img {\n    max-height: 40px;\n  }\n  .title-nav {\n    margin-top: 50px;\n    z-index: 80;\n  }\n  #secondNavbar {\n    height: 24px !important;\n  }\n  #navbarCollapse {\n    justify-content: right;\n  }\n  #navbarCollapse ul#toggles {\n    display: flex;\n    align-content: center;\n  }\n  #navbarCollapse:not(.show) ul#toggles li:not(:first-child) {\n    margin-left: .25rem;\n  }\n  #navbarCollapse.show {\n    padding-bottom: 1em;\n  }\n  #navbarCollapse.show ul#toggles li {\n    margin-top: 5px;\n  }\n  #navbarCollapse.show ul#toggles li {\n    margin-left: 0px;\n  }\n  .iframe iframe {\n    width: 100%;\n    height: 80vh;\n  }\n  #searchdiv {\n    height: 31px;\n  }\n  #theurl {\n    width: 250px;\n  }\n  @media (min-width: 576px) {\n    #theurl {\n      width: 350px;\n    }\n  }\n  @media (min-width: 768px) {\n    #theurl {\n      width: 500px;\n    }\n  }\n  @media (min-width: 992px) {\n    #theurl {\n      width: 600px;\n    }\n  }\n  @media (min-width: 1200px) {\n    #theurl {\n      width: 900px;\n    }\n  }\n  #toggles {\n    align-items: center;\n  }\n  .breadcrumb-row {\n    display: flex;\n    align-items: center;\n    justify-content: center;\n  }\n  div.timeline-wrap div.card {\n    margin-top: 55px;\n  }\n  #calendar-card {\n    overflow-y: auto;\n    max-height: 100%;\n  }\n  div.timeline-wrap div.card-body {\n    display: flex;\n    align-items: center;\n    justify-content: center;\n  }\n  div.timeline-wrap div.card-body div.row {\n    width: 100%;\n    align-items: center;\n    justify-content: center;\n  }\n\n  #calendar-card-body {\n    padding: 0;\n  }\n  .strong {\n    font-weight: bold;\n  }\n  .hidden {\n    color: var(--navbar-background);\n  }\n</style>\n"]}, media: undefined });

    };
    /* scoped */
    const __vue_scope_id__ = undefined;
    /* module identifier */
    const __vue_module_identifier__ = undefined;
    /* functional template */
    const __vue_is_functional_template__ = false;
    /* style inject SSR */
    
    /* style inject shadow dom */
    

    
    const __vue_component__ = /*#__PURE__*/normalizeComponent(
      { render: __vue_render__, staticRenderFns: __vue_staticRenderFns__ },
      __vue_inject_styles__,
      __vue_script__,
      __vue_scope_id__,
      __vue_is_functional_template__,
      __vue_module_identifier__,
      false,
      createInjector,
      undefined,
      undefined
    );

  /*!
   * Vue.js v2.6.14
   * (c) 2014-2021 Evan You
   * Released under the MIT License.
   */
  /*  */

  const emptyObject = Object.freeze({});

  // These helpers produce better VM code in JS engines due to their
  // explicitness and function inlining.
  function isUndef (v) {
    return v === undefined || v === null
  }

  function isDef (v) {
    return v !== undefined && v !== null
  }

  function isTrue (v) {
    return v === true
  }

  function isFalse (v) {
    return v === false
  }

  /**
   * Check if value is primitive.
   */
  function isPrimitive (value) {
    return (
      typeof value === 'string' ||
      typeof value === 'number' ||
      // $flow-disable-line
      typeof value === 'symbol' ||
      typeof value === 'boolean'
    )
  }

  /**
   * Quick object check - this is primarily used to tell
   * Objects from primitive values when we know the value
   * is a JSON-compliant type.
   */
  function isObject (obj) {
    return obj !== null && typeof obj === 'object'
  }

  /**
   * Get the raw type string of a value, e.g., [object Object].
   */
  const _toString = Object.prototype.toString;

  function toRawType (value) {
    return _toString.call(value).slice(8, -1)
  }

  /**
   * Strict object type check. Only returns true
   * for plain JavaScript objects.
   */
  function isPlainObject (obj) {
    return _toString.call(obj) === '[object Object]'
  }

  function isRegExp (v) {
    return _toString.call(v) === '[object RegExp]'
  }

  /**
   * Check if val is a valid array index.
   */
  function isValidArrayIndex (val) {
    const n = parseFloat(String(val));
    return n >= 0 && Math.floor(n) === n && isFinite(val)
  }

  function isPromise (val) {
    return (
      isDef(val) &&
      typeof val.then === 'function' &&
      typeof val.catch === 'function'
    )
  }

  /**
   * Convert a value to a string that is actually rendered.
   */
  function toString (val) {
    return val == null
      ? ''
      : Array.isArray(val) || (isPlainObject(val) && val.toString === _toString)
        ? JSON.stringify(val, null, 2)
        : String(val)
  }

  /**
   * Convert an input value to a number for persistence.
   * If the conversion fails, return original string.
   */
  function toNumber (val) {
    const n = parseFloat(val);
    return isNaN(n) ? val : n
  }

  /**
   * Make a map and return a function for checking if a key
   * is in that map.
   */
  function makeMap (
    str,
    expectsLowerCase
  ) {
    const map = Object.create(null);
    const list = str.split(',');
    for (let i = 0; i < list.length; i++) {
      map[list[i]] = true;
    }
    return expectsLowerCase
      ? val => map[val.toLowerCase()]
      : val => map[val]
  }

  /**
   * Check if a tag is a built-in tag.
   */
  const isBuiltInTag = makeMap('slot,component', true);

  /**
   * Check if an attribute is a reserved attribute.
   */
  const isReservedAttribute = makeMap('key,ref,slot,slot-scope,is');

  /**
   * Remove an item from an array.
   */
  function remove (arr, item) {
    if (arr.length) {
      const index = arr.indexOf(item);
      if (index > -1) {
        return arr.splice(index, 1)
      }
    }
  }

  /**
   * Check whether an object has the property.
   */
  const hasOwnProperty = Object.prototype.hasOwnProperty;
  function hasOwn (obj, key) {
    return hasOwnProperty.call(obj, key)
  }

  /**
   * Create a cached version of a pure function.
   */
  function cached (fn) {
    const cache = Object.create(null);
    return (function cachedFn (str) {
      const hit = cache[str];
      return hit || (cache[str] = fn(str))
    })
  }

  /**
   * Camelize a hyphen-delimited string.
   */
  const camelizeRE = /-(\w)/g;
  const camelize = cached((str) => {
    return str.replace(camelizeRE, (_, c) => c ? c.toUpperCase() : '')
  });

  /**
   * Capitalize a string.
   */
  const capitalize = cached((str) => {
    return str.charAt(0).toUpperCase() + str.slice(1)
  });

  /**
   * Hyphenate a camelCase string.
   */
  const hyphenateRE = /\B([A-Z])/g;
  const hyphenate = cached((str) => {
    return str.replace(hyphenateRE, '-$1').toLowerCase()
  });

  /**
   * Simple bind polyfill for environments that do not support it,
   * e.g., PhantomJS 1.x. Technically, we don't need this anymore
   * since native bind is now performant enough in most browsers.
   * But removing it would mean breaking code that was able to run in
   * PhantomJS 1.x, so this must be kept for backward compatibility.
   */

  /* istanbul ignore next */
  function polyfillBind (fn, ctx) {
    function boundFn (a) {
      const l = arguments.length;
      return l
        ? l > 1
          ? fn.apply(ctx, arguments)
          : fn.call(ctx, a)
        : fn.call(ctx)
    }

    boundFn._length = fn.length;
    return boundFn
  }

  function nativeBind (fn, ctx) {
    return fn.bind(ctx)
  }

  const bind = Function.prototype.bind
    ? nativeBind
    : polyfillBind;

  /**
   * Convert an Array-like object to a real Array.
   */
  function toArray (list, start) {
    start = start || 0;
    let i = list.length - start;
    const ret = new Array(i);
    while (i--) {
      ret[i] = list[i + start];
    }
    return ret
  }

  /**
   * Mix properties into target object.
   */
  function extend (to, _from) {
    for (const key in _from) {
      to[key] = _from[key];
    }
    return to
  }

  /**
   * Merge an Array of Objects into a single Object.
   */
  function toObject (arr) {
    const res = {};
    for (let i = 0; i < arr.length; i++) {
      if (arr[i]) {
        extend(res, arr[i]);
      }
    }
    return res
  }

  /* eslint-disable no-unused-vars */

  /**
   * Perform no operation.
   * Stubbing args to make Flow happy without leaving useless transpiled code
   * with ...rest (https://flow.org/blog/2017/05/07/Strict-Function-Call-Arity/).
   */
  function noop (a, b, c) {}

  /**
   * Always return false.
   */
  const no = (a, b, c) => false;

  /* eslint-enable no-unused-vars */

  /**
   * Return the same value.
   */
  const identity = (_) => _;

  /**
   * Generate a string containing static keys from compiler modules.
   */
  function genStaticKeys (modules) {
    return modules.reduce((keys, m) => {
      return keys.concat(m.staticKeys || [])
    }, []).join(',')
  }

  /**
   * Check if two values are loosely equal - that is,
   * if they are plain objects, do they have the same shape?
   */
  function looseEqual (a, b) {
    if (a === b) return true
    const isObjectA = isObject(a);
    const isObjectB = isObject(b);
    if (isObjectA && isObjectB) {
      try {
        const isArrayA = Array.isArray(a);
        const isArrayB = Array.isArray(b);
        if (isArrayA && isArrayB) {
          return a.length === b.length && a.every((e, i) => {
            return looseEqual(e, b[i])
          })
        } else if (a instanceof Date && b instanceof Date) {
          return a.getTime() === b.getTime()
        } else if (!isArrayA && !isArrayB) {
          const keysA = Object.keys(a);
          const keysB = Object.keys(b);
          return keysA.length === keysB.length && keysA.every(key => {
            return looseEqual(a[key], b[key])
          })
        } else {
          /* istanbul ignore next */
          return false
        }
      } catch (e) {
        /* istanbul ignore next */
        return false
      }
    } else if (!isObjectA && !isObjectB) {
      return String(a) === String(b)
    } else {
      return false
    }
  }

  /**
   * Return the first index at which a loosely equal value can be
   * found in the array (if value is a plain object, the array must
   * contain an object of the same shape), or -1 if it is not present.
   */
  function looseIndexOf (arr, val) {
    for (let i = 0; i < arr.length; i++) {
      if (looseEqual(arr[i], val)) return i
    }
    return -1
  }

  /**
   * Ensure a function is called only once.
   */
  function once (fn) {
    let called = false;
    return function () {
      if (!called) {
        called = true;
        fn.apply(this, arguments);
      }
    }
  }

  const SSR_ATTR = 'data-server-rendered';

  const ASSET_TYPES = [
    'component',
    'directive',
    'filter'
  ];

  const LIFECYCLE_HOOKS = [
    'beforeCreate',
    'created',
    'beforeMount',
    'mounted',
    'beforeUpdate',
    'updated',
    'beforeDestroy',
    'destroyed',
    'activated',
    'deactivated',
    'errorCaptured',
    'serverPrefetch'
  ];

  /*  */



  var config = ({
    /**
     * Option merge strategies (used in core/util/options)
     */
    // $flow-disable-line
    optionMergeStrategies: Object.create(null),

    /**
     * Whether to suppress warnings.
     */
    silent: false,

    /**
     * Show production mode tip message on boot?
     */
    productionTip: "development" !== 'production',

    /**
     * Whether to enable devtools
     */
    devtools: "development" !== 'production',

    /**
     * Whether to record perf
     */
    performance: false,

    /**
     * Error handler for watcher errors
     */
    errorHandler: null,

    /**
     * Warn handler for watcher warns
     */
    warnHandler: null,

    /**
     * Ignore certain custom elements
     */
    ignoredElements: [],

    /**
     * Custom user key aliases for v-on
     */
    // $flow-disable-line
    keyCodes: Object.create(null),

    /**
     * Check if a tag is reserved so that it cannot be registered as a
     * component. This is platform-dependent and may be overwritten.
     */
    isReservedTag: no,

    /**
     * Check if an attribute is reserved so that it cannot be used as a component
     * prop. This is platform-dependent and may be overwritten.
     */
    isReservedAttr: no,

    /**
     * Check if a tag is an unknown element.
     * Platform-dependent.
     */
    isUnknownElement: no,

    /**
     * Get the namespace of an element
     */
    getTagNamespace: noop,

    /**
     * Parse the real tag name for the specific platform.
     */
    parsePlatformTagName: identity,

    /**
     * Check if an attribute must be bound using property, e.g. value
     * Platform-dependent.
     */
    mustUseProp: no,

    /**
     * Perform updates asynchronously. Intended to be used by Vue Test Utils
     * This will significantly reduce performance if set to false.
     */
    async: true,

    /**
     * Exposed for legacy reasons
     */
    _lifecycleHooks: LIFECYCLE_HOOKS
  });

  /*  */

  /**
   * unicode letters used for parsing html tags, component names and property paths.
   * using https://www.w3.org/TR/html53/semantics-scripting.html#potentialcustomelementname
   * skipping \u10000-\uEFFFF due to it freezing up PhantomJS
   */
  const unicodeRegExp = /a-zA-Z\u00B7\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u037D\u037F-\u1FFF\u200C-\u200D\u203F-\u2040\u2070-\u218F\u2C00-\u2FEF\u3001-\uD7FF\uF900-\uFDCF\uFDF0-\uFFFD/;

  /**
   * Check if a string starts with $ or _
   */
  function isReserved (str) {
    const c = (str + '').charCodeAt(0);
    return c === 0x24 || c === 0x5F
  }

  /**
   * Define a property.
   */
  function def (obj, key, val, enumerable) {
    Object.defineProperty(obj, key, {
      value: val,
      enumerable: !!enumerable,
      writable: true,
      configurable: true
    });
  }

  /**
   * Parse simple path.
   */
  const bailRE = new RegExp(`[^${unicodeRegExp.source}.$_\\d]`);
  function parsePath (path) {
    if (bailRE.test(path)) {
      return
    }
    const segments = path.split('.');
    return function (obj) {
      for (let i = 0; i < segments.length; i++) {
        if (!obj) return
        obj = obj[segments[i]];
      }
      return obj
    }
  }

  /*  */

  // can we use __proto__?
  const hasProto = '__proto__' in {};

  // Browser environment sniffing
  const inBrowser = typeof window !== 'undefined';
  const inWeex = typeof WXEnvironment !== 'undefined' && !!WXEnvironment.platform;
  const weexPlatform = inWeex && WXEnvironment.platform.toLowerCase();
  const UA = inBrowser && window.navigator.userAgent.toLowerCase();
  const isIE = UA && /msie|trident/.test(UA);
  const isIE9 = UA && UA.indexOf('msie 9.0') > 0;
  const isEdge = UA && UA.indexOf('edge/') > 0;
  (UA && UA.indexOf('android') > 0) || (weexPlatform === 'android');
  const isIOS = (UA && /iphone|ipad|ipod|ios/.test(UA)) || (weexPlatform === 'ios');
  UA && /chrome\/\d+/.test(UA) && !isEdge;
  UA && /phantomjs/.test(UA);
  const isFF = UA && UA.match(/firefox\/(\d+)/);

  // Firefox has a "watch" function on Object.prototype...
  const nativeWatch = ({}).watch;

  let supportsPassive = false;
  if (inBrowser) {
    try {
      const opts = {};
      Object.defineProperty(opts, 'passive', ({
        get () {
          /* istanbul ignore next */
          supportsPassive = true;
        }
      })); // https://github.com/facebook/flow/issues/285
      window.addEventListener('test-passive', null, opts);
    } catch (e) {}
  }

  // this needs to be lazy-evaled because vue may be required before
  // vue-server-renderer can set VUE_ENV
  let _isServer;
  const isServerRendering = () => {
    if (_isServer === undefined) {
      /* istanbul ignore if */
      if (!inBrowser && !inWeex && typeof global !== 'undefined') {
        // detect presence of vue-server-renderer and avoid
        // Webpack shimming the process
        _isServer = global['process'] && global['process'].env.VUE_ENV === 'server';
      } else {
        _isServer = false;
      }
    }
    return _isServer
  };

  // detect devtools
  const devtools = inBrowser && window.__VUE_DEVTOOLS_GLOBAL_HOOK__;

  /* istanbul ignore next */
  function isNative (Ctor) {
    return typeof Ctor === 'function' && /native code/.test(Ctor.toString())
  }

  const hasSymbol =
    typeof Symbol !== 'undefined' && isNative(Symbol) &&
    typeof Reflect !== 'undefined' && isNative(Reflect.ownKeys);

  let _Set;
  /* istanbul ignore if */ // $flow-disable-line
  if (typeof Set !== 'undefined' && isNative(Set)) {
    // use native Set when available.
    _Set = Set;
  } else {
    // a non-standard Set polyfill that only works with primitive keys.
    _Set = class Set   {
      
      constructor () {
        this.set = Object.create(null);
      }
      has (key) {
        return this.set[key] === true
      }
      add (key) {
        this.set[key] = true;
      }
      clear () {
        this.set = Object.create(null);
      }
    };
  }

  /*  */

  let warn = noop;
  let tip = noop;
  let generateComponentTrace = (noop); // work around flow check
  let formatComponentName = (noop);

  {
    const hasConsole = typeof console !== 'undefined';
    const classifyRE = /(?:^|[-_])(\w)/g;
    const classify = str => str
      .replace(classifyRE, c => c.toUpperCase())
      .replace(/[-_]/g, '');

    warn = (msg, vm) => {
      const trace = vm ? generateComponentTrace(vm) : '';

      if (config.warnHandler) {
        config.warnHandler.call(null, msg, vm, trace);
      } else if (hasConsole && (!config.silent)) {
        console.error(`[Vue warn]: ${msg}${trace}`);
      }
    };

    tip = (msg, vm) => {
      if (hasConsole && (!config.silent)) {
        console.warn(`[Vue tip]: ${msg}` + (
          vm ? generateComponentTrace(vm) : ''
        ));
      }
    };

    formatComponentName = (vm, includeFile) => {
      if (vm.$root === vm) {
        return '<Root>'
      }
      const options = typeof vm === 'function' && vm.cid != null
        ? vm.options
        : vm._isVue
          ? vm.$options || vm.constructor.options
          : vm;
      let name = options.name || options._componentTag;
      const file = options.__file;
      if (!name && file) {
        const match = file.match(/([^/\\]+)\.vue$/);
        name = match && match[1];
      }

      return (
        (name ? `<${classify(name)}>` : `<Anonymous>`) +
        (file && includeFile !== false ? ` at ${file}` : '')
      )
    };

    const repeat = (str, n) => {
      let res = '';
      while (n) {
        if (n % 2 === 1) res += str;
        if (n > 1) str += str;
        n >>= 1;
      }
      return res
    };

    generateComponentTrace = vm => {
      if (vm._isVue && vm.$parent) {
        const tree = [];
        let currentRecursiveSequence = 0;
        while (vm) {
          if (tree.length > 0) {
            const last = tree[tree.length - 1];
            if (last.constructor === vm.constructor) {
              currentRecursiveSequence++;
              vm = vm.$parent;
              continue
            } else if (currentRecursiveSequence > 0) {
              tree[tree.length - 1] = [last, currentRecursiveSequence];
              currentRecursiveSequence = 0;
            }
          }
          tree.push(vm);
          vm = vm.$parent;
        }
        return '\n\nfound in\n\n' + tree
          .map((vm, i) => `${
          i === 0 ? '---> ' : repeat(' ', 5 + i * 2)
        }${
          Array.isArray(vm)
            ? `${formatComponentName(vm[0])}... (${vm[1]} recursive calls)`
            : formatComponentName(vm)
        }`)
          .join('\n')
      } else {
        return `\n\n(found in ${formatComponentName(vm)})`
      }
    };
  }

  /*  */

  let uid = 0;

  /**
   * A dep is an observable that can have multiple
   * directives subscribing to it.
   */
  class Dep {
    
    
    

    constructor () {
      this.id = uid++;
      this.subs = [];
    }

    addSub (sub) {
      this.subs.push(sub);
    }

    removeSub (sub) {
      remove(this.subs, sub);
    }

    depend () {
      if (Dep.target) {
        Dep.target.addDep(this);
      }
    }

    notify () {
      // stabilize the subscriber list first
      const subs = this.subs.slice();
      if (!config.async) {
        // subs aren't sorted in scheduler if not running async
        // we need to sort them now to make sure they fire in correct
        // order
        subs.sort((a, b) => a.id - b.id);
      }
      for (let i = 0, l = subs.length; i < l; i++) {
        subs[i].update();
      }
    }
  }

  // The current target watcher being evaluated.
  // This is globally unique because only one watcher
  // can be evaluated at a time.
  Dep.target = null;
  const targetStack = [];

  function pushTarget (target) {
    targetStack.push(target);
    Dep.target = target;
  }

  function popTarget () {
    targetStack.pop();
    Dep.target = targetStack[targetStack.length - 1];
  }

  /*  */

  class VNode {
    
    
    
    
    
    
     // rendered in this component's scope
    
    
     // component instance
     // component placeholder node

    // strictly internal
     // contains raw HTML? (server only)
     // hoisted static node
     // necessary for enter transition check
     // empty comment placeholder?
     // is a cloned node?
     // is a v-once node?
     // async component factory function
    
    
    
     // real context vm for functional nodes
     // for SSR caching
     // used to store functional render context for devtools
     // functional scope id support

    constructor (
      tag,
      data,
      children,
      text,
      elm,
      context,
      componentOptions,
      asyncFactory
    ) {
      this.tag = tag;
      this.data = data;
      this.children = children;
      this.text = text;
      this.elm = elm;
      this.ns = undefined;
      this.context = context;
      this.fnContext = undefined;
      this.fnOptions = undefined;
      this.fnScopeId = undefined;
      this.key = data && data.key;
      this.componentOptions = componentOptions;
      this.componentInstance = undefined;
      this.parent = undefined;
      this.raw = false;
      this.isStatic = false;
      this.isRootInsert = true;
      this.isComment = false;
      this.isCloned = false;
      this.isOnce = false;
      this.asyncFactory = asyncFactory;
      this.asyncMeta = undefined;
      this.isAsyncPlaceholder = false;
    }

    // DEPRECATED: alias for componentInstance for backwards compat.
    /* istanbul ignore next */
    get child () {
      return this.componentInstance
    }
  }

  const createEmptyVNode = (text = '') => {
    const node = new VNode();
    node.text = text;
    node.isComment = true;
    return node
  };

  function createTextVNode (val) {
    return new VNode(undefined, undefined, undefined, String(val))
  }

  // optimized shallow clone
  // used for static nodes and slot nodes because they may be reused across
  // multiple renders, cloning them avoids errors when DOM manipulations rely
  // on their elm reference.
  function cloneVNode (vnode) {
    const cloned = new VNode(
      vnode.tag,
      vnode.data,
      // #7975
      // clone children array to avoid mutating original in case of cloning
      // a child.
      vnode.children && vnode.children.slice(),
      vnode.text,
      vnode.elm,
      vnode.context,
      vnode.componentOptions,
      vnode.asyncFactory
    );
    cloned.ns = vnode.ns;
    cloned.isStatic = vnode.isStatic;
    cloned.key = vnode.key;
    cloned.isComment = vnode.isComment;
    cloned.fnContext = vnode.fnContext;
    cloned.fnOptions = vnode.fnOptions;
    cloned.fnScopeId = vnode.fnScopeId;
    cloned.asyncMeta = vnode.asyncMeta;
    cloned.isCloned = true;
    return cloned
  }

  /*
   * not type checking this file because flow doesn't play well with
   * dynamically accessing methods on Array prototype
   */

  const arrayProto = Array.prototype;
  const arrayMethods = Object.create(arrayProto);

  const methodsToPatch = [
    'push',
    'pop',
    'shift',
    'unshift',
    'splice',
    'sort',
    'reverse'
  ];

  /**
   * Intercept mutating methods and emit events
   */
  methodsToPatch.forEach(function (method) {
    // cache original method
    const original = arrayProto[method];
    def(arrayMethods, method, function mutator (...args) {
      const result = original.apply(this, args);
      const ob = this.__ob__;
      let inserted;
      switch (method) {
        case 'push':
        case 'unshift':
          inserted = args;
          break
        case 'splice':
          inserted = args.slice(2);
          break
      }
      if (inserted) ob.observeArray(inserted);
      // notify change
      ob.dep.notify();
      return result
    });
  });

  /*  */

  const arrayKeys = Object.getOwnPropertyNames(arrayMethods);

  /**
   * In some cases we may want to disable observation inside a component's
   * update computation.
   */
  let shouldObserve = true;

  function toggleObserving (value) {
    shouldObserve = value;
  }

  /**
   * Observer class that is attached to each observed
   * object. Once attached, the observer converts the target
   * object's property keys into getter/setters that
   * collect dependencies and dispatch updates.
   */
  class Observer {
    
    
     // number of vms that have this object as root $data

    constructor (value) {
      this.value = value;
      this.dep = new Dep();
      this.vmCount = 0;
      def(value, '__ob__', this);
      if (Array.isArray(value)) {
        if (hasProto) {
          protoAugment(value, arrayMethods);
        } else {
          copyAugment(value, arrayMethods, arrayKeys);
        }
        this.observeArray(value);
      } else {
        this.walk(value);
      }
    }

    /**
     * Walk through all properties and convert them into
     * getter/setters. This method should only be called when
     * value type is Object.
     */
    walk (obj) {
      const keys = Object.keys(obj);
      for (let i = 0; i < keys.length; i++) {
        defineReactive$$1(obj, keys[i]);
      }
    }

    /**
     * Observe a list of Array items.
     */
    observeArray (items) {
      for (let i = 0, l = items.length; i < l; i++) {
        observe(items[i]);
      }
    }
  }

  // helpers

  /**
   * Augment a target Object or Array by intercepting
   * the prototype chain using __proto__
   */
  function protoAugment (target, src) {
    /* eslint-disable no-proto */
    target.__proto__ = src;
    /* eslint-enable no-proto */
  }

  /**
   * Augment a target Object or Array by defining
   * hidden properties.
   */
  /* istanbul ignore next */
  function copyAugment (target, src, keys) {
    for (let i = 0, l = keys.length; i < l; i++) {
      const key = keys[i];
      def(target, key, src[key]);
    }
  }

  /**
   * Attempt to create an observer instance for a value,
   * returns the new observer if successfully observed,
   * or the existing observer if the value already has one.
   */
  function observe (value, asRootData) {
    if (!isObject(value) || value instanceof VNode) {
      return
    }
    let ob;
    if (hasOwn(value, '__ob__') && value.__ob__ instanceof Observer) {
      ob = value.__ob__;
    } else if (
      shouldObserve &&
      !isServerRendering() &&
      (Array.isArray(value) || isPlainObject(value)) &&
      Object.isExtensible(value) &&
      !value._isVue
    ) {
      ob = new Observer(value);
    }
    if (asRootData && ob) {
      ob.vmCount++;
    }
    return ob
  }

  /**
   * Define a reactive property on an Object.
   */
  function defineReactive$$1 (
    obj,
    key,
    val,
    customSetter,
    shallow
  ) {
    const dep = new Dep();

    const property = Object.getOwnPropertyDescriptor(obj, key);
    if (property && property.configurable === false) {
      return
    }

    // cater for pre-defined getter/setters
    const getter = property && property.get;
    const setter = property && property.set;
    if ((!getter || setter) && arguments.length === 2) {
      val = obj[key];
    }

    let childOb = !shallow && observe(val);
    Object.defineProperty(obj, key, {
      enumerable: true,
      configurable: true,
      get: function reactiveGetter () {
        const value = getter ? getter.call(obj) : val;
        if (Dep.target) {
          dep.depend();
          if (childOb) {
            childOb.dep.depend();
            if (Array.isArray(value)) {
              dependArray(value);
            }
          }
        }
        return value
      },
      set: function reactiveSetter (newVal) {
        const value = getter ? getter.call(obj) : val;
        /* eslint-disable no-self-compare */
        if (newVal === value || (newVal !== newVal && value !== value)) {
          return
        }
        /* eslint-enable no-self-compare */
        if (customSetter) {
          customSetter();
        }
        // #7981: for accessor properties without setter
        if (getter && !setter) return
        if (setter) {
          setter.call(obj, newVal);
        } else {
          val = newVal;
        }
        childOb = !shallow && observe(newVal);
        dep.notify();
      }
    });
  }

  /**
   * Set a property on an object. Adds the new property and
   * triggers change notification if the property doesn't
   * already exist.
   */
  function set (target, key, val) {
    if (isUndef(target) || isPrimitive(target)
    ) {
      warn(`Cannot set reactive property on undefined, null, or primitive value: ${(target)}`);
    }
    if (Array.isArray(target) && isValidArrayIndex(key)) {
      target.length = Math.max(target.length, key);
      target.splice(key, 1, val);
      return val
    }
    if (key in target && !(key in Object.prototype)) {
      target[key] = val;
      return val
    }
    const ob = (target).__ob__;
    if (target._isVue || (ob && ob.vmCount)) {
      warn(
        'Avoid adding reactive properties to a Vue instance or its root $data ' +
        'at runtime - declare it upfront in the data option.'
      );
      return val
    }
    if (!ob) {
      target[key] = val;
      return val
    }
    defineReactive$$1(ob.value, key, val);
    ob.dep.notify();
    return val
  }

  /**
   * Delete a property and trigger change if necessary.
   */
  function del (target, key) {
    if (isUndef(target) || isPrimitive(target)
    ) {
      warn(`Cannot delete reactive property on undefined, null, or primitive value: ${(target)}`);
    }
    if (Array.isArray(target) && isValidArrayIndex(key)) {
      target.splice(key, 1);
      return
    }
    const ob = (target).__ob__;
    if (target._isVue || (ob && ob.vmCount)) {
      warn(
        'Avoid deleting properties on a Vue instance or its root $data ' +
        '- just set it to null.'
      );
      return
    }
    if (!hasOwn(target, key)) {
      return
    }
    delete target[key];
    if (!ob) {
      return
    }
    ob.dep.notify();
  }

  /**
   * Collect dependencies on array elements when the array is touched, since
   * we cannot intercept array element access like property getters.
   */
  function dependArray (value) {
    for (let e, i = 0, l = value.length; i < l; i++) {
      e = value[i];
      e && e.__ob__ && e.__ob__.dep.depend();
      if (Array.isArray(e)) {
        dependArray(e);
      }
    }
  }

  /*  */

  /**
   * Option overwriting strategies are functions that handle
   * how to merge a parent option value and a child option
   * value into the final value.
   */
  const strats = config.optionMergeStrategies;

  /**
   * Options with restrictions
   */
  {
    strats.el = strats.propsData = function (parent, child, vm, key) {
      if (!vm) {
        warn(
          `option "${key}" can only be used during instance ` +
          'creation with the `new` keyword.'
        );
      }
      return defaultStrat(parent, child)
    };
  }

  /**
   * Helper that recursively merges two data objects together.
   */
  function mergeData (to, from) {
    if (!from) return to
    let key, toVal, fromVal;

    const keys = hasSymbol
      ? Reflect.ownKeys(from)
      : Object.keys(from);

    for (let i = 0; i < keys.length; i++) {
      key = keys[i];
      // in case the object is already observed...
      if (key === '__ob__') continue
      toVal = to[key];
      fromVal = from[key];
      if (!hasOwn(to, key)) {
        set(to, key, fromVal);
      } else if (
        toVal !== fromVal &&
        isPlainObject(toVal) &&
        isPlainObject(fromVal)
      ) {
        mergeData(toVal, fromVal);
      }
    }
    return to
  }

  /**
   * Data
   */
  function mergeDataOrFn (
    parentVal,
    childVal,
    vm
  ) {
    if (!vm) {
      // in a Vue.extend merge, both should be functions
      if (!childVal) {
        return parentVal
      }
      if (!parentVal) {
        return childVal
      }
      // when parentVal & childVal are both present,
      // we need to return a function that returns the
      // merged result of both functions... no need to
      // check if parentVal is a function here because
      // it has to be a function to pass previous merges.
      return function mergedDataFn () {
        return mergeData(
          typeof childVal === 'function' ? childVal.call(this, this) : childVal,
          typeof parentVal === 'function' ? parentVal.call(this, this) : parentVal
        )
      }
    } else {
      return function mergedInstanceDataFn () {
        // instance merge
        const instanceData = typeof childVal === 'function'
          ? childVal.call(vm, vm)
          : childVal;
        const defaultData = typeof parentVal === 'function'
          ? parentVal.call(vm, vm)
          : parentVal;
        if (instanceData) {
          return mergeData(instanceData, defaultData)
        } else {
          return defaultData
        }
      }
    }
  }

  strats.data = function (
    parentVal,
    childVal,
    vm
  ) {
    if (!vm) {
      if (childVal && typeof childVal !== 'function') {
        warn(
          'The "data" option should be a function ' +
          'that returns a per-instance value in component ' +
          'definitions.',
          vm
        );

        return parentVal
      }
      return mergeDataOrFn(parentVal, childVal)
    }

    return mergeDataOrFn(parentVal, childVal, vm)
  };

  /**
   * Hooks and props are merged as arrays.
   */
  function mergeHook (
    parentVal,
    childVal
  ) {
    const res = childVal
      ? parentVal
        ? parentVal.concat(childVal)
        : Array.isArray(childVal)
          ? childVal
          : [childVal]
      : parentVal;
    return res
      ? dedupeHooks(res)
      : res
  }

  function dedupeHooks (hooks) {
    const res = [];
    for (let i = 0; i < hooks.length; i++) {
      if (res.indexOf(hooks[i]) === -1) {
        res.push(hooks[i]);
      }
    }
    return res
  }

  LIFECYCLE_HOOKS.forEach(hook => {
    strats[hook] = mergeHook;
  });

  /**
   * Assets
   *
   * When a vm is present (instance creation), we need to do
   * a three-way merge between constructor options, instance
   * options and parent options.
   */
  function mergeAssets (
    parentVal,
    childVal,
    vm,
    key
  ) {
    const res = Object.create(parentVal || null);
    if (childVal) {
      assertObjectType(key, childVal, vm);
      return extend(res, childVal)
    } else {
      return res
    }
  }

  ASSET_TYPES.forEach(function (type) {
    strats[type + 's'] = mergeAssets;
  });

  /**
   * Watchers.
   *
   * Watchers hashes should not overwrite one
   * another, so we merge them as arrays.
   */
  strats.watch = function (
    parentVal,
    childVal,
    vm,
    key
  ) {
    // work around Firefox's Object.prototype.watch...
    if (parentVal === nativeWatch) parentVal = undefined;
    if (childVal === nativeWatch) childVal = undefined;
    /* istanbul ignore if */
    if (!childVal) return Object.create(parentVal || null)
    {
      assertObjectType(key, childVal, vm);
    }
    if (!parentVal) return childVal
    const ret = {};
    extend(ret, parentVal);
    for (const key in childVal) {
      let parent = ret[key];
      const child = childVal[key];
      if (parent && !Array.isArray(parent)) {
        parent = [parent];
      }
      ret[key] = parent
        ? parent.concat(child)
        : Array.isArray(child) ? child : [child];
    }
    return ret
  };

  /**
   * Other object hashes.
   */
  strats.props =
  strats.methods =
  strats.inject =
  strats.computed = function (
    parentVal,
    childVal,
    vm,
    key
  ) {
    if (childVal && "development" !== 'production') {
      assertObjectType(key, childVal, vm);
    }
    if (!parentVal) return childVal
    const ret = Object.create(null);
    extend(ret, parentVal);
    if (childVal) extend(ret, childVal);
    return ret
  };
  strats.provide = mergeDataOrFn;

  /**
   * Default strategy.
   */
  const defaultStrat = function (parentVal, childVal) {
    return childVal === undefined
      ? parentVal
      : childVal
  };

  /**
   * Validate component names
   */
  function checkComponents (options) {
    for (const key in options.components) {
      validateComponentName(key);
    }
  }

  function validateComponentName (name) {
    if (!new RegExp(`^[a-zA-Z][\\-\\.0-9_${unicodeRegExp.source}]*$`).test(name)) {
      warn(
        'Invalid component name: "' + name + '". Component names ' +
        'should conform to valid custom element name in html5 specification.'
      );
    }
    if (isBuiltInTag(name) || config.isReservedTag(name)) {
      warn(
        'Do not use built-in or reserved HTML elements as component ' +
        'id: ' + name
      );
    }
  }

  /**
   * Ensure all props option syntax are normalized into the
   * Object-based format.
   */
  function normalizeProps (options, vm) {
    const props = options.props;
    if (!props) return
    const res = {};
    let i, val, name;
    if (Array.isArray(props)) {
      i = props.length;
      while (i--) {
        val = props[i];
        if (typeof val === 'string') {
          name = camelize(val);
          res[name] = { type: null };
        } else {
          warn('props must be strings when using array syntax.');
        }
      }
    } else if (isPlainObject(props)) {
      for (const key in props) {
        val = props[key];
        name = camelize(key);
        res[name] = isPlainObject(val)
          ? val
          : { type: val };
      }
    } else {
      warn(
        `Invalid value for option "props": expected an Array or an Object, ` +
        `but got ${toRawType(props)}.`,
        vm
      );
    }
    options.props = res;
  }

  /**
   * Normalize all injections into Object-based format
   */
  function normalizeInject (options, vm) {
    const inject = options.inject;
    if (!inject) return
    const normalized = options.inject = {};
    if (Array.isArray(inject)) {
      for (let i = 0; i < inject.length; i++) {
        normalized[inject[i]] = { from: inject[i] };
      }
    } else if (isPlainObject(inject)) {
      for (const key in inject) {
        const val = inject[key];
        normalized[key] = isPlainObject(val)
          ? extend({ from: key }, val)
          : { from: val };
      }
    } else {
      warn(
        `Invalid value for option "inject": expected an Array or an Object, ` +
        `but got ${toRawType(inject)}.`,
        vm
      );
    }
  }

  /**
   * Normalize raw function directives into object format.
   */
  function normalizeDirectives (options) {
    const dirs = options.directives;
    if (dirs) {
      for (const key in dirs) {
        const def$$1 = dirs[key];
        if (typeof def$$1 === 'function') {
          dirs[key] = { bind: def$$1, update: def$$1 };
        }
      }
    }
  }

  function assertObjectType (name, value, vm) {
    if (!isPlainObject(value)) {
      warn(
        `Invalid value for option "${name}": expected an Object, ` +
        `but got ${toRawType(value)}.`,
        vm
      );
    }
  }

  /**
   * Merge two option objects into a new one.
   * Core utility used in both instantiation and inheritance.
   */
  function mergeOptions (
    parent,
    child,
    vm
  ) {
    {
      checkComponents(child);
    }

    if (typeof child === 'function') {
      child = child.options;
    }

    normalizeProps(child, vm);
    normalizeInject(child, vm);
    normalizeDirectives(child);

    // Apply extends and mixins on the child options,
    // but only if it is a raw options object that isn't
    // the result of another mergeOptions call.
    // Only merged options has the _base property.
    if (!child._base) {
      if (child.extends) {
        parent = mergeOptions(parent, child.extends, vm);
      }
      if (child.mixins) {
        for (let i = 0, l = child.mixins.length; i < l; i++) {
          parent = mergeOptions(parent, child.mixins[i], vm);
        }
      }
    }

    const options = {};
    let key;
    for (key in parent) {
      mergeField(key);
    }
    for (key in child) {
      if (!hasOwn(parent, key)) {
        mergeField(key);
      }
    }
    function mergeField (key) {
      const strat = strats[key] || defaultStrat;
      options[key] = strat(parent[key], child[key], vm, key);
    }
    return options
  }

  /**
   * Resolve an asset.
   * This function is used because child instances need access
   * to assets defined in its ancestor chain.
   */
  function resolveAsset (
    options,
    type,
    id,
    warnMissing
  ) {
    /* istanbul ignore if */
    if (typeof id !== 'string') {
      return
    }
    const assets = options[type];
    // check local registration variations first
    if (hasOwn(assets, id)) return assets[id]
    const camelizedId = camelize(id);
    if (hasOwn(assets, camelizedId)) return assets[camelizedId]
    const PascalCaseId = capitalize(camelizedId);
    if (hasOwn(assets, PascalCaseId)) return assets[PascalCaseId]
    // fallback to prototype chain
    const res = assets[id] || assets[camelizedId] || assets[PascalCaseId];
    if (warnMissing && !res) {
      warn(
        'Failed to resolve ' + type.slice(0, -1) + ': ' + id,
        options
      );
    }
    return res
  }

  /*  */



  function validateProp (
    key,
    propOptions,
    propsData,
    vm
  ) {
    const prop = propOptions[key];
    const absent = !hasOwn(propsData, key);
    let value = propsData[key];
    // boolean casting
    const booleanIndex = getTypeIndex(Boolean, prop.type);
    if (booleanIndex > -1) {
      if (absent && !hasOwn(prop, 'default')) {
        value = false;
      } else if (value === '' || value === hyphenate(key)) {
        // only cast empty string / same name to boolean if
        // boolean has higher priority
        const stringIndex = getTypeIndex(String, prop.type);
        if (stringIndex < 0 || booleanIndex < stringIndex) {
          value = true;
        }
      }
    }
    // check default value
    if (value === undefined) {
      value = getPropDefaultValue(vm, prop, key);
      // since the default value is a fresh copy,
      // make sure to observe it.
      const prevShouldObserve = shouldObserve;
      toggleObserving(true);
      observe(value);
      toggleObserving(prevShouldObserve);
    }
    {
      assertProp(prop, key, value, vm, absent);
    }
    return value
  }

  /**
   * Get the default value of a prop.
   */
  function getPropDefaultValue (vm, prop, key) {
    // no default, return undefined
    if (!hasOwn(prop, 'default')) {
      return undefined
    }
    const def = prop.default;
    // warn against non-factory defaults for Object & Array
    if (isObject(def)) {
      warn(
        'Invalid default value for prop "' + key + '": ' +
        'Props with type Object/Array must use a factory function ' +
        'to return the default value.',
        vm
      );
    }
    // the raw prop value was also undefined from previous render,
    // return previous default value to avoid unnecessary watcher trigger
    if (vm && vm.$options.propsData &&
      vm.$options.propsData[key] === undefined &&
      vm._props[key] !== undefined
    ) {
      return vm._props[key]
    }
    // call factory function for non-Function types
    // a value is Function if its prototype is function even across different execution context
    return typeof def === 'function' && getType(prop.type) !== 'Function'
      ? def.call(vm)
      : def
  }

  /**
   * Assert whether a prop is valid.
   */
  function assertProp (
    prop,
    name,
    value,
    vm,
    absent
  ) {
    if (prop.required && absent) {
      warn(
        'Missing required prop: "' + name + '"',
        vm
      );
      return
    }
    if (value == null && !prop.required) {
      return
    }
    let type = prop.type;
    let valid = !type || type === true;
    const expectedTypes = [];
    if (type) {
      if (!Array.isArray(type)) {
        type = [type];
      }
      for (let i = 0; i < type.length && !valid; i++) {
        const assertedType = assertType(value, type[i], vm);
        expectedTypes.push(assertedType.expectedType || '');
        valid = assertedType.valid;
      }
    }

    const haveExpectedTypes = expectedTypes.some(t => t);
    if (!valid && haveExpectedTypes) {
      warn(
        getInvalidTypeMessage(name, value, expectedTypes),
        vm
      );
      return
    }
    const validator = prop.validator;
    if (validator) {
      if (!validator(value)) {
        warn(
          'Invalid prop: custom validator check failed for prop "' + name + '".',
          vm
        );
      }
    }
  }

  const simpleCheckRE = /^(String|Number|Boolean|Function|Symbol|BigInt)$/;

  function assertType (value, type, vm) {
    let valid;
    const expectedType = getType(type);
    if (simpleCheckRE.test(expectedType)) {
      const t = typeof value;
      valid = t === expectedType.toLowerCase();
      // for primitive wrapper objects
      if (!valid && t === 'object') {
        valid = value instanceof type;
      }
    } else if (expectedType === 'Object') {
      valid = isPlainObject(value);
    } else if (expectedType === 'Array') {
      valid = Array.isArray(value);
    } else {
      try {
        valid = value instanceof type;
      } catch (e) {
        warn('Invalid prop type: "' + String(type) + '" is not a constructor', vm);
        valid = false;
      }
    }
    return {
      valid,
      expectedType
    }
  }

  const functionTypeCheckRE = /^\s*function (\w+)/;

  /**
   * Use function string name to check built-in types,
   * because a simple equality check will fail when running
   * across different vms / iframes.
   */
  function getType (fn) {
    const match = fn && fn.toString().match(functionTypeCheckRE);
    return match ? match[1] : ''
  }

  function isSameType (a, b) {
    return getType(a) === getType(b)
  }

  function getTypeIndex (type, expectedTypes) {
    if (!Array.isArray(expectedTypes)) {
      return isSameType(expectedTypes, type) ? 0 : -1
    }
    for (let i = 0, len = expectedTypes.length; i < len; i++) {
      if (isSameType(expectedTypes[i], type)) {
        return i
      }
    }
    return -1
  }

  function getInvalidTypeMessage (name, value, expectedTypes) {
    let message = `Invalid prop: type check failed for prop "${name}".` +
      ` Expected ${expectedTypes.map(capitalize).join(', ')}`;
    const expectedType = expectedTypes[0];
    const receivedType = toRawType(value);
    // check if we need to specify expected value
    if (
      expectedTypes.length === 1 &&
      isExplicable(expectedType) &&
      isExplicable(typeof value) &&
      !isBoolean(expectedType, receivedType)
    ) {
      message += ` with value ${styleValue(value, expectedType)}`;
    }
    message += `, got ${receivedType} `;
    // check if we need to specify received value
    if (isExplicable(receivedType)) {
      message += `with value ${styleValue(value, receivedType)}.`;
    }
    return message
  }

  function styleValue (value, type) {
    if (type === 'String') {
      return `"${value}"`
    } else if (type === 'Number') {
      return `${Number(value)}`
    } else {
      return `${value}`
    }
  }

  const EXPLICABLE_TYPES = ['string', 'number', 'boolean'];
  function isExplicable (value) {
    return EXPLICABLE_TYPES.some(elem => value.toLowerCase() === elem)
  }

  function isBoolean (...args) {
    return args.some(elem => elem.toLowerCase() === 'boolean')
  }

  /*  */

  function handleError (err, vm, info) {
    // Deactivate deps tracking while processing error handler to avoid possible infinite rendering.
    // See: https://github.com/vuejs/vuex/issues/1505
    pushTarget();
    try {
      if (vm) {
        let cur = vm;
        while ((cur = cur.$parent)) {
          const hooks = cur.$options.errorCaptured;
          if (hooks) {
            for (let i = 0; i < hooks.length; i++) {
              try {
                const capture = hooks[i].call(cur, err, vm, info) === false;
                if (capture) return
              } catch (e) {
                globalHandleError(e, cur, 'errorCaptured hook');
              }
            }
          }
        }
      }
      globalHandleError(err, vm, info);
    } finally {
      popTarget();
    }
  }

  function invokeWithErrorHandling (
    handler,
    context,
    args,
    vm,
    info
  ) {
    let res;
    try {
      res = args ? handler.apply(context, args) : handler.call(context);
      if (res && !res._isVue && isPromise(res) && !res._handled) {
        res.catch(e => handleError(e, vm, info + ` (Promise/async)`));
        // issue #9511
        // avoid catch triggering multiple times when nested calls
        res._handled = true;
      }
    } catch (e) {
      handleError(e, vm, info);
    }
    return res
  }

  function globalHandleError (err, vm, info) {
    if (config.errorHandler) {
      try {
        return config.errorHandler.call(null, err, vm, info)
      } catch (e) {
        // if the user intentionally throws the original error in the handler,
        // do not log it twice
        if (e !== err) {
          logError(e, null, 'config.errorHandler');
        }
      }
    }
    logError(err, vm, info);
  }

  function logError (err, vm, info) {
    {
      warn(`Error in ${info}: "${err.toString()}"`, vm);
    }
    /* istanbul ignore else */
    if ((inBrowser || inWeex) && typeof console !== 'undefined') {
      console.error(err);
    } else {
      throw err
    }
  }

  /*  */

  let isUsingMicroTask = false;

  const callbacks = [];
  let pending = false;

  function flushCallbacks () {
    pending = false;
    const copies = callbacks.slice(0);
    callbacks.length = 0;
    for (let i = 0; i < copies.length; i++) {
      copies[i]();
    }
  }

  // Here we have async deferring wrappers using microtasks.
  // In 2.5 we used (macro) tasks (in combination with microtasks).
  // However, it has subtle problems when state is changed right before repaint
  // (e.g. #6813, out-in transitions).
  // Also, using (macro) tasks in event handler would cause some weird behaviors
  // that cannot be circumvented (e.g. #7109, #7153, #7546, #7834, #8109).
  // So we now use microtasks everywhere, again.
  // A major drawback of this tradeoff is that there are some scenarios
  // where microtasks have too high a priority and fire in between supposedly
  // sequential events (e.g. #4521, #6690, which have workarounds)
  // or even between bubbling of the same event (#6566).
  let timerFunc;

  // The nextTick behavior leverages the microtask queue, which can be accessed
  // via either native Promise.then or MutationObserver.
  // MutationObserver has wider support, however it is seriously bugged in
  // UIWebView in iOS >= 9.3.3 when triggered in touch event handlers. It
  // completely stops working after triggering a few times... so, if native
  // Promise is available, we will use it:
  /* istanbul ignore next, $flow-disable-line */
  if (typeof Promise !== 'undefined' && isNative(Promise)) {
    const p = Promise.resolve();
    timerFunc = () => {
      p.then(flushCallbacks);
      // In problematic UIWebViews, Promise.then doesn't completely break, but
      // it can get stuck in a weird state where callbacks are pushed into the
      // microtask queue but the queue isn't being flushed, until the browser
      // needs to do some other work, e.g. handle a timer. Therefore we can
      // "force" the microtask queue to be flushed by adding an empty timer.
      if (isIOS) setTimeout(noop);
    };
    isUsingMicroTask = true;
  } else if (!isIE && typeof MutationObserver !== 'undefined' && (
    isNative(MutationObserver) ||
    // PhantomJS and iOS 7.x
    MutationObserver.toString() === '[object MutationObserverConstructor]'
  )) {
    // Use MutationObserver where native Promise is not available,
    // e.g. PhantomJS, iOS7, Android 4.4
    // (#6466 MutationObserver is unreliable in IE11)
    let counter = 1;
    const observer = new MutationObserver(flushCallbacks);
    const textNode = document.createTextNode(String(counter));
    observer.observe(textNode, {
      characterData: true
    });
    timerFunc = () => {
      counter = (counter + 1) % 2;
      textNode.data = String(counter);
    };
    isUsingMicroTask = true;
  } else if (typeof setImmediate !== 'undefined' && isNative(setImmediate)) {
    // Fallback to setImmediate.
    // Technically it leverages the (macro) task queue,
    // but it is still a better choice than setTimeout.
    timerFunc = () => {
      setImmediate(flushCallbacks);
    };
  } else {
    // Fallback to setTimeout.
    timerFunc = () => {
      setTimeout(flushCallbacks, 0);
    };
  }

  function nextTick (cb, ctx) {
    let _resolve;
    callbacks.push(() => {
      if (cb) {
        try {
          cb.call(ctx);
        } catch (e) {
          handleError(e, ctx, 'nextTick');
        }
      } else if (_resolve) {
        _resolve(ctx);
      }
    });
    if (!pending) {
      pending = true;
      timerFunc();
    }
    // $flow-disable-line
    if (!cb && typeof Promise !== 'undefined') {
      return new Promise(resolve => {
        _resolve = resolve;
      })
    }
  }

  /*  */

  let mark;
  let measure;

  {
    const perf = inBrowser && window.performance;
    /* istanbul ignore if */
    if (
      perf &&
      perf.mark &&
      perf.measure &&
      perf.clearMarks &&
      perf.clearMeasures
    ) {
      mark = tag => perf.mark(tag);
      measure = (name, startTag, endTag) => {
        perf.measure(name, startTag, endTag);
        perf.clearMarks(startTag);
        perf.clearMarks(endTag);
        // perf.clearMeasures(name)
      };
    }
  }

  /* not type checking this file because flow doesn't play well with Proxy */

  let initProxy;

  {
    const allowedGlobals = makeMap(
      'Infinity,undefined,NaN,isFinite,isNaN,' +
      'parseFloat,parseInt,decodeURI,decodeURIComponent,encodeURI,encodeURIComponent,' +
      'Math,Number,Date,Array,Object,Boolean,String,RegExp,Map,Set,JSON,Intl,BigInt,' +
      'require' // for Webpack/Browserify
    );

    const warnNonPresent = (target, key) => {
      warn(
        `Property or method "${key}" is not defined on the instance but ` +
        'referenced during render. Make sure that this property is reactive, ' +
        'either in the data option, or for class-based components, by ' +
        'initializing the property. ' +
        'See: https://vuejs.org/v2/guide/reactivity.html#Declaring-Reactive-Properties.',
        target
      );
    };

    const warnReservedPrefix = (target, key) => {
      warn(
        `Property "${key}" must be accessed with "$data.${key}" because ` +
        'properties starting with "$" or "_" are not proxied in the Vue instance to ' +
        'prevent conflicts with Vue internals. ' +
        'See: https://vuejs.org/v2/api/#data',
        target
      );
    };

    const hasProxy =
      typeof Proxy !== 'undefined' && isNative(Proxy);

    if (hasProxy) {
      const isBuiltInModifier = makeMap('stop,prevent,self,ctrl,shift,alt,meta,exact');
      config.keyCodes = new Proxy(config.keyCodes, {
        set (target, key, value) {
          if (isBuiltInModifier(key)) {
            warn(`Avoid overwriting built-in modifier in config.keyCodes: .${key}`);
            return false
          } else {
            target[key] = value;
            return true
          }
        }
      });
    }

    const hasHandler = {
      has (target, key) {
        const has = key in target;
        const isAllowed = allowedGlobals(key) ||
          (typeof key === 'string' && key.charAt(0) === '_' && !(key in target.$data));
        if (!has && !isAllowed) {
          if (key in target.$data) warnReservedPrefix(target, key);
          else warnNonPresent(target, key);
        }
        return has || !isAllowed
      }
    };

    const getHandler = {
      get (target, key) {
        if (typeof key === 'string' && !(key in target)) {
          if (key in target.$data) warnReservedPrefix(target, key);
          else warnNonPresent(target, key);
        }
        return target[key]
      }
    };

    initProxy = function initProxy (vm) {
      if (hasProxy) {
        // determine which proxy handler to use
        const options = vm.$options;
        const handlers = options.render && options.render._withStripped
          ? getHandler
          : hasHandler;
        vm._renderProxy = new Proxy(vm, handlers);
      } else {
        vm._renderProxy = vm;
      }
    };
  }

  /*  */

  const seenObjects = new _Set();

  /**
   * Recursively traverse an object to evoke all converted
   * getters, so that every nested property inside the object
   * is collected as a "deep" dependency.
   */
  function traverse (val) {
    _traverse(val, seenObjects);
    seenObjects.clear();
  }

  function _traverse (val, seen) {
    let i, keys;
    const isA = Array.isArray(val);
    if ((!isA && !isObject(val)) || Object.isFrozen(val) || val instanceof VNode) {
      return
    }
    if (val.__ob__) {
      const depId = val.__ob__.dep.id;
      if (seen.has(depId)) {
        return
      }
      seen.add(depId);
    }
    if (isA) {
      i = val.length;
      while (i--) _traverse(val[i], seen);
    } else {
      keys = Object.keys(val);
      i = keys.length;
      while (i--) _traverse(val[keys[i]], seen);
    }
  }

  /*  */

  const normalizeEvent = cached((name) => {
    const passive = name.charAt(0) === '&';
    name = passive ? name.slice(1) : name;
    const once$$1 = name.charAt(0) === '~'; // Prefixed last, checked first
    name = once$$1 ? name.slice(1) : name;
    const capture = name.charAt(0) === '!';
    name = capture ? name.slice(1) : name;
    return {
      name,
      once: once$$1,
      capture,
      passive
    }
  });

  function createFnInvoker (fns, vm) {
    function invoker () {
      const fns = invoker.fns;
      if (Array.isArray(fns)) {
        const cloned = fns.slice();
        for (let i = 0; i < cloned.length; i++) {
          invokeWithErrorHandling(cloned[i], null, arguments, vm, `v-on handler`);
        }
      } else {
        // return handler return value for single handlers
        return invokeWithErrorHandling(fns, null, arguments, vm, `v-on handler`)
      }
    }
    invoker.fns = fns;
    return invoker
  }

  function updateListeners (
    on,
    oldOn,
    add,
    remove$$1,
    createOnceHandler,
    vm
  ) {
    let name, cur, old, event;
    for (name in on) {
      cur = on[name];
      old = oldOn[name];
      event = normalizeEvent(name);
      if (isUndef(cur)) {
        warn(
          `Invalid handler for event "${event.name}": got ` + String(cur),
          vm
        );
      } else if (isUndef(old)) {
        if (isUndef(cur.fns)) {
          cur = on[name] = createFnInvoker(cur, vm);
        }
        if (isTrue(event.once)) {
          cur = on[name] = createOnceHandler(event.name, cur, event.capture);
        }
        add(event.name, cur, event.capture, event.passive, event.params);
      } else if (cur !== old) {
        old.fns = cur;
        on[name] = old;
      }
    }
    for (name in oldOn) {
      if (isUndef(on[name])) {
        event = normalizeEvent(name);
        remove$$1(event.name, oldOn[name], event.capture);
      }
    }
  }

  /*  */

  function mergeVNodeHook (def, hookKey, hook) {
    if (def instanceof VNode) {
      def = def.data.hook || (def.data.hook = {});
    }
    let invoker;
    const oldHook = def[hookKey];

    function wrappedHook () {
      hook.apply(this, arguments);
      // important: remove merged hook to ensure it's called only once
      // and prevent memory leak
      remove(invoker.fns, wrappedHook);
    }

    if (isUndef(oldHook)) {
      // no existing hook
      invoker = createFnInvoker([wrappedHook]);
    } else {
      /* istanbul ignore if */
      if (isDef(oldHook.fns) && isTrue(oldHook.merged)) {
        // already a merged invoker
        invoker = oldHook;
        invoker.fns.push(wrappedHook);
      } else {
        // existing plain hook
        invoker = createFnInvoker([oldHook, wrappedHook]);
      }
    }

    invoker.merged = true;
    def[hookKey] = invoker;
  }

  /*  */

  function extractPropsFromVNodeData (
    data,
    Ctor,
    tag
  ) {
    // we are only extracting raw values here.
    // validation and default values are handled in the child
    // component itself.
    const propOptions = Ctor.options.props;
    if (isUndef(propOptions)) {
      return
    }
    const res = {};
    const { attrs, props } = data;
    if (isDef(attrs) || isDef(props)) {
      for (const key in propOptions) {
        const altKey = hyphenate(key);
        {
          const keyInLowerCase = key.toLowerCase();
          if (
            key !== keyInLowerCase &&
            attrs && hasOwn(attrs, keyInLowerCase)
          ) {
            tip(
              `Prop "${keyInLowerCase}" is passed to component ` +
              `${formatComponentName(tag || Ctor)}, but the declared prop name is` +
              ` "${key}". ` +
              `Note that HTML attributes are case-insensitive and camelCased ` +
              `props need to use their kebab-case equivalents when using in-DOM ` +
              `templates. You should probably use "${altKey}" instead of "${key}".`
            );
          }
        }
        checkProp(res, props, key, altKey, true) ||
        checkProp(res, attrs, key, altKey, false);
      }
    }
    return res
  }

  function checkProp (
    res,
    hash,
    key,
    altKey,
    preserve
  ) {
    if (isDef(hash)) {
      if (hasOwn(hash, key)) {
        res[key] = hash[key];
        if (!preserve) {
          delete hash[key];
        }
        return true
      } else if (hasOwn(hash, altKey)) {
        res[key] = hash[altKey];
        if (!preserve) {
          delete hash[altKey];
        }
        return true
      }
    }
    return false
  }

  /*  */

  // The template compiler attempts to minimize the need for normalization by
  // statically analyzing the template at compile time.
  //
  // For plain HTML markup, normalization can be completely skipped because the
  // generated render function is guaranteed to return Array<VNode>. There are
  // two cases where extra normalization is needed:

  // 1. When the children contains components - because a functional component
  // may return an Array instead of a single root. In this case, just a simple
  // normalization is needed - if any child is an Array, we flatten the whole
  // thing with Array.prototype.concat. It is guaranteed to be only 1-level deep
  // because functional components already normalize their own children.
  function simpleNormalizeChildren (children) {
    for (let i = 0; i < children.length; i++) {
      if (Array.isArray(children[i])) {
        return Array.prototype.concat.apply([], children)
      }
    }
    return children
  }

  // 2. When the children contains constructs that always generated nested Arrays,
  // e.g. <template>, <slot>, v-for, or when the children is provided by user
  // with hand-written render functions / JSX. In such cases a full normalization
  // is needed to cater to all possible types of children values.
  function normalizeChildren (children) {
    return isPrimitive(children)
      ? [createTextVNode(children)]
      : Array.isArray(children)
        ? normalizeArrayChildren(children)
        : undefined
  }

  function isTextNode (node) {
    return isDef(node) && isDef(node.text) && isFalse(node.isComment)
  }

  function normalizeArrayChildren (children, nestedIndex) {
    const res = [];
    let i, c, lastIndex, last;
    for (i = 0; i < children.length; i++) {
      c = children[i];
      if (isUndef(c) || typeof c === 'boolean') continue
      lastIndex = res.length - 1;
      last = res[lastIndex];
      //  nested
      if (Array.isArray(c)) {
        if (c.length > 0) {
          c = normalizeArrayChildren(c, `${nestedIndex || ''}_${i}`);
          // merge adjacent text nodes
          if (isTextNode(c[0]) && isTextNode(last)) {
            res[lastIndex] = createTextVNode(last.text + (c[0]).text);
            c.shift();
          }
          res.push.apply(res, c);
        }
      } else if (isPrimitive(c)) {
        if (isTextNode(last)) {
          // merge adjacent text nodes
          // this is necessary for SSR hydration because text nodes are
          // essentially merged when rendered to HTML strings
          res[lastIndex] = createTextVNode(last.text + c);
        } else if (c !== '') {
          // convert primitive to vnode
          res.push(createTextVNode(c));
        }
      } else {
        if (isTextNode(c) && isTextNode(last)) {
          // merge adjacent text nodes
          res[lastIndex] = createTextVNode(last.text + c.text);
        } else {
          // default key for nested array children (likely generated by v-for)
          if (isTrue(children._isVList) &&
            isDef(c.tag) &&
            isUndef(c.key) &&
            isDef(nestedIndex)) {
            c.key = `__vlist${nestedIndex}_${i}__`;
          }
          res.push(c);
        }
      }
    }
    return res
  }

  /*  */

  function initProvide (vm) {
    const provide = vm.$options.provide;
    if (provide) {
      vm._provided = typeof provide === 'function'
        ? provide.call(vm)
        : provide;
    }
  }

  function initInjections (vm) {
    const result = resolveInject(vm.$options.inject, vm);
    if (result) {
      toggleObserving(false);
      Object.keys(result).forEach(key => {
        /* istanbul ignore else */
        {
          defineReactive$$1(vm, key, result[key], () => {
            warn(
              `Avoid mutating an injected value directly since the changes will be ` +
              `overwritten whenever the provided component re-renders. ` +
              `injection being mutated: "${key}"`,
              vm
            );
          });
        }
      });
      toggleObserving(true);
    }
  }

  function resolveInject (inject, vm) {
    if (inject) {
      // inject is :any because flow is not smart enough to figure out cached
      const result = Object.create(null);
      const keys = hasSymbol
        ? Reflect.ownKeys(inject)
        : Object.keys(inject);

      for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        // #6574 in case the inject object is observed...
        if (key === '__ob__') continue
        const provideKey = inject[key].from;
        let source = vm;
        while (source) {
          if (source._provided && hasOwn(source._provided, provideKey)) {
            result[key] = source._provided[provideKey];
            break
          }
          source = source.$parent;
        }
        if (!source) {
          if ('default' in inject[key]) {
            const provideDefault = inject[key].default;
            result[key] = typeof provideDefault === 'function'
              ? provideDefault.call(vm)
              : provideDefault;
          } else {
            warn(`Injection "${key}" not found`, vm);
          }
        }
      }
      return result
    }
  }

  /*  */



  /**
   * Runtime helper for resolving raw children VNodes into a slot object.
   */
  function resolveSlots (
    children,
    context
  ) {
    if (!children || !children.length) {
      return {}
    }
    const slots = {};
    for (let i = 0, l = children.length; i < l; i++) {
      const child = children[i];
      const data = child.data;
      // remove slot attribute if the node is resolved as a Vue slot node
      if (data && data.attrs && data.attrs.slot) {
        delete data.attrs.slot;
      }
      // named slots should only be respected if the vnode was rendered in the
      // same context.
      if ((child.context === context || child.fnContext === context) &&
        data && data.slot != null
      ) {
        const name = data.slot;
        const slot = (slots[name] || (slots[name] = []));
        if (child.tag === 'template') {
          slot.push.apply(slot, child.children || []);
        } else {
          slot.push(child);
        }
      } else {
        (slots.default || (slots.default = [])).push(child);
      }
    }
    // ignore slots that contains only whitespace
    for (const name in slots) {
      if (slots[name].every(isWhitespace)) {
        delete slots[name];
      }
    }
    return slots
  }

  function isWhitespace (node) {
    return (node.isComment && !node.asyncFactory) || node.text === ' '
  }

  /*  */

  function isAsyncPlaceholder (node) {
    return node.isComment && node.asyncFactory
  }

  /*  */

  function normalizeScopedSlots (
    slots,
    normalSlots,
    prevSlots
  ) {
    let res;
    const hasNormalSlots = Object.keys(normalSlots).length > 0;
    const isStable = slots ? !!slots.$stable : !hasNormalSlots;
    const key = slots && slots.$key;
    if (!slots) {
      res = {};
    } else if (slots._normalized) {
      // fast path 1: child component re-render only, parent did not change
      return slots._normalized
    } else if (
      isStable &&
      prevSlots &&
      prevSlots !== emptyObject &&
      key === prevSlots.$key &&
      !hasNormalSlots &&
      !prevSlots.$hasNormal
    ) {
      // fast path 2: stable scoped slots w/ no normal slots to proxy,
      // only need to normalize once
      return prevSlots
    } else {
      res = {};
      for (const key in slots) {
        if (slots[key] && key[0] !== '$') {
          res[key] = normalizeScopedSlot(normalSlots, key, slots[key]);
        }
      }
    }
    // expose normal slots on scopedSlots
    for (const key in normalSlots) {
      if (!(key in res)) {
        res[key] = proxyNormalSlot(normalSlots, key);
      }
    }
    // avoriaz seems to mock a non-extensible $scopedSlots object
    // and when that is passed down this would cause an error
    if (slots && Object.isExtensible(slots)) {
      (slots)._normalized = res;
    }
    def(res, '$stable', isStable);
    def(res, '$key', key);
    def(res, '$hasNormal', hasNormalSlots);
    return res
  }

  function normalizeScopedSlot(normalSlots, key, fn) {
    const normalized = function () {
      let res = arguments.length ? fn.apply(null, arguments) : fn({});
      res = res && typeof res === 'object' && !Array.isArray(res)
        ? [res] // single vnode
        : normalizeChildren(res);
      let vnode = res && res[0];
      return res && (
        !vnode ||
        (res.length === 1 && vnode.isComment && !isAsyncPlaceholder(vnode)) // #9658, #10391
      ) ? undefined
        : res
    };
    // this is a slot using the new v-slot syntax without scope. although it is
    // compiled as a scoped slot, render fn users would expect it to be present
    // on this.$slots because the usage is semantically a normal slot.
    if (fn.proxy) {
      Object.defineProperty(normalSlots, key, {
        get: normalized,
        enumerable: true,
        configurable: true
      });
    }
    return normalized
  }

  function proxyNormalSlot(slots, key) {
    return () => slots[key]
  }

  /*  */

  /**
   * Runtime helper for rendering v-for lists.
   */
  function renderList (
    val,
    render
  ) {
    let ret, i, l, keys, key;
    if (Array.isArray(val) || typeof val === 'string') {
      ret = new Array(val.length);
      for (i = 0, l = val.length; i < l; i++) {
        ret[i] = render(val[i], i);
      }
    } else if (typeof val === 'number') {
      ret = new Array(val);
      for (i = 0; i < val; i++) {
        ret[i] = render(i + 1, i);
      }
    } else if (isObject(val)) {
      if (hasSymbol && val[Symbol.iterator]) {
        ret = [];
        const iterator = val[Symbol.iterator]();
        let result = iterator.next();
        while (!result.done) {
          ret.push(render(result.value, ret.length));
          result = iterator.next();
        }
      } else {
        keys = Object.keys(val);
        ret = new Array(keys.length);
        for (i = 0, l = keys.length; i < l; i++) {
          key = keys[i];
          ret[i] = render(val[key], key, i);
        }
      }
    }
    if (!isDef(ret)) {
      ret = [];
    }
    (ret)._isVList = true;
    return ret
  }

  /*  */

  /**
   * Runtime helper for rendering <slot>
   */
  function renderSlot (
    name,
    fallbackRender,
    props,
    bindObject
  ) {
    const scopedSlotFn = this.$scopedSlots[name];
    let nodes;
    if (scopedSlotFn) {
      // scoped slot
      props = props || {};
      if (bindObject) {
        if (!isObject(bindObject)) {
          warn('slot v-bind without argument expects an Object', this);
        }
        props = extend(extend({}, bindObject), props);
      }
      nodes =
        scopedSlotFn(props) ||
        (typeof fallbackRender === 'function' ? fallbackRender() : fallbackRender);
    } else {
      nodes =
        this.$slots[name] ||
        (typeof fallbackRender === 'function' ? fallbackRender() : fallbackRender);
    }

    const target = props && props.slot;
    if (target) {
      return this.$createElement('template', { slot: target }, nodes)
    } else {
      return nodes
    }
  }

  /*  */

  /**
   * Runtime helper for resolving filters
   */
  function resolveFilter (id) {
    return resolveAsset(this.$options, 'filters', id, true) || identity
  }

  /*  */

  function isKeyNotMatch (expect, actual) {
    if (Array.isArray(expect)) {
      return expect.indexOf(actual) === -1
    } else {
      return expect !== actual
    }
  }

  /**
   * Runtime helper for checking keyCodes from config.
   * exposed as Vue.prototype._k
   * passing in eventKeyName as last argument separately for backwards compat
   */
  function checkKeyCodes (
    eventKeyCode,
    key,
    builtInKeyCode,
    eventKeyName,
    builtInKeyName
  ) {
    const mappedKeyCode = config.keyCodes[key] || builtInKeyCode;
    if (builtInKeyName && eventKeyName && !config.keyCodes[key]) {
      return isKeyNotMatch(builtInKeyName, eventKeyName)
    } else if (mappedKeyCode) {
      return isKeyNotMatch(mappedKeyCode, eventKeyCode)
    } else if (eventKeyName) {
      return hyphenate(eventKeyName) !== key
    }
    return eventKeyCode === undefined
  }

  /*  */

  /**
   * Runtime helper for merging v-bind="object" into a VNode's data.
   */
  function bindObjectProps (
    data,
    tag,
    value,
    asProp,
    isSync
  ) {
    if (value) {
      if (!isObject(value)) {
        warn(
          'v-bind without argument expects an Object or Array value',
          this
        );
      } else {
        if (Array.isArray(value)) {
          value = toObject(value);
        }
        let hash;
        for (const key in value) {
          if (
            key === 'class' ||
            key === 'style' ||
            isReservedAttribute(key)
          ) {
            hash = data;
          } else {
            const type = data.attrs && data.attrs.type;
            hash = asProp || config.mustUseProp(tag, type, key)
              ? data.domProps || (data.domProps = {})
              : data.attrs || (data.attrs = {});
          }
          const camelizedKey = camelize(key);
          const hyphenatedKey = hyphenate(key);
          if (!(camelizedKey in hash) && !(hyphenatedKey in hash)) {
            hash[key] = value[key];

            if (isSync) {
              const on = data.on || (data.on = {});
              on[`update:${key}`] = function ($event) {
                value[key] = $event;
              };
            }
          }
        }
      }
    }
    return data
  }

  /*  */

  /**
   * Runtime helper for rendering static trees.
   */
  function renderStatic (
    index,
    isInFor
  ) {
    const cached = this._staticTrees || (this._staticTrees = []);
    let tree = cached[index];
    // if has already-rendered static tree and not inside v-for,
    // we can reuse the same tree.
    if (tree && !isInFor) {
      return tree
    }
    // otherwise, render a fresh tree.
    tree = cached[index] = this.$options.staticRenderFns[index].call(
      this._renderProxy,
      null,
      this // for render fns generated for functional component templates
    );
    markStatic(tree, `__static__${index}`, false);
    return tree
  }

  /**
   * Runtime helper for v-once.
   * Effectively it means marking the node as static with a unique key.
   */
  function markOnce (
    tree,
    index,
    key
  ) {
    markStatic(tree, `__once__${index}${key ? `_${key}` : ``}`, true);
    return tree
  }

  function markStatic (
    tree,
    key,
    isOnce
  ) {
    if (Array.isArray(tree)) {
      for (let i = 0; i < tree.length; i++) {
        if (tree[i] && typeof tree[i] !== 'string') {
          markStaticNode(tree[i], `${key}_${i}`, isOnce);
        }
      }
    } else {
      markStaticNode(tree, key, isOnce);
    }
  }

  function markStaticNode (node, key, isOnce) {
    node.isStatic = true;
    node.key = key;
    node.isOnce = isOnce;
  }

  /*  */

  function bindObjectListeners (data, value) {
    if (value) {
      if (!isPlainObject(value)) {
        warn(
          'v-on without argument expects an Object value',
          this
        );
      } else {
        const on = data.on = data.on ? extend({}, data.on) : {};
        for (const key in value) {
          const existing = on[key];
          const ours = value[key];
          on[key] = existing ? [].concat(existing, ours) : ours;
        }
      }
    }
    return data
  }

  /*  */

  function resolveScopedSlots (
    fns, // see flow/vnode
    res,
    // the following are added in 2.6
    hasDynamicKeys,
    contentHashKey
  ) {
    res = res || { $stable: !hasDynamicKeys };
    for (let i = 0; i < fns.length; i++) {
      const slot = fns[i];
      if (Array.isArray(slot)) {
        resolveScopedSlots(slot, res, hasDynamicKeys);
      } else if (slot) {
        // marker for reverse proxying v-slot without scope on this.$slots
        if (slot.proxy) {
          slot.fn.proxy = true;
        }
        res[slot.key] = slot.fn;
      }
    }
    if (contentHashKey) {
      (res).$key = contentHashKey;
    }
    return res
  }

  /*  */

  function bindDynamicKeys (baseObj, values) {
    for (let i = 0; i < values.length; i += 2) {
      const key = values[i];
      if (typeof key === 'string' && key) {
        baseObj[values[i]] = values[i + 1];
      } else if (key !== '' && key !== null) {
        // null is a special value for explicitly removing a binding
        warn(
          `Invalid value for dynamic directive argument (expected string or null): ${key}`,
          this
        );
      }
    }
    return baseObj
  }

  // helper to dynamically append modifier runtime markers to event names.
  // ensure only append when value is already string, otherwise it will be cast
  // to string and cause the type check to miss.
  function prependModifier (value, symbol) {
    return typeof value === 'string' ? symbol + value : value
  }

  /*  */

  function installRenderHelpers (target) {
    target._o = markOnce;
    target._n = toNumber;
    target._s = toString;
    target._l = renderList;
    target._t = renderSlot;
    target._q = looseEqual;
    target._i = looseIndexOf;
    target._m = renderStatic;
    target._f = resolveFilter;
    target._k = checkKeyCodes;
    target._b = bindObjectProps;
    target._v = createTextVNode;
    target._e = createEmptyVNode;
    target._u = resolveScopedSlots;
    target._g = bindObjectListeners;
    target._d = bindDynamicKeys;
    target._p = prependModifier;
  }

  /*  */

  function FunctionalRenderContext (
    data,
    props,
    children,
    parent,
    Ctor
  ) {
    const options = Ctor.options;
    // ensure the createElement function in functional components
    // gets a unique context - this is necessary for correct named slot check
    let contextVm;
    if (hasOwn(parent, '_uid')) {
      contextVm = Object.create(parent);
      // $flow-disable-line
      contextVm._original = parent;
    } else {
      // the context vm passed in is a functional context as well.
      // in this case we want to make sure we are able to get a hold to the
      // real context instance.
      contextVm = parent;
      // $flow-disable-line
      parent = parent._original;
    }
    const isCompiled = isTrue(options._compiled);
    const needNormalization = !isCompiled;

    this.data = data;
    this.props = props;
    this.children = children;
    this.parent = parent;
    this.listeners = data.on || emptyObject;
    this.injections = resolveInject(options.inject, parent);
    this.slots = () => {
      if (!this.$slots) {
        normalizeScopedSlots(
          data.scopedSlots,
          this.$slots = resolveSlots(children, parent)
        );
      }
      return this.$slots
    };

    Object.defineProperty(this, 'scopedSlots', ({
      enumerable: true,
      get () {
        return normalizeScopedSlots(data.scopedSlots, this.slots())
      }
    }));

    // support for compiled functional template
    if (isCompiled) {
      // exposing $options for renderStatic()
      this.$options = options;
      // pre-resolve slots for renderSlot()
      this.$slots = this.slots();
      this.$scopedSlots = normalizeScopedSlots(data.scopedSlots, this.$slots);
    }

    if (options._scopeId) {
      this._c = (a, b, c, d) => {
        const vnode = createElement(contextVm, a, b, c, d, needNormalization);
        if (vnode && !Array.isArray(vnode)) {
          vnode.fnScopeId = options._scopeId;
          vnode.fnContext = parent;
        }
        return vnode
      };
    } else {
      this._c = (a, b, c, d) => createElement(contextVm, a, b, c, d, needNormalization);
    }
  }

  installRenderHelpers(FunctionalRenderContext.prototype);

  function createFunctionalComponent (
    Ctor,
    propsData,
    data,
    contextVm,
    children
  ) {
    const options = Ctor.options;
    const props = {};
    const propOptions = options.props;
    if (isDef(propOptions)) {
      for (const key in propOptions) {
        props[key] = validateProp(key, propOptions, propsData || emptyObject);
      }
    } else {
      if (isDef(data.attrs)) mergeProps(props, data.attrs);
      if (isDef(data.props)) mergeProps(props, data.props);
    }

    const renderContext = new FunctionalRenderContext(
      data,
      props,
      children,
      contextVm,
      Ctor
    );

    const vnode = options.render.call(null, renderContext._c, renderContext);

    if (vnode instanceof VNode) {
      return cloneAndMarkFunctionalResult(vnode, data, renderContext.parent, options, renderContext)
    } else if (Array.isArray(vnode)) {
      const vnodes = normalizeChildren(vnode) || [];
      const res = new Array(vnodes.length);
      for (let i = 0; i < vnodes.length; i++) {
        res[i] = cloneAndMarkFunctionalResult(vnodes[i], data, renderContext.parent, options, renderContext);
      }
      return res
    }
  }

  function cloneAndMarkFunctionalResult (vnode, data, contextVm, options, renderContext) {
    // #7817 clone node before setting fnContext, otherwise if the node is reused
    // (e.g. it was from a cached normal slot) the fnContext causes named slots
    // that should not be matched to match.
    const clone = cloneVNode(vnode);
    clone.fnContext = contextVm;
    clone.fnOptions = options;
    {
      (clone.devtoolsMeta = clone.devtoolsMeta || {}).renderContext = renderContext;
    }
    if (data.slot) {
      (clone.data || (clone.data = {})).slot = data.slot;
    }
    return clone
  }

  function mergeProps (to, from) {
    for (const key in from) {
      to[camelize(key)] = from[key];
    }
  }

  /*  */

  /*  */

  /*  */

  /*  */

  // inline hooks to be invoked on component VNodes during patch
  const componentVNodeHooks = {
    init (vnode, hydrating) {
      if (
        vnode.componentInstance &&
        !vnode.componentInstance._isDestroyed &&
        vnode.data.keepAlive
      ) {
        // kept-alive components, treat as a patch
        const mountedNode = vnode; // work around flow
        componentVNodeHooks.prepatch(mountedNode, mountedNode);
      } else {
        const child = vnode.componentInstance = createComponentInstanceForVnode(
          vnode,
          activeInstance
        );
        child.$mount(hydrating ? vnode.elm : undefined, hydrating);
      }
    },

    prepatch (oldVnode, vnode) {
      const options = vnode.componentOptions;
      const child = vnode.componentInstance = oldVnode.componentInstance;
      updateChildComponent(
        child,
        options.propsData, // updated props
        options.listeners, // updated listeners
        vnode, // new parent vnode
        options.children // new children
      );
    },

    insert (vnode) {
      const { context, componentInstance } = vnode;
      if (!componentInstance._isMounted) {
        componentInstance._isMounted = true;
        callHook(componentInstance, 'mounted');
      }
      if (vnode.data.keepAlive) {
        if (context._isMounted) {
          // vue-router#1212
          // During updates, a kept-alive component's child components may
          // change, so directly walking the tree here may call activated hooks
          // on incorrect children. Instead we push them into a queue which will
          // be processed after the whole patch process ended.
          queueActivatedComponent(componentInstance);
        } else {
          activateChildComponent(componentInstance, true /* direct */);
        }
      }
    },

    destroy (vnode) {
      const { componentInstance } = vnode;
      if (!componentInstance._isDestroyed) {
        if (!vnode.data.keepAlive) {
          componentInstance.$destroy();
        } else {
          deactivateChildComponent(componentInstance, true /* direct */);
        }
      }
    }
  };

  const hooksToMerge = Object.keys(componentVNodeHooks);

  function createComponent (
    Ctor,
    data,
    context,
    children,
    tag
  ) {
    if (isUndef(Ctor)) {
      return
    }

    const baseCtor = context.$options._base;

    // plain options object: turn it into a constructor
    if (isObject(Ctor)) {
      Ctor = baseCtor.extend(Ctor);
    }

    // if at this stage it's not a constructor or an async component factory,
    // reject.
    if (typeof Ctor !== 'function') {
      {
        warn(`Invalid Component definition: ${String(Ctor)}`, context);
      }
      return
    }

    // async component
    let asyncFactory;
    if (isUndef(Ctor.cid)) {
      asyncFactory = Ctor;
      Ctor = resolveAsyncComponent(asyncFactory, baseCtor);
      if (Ctor === undefined) {
        // return a placeholder node for async component, which is rendered
        // as a comment node but preserves all the raw information for the node.
        // the information will be used for async server-rendering and hydration.
        return createAsyncPlaceholder(
          asyncFactory,
          data,
          context,
          children,
          tag
        )
      }
    }

    data = data || {};

    // resolve constructor options in case global mixins are applied after
    // component constructor creation
    resolveConstructorOptions(Ctor);

    // transform component v-model data into props & events
    if (isDef(data.model)) {
      transformModel(Ctor.options, data);
    }

    // extract props
    const propsData = extractPropsFromVNodeData(data, Ctor, tag);

    // functional component
    if (isTrue(Ctor.options.functional)) {
      return createFunctionalComponent(Ctor, propsData, data, context, children)
    }

    // extract listeners, since these needs to be treated as
    // child component listeners instead of DOM listeners
    const listeners = data.on;
    // replace with listeners with .native modifier
    // so it gets processed during parent component patch.
    data.on = data.nativeOn;

    if (isTrue(Ctor.options.abstract)) {
      // abstract components do not keep anything
      // other than props & listeners & slot

      // work around flow
      const slot = data.slot;
      data = {};
      if (slot) {
        data.slot = slot;
      }
    }

    // install component management hooks onto the placeholder node
    installComponentHooks(data);

    // return a placeholder vnode
    const name = Ctor.options.name || tag;
    const vnode = new VNode(
      `vue-component-${Ctor.cid}${name ? `-${name}` : ''}`,
      data, undefined, undefined, undefined, context,
      { Ctor, propsData, listeners, tag, children },
      asyncFactory
    );

    return vnode
  }

  function createComponentInstanceForVnode (
    // we know it's MountedComponentVNode but flow doesn't
    vnode,
    // activeInstance in lifecycle state
    parent
  ) {
    const options = {
      _isComponent: true,
      _parentVnode: vnode,
      parent
    };
    // check inline-template render functions
    const inlineTemplate = vnode.data.inlineTemplate;
    if (isDef(inlineTemplate)) {
      options.render = inlineTemplate.render;
      options.staticRenderFns = inlineTemplate.staticRenderFns;
    }
    return new vnode.componentOptions.Ctor(options)
  }

  function installComponentHooks (data) {
    const hooks = data.hook || (data.hook = {});
    for (let i = 0; i < hooksToMerge.length; i++) {
      const key = hooksToMerge[i];
      const existing = hooks[key];
      const toMerge = componentVNodeHooks[key];
      if (existing !== toMerge && !(existing && existing._merged)) {
        hooks[key] = existing ? mergeHook$1(toMerge, existing) : toMerge;
      }
    }
  }

  function mergeHook$1 (f1, f2) {
    const merged = (a, b) => {
      // flow complains about extra args which is why we use any
      f1(a, b);
      f2(a, b);
    };
    merged._merged = true;
    return merged
  }

  // transform component v-model info (value and callback) into
  // prop and event handler respectively.
  function transformModel (options, data) {
    const prop = (options.model && options.model.prop) || 'value';
    const event = (options.model && options.model.event) || 'input'
    ;(data.attrs || (data.attrs = {}))[prop] = data.model.value;
    const on = data.on || (data.on = {});
    const existing = on[event];
    const callback = data.model.callback;
    if (isDef(existing)) {
      if (
        Array.isArray(existing)
          ? existing.indexOf(callback) === -1
          : existing !== callback
      ) {
        on[event] = [callback].concat(existing);
      }
    } else {
      on[event] = callback;
    }
  }

  /*  */

  const SIMPLE_NORMALIZE = 1;
  const ALWAYS_NORMALIZE = 2;

  // wrapper function for providing a more flexible interface
  // without getting yelled at by flow
  function createElement (
    context,
    tag,
    data,
    children,
    normalizationType,
    alwaysNormalize
  ) {
    if (Array.isArray(data) || isPrimitive(data)) {
      normalizationType = children;
      children = data;
      data = undefined;
    }
    if (isTrue(alwaysNormalize)) {
      normalizationType = ALWAYS_NORMALIZE;
    }
    return _createElement(context, tag, data, children, normalizationType)
  }

  function _createElement (
    context,
    tag,
    data,
    children,
    normalizationType
  ) {
    if (isDef(data) && isDef((data).__ob__)) {
      warn(
        `Avoid using observed data object as vnode data: ${JSON.stringify(data)}\n` +
        'Always create fresh vnode data objects in each render!',
        context
      );
      return createEmptyVNode()
    }
    // object syntax in v-bind
    if (isDef(data) && isDef(data.is)) {
      tag = data.is;
    }
    if (!tag) {
      // in case of component :is set to falsy value
      return createEmptyVNode()
    }
    // warn against non-primitive key
    if (isDef(data) && isDef(data.key) && !isPrimitive(data.key)
    ) {
      {
        warn(
          'Avoid using non-primitive value as key, ' +
          'use string/number value instead.',
          context
        );
      }
    }
    // support single function children as default scoped slot
    if (Array.isArray(children) &&
      typeof children[0] === 'function'
    ) {
      data = data || {};
      data.scopedSlots = { default: children[0] };
      children.length = 0;
    }
    if (normalizationType === ALWAYS_NORMALIZE) {
      children = normalizeChildren(children);
    } else if (normalizationType === SIMPLE_NORMALIZE) {
      children = simpleNormalizeChildren(children);
    }
    let vnode, ns;
    if (typeof tag === 'string') {
      let Ctor;
      ns = (context.$vnode && context.$vnode.ns) || config.getTagNamespace(tag);
      if (config.isReservedTag(tag)) {
        // platform built-in elements
        if (isDef(data) && isDef(data.nativeOn) && data.tag !== 'component') {
          warn(
            `The .native modifier for v-on is only valid on components but it was used on <${tag}>.`,
            context
          );
        }
        vnode = new VNode(
          config.parsePlatformTagName(tag), data, children,
          undefined, undefined, context
        );
      } else if ((!data || !data.pre) && isDef(Ctor = resolveAsset(context.$options, 'components', tag))) {
        // component
        vnode = createComponent(Ctor, data, context, children, tag);
      } else {
        // unknown or unlisted namespaced elements
        // check at runtime because it may get assigned a namespace when its
        // parent normalizes children
        vnode = new VNode(
          tag, data, children,
          undefined, undefined, context
        );
      }
    } else {
      // direct component options / constructor
      vnode = createComponent(tag, data, context, children);
    }
    if (Array.isArray(vnode)) {
      return vnode
    } else if (isDef(vnode)) {
      if (isDef(ns)) applyNS(vnode, ns);
      if (isDef(data)) registerDeepBindings(data);
      return vnode
    } else {
      return createEmptyVNode()
    }
  }

  function applyNS (vnode, ns, force) {
    vnode.ns = ns;
    if (vnode.tag === 'foreignObject') {
      // use default namespace inside foreignObject
      ns = undefined;
      force = true;
    }
    if (isDef(vnode.children)) {
      for (let i = 0, l = vnode.children.length; i < l; i++) {
        const child = vnode.children[i];
        if (isDef(child.tag) && (
          isUndef(child.ns) || (isTrue(force) && child.tag !== 'svg'))) {
          applyNS(child, ns, force);
        }
      }
    }
  }

  // ref #5318
  // necessary to ensure parent re-render when deep bindings like :style and
  // :class are used on slot nodes
  function registerDeepBindings (data) {
    if (isObject(data.style)) {
      traverse(data.style);
    }
    if (isObject(data.class)) {
      traverse(data.class);
    }
  }

  /*  */

  function initRender (vm) {
    vm._vnode = null; // the root of the child tree
    vm._staticTrees = null; // v-once cached trees
    const options = vm.$options;
    const parentVnode = vm.$vnode = options._parentVnode; // the placeholder node in parent tree
    const renderContext = parentVnode && parentVnode.context;
    vm.$slots = resolveSlots(options._renderChildren, renderContext);
    vm.$scopedSlots = emptyObject;
    // bind the createElement fn to this instance
    // so that we get proper render context inside it.
    // args order: tag, data, children, normalizationType, alwaysNormalize
    // internal version is used by render functions compiled from templates
    vm._c = (a, b, c, d) => createElement(vm, a, b, c, d, false);
    // normalization is always applied for the public version, used in
    // user-written render functions.
    vm.$createElement = (a, b, c, d) => createElement(vm, a, b, c, d, true);

    // $attrs & $listeners are exposed for easier HOC creation.
    // they need to be reactive so that HOCs using them are always updated
    const parentData = parentVnode && parentVnode.data;

    /* istanbul ignore else */
    {
      defineReactive$$1(vm, '$attrs', parentData && parentData.attrs || emptyObject, () => {
        !isUpdatingChildComponent && warn(`$attrs is readonly.`, vm);
      }, true);
      defineReactive$$1(vm, '$listeners', options._parentListeners || emptyObject, () => {
        !isUpdatingChildComponent && warn(`$listeners is readonly.`, vm);
      }, true);
    }
  }

  let currentRenderingInstance = null;

  function renderMixin (Vue) {
    // install runtime convenience helpers
    installRenderHelpers(Vue.prototype);

    Vue.prototype.$nextTick = function (fn) {
      return nextTick(fn, this)
    };

    Vue.prototype._render = function () {
      const vm = this;
      const { render, _parentVnode } = vm.$options;

      if (_parentVnode) {
        vm.$scopedSlots = normalizeScopedSlots(
          _parentVnode.data.scopedSlots,
          vm.$slots,
          vm.$scopedSlots
        );
      }

      // set parent vnode. this allows render functions to have access
      // to the data on the placeholder node.
      vm.$vnode = _parentVnode;
      // render self
      let vnode;
      try {
        // There's no need to maintain a stack because all render fns are called
        // separately from one another. Nested component's render fns are called
        // when parent component is patched.
        currentRenderingInstance = vm;
        vnode = render.call(vm._renderProxy, vm.$createElement);
      } catch (e) {
        handleError(e, vm, `render`);
        // return error render result,
        // or previous vnode to prevent render error causing blank component
        /* istanbul ignore else */
        if (vm.$options.renderError) {
          try {
            vnode = vm.$options.renderError.call(vm._renderProxy, vm.$createElement, e);
          } catch (e) {
            handleError(e, vm, `renderError`);
            vnode = vm._vnode;
          }
        } else {
          vnode = vm._vnode;
        }
      } finally {
        currentRenderingInstance = null;
      }
      // if the returned array contains only a single node, allow it
      if (Array.isArray(vnode) && vnode.length === 1) {
        vnode = vnode[0];
      }
      // return empty vnode in case the render function errored out
      if (!(vnode instanceof VNode)) {
        if (Array.isArray(vnode)) {
          warn(
            'Multiple root nodes returned from render function. Render function ' +
            'should return a single root node.',
            vm
          );
        }
        vnode = createEmptyVNode();
      }
      // set parent
      vnode.parent = _parentVnode;
      return vnode
    };
  }

  /*  */

  function ensureCtor (comp, base) {
    if (
      comp.__esModule ||
      (hasSymbol && comp[Symbol.toStringTag] === 'Module')
    ) {
      comp = comp.default;
    }
    return isObject(comp)
      ? base.extend(comp)
      : comp
  }

  function createAsyncPlaceholder (
    factory,
    data,
    context,
    children,
    tag
  ) {
    const node = createEmptyVNode();
    node.asyncFactory = factory;
    node.asyncMeta = { data, context, children, tag };
    return node
  }

  function resolveAsyncComponent (
    factory,
    baseCtor
  ) {
    if (isTrue(factory.error) && isDef(factory.errorComp)) {
      return factory.errorComp
    }

    if (isDef(factory.resolved)) {
      return factory.resolved
    }

    const owner = currentRenderingInstance;
    if (owner && isDef(factory.owners) && factory.owners.indexOf(owner) === -1) {
      // already pending
      factory.owners.push(owner);
    }

    if (isTrue(factory.loading) && isDef(factory.loadingComp)) {
      return factory.loadingComp
    }

    if (owner && !isDef(factory.owners)) {
      const owners = factory.owners = [owner];
      let sync = true;
      let timerLoading = null;
      let timerTimeout = null

      ;(owner).$on('hook:destroyed', () => remove(owners, owner));

      const forceRender = (renderCompleted) => {
        for (let i = 0, l = owners.length; i < l; i++) {
          (owners[i]).$forceUpdate();
        }

        if (renderCompleted) {
          owners.length = 0;
          if (timerLoading !== null) {
            clearTimeout(timerLoading);
            timerLoading = null;
          }
          if (timerTimeout !== null) {
            clearTimeout(timerTimeout);
            timerTimeout = null;
          }
        }
      };

      const resolve = once((res) => {
        // cache resolved
        factory.resolved = ensureCtor(res, baseCtor);
        // invoke callbacks only if this is not a synchronous resolve
        // (async resolves are shimmed as synchronous during SSR)
        if (!sync) {
          forceRender(true);
        } else {
          owners.length = 0;
        }
      });

      const reject = once(reason => {
        warn(
          `Failed to resolve async component: ${String(factory)}` +
          (reason ? `\nReason: ${reason}` : '')
        );
        if (isDef(factory.errorComp)) {
          factory.error = true;
          forceRender(true);
        }
      });

      const res = factory(resolve, reject);

      if (isObject(res)) {
        if (isPromise(res)) {
          // () => Promise
          if (isUndef(factory.resolved)) {
            res.then(resolve, reject);
          }
        } else if (isPromise(res.component)) {
          res.component.then(resolve, reject);

          if (isDef(res.error)) {
            factory.errorComp = ensureCtor(res.error, baseCtor);
          }

          if (isDef(res.loading)) {
            factory.loadingComp = ensureCtor(res.loading, baseCtor);
            if (res.delay === 0) {
              factory.loading = true;
            } else {
              timerLoading = setTimeout(() => {
                timerLoading = null;
                if (isUndef(factory.resolved) && isUndef(factory.error)) {
                  factory.loading = true;
                  forceRender(false);
                }
              }, res.delay || 200);
            }
          }

          if (isDef(res.timeout)) {
            timerTimeout = setTimeout(() => {
              timerTimeout = null;
              if (isUndef(factory.resolved)) {
                reject(
                  `timeout (${res.timeout}ms)`
                );
              }
            }, res.timeout);
          }
        }
      }

      sync = false;
      // return in case resolved synchronously
      return factory.loading
        ? factory.loadingComp
        : factory.resolved
    }
  }

  /*  */

  function getFirstComponentChild (children) {
    if (Array.isArray(children)) {
      for (let i = 0; i < children.length; i++) {
        const c = children[i];
        if (isDef(c) && (isDef(c.componentOptions) || isAsyncPlaceholder(c))) {
          return c
        }
      }
    }
  }

  /*  */

  /*  */

  function initEvents (vm) {
    vm._events = Object.create(null);
    vm._hasHookEvent = false;
    // init parent attached events
    const listeners = vm.$options._parentListeners;
    if (listeners) {
      updateComponentListeners(vm, listeners);
    }
  }

  let target;

  function add (event, fn) {
    target.$on(event, fn);
  }

  function remove$1 (event, fn) {
    target.$off(event, fn);
  }

  function createOnceHandler (event, fn) {
    const _target = target;
    return function onceHandler () {
      const res = fn.apply(null, arguments);
      if (res !== null) {
        _target.$off(event, onceHandler);
      }
    }
  }

  function updateComponentListeners (
    vm,
    listeners,
    oldListeners
  ) {
    target = vm;
    updateListeners(listeners, oldListeners || {}, add, remove$1, createOnceHandler, vm);
    target = undefined;
  }

  function eventsMixin (Vue) {
    const hookRE = /^hook:/;
    Vue.prototype.$on = function (event, fn) {
      const vm = this;
      if (Array.isArray(event)) {
        for (let i = 0, l = event.length; i < l; i++) {
          vm.$on(event[i], fn);
        }
      } else {
        (vm._events[event] || (vm._events[event] = [])).push(fn);
        // optimize hook:event cost by using a boolean flag marked at registration
        // instead of a hash lookup
        if (hookRE.test(event)) {
          vm._hasHookEvent = true;
        }
      }
      return vm
    };

    Vue.prototype.$once = function (event, fn) {
      const vm = this;
      function on () {
        vm.$off(event, on);
        fn.apply(vm, arguments);
      }
      on.fn = fn;
      vm.$on(event, on);
      return vm
    };

    Vue.prototype.$off = function (event, fn) {
      const vm = this;
      // all
      if (!arguments.length) {
        vm._events = Object.create(null);
        return vm
      }
      // array of events
      if (Array.isArray(event)) {
        for (let i = 0, l = event.length; i < l; i++) {
          vm.$off(event[i], fn);
        }
        return vm
      }
      // specific event
      const cbs = vm._events[event];
      if (!cbs) {
        return vm
      }
      if (!fn) {
        vm._events[event] = null;
        return vm
      }
      // specific handler
      let cb;
      let i = cbs.length;
      while (i--) {
        cb = cbs[i];
        if (cb === fn || cb.fn === fn) {
          cbs.splice(i, 1);
          break
        }
      }
      return vm
    };

    Vue.prototype.$emit = function (event) {
      const vm = this;
      {
        const lowerCaseEvent = event.toLowerCase();
        if (lowerCaseEvent !== event && vm._events[lowerCaseEvent]) {
          tip(
            `Event "${lowerCaseEvent}" is emitted in component ` +
            `${formatComponentName(vm)} but the handler is registered for "${event}". ` +
            `Note that HTML attributes are case-insensitive and you cannot use ` +
            `v-on to listen to camelCase events when using in-DOM templates. ` +
            `You should probably use "${hyphenate(event)}" instead of "${event}".`
          );
        }
      }
      let cbs = vm._events[event];
      if (cbs) {
        cbs = cbs.length > 1 ? toArray(cbs) : cbs;
        const args = toArray(arguments, 1);
        const info = `event handler for "${event}"`;
        for (let i = 0, l = cbs.length; i < l; i++) {
          invokeWithErrorHandling(cbs[i], vm, args, vm, info);
        }
      }
      return vm
    };
  }

  /*  */

  let activeInstance = null;
  let isUpdatingChildComponent = false;

  function setActiveInstance(vm) {
    const prevActiveInstance = activeInstance;
    activeInstance = vm;
    return () => {
      activeInstance = prevActiveInstance;
    }
  }

  function initLifecycle (vm) {
    const options = vm.$options;

    // locate first non-abstract parent
    let parent = options.parent;
    if (parent && !options.abstract) {
      while (parent.$options.abstract && parent.$parent) {
        parent = parent.$parent;
      }
      parent.$children.push(vm);
    }

    vm.$parent = parent;
    vm.$root = parent ? parent.$root : vm;

    vm.$children = [];
    vm.$refs = {};

    vm._watcher = null;
    vm._inactive = null;
    vm._directInactive = false;
    vm._isMounted = false;
    vm._isDestroyed = false;
    vm._isBeingDestroyed = false;
  }

  function lifecycleMixin (Vue) {
    Vue.prototype._update = function (vnode, hydrating) {
      const vm = this;
      const prevEl = vm.$el;
      const prevVnode = vm._vnode;
      const restoreActiveInstance = setActiveInstance(vm);
      vm._vnode = vnode;
      // Vue.prototype.__patch__ is injected in entry points
      // based on the rendering backend used.
      if (!prevVnode) {
        // initial render
        vm.$el = vm.__patch__(vm.$el, vnode, hydrating, false /* removeOnly */);
      } else {
        // updates
        vm.$el = vm.__patch__(prevVnode, vnode);
      }
      restoreActiveInstance();
      // update __vue__ reference
      if (prevEl) {
        prevEl.__vue__ = null;
      }
      if (vm.$el) {
        vm.$el.__vue__ = vm;
      }
      // if parent is an HOC, update its $el as well
      if (vm.$vnode && vm.$parent && vm.$vnode === vm.$parent._vnode) {
        vm.$parent.$el = vm.$el;
      }
      // updated hook is called by the scheduler to ensure that children are
      // updated in a parent's updated hook.
    };

    Vue.prototype.$forceUpdate = function () {
      const vm = this;
      if (vm._watcher) {
        vm._watcher.update();
      }
    };

    Vue.prototype.$destroy = function () {
      const vm = this;
      if (vm._isBeingDestroyed) {
        return
      }
      callHook(vm, 'beforeDestroy');
      vm._isBeingDestroyed = true;
      // remove self from parent
      const parent = vm.$parent;
      if (parent && !parent._isBeingDestroyed && !vm.$options.abstract) {
        remove(parent.$children, vm);
      }
      // teardown watchers
      if (vm._watcher) {
        vm._watcher.teardown();
      }
      let i = vm._watchers.length;
      while (i--) {
        vm._watchers[i].teardown();
      }
      // remove reference from data ob
      // frozen object may not have observer.
      if (vm._data.__ob__) {
        vm._data.__ob__.vmCount--;
      }
      // call the last hook...
      vm._isDestroyed = true;
      // invoke destroy hooks on current rendered tree
      vm.__patch__(vm._vnode, null);
      // fire destroyed hook
      callHook(vm, 'destroyed');
      // turn off all instance listeners.
      vm.$off();
      // remove __vue__ reference
      if (vm.$el) {
        vm.$el.__vue__ = null;
      }
      // release circular reference (#6759)
      if (vm.$vnode) {
        vm.$vnode.parent = null;
      }
    };
  }

  function mountComponent (
    vm,
    el,
    hydrating
  ) {
    vm.$el = el;
    if (!vm.$options.render) {
      vm.$options.render = createEmptyVNode;
      {
        /* istanbul ignore if */
        if ((vm.$options.template && vm.$options.template.charAt(0) !== '#') ||
          vm.$options.el || el) {
          warn(
            'You are using the runtime-only build of Vue where the template ' +
            'compiler is not available. Either pre-compile the templates into ' +
            'render functions, or use the compiler-included build.',
            vm
          );
        } else {
          warn(
            'Failed to mount component: template or render function not defined.',
            vm
          );
        }
      }
    }
    callHook(vm, 'beforeMount');

    let updateComponent;
    /* istanbul ignore if */
    if (config.performance && mark) {
      updateComponent = () => {
        const name = vm._name;
        const id = vm._uid;
        const startTag = `vue-perf-start:${id}`;
        const endTag = `vue-perf-end:${id}`;

        mark(startTag);
        const vnode = vm._render();
        mark(endTag);
        measure(`vue ${name} render`, startTag, endTag);

        mark(startTag);
        vm._update(vnode, hydrating);
        mark(endTag);
        measure(`vue ${name} patch`, startTag, endTag);
      };
    } else {
      updateComponent = () => {
        vm._update(vm._render(), hydrating);
      };
    }

    // we set this to vm._watcher inside the watcher's constructor
    // since the watcher's initial patch may call $forceUpdate (e.g. inside child
    // component's mounted hook), which relies on vm._watcher being already defined
    new Watcher(vm, updateComponent, noop, {
      before () {
        if (vm._isMounted && !vm._isDestroyed) {
          callHook(vm, 'beforeUpdate');
        }
      }
    }, true /* isRenderWatcher */);
    hydrating = false;

    // manually mounted instance, call mounted on self
    // mounted is called for render-created child components in its inserted hook
    if (vm.$vnode == null) {
      vm._isMounted = true;
      callHook(vm, 'mounted');
    }
    return vm
  }

  function updateChildComponent (
    vm,
    propsData,
    listeners,
    parentVnode,
    renderChildren
  ) {
    {
      isUpdatingChildComponent = true;
    }

    // determine whether component has slot children
    // we need to do this before overwriting $options._renderChildren.

    // check if there are dynamic scopedSlots (hand-written or compiled but with
    // dynamic slot names). Static scoped slots compiled from template has the
    // "$stable" marker.
    const newScopedSlots = parentVnode.data.scopedSlots;
    const oldScopedSlots = vm.$scopedSlots;
    const hasDynamicScopedSlot = !!(
      (newScopedSlots && !newScopedSlots.$stable) ||
      (oldScopedSlots !== emptyObject && !oldScopedSlots.$stable) ||
      (newScopedSlots && vm.$scopedSlots.$key !== newScopedSlots.$key) ||
      (!newScopedSlots && vm.$scopedSlots.$key)
    );

    // Any static slot children from the parent may have changed during parent's
    // update. Dynamic scoped slots may also have changed. In such cases, a forced
    // update is necessary to ensure correctness.
    const needsForceUpdate = !!(
      renderChildren ||               // has new static slots
      vm.$options._renderChildren ||  // has old static slots
      hasDynamicScopedSlot
    );

    vm.$options._parentVnode = parentVnode;
    vm.$vnode = parentVnode; // update vm's placeholder node without re-render

    if (vm._vnode) { // update child tree's parent
      vm._vnode.parent = parentVnode;
    }
    vm.$options._renderChildren = renderChildren;

    // update $attrs and $listeners hash
    // these are also reactive so they may trigger child update if the child
    // used them during render
    vm.$attrs = parentVnode.data.attrs || emptyObject;
    vm.$listeners = listeners || emptyObject;

    // update props
    if (propsData && vm.$options.props) {
      toggleObserving(false);
      const props = vm._props;
      const propKeys = vm.$options._propKeys || [];
      for (let i = 0; i < propKeys.length; i++) {
        const key = propKeys[i];
        const propOptions = vm.$options.props; // wtf flow?
        props[key] = validateProp(key, propOptions, propsData, vm);
      }
      toggleObserving(true);
      // keep a copy of raw propsData
      vm.$options.propsData = propsData;
    }

    // update listeners
    listeners = listeners || emptyObject;
    const oldListeners = vm.$options._parentListeners;
    vm.$options._parentListeners = listeners;
    updateComponentListeners(vm, listeners, oldListeners);

    // resolve slots + force update if has children
    if (needsForceUpdate) {
      vm.$slots = resolveSlots(renderChildren, parentVnode.context);
      vm.$forceUpdate();
    }

    {
      isUpdatingChildComponent = false;
    }
  }

  function isInInactiveTree (vm) {
    while (vm && (vm = vm.$parent)) {
      if (vm._inactive) return true
    }
    return false
  }

  function activateChildComponent (vm, direct) {
    if (direct) {
      vm._directInactive = false;
      if (isInInactiveTree(vm)) {
        return
      }
    } else if (vm._directInactive) {
      return
    }
    if (vm._inactive || vm._inactive === null) {
      vm._inactive = false;
      for (let i = 0; i < vm.$children.length; i++) {
        activateChildComponent(vm.$children[i]);
      }
      callHook(vm, 'activated');
    }
  }

  function deactivateChildComponent (vm, direct) {
    if (direct) {
      vm._directInactive = true;
      if (isInInactiveTree(vm)) {
        return
      }
    }
    if (!vm._inactive) {
      vm._inactive = true;
      for (let i = 0; i < vm.$children.length; i++) {
        deactivateChildComponent(vm.$children[i]);
      }
      callHook(vm, 'deactivated');
    }
  }

  function callHook (vm, hook) {
    // #7573 disable dep collection when invoking lifecycle hooks
    pushTarget();
    const handlers = vm.$options[hook];
    const info = `${hook} hook`;
    if (handlers) {
      for (let i = 0, j = handlers.length; i < j; i++) {
        invokeWithErrorHandling(handlers[i], vm, null, vm, info);
      }
    }
    if (vm._hasHookEvent) {
      vm.$emit('hook:' + hook);
    }
    popTarget();
  }

  /*  */

  const MAX_UPDATE_COUNT = 100;

  const queue = [];
  const activatedChildren = [];
  let has = {};
  let circular = {};
  let waiting = false;
  let flushing = false;
  let index = 0;

  /**
   * Reset the scheduler's state.
   */
  function resetSchedulerState () {
    index = queue.length = activatedChildren.length = 0;
    has = {};
    {
      circular = {};
    }
    waiting = flushing = false;
  }

  // Async edge case #6566 requires saving the timestamp when event listeners are
  // attached. However, calling performance.now() has a perf overhead especially
  // if the page has thousands of event listeners. Instead, we take a timestamp
  // every time the scheduler flushes and use that for all event listeners
  // attached during that flush.
  let currentFlushTimestamp = 0;

  // Async edge case fix requires storing an event listener's attach timestamp.
  let getNow = Date.now;

  // Determine what event timestamp the browser is using. Annoyingly, the
  // timestamp can either be hi-res (relative to page load) or low-res
  // (relative to UNIX epoch), so in order to compare time we have to use the
  // same timestamp type when saving the flush timestamp.
  // All IE versions use low-res event timestamps, and have problematic clock
  // implementations (#9632)
  if (inBrowser && !isIE) {
    const performance = window.performance;
    if (
      performance &&
      typeof performance.now === 'function' &&
      getNow() > document.createEvent('Event').timeStamp
    ) {
      // if the event timestamp, although evaluated AFTER the Date.now(), is
      // smaller than it, it means the event is using a hi-res timestamp,
      // and we need to use the hi-res version for event listener timestamps as
      // well.
      getNow = () => performance.now();
    }
  }

  /**
   * Flush both queues and run the watchers.
   */
  function flushSchedulerQueue () {
    currentFlushTimestamp = getNow();
    flushing = true;
    let watcher, id;

    // Sort queue before flush.
    // This ensures that:
    // 1. Components are updated from parent to child. (because parent is always
    //    created before the child)
    // 2. A component's user watchers are run before its render watcher (because
    //    user watchers are created before the render watcher)
    // 3. If a component is destroyed during a parent component's watcher run,
    //    its watchers can be skipped.
    queue.sort((a, b) => a.id - b.id);

    // do not cache length because more watchers might be pushed
    // as we run existing watchers
    for (index = 0; index < queue.length; index++) {
      watcher = queue[index];
      if (watcher.before) {
        watcher.before();
      }
      id = watcher.id;
      has[id] = null;
      watcher.run();
      // in dev build, check and stop circular updates.
      if (has[id] != null) {
        circular[id] = (circular[id] || 0) + 1;
        if (circular[id] > MAX_UPDATE_COUNT) {
          warn(
            'You may have an infinite update loop ' + (
              watcher.user
                ? `in watcher with expression "${watcher.expression}"`
                : `in a component render function.`
            ),
            watcher.vm
          );
          break
        }
      }
    }

    // keep copies of post queues before resetting state
    const activatedQueue = activatedChildren.slice();
    const updatedQueue = queue.slice();

    resetSchedulerState();

    // call component updated and activated hooks
    callActivatedHooks(activatedQueue);
    callUpdatedHooks(updatedQueue);

    // devtool hook
    /* istanbul ignore if */
    if (devtools && config.devtools) {
      devtools.emit('flush');
    }
  }

  function callUpdatedHooks (queue) {
    let i = queue.length;
    while (i--) {
      const watcher = queue[i];
      const vm = watcher.vm;
      if (vm._watcher === watcher && vm._isMounted && !vm._isDestroyed) {
        callHook(vm, 'updated');
      }
    }
  }

  /**
   * Queue a kept-alive component that was activated during patch.
   * The queue will be processed after the entire tree has been patched.
   */
  function queueActivatedComponent (vm) {
    // setting _inactive to false here so that a render function can
    // rely on checking whether it's in an inactive tree (e.g. router-view)
    vm._inactive = false;
    activatedChildren.push(vm);
  }

  function callActivatedHooks (queue) {
    for (let i = 0; i < queue.length; i++) {
      queue[i]._inactive = true;
      activateChildComponent(queue[i], true /* true */);
    }
  }

  /**
   * Push a watcher into the watcher queue.
   * Jobs with duplicate IDs will be skipped unless it's
   * pushed when the queue is being flushed.
   */
  function queueWatcher (watcher) {
    const id = watcher.id;
    if (has[id] == null) {
      has[id] = true;
      if (!flushing) {
        queue.push(watcher);
      } else {
        // if already flushing, splice the watcher based on its id
        // if already past its id, it will be run next immediately.
        let i = queue.length - 1;
        while (i > index && queue[i].id > watcher.id) {
          i--;
        }
        queue.splice(i + 1, 0, watcher);
      }
      // queue the flush
      if (!waiting) {
        waiting = true;

        if (!config.async) {
          flushSchedulerQueue();
          return
        }
        nextTick(flushSchedulerQueue);
      }
    }
  }

  /*  */



  let uid$2 = 0;

  /**
   * A watcher parses an expression, collects dependencies,
   * and fires callback when the expression value changes.
   * This is used for both the $watch() api and directives.
   */
  class Watcher {
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    

    constructor (
      vm,
      expOrFn,
      cb,
      options,
      isRenderWatcher
    ) {
      this.vm = vm;
      if (isRenderWatcher) {
        vm._watcher = this;
      }
      vm._watchers.push(this);
      // options
      if (options) {
        this.deep = !!options.deep;
        this.user = !!options.user;
        this.lazy = !!options.lazy;
        this.sync = !!options.sync;
        this.before = options.before;
      } else {
        this.deep = this.user = this.lazy = this.sync = false;
      }
      this.cb = cb;
      this.id = ++uid$2; // uid for batching
      this.active = true;
      this.dirty = this.lazy; // for lazy watchers
      this.deps = [];
      this.newDeps = [];
      this.depIds = new _Set();
      this.newDepIds = new _Set();
      this.expression = expOrFn.toString();
      // parse expression for getter
      if (typeof expOrFn === 'function') {
        this.getter = expOrFn;
      } else {
        this.getter = parsePath(expOrFn);
        if (!this.getter) {
          this.getter = noop;
          warn(
            `Failed watching path: "${expOrFn}" ` +
            'Watcher only accepts simple dot-delimited paths. ' +
            'For full control, use a function instead.',
            vm
          );
        }
      }
      this.value = this.lazy
        ? undefined
        : this.get();
    }

    /**
     * Evaluate the getter, and re-collect dependencies.
     */
    get () {
      pushTarget(this);
      let value;
      const vm = this.vm;
      try {
        value = this.getter.call(vm, vm);
      } catch (e) {
        if (this.user) {
          handleError(e, vm, `getter for watcher "${this.expression}"`);
        } else {
          throw e
        }
      } finally {
        // "touch" every property so they are all tracked as
        // dependencies for deep watching
        if (this.deep) {
          traverse(value);
        }
        popTarget();
        this.cleanupDeps();
      }
      return value
    }

    /**
     * Add a dependency to this directive.
     */
    addDep (dep) {
      const id = dep.id;
      if (!this.newDepIds.has(id)) {
        this.newDepIds.add(id);
        this.newDeps.push(dep);
        if (!this.depIds.has(id)) {
          dep.addSub(this);
        }
      }
    }

    /**
     * Clean up for dependency collection.
     */
    cleanupDeps () {
      let i = this.deps.length;
      while (i--) {
        const dep = this.deps[i];
        if (!this.newDepIds.has(dep.id)) {
          dep.removeSub(this);
        }
      }
      let tmp = this.depIds;
      this.depIds = this.newDepIds;
      this.newDepIds = tmp;
      this.newDepIds.clear();
      tmp = this.deps;
      this.deps = this.newDeps;
      this.newDeps = tmp;
      this.newDeps.length = 0;
    }

    /**
     * Subscriber interface.
     * Will be called when a dependency changes.
     */
    update () {
      /* istanbul ignore else */
      if (this.lazy) {
        this.dirty = true;
      } else if (this.sync) {
        this.run();
      } else {
        queueWatcher(this);
      }
    }

    /**
     * Scheduler job interface.
     * Will be called by the scheduler.
     */
    run () {
      if (this.active) {
        const value = this.get();
        if (
          value !== this.value ||
          // Deep watchers and watchers on Object/Arrays should fire even
          // when the value is the same, because the value may
          // have mutated.
          isObject(value) ||
          this.deep
        ) {
          // set new value
          const oldValue = this.value;
          this.value = value;
          if (this.user) {
            const info = `callback for watcher "${this.expression}"`;
            invokeWithErrorHandling(this.cb, this.vm, [value, oldValue], this.vm, info);
          } else {
            this.cb.call(this.vm, value, oldValue);
          }
        }
      }
    }

    /**
     * Evaluate the value of the watcher.
     * This only gets called for lazy watchers.
     */
    evaluate () {
      this.value = this.get();
      this.dirty = false;
    }

    /**
     * Depend on all deps collected by this watcher.
     */
    depend () {
      let i = this.deps.length;
      while (i--) {
        this.deps[i].depend();
      }
    }

    /**
     * Remove self from all dependencies' subscriber list.
     */
    teardown () {
      if (this.active) {
        // remove self from vm's watcher list
        // this is a somewhat expensive operation so we skip it
        // if the vm is being destroyed.
        if (!this.vm._isBeingDestroyed) {
          remove(this.vm._watchers, this);
        }
        let i = this.deps.length;
        while (i--) {
          this.deps[i].removeSub(this);
        }
        this.active = false;
      }
    }
  }

  /*  */

  const sharedPropertyDefinition = {
    enumerable: true,
    configurable: true,
    get: noop,
    set: noop
  };

  function proxy (target, sourceKey, key) {
    sharedPropertyDefinition.get = function proxyGetter () {
      return this[sourceKey][key]
    };
    sharedPropertyDefinition.set = function proxySetter (val) {
      this[sourceKey][key] = val;
    };
    Object.defineProperty(target, key, sharedPropertyDefinition);
  }

  function initState (vm) {
    vm._watchers = [];
    const opts = vm.$options;
    if (opts.props) initProps(vm, opts.props);
    if (opts.methods) initMethods(vm, opts.methods);
    if (opts.data) {
      initData(vm);
    } else {
      observe(vm._data = {}, true /* asRootData */);
    }
    if (opts.computed) initComputed(vm, opts.computed);
    if (opts.watch && opts.watch !== nativeWatch) {
      initWatch(vm, opts.watch);
    }
  }

  function initProps (vm, propsOptions) {
    const propsData = vm.$options.propsData || {};
    const props = vm._props = {};
    // cache prop keys so that future props updates can iterate using Array
    // instead of dynamic object key enumeration.
    const keys = vm.$options._propKeys = [];
    const isRoot = !vm.$parent;
    // root instance props should be converted
    if (!isRoot) {
      toggleObserving(false);
    }
    for (const key in propsOptions) {
      keys.push(key);
      const value = validateProp(key, propsOptions, propsData, vm);
      /* istanbul ignore else */
      {
        const hyphenatedKey = hyphenate(key);
        if (isReservedAttribute(hyphenatedKey) ||
            config.isReservedAttr(hyphenatedKey)) {
          warn(
            `"${hyphenatedKey}" is a reserved attribute and cannot be used as component prop.`,
            vm
          );
        }
        defineReactive$$1(props, key, value, () => {
          if (!isRoot && !isUpdatingChildComponent) {
            warn(
              `Avoid mutating a prop directly since the value will be ` +
              `overwritten whenever the parent component re-renders. ` +
              `Instead, use a data or computed property based on the prop's ` +
              `value. Prop being mutated: "${key}"`,
              vm
            );
          }
        });
      }
      // static props are already proxied on the component's prototype
      // during Vue.extend(). We only need to proxy props defined at
      // instantiation here.
      if (!(key in vm)) {
        proxy(vm, `_props`, key);
      }
    }
    toggleObserving(true);
  }

  function initData (vm) {
    let data = vm.$options.data;
    data = vm._data = typeof data === 'function'
      ? getData(data, vm)
      : data || {};
    if (!isPlainObject(data)) {
      data = {};
      warn(
        'data functions should return an object:\n' +
        'https://vuejs.org/v2/guide/components.html#data-Must-Be-a-Function',
        vm
      );
    }
    // proxy data on instance
    const keys = Object.keys(data);
    const props = vm.$options.props;
    const methods = vm.$options.methods;
    let i = keys.length;
    while (i--) {
      const key = keys[i];
      {
        if (methods && hasOwn(methods, key)) {
          warn(
            `Method "${key}" has already been defined as a data property.`,
            vm
          );
        }
      }
      if (props && hasOwn(props, key)) {
        warn(
          `The data property "${key}" is already declared as a prop. ` +
          `Use prop default value instead.`,
          vm
        );
      } else if (!isReserved(key)) {
        proxy(vm, `_data`, key);
      }
    }
    // observe data
    observe(data, true /* asRootData */);
  }

  function getData (data, vm) {
    // #7573 disable dep collection when invoking data getters
    pushTarget();
    try {
      return data.call(vm, vm)
    } catch (e) {
      handleError(e, vm, `data()`);
      return {}
    } finally {
      popTarget();
    }
  }

  const computedWatcherOptions = { lazy: true };

  function initComputed (vm, computed) {
    // $flow-disable-line
    const watchers = vm._computedWatchers = Object.create(null);
    // computed properties are just getters during SSR
    const isSSR = isServerRendering();

    for (const key in computed) {
      const userDef = computed[key];
      const getter = typeof userDef === 'function' ? userDef : userDef.get;
      if (getter == null) {
        warn(
          `Getter is missing for computed property "${key}".`,
          vm
        );
      }

      if (!isSSR) {
        // create internal watcher for the computed property.
        watchers[key] = new Watcher(
          vm,
          getter || noop,
          noop,
          computedWatcherOptions
        );
      }

      // component-defined computed properties are already defined on the
      // component prototype. We only need to define computed properties defined
      // at instantiation here.
      if (!(key in vm)) {
        defineComputed(vm, key, userDef);
      } else {
        if (key in vm.$data) {
          warn(`The computed property "${key}" is already defined in data.`, vm);
        } else if (vm.$options.props && key in vm.$options.props) {
          warn(`The computed property "${key}" is already defined as a prop.`, vm);
        } else if (vm.$options.methods && key in vm.$options.methods) {
          warn(`The computed property "${key}" is already defined as a method.`, vm);
        }
      }
    }
  }

  function defineComputed (
    target,
    key,
    userDef
  ) {
    const shouldCache = !isServerRendering();
    if (typeof userDef === 'function') {
      sharedPropertyDefinition.get = shouldCache
        ? createComputedGetter(key)
        : createGetterInvoker(userDef);
      sharedPropertyDefinition.set = noop;
    } else {
      sharedPropertyDefinition.get = userDef.get
        ? shouldCache && userDef.cache !== false
          ? createComputedGetter(key)
          : createGetterInvoker(userDef.get)
        : noop;
      sharedPropertyDefinition.set = userDef.set || noop;
    }
    if (sharedPropertyDefinition.set === noop) {
      sharedPropertyDefinition.set = function () {
        warn(
          `Computed property "${key}" was assigned to but it has no setter.`,
          this
        );
      };
    }
    Object.defineProperty(target, key, sharedPropertyDefinition);
  }

  function createComputedGetter (key) {
    return function computedGetter () {
      const watcher = this._computedWatchers && this._computedWatchers[key];
      if (watcher) {
        if (watcher.dirty) {
          watcher.evaluate();
        }
        if (Dep.target) {
          watcher.depend();
        }
        return watcher.value
      }
    }
  }

  function createGetterInvoker(fn) {
    return function computedGetter () {
      return fn.call(this, this)
    }
  }

  function initMethods (vm, methods) {
    const props = vm.$options.props;
    for (const key in methods) {
      {
        if (typeof methods[key] !== 'function') {
          warn(
            `Method "${key}" has type "${typeof methods[key]}" in the component definition. ` +
            `Did you reference the function correctly?`,
            vm
          );
        }
        if (props && hasOwn(props, key)) {
          warn(
            `Method "${key}" has already been defined as a prop.`,
            vm
          );
        }
        if ((key in vm) && isReserved(key)) {
          warn(
            `Method "${key}" conflicts with an existing Vue instance method. ` +
            `Avoid defining component methods that start with _ or $.`
          );
        }
      }
      vm[key] = typeof methods[key] !== 'function' ? noop : bind(methods[key], vm);
    }
  }

  function initWatch (vm, watch) {
    for (const key in watch) {
      const handler = watch[key];
      if (Array.isArray(handler)) {
        for (let i = 0; i < handler.length; i++) {
          createWatcher(vm, key, handler[i]);
        }
      } else {
        createWatcher(vm, key, handler);
      }
    }
  }

  function createWatcher (
    vm,
    expOrFn,
    handler,
    options
  ) {
    if (isPlainObject(handler)) {
      options = handler;
      handler = handler.handler;
    }
    if (typeof handler === 'string') {
      handler = vm[handler];
    }
    return vm.$watch(expOrFn, handler, options)
  }

  function stateMixin (Vue) {
    // flow somehow has problems with directly declared definition object
    // when using Object.defineProperty, so we have to procedurally build up
    // the object here.
    const dataDef = {};
    dataDef.get = function () { return this._data };
    const propsDef = {};
    propsDef.get = function () { return this._props };
    {
      dataDef.set = function () {
        warn(
          'Avoid replacing instance root $data. ' +
          'Use nested data properties instead.',
          this
        );
      };
      propsDef.set = function () {
        warn(`$props is readonly.`, this);
      };
    }
    Object.defineProperty(Vue.prototype, '$data', dataDef);
    Object.defineProperty(Vue.prototype, '$props', propsDef);

    Vue.prototype.$set = set;
    Vue.prototype.$delete = del;

    Vue.prototype.$watch = function (
      expOrFn,
      cb,
      options
    ) {
      const vm = this;
      if (isPlainObject(cb)) {
        return createWatcher(vm, expOrFn, cb, options)
      }
      options = options || {};
      options.user = true;
      const watcher = new Watcher(vm, expOrFn, cb, options);
      if (options.immediate) {
        const info = `callback for immediate watcher "${watcher.expression}"`;
        pushTarget();
        invokeWithErrorHandling(cb, vm, [watcher.value], vm, info);
        popTarget();
      }
      return function unwatchFn () {
        watcher.teardown();
      }
    };
  }

  /*  */

  let uid$3 = 0;

  function initMixin (Vue) {
    Vue.prototype._init = function (options) {
      const vm = this;
      // a uid
      vm._uid = uid$3++;

      let startTag, endTag;
      /* istanbul ignore if */
      if (config.performance && mark) {
        startTag = `vue-perf-start:${vm._uid}`;
        endTag = `vue-perf-end:${vm._uid}`;
        mark(startTag);
      }

      // a flag to avoid this being observed
      vm._isVue = true;
      // merge options
      if (options && options._isComponent) {
        // optimize internal component instantiation
        // since dynamic options merging is pretty slow, and none of the
        // internal component options needs special treatment.
        initInternalComponent(vm, options);
      } else {
        vm.$options = mergeOptions(
          resolveConstructorOptions(vm.constructor),
          options || {},
          vm
        );
      }
      /* istanbul ignore else */
      {
        initProxy(vm);
      }
      // expose real self
      vm._self = vm;
      initLifecycle(vm);
      initEvents(vm);
      initRender(vm);
      callHook(vm, 'beforeCreate');
      initInjections(vm); // resolve injections before data/props
      initState(vm);
      initProvide(vm); // resolve provide after data/props
      callHook(vm, 'created');

      /* istanbul ignore if */
      if (config.performance && mark) {
        vm._name = formatComponentName(vm, false);
        mark(endTag);
        measure(`vue ${vm._name} init`, startTag, endTag);
      }

      if (vm.$options.el) {
        vm.$mount(vm.$options.el);
      }
    };
  }

  function initInternalComponent (vm, options) {
    const opts = vm.$options = Object.create(vm.constructor.options);
    // doing this because it's faster than dynamic enumeration.
    const parentVnode = options._parentVnode;
    opts.parent = options.parent;
    opts._parentVnode = parentVnode;

    const vnodeComponentOptions = parentVnode.componentOptions;
    opts.propsData = vnodeComponentOptions.propsData;
    opts._parentListeners = vnodeComponentOptions.listeners;
    opts._renderChildren = vnodeComponentOptions.children;
    opts._componentTag = vnodeComponentOptions.tag;

    if (options.render) {
      opts.render = options.render;
      opts.staticRenderFns = options.staticRenderFns;
    }
  }

  function resolveConstructorOptions (Ctor) {
    let options = Ctor.options;
    if (Ctor.super) {
      const superOptions = resolveConstructorOptions(Ctor.super);
      const cachedSuperOptions = Ctor.superOptions;
      if (superOptions !== cachedSuperOptions) {
        // super option changed,
        // need to resolve new options.
        Ctor.superOptions = superOptions;
        // check if there are any late-modified/attached options (#4976)
        const modifiedOptions = resolveModifiedOptions(Ctor);
        // update base extend options
        if (modifiedOptions) {
          extend(Ctor.extendOptions, modifiedOptions);
        }
        options = Ctor.options = mergeOptions(superOptions, Ctor.extendOptions);
        if (options.name) {
          options.components[options.name] = Ctor;
        }
      }
    }
    return options
  }

  function resolveModifiedOptions (Ctor) {
    let modified;
    const latest = Ctor.options;
    const sealed = Ctor.sealedOptions;
    for (const key in latest) {
      if (latest[key] !== sealed[key]) {
        if (!modified) modified = {};
        modified[key] = latest[key];
      }
    }
    return modified
  }

  function Vue (options) {
    if (!(this instanceof Vue)
    ) {
      warn('Vue is a constructor and should be called with the `new` keyword');
    }
    this._init(options);
  }

  initMixin(Vue);
  stateMixin(Vue);
  eventsMixin(Vue);
  lifecycleMixin(Vue);
  renderMixin(Vue);

  /*  */

  function initUse (Vue) {
    Vue.use = function (plugin) {
      const installedPlugins = (this._installedPlugins || (this._installedPlugins = []));
      if (installedPlugins.indexOf(plugin) > -1) {
        return this
      }

      // additional parameters
      const args = toArray(arguments, 1);
      args.unshift(this);
      if (typeof plugin.install === 'function') {
        plugin.install.apply(plugin, args);
      } else if (typeof plugin === 'function') {
        plugin.apply(null, args);
      }
      installedPlugins.push(plugin);
      return this
    };
  }

  /*  */

  function initMixin$1 (Vue) {
    Vue.mixin = function (mixin) {
      this.options = mergeOptions(this.options, mixin);
      return this
    };
  }

  /*  */

  function initExtend (Vue) {
    /**
     * Each instance constructor, including Vue, has a unique
     * cid. This enables us to create wrapped "child
     * constructors" for prototypal inheritance and cache them.
     */
    Vue.cid = 0;
    let cid = 1;

    /**
     * Class inheritance
     */
    Vue.extend = function (extendOptions) {
      extendOptions = extendOptions || {};
      const Super = this;
      const SuperId = Super.cid;
      const cachedCtors = extendOptions._Ctor || (extendOptions._Ctor = {});
      if (cachedCtors[SuperId]) {
        return cachedCtors[SuperId]
      }

      const name = extendOptions.name || Super.options.name;
      if (name) {
        validateComponentName(name);
      }

      const Sub = function VueComponent (options) {
        this._init(options);
      };
      Sub.prototype = Object.create(Super.prototype);
      Sub.prototype.constructor = Sub;
      Sub.cid = cid++;
      Sub.options = mergeOptions(
        Super.options,
        extendOptions
      );
      Sub['super'] = Super;

      // For props and computed properties, we define the proxy getters on
      // the Vue instances at extension time, on the extended prototype. This
      // avoids Object.defineProperty calls for each instance created.
      if (Sub.options.props) {
        initProps$1(Sub);
      }
      if (Sub.options.computed) {
        initComputed$1(Sub);
      }

      // allow further extension/mixin/plugin usage
      Sub.extend = Super.extend;
      Sub.mixin = Super.mixin;
      Sub.use = Super.use;

      // create asset registers, so extended classes
      // can have their private assets too.
      ASSET_TYPES.forEach(function (type) {
        Sub[type] = Super[type];
      });
      // enable recursive self-lookup
      if (name) {
        Sub.options.components[name] = Sub;
      }

      // keep a reference to the super options at extension time.
      // later at instantiation we can check if Super's options have
      // been updated.
      Sub.superOptions = Super.options;
      Sub.extendOptions = extendOptions;
      Sub.sealedOptions = extend({}, Sub.options);

      // cache constructor
      cachedCtors[SuperId] = Sub;
      return Sub
    };
  }

  function initProps$1 (Comp) {
    const props = Comp.options.props;
    for (const key in props) {
      proxy(Comp.prototype, `_props`, key);
    }
  }

  function initComputed$1 (Comp) {
    const computed = Comp.options.computed;
    for (const key in computed) {
      defineComputed(Comp.prototype, key, computed[key]);
    }
  }

  /*  */

  function initAssetRegisters (Vue) {
    /**
     * Create asset registration methods.
     */
    ASSET_TYPES.forEach(type => {
      Vue[type] = function (
        id,
        definition
      ) {
        if (!definition) {
          return this.options[type + 's'][id]
        } else {
          /* istanbul ignore if */
          if (type === 'component') {
            validateComponentName(id);
          }
          if (type === 'component' && isPlainObject(definition)) {
            definition.name = definition.name || id;
            definition = this.options._base.extend(definition);
          }
          if (type === 'directive' && typeof definition === 'function') {
            definition = { bind: definition, update: definition };
          }
          this.options[type + 's'][id] = definition;
          return definition
        }
      };
    });
  }

  /*  */





  function getComponentName (opts) {
    return opts && (opts.Ctor.options.name || opts.tag)
  }

  function matches (pattern, name) {
    if (Array.isArray(pattern)) {
      return pattern.indexOf(name) > -1
    } else if (typeof pattern === 'string') {
      return pattern.split(',').indexOf(name) > -1
    } else if (isRegExp(pattern)) {
      return pattern.test(name)
    }
    /* istanbul ignore next */
    return false
  }

  function pruneCache (keepAliveInstance, filter) {
    const { cache, keys, _vnode } = keepAliveInstance;
    for (const key in cache) {
      const entry = cache[key];
      if (entry) {
        const name = entry.name;
        if (name && !filter(name)) {
          pruneCacheEntry(cache, key, keys, _vnode);
        }
      }
    }
  }

  function pruneCacheEntry (
    cache,
    key,
    keys,
    current
  ) {
    const entry = cache[key];
    if (entry && (!current || entry.tag !== current.tag)) {
      entry.componentInstance.$destroy();
    }
    cache[key] = null;
    remove(keys, key);
  }

  const patternTypes = [String, RegExp, Array];

  var KeepAlive = {
    name: 'keep-alive',
    abstract: true,

    props: {
      include: patternTypes,
      exclude: patternTypes,
      max: [String, Number]
    },

    methods: {
      cacheVNode() {
        const { cache, keys, vnodeToCache, keyToCache } = this;
        if (vnodeToCache) {
          const { tag, componentInstance, componentOptions } = vnodeToCache;
          cache[keyToCache] = {
            name: getComponentName(componentOptions),
            tag,
            componentInstance,
          };
          keys.push(keyToCache);
          // prune oldest entry
          if (this.max && keys.length > parseInt(this.max)) {
            pruneCacheEntry(cache, keys[0], keys, this._vnode);
          }
          this.vnodeToCache = null;
        }
      }
    },

    created () {
      this.cache = Object.create(null);
      this.keys = [];
    },

    destroyed () {
      for (const key in this.cache) {
        pruneCacheEntry(this.cache, key, this.keys);
      }
    },

    mounted () {
      this.cacheVNode();
      this.$watch('include', val => {
        pruneCache(this, name => matches(val, name));
      });
      this.$watch('exclude', val => {
        pruneCache(this, name => !matches(val, name));
      });
    },

    updated () {
      this.cacheVNode();
    },

    render () {
      const slot = this.$slots.default;
      const vnode = getFirstComponentChild(slot);
      const componentOptions = vnode && vnode.componentOptions;
      if (componentOptions) {
        // check pattern
        const name = getComponentName(componentOptions);
        const { include, exclude } = this;
        if (
          // not included
          (include && (!name || !matches(include, name))) ||
          // excluded
          (exclude && name && matches(exclude, name))
        ) {
          return vnode
        }

        const { cache, keys } = this;
        const key = vnode.key == null
          // same constructor may get registered as different local components
          // so cid alone is not enough (#3269)
          ? componentOptions.Ctor.cid + (componentOptions.tag ? `::${componentOptions.tag}` : '')
          : vnode.key;
        if (cache[key]) {
          vnode.componentInstance = cache[key].componentInstance;
          // make current key freshest
          remove(keys, key);
          keys.push(key);
        } else {
          // delay setting the cache until update
          this.vnodeToCache = vnode;
          this.keyToCache = key;
        }

        vnode.data.keepAlive = true;
      }
      return vnode || (slot && slot[0])
    }
  };

  var builtInComponents = {
    KeepAlive
  };

  /*  */

  function initGlobalAPI (Vue) {
    // config
    const configDef = {};
    configDef.get = () => config;
    {
      configDef.set = () => {
        warn(
          'Do not replace the Vue.config object, set individual fields instead.'
        );
      };
    }
    Object.defineProperty(Vue, 'config', configDef);

    // exposed util methods.
    // NOTE: these are not considered part of the public API - avoid relying on
    // them unless you are aware of the risk.
    Vue.util = {
      warn,
      extend,
      mergeOptions,
      defineReactive: defineReactive$$1
    };

    Vue.set = set;
    Vue.delete = del;
    Vue.nextTick = nextTick;

    // 2.6 explicit observable API
    Vue.observable = (obj) => {
      observe(obj);
      return obj
    };

    Vue.options = Object.create(null);
    ASSET_TYPES.forEach(type => {
      Vue.options[type + 's'] = Object.create(null);
    });

    // this is used to identify the "base" constructor to extend all plain-object
    // components with in Weex's multi-instance scenarios.
    Vue.options._base = Vue;

    extend(Vue.options.components, builtInComponents);

    initUse(Vue);
    initMixin$1(Vue);
    initExtend(Vue);
    initAssetRegisters(Vue);
  }

  initGlobalAPI(Vue);

  Object.defineProperty(Vue.prototype, '$isServer', {
    get: isServerRendering
  });

  Object.defineProperty(Vue.prototype, '$ssrContext', {
    get () {
      /* istanbul ignore next */
      return this.$vnode && this.$vnode.ssrContext
    }
  });

  // expose FunctionalRenderContext for ssr runtime helper installation
  Object.defineProperty(Vue, 'FunctionalRenderContext', {
    value: FunctionalRenderContext
  });

  Vue.version = '2.6.14';

  /*  */

  // these are reserved for web because they are directly compiled away
  // during template compilation
  const isReservedAttr = makeMap('style,class');

  // attributes that should be using props for binding
  const acceptValue = makeMap('input,textarea,option,select,progress');
  const mustUseProp = (tag, type, attr) => {
    return (
      (attr === 'value' && acceptValue(tag)) && type !== 'button' ||
      (attr === 'selected' && tag === 'option') ||
      (attr === 'checked' && tag === 'input') ||
      (attr === 'muted' && tag === 'video')
    )
  };

  const isEnumeratedAttr = makeMap('contenteditable,draggable,spellcheck');

  const isValidContentEditableValue = makeMap('events,caret,typing,plaintext-only');

  const convertEnumeratedValue = (key, value) => {
    return isFalsyAttrValue(value) || value === 'false'
      ? 'false'
      // allow arbitrary string value for contenteditable
      : key === 'contenteditable' && isValidContentEditableValue(value)
        ? value
        : 'true'
  };

  const isBooleanAttr = makeMap(
    'allowfullscreen,async,autofocus,autoplay,checked,compact,controls,declare,' +
    'default,defaultchecked,defaultmuted,defaultselected,defer,disabled,' +
    'enabled,formnovalidate,hidden,indeterminate,inert,ismap,itemscope,loop,multiple,' +
    'muted,nohref,noresize,noshade,novalidate,nowrap,open,pauseonexit,readonly,' +
    'required,reversed,scoped,seamless,selected,sortable,' +
    'truespeed,typemustmatch,visible'
  );

  const xlinkNS = 'http://www.w3.org/1999/xlink';

  const isXlink = (name) => {
    return name.charAt(5) === ':' && name.slice(0, 5) === 'xlink'
  };

  const getXlinkProp = (name) => {
    return isXlink(name) ? name.slice(6, name.length) : ''
  };

  const isFalsyAttrValue = (val) => {
    return val == null || val === false
  };

  /*  */

  function genClassForVnode (vnode) {
    let data = vnode.data;
    let parentNode = vnode;
    let childNode = vnode;
    while (isDef(childNode.componentInstance)) {
      childNode = childNode.componentInstance._vnode;
      if (childNode && childNode.data) {
        data = mergeClassData(childNode.data, data);
      }
    }
    while (isDef(parentNode = parentNode.parent)) {
      if (parentNode && parentNode.data) {
        data = mergeClassData(data, parentNode.data);
      }
    }
    return renderClass(data.staticClass, data.class)
  }

  function mergeClassData (child, parent) {
    return {
      staticClass: concat(child.staticClass, parent.staticClass),
      class: isDef(child.class)
        ? [child.class, parent.class]
        : parent.class
    }
  }

  function renderClass (
    staticClass,
    dynamicClass
  ) {
    if (isDef(staticClass) || isDef(dynamicClass)) {
      return concat(staticClass, stringifyClass(dynamicClass))
    }
    /* istanbul ignore next */
    return ''
  }

  function concat (a, b) {
    return a ? b ? (a + ' ' + b) : a : (b || '')
  }

  function stringifyClass (value) {
    if (Array.isArray(value)) {
      return stringifyArray(value)
    }
    if (isObject(value)) {
      return stringifyObject(value)
    }
    if (typeof value === 'string') {
      return value
    }
    /* istanbul ignore next */
    return ''
  }

  function stringifyArray (value) {
    let res = '';
    let stringified;
    for (let i = 0, l = value.length; i < l; i++) {
      if (isDef(stringified = stringifyClass(value[i])) && stringified !== '') {
        if (res) res += ' ';
        res += stringified;
      }
    }
    return res
  }

  function stringifyObject (value) {
    let res = '';
    for (const key in value) {
      if (value[key]) {
        if (res) res += ' ';
        res += key;
      }
    }
    return res
  }

  /*  */

  const namespaceMap = {
    svg: 'http://www.w3.org/2000/svg',
    math: 'http://www.w3.org/1998/Math/MathML'
  };

  const isHTMLTag = makeMap(
    'html,body,base,head,link,meta,style,title,' +
    'address,article,aside,footer,header,h1,h2,h3,h4,h5,h6,hgroup,nav,section,' +
    'div,dd,dl,dt,figcaption,figure,picture,hr,img,li,main,ol,p,pre,ul,' +
    'a,b,abbr,bdi,bdo,br,cite,code,data,dfn,em,i,kbd,mark,q,rp,rt,rtc,ruby,' +
    's,samp,small,span,strong,sub,sup,time,u,var,wbr,area,audio,map,track,video,' +
    'embed,object,param,source,canvas,script,noscript,del,ins,' +
    'caption,col,colgroup,table,thead,tbody,td,th,tr,' +
    'button,datalist,fieldset,form,input,label,legend,meter,optgroup,option,' +
    'output,progress,select,textarea,' +
    'details,dialog,menu,menuitem,summary,' +
    'content,element,shadow,template,blockquote,iframe,tfoot'
  );

  // this map is intentionally selective, only covering SVG elements that may
  // contain child elements.
  const isSVG = makeMap(
    'svg,animate,circle,clippath,cursor,defs,desc,ellipse,filter,font-face,' +
    'foreignobject,g,glyph,image,line,marker,mask,missing-glyph,path,pattern,' +
    'polygon,polyline,rect,switch,symbol,text,textpath,tspan,use,view',
    true
  );

  const isPreTag = (tag) => tag === 'pre';

  const isReservedTag = (tag) => {
    return isHTMLTag(tag) || isSVG(tag)
  };

  function getTagNamespace (tag) {
    if (isSVG(tag)) {
      return 'svg'
    }
    // basic support for MathML
    // note it doesn't support other MathML elements being component roots
    if (tag === 'math') {
      return 'math'
    }
  }

  const unknownElementCache = Object.create(null);
  function isUnknownElement (tag) {
    /* istanbul ignore if */
    if (!inBrowser) {
      return true
    }
    if (isReservedTag(tag)) {
      return false
    }
    tag = tag.toLowerCase();
    /* istanbul ignore if */
    if (unknownElementCache[tag] != null) {
      return unknownElementCache[tag]
    }
    const el = document.createElement(tag);
    if (tag.indexOf('-') > -1) {
      // http://stackoverflow.com/a/28210364/1070244
      return (unknownElementCache[tag] = (
        el.constructor === window.HTMLUnknownElement ||
        el.constructor === window.HTMLElement
      ))
    } else {
      return (unknownElementCache[tag] = /HTMLUnknownElement/.test(el.toString()))
    }
  }

  const isTextInputType = makeMap('text,number,password,search,email,tel,url');

  /*  */

  /**
   * Query an element selector if it's not an element already.
   */
  function query (el) {
    if (typeof el === 'string') {
      const selected = document.querySelector(el);
      if (!selected) {
        warn(
          'Cannot find element: ' + el
        );
        return document.createElement('div')
      }
      return selected
    } else {
      return el
    }
  }

  /*  */

  function createElement$1 (tagName, vnode) {
    const elm = document.createElement(tagName);
    if (tagName !== 'select') {
      return elm
    }
    // false or null will remove the attribute but undefined will not
    if (vnode.data && vnode.data.attrs && vnode.data.attrs.multiple !== undefined) {
      elm.setAttribute('multiple', 'multiple');
    }
    return elm
  }

  function createElementNS (namespace, tagName) {
    return document.createElementNS(namespaceMap[namespace], tagName)
  }

  function createTextNode (text) {
    return document.createTextNode(text)
  }

  function createComment (text) {
    return document.createComment(text)
  }

  function insertBefore (parentNode, newNode, referenceNode) {
    parentNode.insertBefore(newNode, referenceNode);
  }

  function removeChild (node, child) {
    node.removeChild(child);
  }

  function appendChild (node, child) {
    node.appendChild(child);
  }

  function parentNode (node) {
    return node.parentNode
  }

  function nextSibling (node) {
    return node.nextSibling
  }

  function tagName (node) {
    return node.tagName
  }

  function setTextContent (node, text) {
    node.textContent = text;
  }

  function setStyleScope (node, scopeId) {
    node.setAttribute(scopeId, '');
  }

  var nodeOps = /*#__PURE__*/Object.freeze({
    createElement: createElement$1,
    createElementNS: createElementNS,
    createTextNode: createTextNode,
    createComment: createComment,
    insertBefore: insertBefore,
    removeChild: removeChild,
    appendChild: appendChild,
    parentNode: parentNode,
    nextSibling: nextSibling,
    tagName: tagName,
    setTextContent: setTextContent,
    setStyleScope: setStyleScope
  });

  /*  */

  var ref = {
    create (_, vnode) {
      registerRef(vnode);
    },
    update (oldVnode, vnode) {
      if (oldVnode.data.ref !== vnode.data.ref) {
        registerRef(oldVnode, true);
        registerRef(vnode);
      }
    },
    destroy (vnode) {
      registerRef(vnode, true);
    }
  };

  function registerRef (vnode, isRemoval) {
    const key = vnode.data.ref;
    if (!isDef(key)) return

    const vm = vnode.context;
    const ref = vnode.componentInstance || vnode.elm;
    const refs = vm.$refs;
    if (isRemoval) {
      if (Array.isArray(refs[key])) {
        remove(refs[key], ref);
      } else if (refs[key] === ref) {
        refs[key] = undefined;
      }
    } else {
      if (vnode.data.refInFor) {
        if (!Array.isArray(refs[key])) {
          refs[key] = [ref];
        } else if (refs[key].indexOf(ref) < 0) {
          // $flow-disable-line
          refs[key].push(ref);
        }
      } else {
        refs[key] = ref;
      }
    }
  }

  /**
   * Virtual DOM patching algorithm based on Snabbdom by
   * Simon Friis Vindum (@paldepind)
   * Licensed under the MIT License
   * https://github.com/paldepind/snabbdom/blob/master/LICENSE
   *
   * modified by Evan You (@yyx990803)
   *
   * Not type-checking this because this file is perf-critical and the cost
   * of making flow understand it is not worth it.
   */

  const emptyNode = new VNode('', {}, []);

  const hooks = ['create', 'activate', 'update', 'remove', 'destroy'];

  function sameVnode (a, b) {
    return (
      a.key === b.key &&
      a.asyncFactory === b.asyncFactory && (
        (
          a.tag === b.tag &&
          a.isComment === b.isComment &&
          isDef(a.data) === isDef(b.data) &&
          sameInputType(a, b)
        ) || (
          isTrue(a.isAsyncPlaceholder) &&
          isUndef(b.asyncFactory.error)
        )
      )
    )
  }

  function sameInputType (a, b) {
    if (a.tag !== 'input') return true
    let i;
    const typeA = isDef(i = a.data) && isDef(i = i.attrs) && i.type;
    const typeB = isDef(i = b.data) && isDef(i = i.attrs) && i.type;
    return typeA === typeB || isTextInputType(typeA) && isTextInputType(typeB)
  }

  function createKeyToOldIdx (children, beginIdx, endIdx) {
    let i, key;
    const map = {};
    for (i = beginIdx; i <= endIdx; ++i) {
      key = children[i].key;
      if (isDef(key)) map[key] = i;
    }
    return map
  }

  function createPatchFunction (backend) {
    let i, j;
    const cbs = {};

    const { modules, nodeOps } = backend;

    for (i = 0; i < hooks.length; ++i) {
      cbs[hooks[i]] = [];
      for (j = 0; j < modules.length; ++j) {
        if (isDef(modules[j][hooks[i]])) {
          cbs[hooks[i]].push(modules[j][hooks[i]]);
        }
      }
    }

    function emptyNodeAt (elm) {
      return new VNode(nodeOps.tagName(elm).toLowerCase(), {}, [], undefined, elm)
    }

    function createRmCb (childElm, listeners) {
      function remove$$1 () {
        if (--remove$$1.listeners === 0) {
          removeNode(childElm);
        }
      }
      remove$$1.listeners = listeners;
      return remove$$1
    }

    function removeNode (el) {
      const parent = nodeOps.parentNode(el);
      // element may have already been removed due to v-html / v-text
      if (isDef(parent)) {
        nodeOps.removeChild(parent, el);
      }
    }

    function isUnknownElement$$1 (vnode, inVPre) {
      return (
        !inVPre &&
        !vnode.ns &&
        !(
          config.ignoredElements.length &&
          config.ignoredElements.some(ignore => {
            return isRegExp(ignore)
              ? ignore.test(vnode.tag)
              : ignore === vnode.tag
          })
        ) &&
        config.isUnknownElement(vnode.tag)
      )
    }

    let creatingElmInVPre = 0;

    function createElm (
      vnode,
      insertedVnodeQueue,
      parentElm,
      refElm,
      nested,
      ownerArray,
      index
    ) {
      if (isDef(vnode.elm) && isDef(ownerArray)) {
        // This vnode was used in a previous render!
        // now it's used as a new node, overwriting its elm would cause
        // potential patch errors down the road when it's used as an insertion
        // reference node. Instead, we clone the node on-demand before creating
        // associated DOM element for it.
        vnode = ownerArray[index] = cloneVNode(vnode);
      }

      vnode.isRootInsert = !nested; // for transition enter check
      if (createComponent(vnode, insertedVnodeQueue, parentElm, refElm)) {
        return
      }

      const data = vnode.data;
      const children = vnode.children;
      const tag = vnode.tag;
      if (isDef(tag)) {
        {
          if (data && data.pre) {
            creatingElmInVPre++;
          }
          if (isUnknownElement$$1(vnode, creatingElmInVPre)) {
            warn(
              'Unknown custom element: <' + tag + '> - did you ' +
              'register the component correctly? For recursive components, ' +
              'make sure to provide the "name" option.',
              vnode.context
            );
          }
        }

        vnode.elm = vnode.ns
          ? nodeOps.createElementNS(vnode.ns, tag)
          : nodeOps.createElement(tag, vnode);
        setScope(vnode);

        /* istanbul ignore if */
        {
          createChildren(vnode, children, insertedVnodeQueue);
          if (isDef(data)) {
            invokeCreateHooks(vnode, insertedVnodeQueue);
          }
          insert(parentElm, vnode.elm, refElm);
        }

        if (data && data.pre) {
          creatingElmInVPre--;
        }
      } else if (isTrue(vnode.isComment)) {
        vnode.elm = nodeOps.createComment(vnode.text);
        insert(parentElm, vnode.elm, refElm);
      } else {
        vnode.elm = nodeOps.createTextNode(vnode.text);
        insert(parentElm, vnode.elm, refElm);
      }
    }

    function createComponent (vnode, insertedVnodeQueue, parentElm, refElm) {
      let i = vnode.data;
      if (isDef(i)) {
        const isReactivated = isDef(vnode.componentInstance) && i.keepAlive;
        if (isDef(i = i.hook) && isDef(i = i.init)) {
          i(vnode, false /* hydrating */);
        }
        // after calling the init hook, if the vnode is a child component
        // it should've created a child instance and mounted it. the child
        // component also has set the placeholder vnode's elm.
        // in that case we can just return the element and be done.
        if (isDef(vnode.componentInstance)) {
          initComponent(vnode, insertedVnodeQueue);
          insert(parentElm, vnode.elm, refElm);
          if (isTrue(isReactivated)) {
            reactivateComponent(vnode, insertedVnodeQueue, parentElm, refElm);
          }
          return true
        }
      }
    }

    function initComponent (vnode, insertedVnodeQueue) {
      if (isDef(vnode.data.pendingInsert)) {
        insertedVnodeQueue.push.apply(insertedVnodeQueue, vnode.data.pendingInsert);
        vnode.data.pendingInsert = null;
      }
      vnode.elm = vnode.componentInstance.$el;
      if (isPatchable(vnode)) {
        invokeCreateHooks(vnode, insertedVnodeQueue);
        setScope(vnode);
      } else {
        // empty component root.
        // skip all element-related modules except for ref (#3455)
        registerRef(vnode);
        // make sure to invoke the insert hook
        insertedVnodeQueue.push(vnode);
      }
    }

    function reactivateComponent (vnode, insertedVnodeQueue, parentElm, refElm) {
      let i;
      // hack for #4339: a reactivated component with inner transition
      // does not trigger because the inner node's created hooks are not called
      // again. It's not ideal to involve module-specific logic in here but
      // there doesn't seem to be a better way to do it.
      let innerNode = vnode;
      while (innerNode.componentInstance) {
        innerNode = innerNode.componentInstance._vnode;
        if (isDef(i = innerNode.data) && isDef(i = i.transition)) {
          for (i = 0; i < cbs.activate.length; ++i) {
            cbs.activate[i](emptyNode, innerNode);
          }
          insertedVnodeQueue.push(innerNode);
          break
        }
      }
      // unlike a newly created component,
      // a reactivated keep-alive component doesn't insert itself
      insert(parentElm, vnode.elm, refElm);
    }

    function insert (parent, elm, ref$$1) {
      if (isDef(parent)) {
        if (isDef(ref$$1)) {
          if (nodeOps.parentNode(ref$$1) === parent) {
            nodeOps.insertBefore(parent, elm, ref$$1);
          }
        } else {
          nodeOps.appendChild(parent, elm);
        }
      }
    }

    function createChildren (vnode, children, insertedVnodeQueue) {
      if (Array.isArray(children)) {
        {
          checkDuplicateKeys(children);
        }
        for (let i = 0; i < children.length; ++i) {
          createElm(children[i], insertedVnodeQueue, vnode.elm, null, true, children, i);
        }
      } else if (isPrimitive(vnode.text)) {
        nodeOps.appendChild(vnode.elm, nodeOps.createTextNode(String(vnode.text)));
      }
    }

    function isPatchable (vnode) {
      while (vnode.componentInstance) {
        vnode = vnode.componentInstance._vnode;
      }
      return isDef(vnode.tag)
    }

    function invokeCreateHooks (vnode, insertedVnodeQueue) {
      for (let i = 0; i < cbs.create.length; ++i) {
        cbs.create[i](emptyNode, vnode);
      }
      i = vnode.data.hook; // Reuse variable
      if (isDef(i)) {
        if (isDef(i.create)) i.create(emptyNode, vnode);
        if (isDef(i.insert)) insertedVnodeQueue.push(vnode);
      }
    }

    // set scope id attribute for scoped CSS.
    // this is implemented as a special case to avoid the overhead
    // of going through the normal attribute patching process.
    function setScope (vnode) {
      let i;
      if (isDef(i = vnode.fnScopeId)) {
        nodeOps.setStyleScope(vnode.elm, i);
      } else {
        let ancestor = vnode;
        while (ancestor) {
          if (isDef(i = ancestor.context) && isDef(i = i.$options._scopeId)) {
            nodeOps.setStyleScope(vnode.elm, i);
          }
          ancestor = ancestor.parent;
        }
      }
      // for slot content they should also get the scopeId from the host instance.
      if (isDef(i = activeInstance) &&
        i !== vnode.context &&
        i !== vnode.fnContext &&
        isDef(i = i.$options._scopeId)
      ) {
        nodeOps.setStyleScope(vnode.elm, i);
      }
    }

    function addVnodes (parentElm, refElm, vnodes, startIdx, endIdx, insertedVnodeQueue) {
      for (; startIdx <= endIdx; ++startIdx) {
        createElm(vnodes[startIdx], insertedVnodeQueue, parentElm, refElm, false, vnodes, startIdx);
      }
    }

    function invokeDestroyHook (vnode) {
      let i, j;
      const data = vnode.data;
      if (isDef(data)) {
        if (isDef(i = data.hook) && isDef(i = i.destroy)) i(vnode);
        for (i = 0; i < cbs.destroy.length; ++i) cbs.destroy[i](vnode);
      }
      if (isDef(i = vnode.children)) {
        for (j = 0; j < vnode.children.length; ++j) {
          invokeDestroyHook(vnode.children[j]);
        }
      }
    }

    function removeVnodes (vnodes, startIdx, endIdx) {
      for (; startIdx <= endIdx; ++startIdx) {
        const ch = vnodes[startIdx];
        if (isDef(ch)) {
          if (isDef(ch.tag)) {
            removeAndInvokeRemoveHook(ch);
            invokeDestroyHook(ch);
          } else { // Text node
            removeNode(ch.elm);
          }
        }
      }
    }

    function removeAndInvokeRemoveHook (vnode, rm) {
      if (isDef(rm) || isDef(vnode.data)) {
        let i;
        const listeners = cbs.remove.length + 1;
        if (isDef(rm)) {
          // we have a recursively passed down rm callback
          // increase the listeners count
          rm.listeners += listeners;
        } else {
          // directly removing
          rm = createRmCb(vnode.elm, listeners);
        }
        // recursively invoke hooks on child component root node
        if (isDef(i = vnode.componentInstance) && isDef(i = i._vnode) && isDef(i.data)) {
          removeAndInvokeRemoveHook(i, rm);
        }
        for (i = 0; i < cbs.remove.length; ++i) {
          cbs.remove[i](vnode, rm);
        }
        if (isDef(i = vnode.data.hook) && isDef(i = i.remove)) {
          i(vnode, rm);
        } else {
          rm();
        }
      } else {
        removeNode(vnode.elm);
      }
    }

    function updateChildren (parentElm, oldCh, newCh, insertedVnodeQueue, removeOnly) {
      let oldStartIdx = 0;
      let newStartIdx = 0;
      let oldEndIdx = oldCh.length - 1;
      let oldStartVnode = oldCh[0];
      let oldEndVnode = oldCh[oldEndIdx];
      let newEndIdx = newCh.length - 1;
      let newStartVnode = newCh[0];
      let newEndVnode = newCh[newEndIdx];
      let oldKeyToIdx, idxInOld, vnodeToMove, refElm;

      // removeOnly is a special flag used only by <transition-group>
      // to ensure removed elements stay in correct relative positions
      // during leaving transitions
      const canMove = !removeOnly;

      {
        checkDuplicateKeys(newCh);
      }

      while (oldStartIdx <= oldEndIdx && newStartIdx <= newEndIdx) {
        if (isUndef(oldStartVnode)) {
          oldStartVnode = oldCh[++oldStartIdx]; // Vnode has been moved left
        } else if (isUndef(oldEndVnode)) {
          oldEndVnode = oldCh[--oldEndIdx];
        } else if (sameVnode(oldStartVnode, newStartVnode)) {
          patchVnode(oldStartVnode, newStartVnode, insertedVnodeQueue, newCh, newStartIdx);
          oldStartVnode = oldCh[++oldStartIdx];
          newStartVnode = newCh[++newStartIdx];
        } else if (sameVnode(oldEndVnode, newEndVnode)) {
          patchVnode(oldEndVnode, newEndVnode, insertedVnodeQueue, newCh, newEndIdx);
          oldEndVnode = oldCh[--oldEndIdx];
          newEndVnode = newCh[--newEndIdx];
        } else if (sameVnode(oldStartVnode, newEndVnode)) { // Vnode moved right
          patchVnode(oldStartVnode, newEndVnode, insertedVnodeQueue, newCh, newEndIdx);
          canMove && nodeOps.insertBefore(parentElm, oldStartVnode.elm, nodeOps.nextSibling(oldEndVnode.elm));
          oldStartVnode = oldCh[++oldStartIdx];
          newEndVnode = newCh[--newEndIdx];
        } else if (sameVnode(oldEndVnode, newStartVnode)) { // Vnode moved left
          patchVnode(oldEndVnode, newStartVnode, insertedVnodeQueue, newCh, newStartIdx);
          canMove && nodeOps.insertBefore(parentElm, oldEndVnode.elm, oldStartVnode.elm);
          oldEndVnode = oldCh[--oldEndIdx];
          newStartVnode = newCh[++newStartIdx];
        } else {
          if (isUndef(oldKeyToIdx)) oldKeyToIdx = createKeyToOldIdx(oldCh, oldStartIdx, oldEndIdx);
          idxInOld = isDef(newStartVnode.key)
            ? oldKeyToIdx[newStartVnode.key]
            : findIdxInOld(newStartVnode, oldCh, oldStartIdx, oldEndIdx);
          if (isUndef(idxInOld)) { // New element
            createElm(newStartVnode, insertedVnodeQueue, parentElm, oldStartVnode.elm, false, newCh, newStartIdx);
          } else {
            vnodeToMove = oldCh[idxInOld];
            if (sameVnode(vnodeToMove, newStartVnode)) {
              patchVnode(vnodeToMove, newStartVnode, insertedVnodeQueue, newCh, newStartIdx);
              oldCh[idxInOld] = undefined;
              canMove && nodeOps.insertBefore(parentElm, vnodeToMove.elm, oldStartVnode.elm);
            } else {
              // same key but different element. treat as new element
              createElm(newStartVnode, insertedVnodeQueue, parentElm, oldStartVnode.elm, false, newCh, newStartIdx);
            }
          }
          newStartVnode = newCh[++newStartIdx];
        }
      }
      if (oldStartIdx > oldEndIdx) {
        refElm = isUndef(newCh[newEndIdx + 1]) ? null : newCh[newEndIdx + 1].elm;
        addVnodes(parentElm, refElm, newCh, newStartIdx, newEndIdx, insertedVnodeQueue);
      } else if (newStartIdx > newEndIdx) {
        removeVnodes(oldCh, oldStartIdx, oldEndIdx);
      }
    }

    function checkDuplicateKeys (children) {
      const seenKeys = {};
      for (let i = 0; i < children.length; i++) {
        const vnode = children[i];
        const key = vnode.key;
        if (isDef(key)) {
          if (seenKeys[key]) {
            warn(
              `Duplicate keys detected: '${key}'. This may cause an update error.`,
              vnode.context
            );
          } else {
            seenKeys[key] = true;
          }
        }
      }
    }

    function findIdxInOld (node, oldCh, start, end) {
      for (let i = start; i < end; i++) {
        const c = oldCh[i];
        if (isDef(c) && sameVnode(node, c)) return i
      }
    }

    function patchVnode (
      oldVnode,
      vnode,
      insertedVnodeQueue,
      ownerArray,
      index,
      removeOnly
    ) {
      if (oldVnode === vnode) {
        return
      }

      if (isDef(vnode.elm) && isDef(ownerArray)) {
        // clone reused vnode
        vnode = ownerArray[index] = cloneVNode(vnode);
      }

      const elm = vnode.elm = oldVnode.elm;

      if (isTrue(oldVnode.isAsyncPlaceholder)) {
        if (isDef(vnode.asyncFactory.resolved)) {
          hydrate(oldVnode.elm, vnode, insertedVnodeQueue);
        } else {
          vnode.isAsyncPlaceholder = true;
        }
        return
      }

      // reuse element for static trees.
      // note we only do this if the vnode is cloned -
      // if the new node is not cloned it means the render functions have been
      // reset by the hot-reload-api and we need to do a proper re-render.
      if (isTrue(vnode.isStatic) &&
        isTrue(oldVnode.isStatic) &&
        vnode.key === oldVnode.key &&
        (isTrue(vnode.isCloned) || isTrue(vnode.isOnce))
      ) {
        vnode.componentInstance = oldVnode.componentInstance;
        return
      }

      let i;
      const data = vnode.data;
      if (isDef(data) && isDef(i = data.hook) && isDef(i = i.prepatch)) {
        i(oldVnode, vnode);
      }

      const oldCh = oldVnode.children;
      const ch = vnode.children;
      if (isDef(data) && isPatchable(vnode)) {
        for (i = 0; i < cbs.update.length; ++i) cbs.update[i](oldVnode, vnode);
        if (isDef(i = data.hook) && isDef(i = i.update)) i(oldVnode, vnode);
      }
      if (isUndef(vnode.text)) {
        if (isDef(oldCh) && isDef(ch)) {
          if (oldCh !== ch) updateChildren(elm, oldCh, ch, insertedVnodeQueue, removeOnly);
        } else if (isDef(ch)) {
          {
            checkDuplicateKeys(ch);
          }
          if (isDef(oldVnode.text)) nodeOps.setTextContent(elm, '');
          addVnodes(elm, null, ch, 0, ch.length - 1, insertedVnodeQueue);
        } else if (isDef(oldCh)) {
          removeVnodes(oldCh, 0, oldCh.length - 1);
        } else if (isDef(oldVnode.text)) {
          nodeOps.setTextContent(elm, '');
        }
      } else if (oldVnode.text !== vnode.text) {
        nodeOps.setTextContent(elm, vnode.text);
      }
      if (isDef(data)) {
        if (isDef(i = data.hook) && isDef(i = i.postpatch)) i(oldVnode, vnode);
      }
    }

    function invokeInsertHook (vnode, queue, initial) {
      // delay insert hooks for component root nodes, invoke them after the
      // element is really inserted
      if (isTrue(initial) && isDef(vnode.parent)) {
        vnode.parent.data.pendingInsert = queue;
      } else {
        for (let i = 0; i < queue.length; ++i) {
          queue[i].data.hook.insert(queue[i]);
        }
      }
    }

    let hydrationBailed = false;
    // list of modules that can skip create hook during hydration because they
    // are already rendered on the client or has no need for initialization
    // Note: style is excluded because it relies on initial clone for future
    // deep updates (#7063).
    const isRenderedModule = makeMap('attrs,class,staticClass,staticStyle,key');

    // Note: this is a browser-only function so we can assume elms are DOM nodes.
    function hydrate (elm, vnode, insertedVnodeQueue, inVPre) {
      let i;
      const { tag, data, children } = vnode;
      inVPre = inVPre || (data && data.pre);
      vnode.elm = elm;

      if (isTrue(vnode.isComment) && isDef(vnode.asyncFactory)) {
        vnode.isAsyncPlaceholder = true;
        return true
      }
      // assert node match
      {
        if (!assertNodeMatch(elm, vnode, inVPre)) {
          return false
        }
      }
      if (isDef(data)) {
        if (isDef(i = data.hook) && isDef(i = i.init)) i(vnode, true /* hydrating */);
        if (isDef(i = vnode.componentInstance)) {
          // child component. it should have hydrated its own tree.
          initComponent(vnode, insertedVnodeQueue);
          return true
        }
      }
      if (isDef(tag)) {
        if (isDef(children)) {
          // empty element, allow client to pick up and populate children
          if (!elm.hasChildNodes()) {
            createChildren(vnode, children, insertedVnodeQueue);
          } else {
            // v-html and domProps: innerHTML
            if (isDef(i = data) && isDef(i = i.domProps) && isDef(i = i.innerHTML)) {
              if (i !== elm.innerHTML) {
                /* istanbul ignore if */
                if (typeof console !== 'undefined' &&
                  !hydrationBailed
                ) {
                  hydrationBailed = true;
                  console.warn('Parent: ', elm);
                  console.warn('server innerHTML: ', i);
                  console.warn('client innerHTML: ', elm.innerHTML);
                }
                return false
              }
            } else {
              // iterate and compare children lists
              let childrenMatch = true;
              let childNode = elm.firstChild;
              for (let i = 0; i < children.length; i++) {
                if (!childNode || !hydrate(childNode, children[i], insertedVnodeQueue, inVPre)) {
                  childrenMatch = false;
                  break
                }
                childNode = childNode.nextSibling;
              }
              // if childNode is not null, it means the actual childNodes list is
              // longer than the virtual children list.
              if (!childrenMatch || childNode) {
                /* istanbul ignore if */
                if (typeof console !== 'undefined' &&
                  !hydrationBailed
                ) {
                  hydrationBailed = true;
                  console.warn('Parent: ', elm);
                  console.warn('Mismatching childNodes vs. VNodes: ', elm.childNodes, children);
                }
                return false
              }
            }
          }
        }
        if (isDef(data)) {
          let fullInvoke = false;
          for (const key in data) {
            if (!isRenderedModule(key)) {
              fullInvoke = true;
              invokeCreateHooks(vnode, insertedVnodeQueue);
              break
            }
          }
          if (!fullInvoke && data['class']) {
            // ensure collecting deps for deep class bindings for future updates
            traverse(data['class']);
          }
        }
      } else if (elm.data !== vnode.text) {
        elm.data = vnode.text;
      }
      return true
    }

    function assertNodeMatch (node, vnode, inVPre) {
      if (isDef(vnode.tag)) {
        return vnode.tag.indexOf('vue-component') === 0 || (
          !isUnknownElement$$1(vnode, inVPre) &&
          vnode.tag.toLowerCase() === (node.tagName && node.tagName.toLowerCase())
        )
      } else {
        return node.nodeType === (vnode.isComment ? 8 : 3)
      }
    }

    return function patch (oldVnode, vnode, hydrating, removeOnly) {
      if (isUndef(vnode)) {
        if (isDef(oldVnode)) invokeDestroyHook(oldVnode);
        return
      }

      let isInitialPatch = false;
      const insertedVnodeQueue = [];

      if (isUndef(oldVnode)) {
        // empty mount (likely as component), create new root element
        isInitialPatch = true;
        createElm(vnode, insertedVnodeQueue);
      } else {
        const isRealElement = isDef(oldVnode.nodeType);
        if (!isRealElement && sameVnode(oldVnode, vnode)) {
          // patch existing root node
          patchVnode(oldVnode, vnode, insertedVnodeQueue, null, null, removeOnly);
        } else {
          if (isRealElement) {
            // mounting to a real element
            // check if this is server-rendered content and if we can perform
            // a successful hydration.
            if (oldVnode.nodeType === 1 && oldVnode.hasAttribute(SSR_ATTR)) {
              oldVnode.removeAttribute(SSR_ATTR);
              hydrating = true;
            }
            if (isTrue(hydrating)) {
              if (hydrate(oldVnode, vnode, insertedVnodeQueue)) {
                invokeInsertHook(vnode, insertedVnodeQueue, true);
                return oldVnode
              } else {
                warn(
                  'The client-side rendered virtual DOM tree is not matching ' +
                  'server-rendered content. This is likely caused by incorrect ' +
                  'HTML markup, for example nesting block-level elements inside ' +
                  '<p>, or missing <tbody>. Bailing hydration and performing ' +
                  'full client-side render.'
                );
              }
            }
            // either not server-rendered, or hydration failed.
            // create an empty node and replace it
            oldVnode = emptyNodeAt(oldVnode);
          }

          // replacing existing element
          const oldElm = oldVnode.elm;
          const parentElm = nodeOps.parentNode(oldElm);

          // create new node
          createElm(
            vnode,
            insertedVnodeQueue,
            // extremely rare edge case: do not insert if old element is in a
            // leaving transition. Only happens when combining transition +
            // keep-alive + HOCs. (#4590)
            oldElm._leaveCb ? null : parentElm,
            nodeOps.nextSibling(oldElm)
          );

          // update parent placeholder node element, recursively
          if (isDef(vnode.parent)) {
            let ancestor = vnode.parent;
            const patchable = isPatchable(vnode);
            while (ancestor) {
              for (let i = 0; i < cbs.destroy.length; ++i) {
                cbs.destroy[i](ancestor);
              }
              ancestor.elm = vnode.elm;
              if (patchable) {
                for (let i = 0; i < cbs.create.length; ++i) {
                  cbs.create[i](emptyNode, ancestor);
                }
                // #6513
                // invoke insert hooks that may have been merged by create hooks.
                // e.g. for directives that uses the "inserted" hook.
                const insert = ancestor.data.hook.insert;
                if (insert.merged) {
                  // start at index 1 to avoid re-invoking component mounted hook
                  for (let i = 1; i < insert.fns.length; i++) {
                    insert.fns[i]();
                  }
                }
              } else {
                registerRef(ancestor);
              }
              ancestor = ancestor.parent;
            }
          }

          // destroy old node
          if (isDef(parentElm)) {
            removeVnodes([oldVnode], 0, 0);
          } else if (isDef(oldVnode.tag)) {
            invokeDestroyHook(oldVnode);
          }
        }
      }

      invokeInsertHook(vnode, insertedVnodeQueue, isInitialPatch);
      return vnode.elm
    }
  }

  /*  */

  var directives = {
    create: updateDirectives,
    update: updateDirectives,
    destroy: function unbindDirectives (vnode) {
      updateDirectives(vnode, emptyNode);
    }
  };

  function updateDirectives (oldVnode, vnode) {
    if (oldVnode.data.directives || vnode.data.directives) {
      _update(oldVnode, vnode);
    }
  }

  function _update (oldVnode, vnode) {
    const isCreate = oldVnode === emptyNode;
    const isDestroy = vnode === emptyNode;
    const oldDirs = normalizeDirectives$1(oldVnode.data.directives, oldVnode.context);
    const newDirs = normalizeDirectives$1(vnode.data.directives, vnode.context);

    const dirsWithInsert = [];
    const dirsWithPostpatch = [];

    let key, oldDir, dir;
    for (key in newDirs) {
      oldDir = oldDirs[key];
      dir = newDirs[key];
      if (!oldDir) {
        // new directive, bind
        callHook$1(dir, 'bind', vnode, oldVnode);
        if (dir.def && dir.def.inserted) {
          dirsWithInsert.push(dir);
        }
      } else {
        // existing directive, update
        dir.oldValue = oldDir.value;
        dir.oldArg = oldDir.arg;
        callHook$1(dir, 'update', vnode, oldVnode);
        if (dir.def && dir.def.componentUpdated) {
          dirsWithPostpatch.push(dir);
        }
      }
    }

    if (dirsWithInsert.length) {
      const callInsert = () => {
        for (let i = 0; i < dirsWithInsert.length; i++) {
          callHook$1(dirsWithInsert[i], 'inserted', vnode, oldVnode);
        }
      };
      if (isCreate) {
        mergeVNodeHook(vnode, 'insert', callInsert);
      } else {
        callInsert();
      }
    }

    if (dirsWithPostpatch.length) {
      mergeVNodeHook(vnode, 'postpatch', () => {
        for (let i = 0; i < dirsWithPostpatch.length; i++) {
          callHook$1(dirsWithPostpatch[i], 'componentUpdated', vnode, oldVnode);
        }
      });
    }

    if (!isCreate) {
      for (key in oldDirs) {
        if (!newDirs[key]) {
          // no longer present, unbind
          callHook$1(oldDirs[key], 'unbind', oldVnode, oldVnode, isDestroy);
        }
      }
    }
  }

  const emptyModifiers = Object.create(null);

  function normalizeDirectives$1 (
    dirs,
    vm
  ) {
    const res = Object.create(null);
    if (!dirs) {
      // $flow-disable-line
      return res
    }
    let i, dir;
    for (i = 0; i < dirs.length; i++) {
      dir = dirs[i];
      if (!dir.modifiers) {
        // $flow-disable-line
        dir.modifiers = emptyModifiers;
      }
      res[getRawDirName(dir)] = dir;
      dir.def = resolveAsset(vm.$options, 'directives', dir.name, true);
    }
    // $flow-disable-line
    return res
  }

  function getRawDirName (dir) {
    return dir.rawName || `${dir.name}.${Object.keys(dir.modifiers || {}).join('.')}`
  }

  function callHook$1 (dir, hook, vnode, oldVnode, isDestroy) {
    const fn = dir.def && dir.def[hook];
    if (fn) {
      try {
        fn(vnode.elm, dir, vnode, oldVnode, isDestroy);
      } catch (e) {
        handleError(e, vnode.context, `directive ${dir.name} ${hook} hook`);
      }
    }
  }

  var baseModules = [
    ref,
    directives
  ];

  /*  */

  function updateAttrs (oldVnode, vnode) {
    const opts = vnode.componentOptions;
    if (isDef(opts) && opts.Ctor.options.inheritAttrs === false) {
      return
    }
    if (isUndef(oldVnode.data.attrs) && isUndef(vnode.data.attrs)) {
      return
    }
    let key, cur, old;
    const elm = vnode.elm;
    const oldAttrs = oldVnode.data.attrs || {};
    let attrs = vnode.data.attrs || {};
    // clone observed objects, as the user probably wants to mutate it
    if (isDef(attrs.__ob__)) {
      attrs = vnode.data.attrs = extend({}, attrs);
    }

    for (key in attrs) {
      cur = attrs[key];
      old = oldAttrs[key];
      if (old !== cur) {
        setAttr(elm, key, cur, vnode.data.pre);
      }
    }
    // #4391: in IE9, setting type can reset value for input[type=radio]
    // #6666: IE/Edge forces progress value down to 1 before setting a max
    /* istanbul ignore if */
    if ((isIE || isEdge) && attrs.value !== oldAttrs.value) {
      setAttr(elm, 'value', attrs.value);
    }
    for (key in oldAttrs) {
      if (isUndef(attrs[key])) {
        if (isXlink(key)) {
          elm.removeAttributeNS(xlinkNS, getXlinkProp(key));
        } else if (!isEnumeratedAttr(key)) {
          elm.removeAttribute(key);
        }
      }
    }
  }

  function setAttr (el, key, value, isInPre) {
    if (isInPre || el.tagName.indexOf('-') > -1) {
      baseSetAttr(el, key, value);
    } else if (isBooleanAttr(key)) {
      // set attribute for blank value
      // e.g. <option disabled>Select one</option>
      if (isFalsyAttrValue(value)) {
        el.removeAttribute(key);
      } else {
        // technically allowfullscreen is a boolean attribute for <iframe>,
        // but Flash expects a value of "true" when used on <embed> tag
        value = key === 'allowfullscreen' && el.tagName === 'EMBED'
          ? 'true'
          : key;
        el.setAttribute(key, value);
      }
    } else if (isEnumeratedAttr(key)) {
      el.setAttribute(key, convertEnumeratedValue(key, value));
    } else if (isXlink(key)) {
      if (isFalsyAttrValue(value)) {
        el.removeAttributeNS(xlinkNS, getXlinkProp(key));
      } else {
        el.setAttributeNS(xlinkNS, key, value);
      }
    } else {
      baseSetAttr(el, key, value);
    }
  }

  function baseSetAttr (el, key, value) {
    if (isFalsyAttrValue(value)) {
      el.removeAttribute(key);
    } else {
      // #7138: IE10 & 11 fires input event when setting placeholder on
      // <textarea>... block the first input event and remove the blocker
      // immediately.
      /* istanbul ignore if */
      if (
        isIE && !isIE9 &&
        el.tagName === 'TEXTAREA' &&
        key === 'placeholder' && value !== '' && !el.__ieph
      ) {
        const blocker = e => {
          e.stopImmediatePropagation();
          el.removeEventListener('input', blocker);
        };
        el.addEventListener('input', blocker);
        // $flow-disable-line
        el.__ieph = true; /* IE placeholder patched */
      }
      el.setAttribute(key, value);
    }
  }

  var attrs = {
    create: updateAttrs,
    update: updateAttrs
  };

  /*  */

  function updateClass (oldVnode, vnode) {
    const el = vnode.elm;
    const data = vnode.data;
    const oldData = oldVnode.data;
    if (
      isUndef(data.staticClass) &&
      isUndef(data.class) && (
        isUndef(oldData) || (
          isUndef(oldData.staticClass) &&
          isUndef(oldData.class)
        )
      )
    ) {
      return
    }

    let cls = genClassForVnode(vnode);

    // handle transition classes
    const transitionClass = el._transitionClasses;
    if (isDef(transitionClass)) {
      cls = concat(cls, stringifyClass(transitionClass));
    }

    // set the class
    if (cls !== el._prevClass) {
      el.setAttribute('class', cls);
      el._prevClass = cls;
    }
  }

  var klass = {
    create: updateClass,
    update: updateClass
  };

  /*  */

  const validDivisionCharRE = /[\w).+\-_$\]]/;

  function parseFilters (exp) {
    let inSingle = false;
    let inDouble = false;
    let inTemplateString = false;
    let inRegex = false;
    let curly = 0;
    let square = 0;
    let paren = 0;
    let lastFilterIndex = 0;
    let c, prev, i, expression, filters;

    for (i = 0; i < exp.length; i++) {
      prev = c;
      c = exp.charCodeAt(i);
      if (inSingle) {
        if (c === 0x27 && prev !== 0x5C) inSingle = false;
      } else if (inDouble) {
        if (c === 0x22 && prev !== 0x5C) inDouble = false;
      } else if (inTemplateString) {
        if (c === 0x60 && prev !== 0x5C) inTemplateString = false;
      } else if (inRegex) {
        if (c === 0x2f && prev !== 0x5C) inRegex = false;
      } else if (
        c === 0x7C && // pipe
        exp.charCodeAt(i + 1) !== 0x7C &&
        exp.charCodeAt(i - 1) !== 0x7C &&
        !curly && !square && !paren
      ) {
        if (expression === undefined) {
          // first filter, end of expression
          lastFilterIndex = i + 1;
          expression = exp.slice(0, i).trim();
        } else {
          pushFilter();
        }
      } else {
        switch (c) {
          case 0x22: inDouble = true; break         // "
          case 0x27: inSingle = true; break         // '
          case 0x60: inTemplateString = true; break // `
          case 0x28: paren++; break                 // (
          case 0x29: paren--; break                 // )
          case 0x5B: square++; break                // [
          case 0x5D: square--; break                // ]
          case 0x7B: curly++; break                 // {
          case 0x7D: curly--; break                 // }
        }
        if (c === 0x2f) { // /
          let j = i - 1;
          let p;
          // find first non-whitespace prev char
          for (; j >= 0; j--) {
            p = exp.charAt(j);
            if (p !== ' ') break
          }
          if (!p || !validDivisionCharRE.test(p)) {
            inRegex = true;
          }
        }
      }
    }

    if (expression === undefined) {
      expression = exp.slice(0, i).trim();
    } else if (lastFilterIndex !== 0) {
      pushFilter();
    }

    function pushFilter () {
      (filters || (filters = [])).push(exp.slice(lastFilterIndex, i).trim());
      lastFilterIndex = i + 1;
    }

    if (filters) {
      for (i = 0; i < filters.length; i++) {
        expression = wrapFilter(expression, filters[i]);
      }
    }

    return expression
  }

  function wrapFilter (exp, filter) {
    const i = filter.indexOf('(');
    if (i < 0) {
      // _f: resolveFilter
      return `_f("${filter}")(${exp})`
    } else {
      const name = filter.slice(0, i);
      const args = filter.slice(i + 1);
      return `_f("${name}")(${exp}${args !== ')' ? ',' + args : args}`
    }
  }

  /*  */



  /* eslint-disable no-unused-vars */
  function baseWarn (msg, range) {
    console.error(`[Vue compiler]: ${msg}`);
  }
  /* eslint-enable no-unused-vars */

  function pluckModuleFunction (
    modules,
    key
  ) {
    return modules
      ? modules.map(m => m[key]).filter(_ => _)
      : []
  }

  function addProp (el, name, value, range, dynamic) {
    (el.props || (el.props = [])).push(rangeSetItem({ name, value, dynamic }, range));
    el.plain = false;
  }

  function addAttr (el, name, value, range, dynamic) {
    const attrs = dynamic
      ? (el.dynamicAttrs || (el.dynamicAttrs = []))
      : (el.attrs || (el.attrs = []));
    attrs.push(rangeSetItem({ name, value, dynamic }, range));
    el.plain = false;
  }

  // add a raw attr (use this in preTransforms)
  function addRawAttr (el, name, value, range) {
    el.attrsMap[name] = value;
    el.attrsList.push(rangeSetItem({ name, value }, range));
  }

  function addDirective (
    el,
    name,
    rawName,
    value,
    arg,
    isDynamicArg,
    modifiers,
    range
  ) {
    (el.directives || (el.directives = [])).push(rangeSetItem({
      name,
      rawName,
      value,
      arg,
      isDynamicArg,
      modifiers
    }, range));
    el.plain = false;
  }

  function prependModifierMarker (symbol, name, dynamic) {
    return dynamic
      ? `_p(${name},"${symbol}")`
      : symbol + name // mark the event as captured
  }

  function addHandler (
    el,
    name,
    value,
    modifiers,
    important,
    warn,
    range,
    dynamic
  ) {
    modifiers = modifiers || emptyObject;
    // warn prevent and passive modifier
    /* istanbul ignore if */
    if (
      warn &&
      modifiers.prevent && modifiers.passive
    ) {
      warn(
        'passive and prevent can\'t be used together. ' +
        'Passive handler can\'t prevent default event.',
        range
      );
    }

    // normalize click.right and click.middle since they don't actually fire
    // this is technically browser-specific, but at least for now browsers are
    // the only target envs that have right/middle clicks.
    if (modifiers.right) {
      if (dynamic) {
        name = `(${name})==='click'?'contextmenu':(${name})`;
      } else if (name === 'click') {
        name = 'contextmenu';
        delete modifiers.right;
      }
    } else if (modifiers.middle) {
      if (dynamic) {
        name = `(${name})==='click'?'mouseup':(${name})`;
      } else if (name === 'click') {
        name = 'mouseup';
      }
    }

    // check capture modifier
    if (modifiers.capture) {
      delete modifiers.capture;
      name = prependModifierMarker('!', name, dynamic);
    }
    if (modifiers.once) {
      delete modifiers.once;
      name = prependModifierMarker('~', name, dynamic);
    }
    /* istanbul ignore if */
    if (modifiers.passive) {
      delete modifiers.passive;
      name = prependModifierMarker('&', name, dynamic);
    }

    let events;
    if (modifiers.native) {
      delete modifiers.native;
      events = el.nativeEvents || (el.nativeEvents = {});
    } else {
      events = el.events || (el.events = {});
    }

    const newHandler = rangeSetItem({ value: value.trim(), dynamic }, range);
    if (modifiers !== emptyObject) {
      newHandler.modifiers = modifiers;
    }

    const handlers = events[name];
    /* istanbul ignore if */
    if (Array.isArray(handlers)) {
      important ? handlers.unshift(newHandler) : handlers.push(newHandler);
    } else if (handlers) {
      events[name] = important ? [newHandler, handlers] : [handlers, newHandler];
    } else {
      events[name] = newHandler;
    }

    el.plain = false;
  }

  function getRawBindingAttr (
    el,
    name
  ) {
    return el.rawAttrsMap[':' + name] ||
      el.rawAttrsMap['v-bind:' + name] ||
      el.rawAttrsMap[name]
  }

  function getBindingAttr (
    el,
    name,
    getStatic
  ) {
    const dynamicValue =
      getAndRemoveAttr(el, ':' + name) ||
      getAndRemoveAttr(el, 'v-bind:' + name);
    if (dynamicValue != null) {
      return parseFilters(dynamicValue)
    } else if (getStatic !== false) {
      const staticValue = getAndRemoveAttr(el, name);
      if (staticValue != null) {
        return JSON.stringify(staticValue)
      }
    }
  }

  // note: this only removes the attr from the Array (attrsList) so that it
  // doesn't get processed by processAttrs.
  // By default it does NOT remove it from the map (attrsMap) because the map is
  // needed during codegen.
  function getAndRemoveAttr (
    el,
    name,
    removeFromMap
  ) {
    let val;
    if ((val = el.attrsMap[name]) != null) {
      const list = el.attrsList;
      for (let i = 0, l = list.length; i < l; i++) {
        if (list[i].name === name) {
          list.splice(i, 1);
          break
        }
      }
    }
    if (removeFromMap) {
      delete el.attrsMap[name];
    }
    return val
  }

  function getAndRemoveAttrByRegex (
    el,
    name
  ) {
    const list = el.attrsList;
    for (let i = 0, l = list.length; i < l; i++) {
      const attr = list[i];
      if (name.test(attr.name)) {
        list.splice(i, 1);
        return attr
      }
    }
  }

  function rangeSetItem (
    item,
    range
  ) {
    if (range) {
      if (range.start != null) {
        item.start = range.start;
      }
      if (range.end != null) {
        item.end = range.end;
      }
    }
    return item
  }

  /*  */

  /**
   * Cross-platform code generation for component v-model
   */
  function genComponentModel (
    el,
    value,
    modifiers
  ) {
    const { number, trim } = modifiers || {};

    const baseValueExpression = '$$v';
    let valueExpression = baseValueExpression;
    if (trim) {
      valueExpression =
        `(typeof ${baseValueExpression} === 'string'` +
        `? ${baseValueExpression}.trim()` +
        `: ${baseValueExpression})`;
    }
    if (number) {
      valueExpression = `_n(${valueExpression})`;
    }
    const assignment = genAssignmentCode(value, valueExpression);

    el.model = {
      value: `(${value})`,
      expression: JSON.stringify(value),
      callback: `function (${baseValueExpression}) {${assignment}}`
    };
  }

  /**
   * Cross-platform codegen helper for generating v-model value assignment code.
   */
  function genAssignmentCode (
    value,
    assignment
  ) {
    const res = parseModel(value);
    if (res.key === null) {
      return `${value}=${assignment}`
    } else {
      return `$set(${res.exp}, ${res.key}, ${assignment})`
    }
  }

  /**
   * Parse a v-model expression into a base path and a final key segment.
   * Handles both dot-path and possible square brackets.
   *
   * Possible cases:
   *
   * - test
   * - test[key]
   * - test[test1[key]]
   * - test["a"][key]
   * - xxx.test[a[a].test1[key]]
   * - test.xxx.a["asa"][test1[key]]
   *
   */

  let len, str, chr, index$1, expressionPos, expressionEndPos;



  function parseModel (val) {
    // Fix https://github.com/vuejs/vue/pull/7730
    // allow v-model="obj.val " (trailing whitespace)
    val = val.trim();
    len = val.length;

    if (val.indexOf('[') < 0 || val.lastIndexOf(']') < len - 1) {
      index$1 = val.lastIndexOf('.');
      if (index$1 > -1) {
        return {
          exp: val.slice(0, index$1),
          key: '"' + val.slice(index$1 + 1) + '"'
        }
      } else {
        return {
          exp: val,
          key: null
        }
      }
    }

    str = val;
    index$1 = expressionPos = expressionEndPos = 0;

    while (!eof()) {
      chr = next();
      /* istanbul ignore if */
      if (isStringStart(chr)) {
        parseString(chr);
      } else if (chr === 0x5B) {
        parseBracket(chr);
      }
    }

    return {
      exp: val.slice(0, expressionPos),
      key: val.slice(expressionPos + 1, expressionEndPos)
    }
  }

  function next () {
    return str.charCodeAt(++index$1)
  }

  function eof () {
    return index$1 >= len
  }

  function isStringStart (chr) {
    return chr === 0x22 || chr === 0x27
  }

  function parseBracket (chr) {
    let inBracket = 1;
    expressionPos = index$1;
    while (!eof()) {
      chr = next();
      if (isStringStart(chr)) {
        parseString(chr);
        continue
      }
      if (chr === 0x5B) inBracket++;
      if (chr === 0x5D) inBracket--;
      if (inBracket === 0) {
        expressionEndPos = index$1;
        break
      }
    }
  }

  function parseString (chr) {
    const stringQuote = chr;
    while (!eof()) {
      chr = next();
      if (chr === stringQuote) {
        break
      }
    }
  }

  /*  */

  let warn$1;

  // in some cases, the event used has to be determined at runtime
  // so we used some reserved tokens during compile.
  const RANGE_TOKEN = '__r';
  const CHECKBOX_RADIO_TOKEN = '__c';

  function model (
    el,
    dir,
    _warn
  ) {
    warn$1 = _warn;
    const value = dir.value;
    const modifiers = dir.modifiers;
    const tag = el.tag;
    const type = el.attrsMap.type;

    {
      // inputs with type="file" are read only and setting the input's
      // value will throw an error.
      if (tag === 'input' && type === 'file') {
        warn$1(
          `<${el.tag} v-model="${value}" type="file">:\n` +
          `File inputs are read only. Use a v-on:change listener instead.`,
          el.rawAttrsMap['v-model']
        );
      }
    }

    if (el.component) {
      genComponentModel(el, value, modifiers);
      // component v-model doesn't need extra runtime
      return false
    } else if (tag === 'select') {
      genSelect(el, value, modifiers);
    } else if (tag === 'input' && type === 'checkbox') {
      genCheckboxModel(el, value, modifiers);
    } else if (tag === 'input' && type === 'radio') {
      genRadioModel(el, value, modifiers);
    } else if (tag === 'input' || tag === 'textarea') {
      genDefaultModel(el, value, modifiers);
    } else if (!config.isReservedTag(tag)) {
      genComponentModel(el, value, modifiers);
      // component v-model doesn't need extra runtime
      return false
    } else {
      warn$1(
        `<${el.tag} v-model="${value}">: ` +
        `v-model is not supported on this element type. ` +
        'If you are working with contenteditable, it\'s recommended to ' +
        'wrap a library dedicated for that purpose inside a custom component.',
        el.rawAttrsMap['v-model']
      );
    }

    // ensure runtime directive metadata
    return true
  }

  function genCheckboxModel (
    el,
    value,
    modifiers
  ) {
    const number = modifiers && modifiers.number;
    const valueBinding = getBindingAttr(el, 'value') || 'null';
    const trueValueBinding = getBindingAttr(el, 'true-value') || 'true';
    const falseValueBinding = getBindingAttr(el, 'false-value') || 'false';
    addProp(el, 'checked',
      `Array.isArray(${value})` +
      `?_i(${value},${valueBinding})>-1` + (
        trueValueBinding === 'true'
          ? `:(${value})`
          : `:_q(${value},${trueValueBinding})`
      )
    );
    addHandler(el, 'change',
      `var $$a=${value},` +
          '$$el=$event.target,' +
          `$$c=$$el.checked?(${trueValueBinding}):(${falseValueBinding});` +
      'if(Array.isArray($$a)){' +
        `var $$v=${number ? '_n(' + valueBinding + ')' : valueBinding},` +
            '$$i=_i($$a,$$v);' +
        `if($$el.checked){$$i<0&&(${genAssignmentCode(value, '$$a.concat([$$v])')})}` +
        `else{$$i>-1&&(${genAssignmentCode(value, '$$a.slice(0,$$i).concat($$a.slice($$i+1))')})}` +
      `}else{${genAssignmentCode(value, '$$c')}}`,
      null, true
    );
  }

  function genRadioModel (
    el,
    value,
    modifiers
  ) {
    const number = modifiers && modifiers.number;
    let valueBinding = getBindingAttr(el, 'value') || 'null';
    valueBinding = number ? `_n(${valueBinding})` : valueBinding;
    addProp(el, 'checked', `_q(${value},${valueBinding})`);
    addHandler(el, 'change', genAssignmentCode(value, valueBinding), null, true);
  }

  function genSelect (
    el,
    value,
    modifiers
  ) {
    const number = modifiers && modifiers.number;
    const selectedVal = `Array.prototype.filter` +
      `.call($event.target.options,function(o){return o.selected})` +
      `.map(function(o){var val = "_value" in o ? o._value : o.value;` +
      `return ${number ? '_n(val)' : 'val'}})`;

    const assignment = '$event.target.multiple ? $$selectedVal : $$selectedVal[0]';
    let code = `var $$selectedVal = ${selectedVal};`;
    code = `${code} ${genAssignmentCode(value, assignment)}`;
    addHandler(el, 'change', code, null, true);
  }

  function genDefaultModel (
    el,
    value,
    modifiers
  ) {
    const type = el.attrsMap.type;

    // warn if v-bind:value conflicts with v-model
    // except for inputs with v-bind:type
    {
      const value = el.attrsMap['v-bind:value'] || el.attrsMap[':value'];
      const typeBinding = el.attrsMap['v-bind:type'] || el.attrsMap[':type'];
      if (value && !typeBinding) {
        const binding = el.attrsMap['v-bind:value'] ? 'v-bind:value' : ':value';
        warn$1(
          `${binding}="${value}" conflicts with v-model on the same element ` +
          'because the latter already expands to a value binding internally',
          el.rawAttrsMap[binding]
        );
      }
    }

    const { lazy, number, trim } = modifiers || {};
    const needCompositionGuard = !lazy && type !== 'range';
    const event = lazy
      ? 'change'
      : type === 'range'
        ? RANGE_TOKEN
        : 'input';

    let valueExpression = '$event.target.value';
    if (trim) {
      valueExpression = `$event.target.value.trim()`;
    }
    if (number) {
      valueExpression = `_n(${valueExpression})`;
    }

    let code = genAssignmentCode(value, valueExpression);
    if (needCompositionGuard) {
      code = `if($event.target.composing)return;${code}`;
    }

    addProp(el, 'value', `(${value})`);
    addHandler(el, event, code, null, true);
    if (trim || number) {
      addHandler(el, 'blur', '$forceUpdate()');
    }
  }

  /*  */

  // normalize v-model event tokens that can only be determined at runtime.
  // it's important to place the event as the first in the array because
  // the whole point is ensuring the v-model callback gets called before
  // user-attached handlers.
  function normalizeEvents (on) {
    /* istanbul ignore if */
    if (isDef(on[RANGE_TOKEN])) {
      // IE input[type=range] only supports `change` event
      const event = isIE ? 'change' : 'input';
      on[event] = [].concat(on[RANGE_TOKEN], on[event] || []);
      delete on[RANGE_TOKEN];
    }
    // This was originally intended to fix #4521 but no longer necessary
    // after 2.5. Keeping it for backwards compat with generated code from < 2.4
    /* istanbul ignore if */
    if (isDef(on[CHECKBOX_RADIO_TOKEN])) {
      on.change = [].concat(on[CHECKBOX_RADIO_TOKEN], on.change || []);
      delete on[CHECKBOX_RADIO_TOKEN];
    }
  }

  let target$1;

  function createOnceHandler$1 (event, handler, capture) {
    const _target = target$1; // save current target element in closure
    return function onceHandler () {
      const res = handler.apply(null, arguments);
      if (res !== null) {
        remove$2(event, onceHandler, capture, _target);
      }
    }
  }

  // #9446: Firefox <= 53 (in particular, ESR 52) has incorrect Event.timeStamp
  // implementation and does not fire microtasks in between event propagation, so
  // safe to exclude.
  const useMicrotaskFix = isUsingMicroTask && !(isFF && Number(isFF[1]) <= 53);

  function add$1 (
    name,
    handler,
    capture,
    passive
  ) {
    // async edge case #6566: inner click event triggers patch, event handler
    // attached to outer element during patch, and triggered again. This
    // happens because browsers fire microtask ticks between event propagation.
    // the solution is simple: we save the timestamp when a handler is attached,
    // and the handler would only fire if the event passed to it was fired
    // AFTER it was attached.
    if (useMicrotaskFix) {
      const attachedTimestamp = currentFlushTimestamp;
      const original = handler;
      handler = original._wrapper = function (e) {
        if (
          // no bubbling, should always fire.
          // this is just a safety net in case event.timeStamp is unreliable in
          // certain weird environments...
          e.target === e.currentTarget ||
          // event is fired after handler attachment
          e.timeStamp >= attachedTimestamp ||
          // bail for environments that have buggy event.timeStamp implementations
          // #9462 iOS 9 bug: event.timeStamp is 0 after history.pushState
          // #9681 QtWebEngine event.timeStamp is negative value
          e.timeStamp <= 0 ||
          // #9448 bail if event is fired in another document in a multi-page
          // electron/nw.js app, since event.timeStamp will be using a different
          // starting reference
          e.target.ownerDocument !== document
        ) {
          return original.apply(this, arguments)
        }
      };
    }
    target$1.addEventListener(
      name,
      handler,
      supportsPassive
        ? { capture, passive }
        : capture
    );
  }

  function remove$2 (
    name,
    handler,
    capture,
    _target
  ) {
    (_target || target$1).removeEventListener(
      name,
      handler._wrapper || handler,
      capture
    );
  }

  function updateDOMListeners (oldVnode, vnode) {
    if (isUndef(oldVnode.data.on) && isUndef(vnode.data.on)) {
      return
    }
    const on = vnode.data.on || {};
    const oldOn = oldVnode.data.on || {};
    target$1 = vnode.elm;
    normalizeEvents(on);
    updateListeners(on, oldOn, add$1, remove$2, createOnceHandler$1, vnode.context);
    target$1 = undefined;
  }

  var events = {
    create: updateDOMListeners,
    update: updateDOMListeners
  };

  /*  */

  let svgContainer;

  function updateDOMProps (oldVnode, vnode) {
    if (isUndef(oldVnode.data.domProps) && isUndef(vnode.data.domProps)) {
      return
    }
    let key, cur;
    const elm = vnode.elm;
    const oldProps = oldVnode.data.domProps || {};
    let props = vnode.data.domProps || {};
    // clone observed objects, as the user probably wants to mutate it
    if (isDef(props.__ob__)) {
      props = vnode.data.domProps = extend({}, props);
    }

    for (key in oldProps) {
      if (!(key in props)) {
        elm[key] = '';
      }
    }

    for (key in props) {
      cur = props[key];
      // ignore children if the node has textContent or innerHTML,
      // as these will throw away existing DOM nodes and cause removal errors
      // on subsequent patches (#3360)
      if (key === 'textContent' || key === 'innerHTML') {
        if (vnode.children) vnode.children.length = 0;
        if (cur === oldProps[key]) continue
        // #6601 work around Chrome version <= 55 bug where single textNode
        // replaced by innerHTML/textContent retains its parentNode property
        if (elm.childNodes.length === 1) {
          elm.removeChild(elm.childNodes[0]);
        }
      }

      if (key === 'value' && elm.tagName !== 'PROGRESS') {
        // store value as _value as well since
        // non-string values will be stringified
        elm._value = cur;
        // avoid resetting cursor position when value is the same
        const strCur = isUndef(cur) ? '' : String(cur);
        if (shouldUpdateValue(elm, strCur)) {
          elm.value = strCur;
        }
      } else if (key === 'innerHTML' && isSVG(elm.tagName) && isUndef(elm.innerHTML)) {
        // IE doesn't support innerHTML for SVG elements
        svgContainer = svgContainer || document.createElement('div');
        svgContainer.innerHTML = `<svg>${cur}</svg>`;
        const svg = svgContainer.firstChild;
        while (elm.firstChild) {
          elm.removeChild(elm.firstChild);
        }
        while (svg.firstChild) {
          elm.appendChild(svg.firstChild);
        }
      } else if (
        // skip the update if old and new VDOM state is the same.
        // `value` is handled separately because the DOM value may be temporarily
        // out of sync with VDOM state due to focus, composition and modifiers.
        // This  #4521 by skipping the unnecessary `checked` update.
        cur !== oldProps[key]
      ) {
        // some property updates can throw
        // e.g. `value` on <progress> w/ non-finite value
        try {
          elm[key] = cur;
        } catch (e) {}
      }
    }
  }

  // check platforms/web/util/attrs.js acceptValue


  function shouldUpdateValue (elm, checkVal) {
    return (!elm.composing && (
      elm.tagName === 'OPTION' ||
      isNotInFocusAndDirty(elm, checkVal) ||
      isDirtyWithModifiers(elm, checkVal)
    ))
  }

  function isNotInFocusAndDirty (elm, checkVal) {
    // return true when textbox (.number and .trim) loses focus and its value is
    // not equal to the updated value
    let notInFocus = true;
    // #6157
    // work around IE bug when accessing document.activeElement in an iframe
    try { notInFocus = document.activeElement !== elm; } catch (e) {}
    return notInFocus && elm.value !== checkVal
  }

  function isDirtyWithModifiers (elm, newVal) {
    const value = elm.value;
    const modifiers = elm._vModifiers; // injected by v-model runtime
    if (isDef(modifiers)) {
      if (modifiers.number) {
        return toNumber(value) !== toNumber(newVal)
      }
      if (modifiers.trim) {
        return value.trim() !== newVal.trim()
      }
    }
    return value !== newVal
  }

  var domProps = {
    create: updateDOMProps,
    update: updateDOMProps
  };

  /*  */

  const parseStyleText = cached(function (cssText) {
    const res = {};
    const listDelimiter = /;(?![^(]*\))/g;
    const propertyDelimiter = /:(.+)/;
    cssText.split(listDelimiter).forEach(function (item) {
      if (item) {
        const tmp = item.split(propertyDelimiter);
        tmp.length > 1 && (res[tmp[0].trim()] = tmp[1].trim());
      }
    });
    return res
  });

  // merge static and dynamic style data on the same vnode
  function normalizeStyleData (data) {
    const style = normalizeStyleBinding(data.style);
    // static style is pre-processed into an object during compilation
    // and is always a fresh object, so it's safe to merge into it
    return data.staticStyle
      ? extend(data.staticStyle, style)
      : style
  }

  // normalize possible array / string values into Object
  function normalizeStyleBinding (bindingStyle) {
    if (Array.isArray(bindingStyle)) {
      return toObject(bindingStyle)
    }
    if (typeof bindingStyle === 'string') {
      return parseStyleText(bindingStyle)
    }
    return bindingStyle
  }

  /**
   * parent component style should be after child's
   * so that parent component's style could override it
   */
  function getStyle (vnode, checkChild) {
    const res = {};
    let styleData;

    if (checkChild) {
      let childNode = vnode;
      while (childNode.componentInstance) {
        childNode = childNode.componentInstance._vnode;
        if (
          childNode && childNode.data &&
          (styleData = normalizeStyleData(childNode.data))
        ) {
          extend(res, styleData);
        }
      }
    }

    if ((styleData = normalizeStyleData(vnode.data))) {
      extend(res, styleData);
    }

    let parentNode = vnode;
    while ((parentNode = parentNode.parent)) {
      if (parentNode.data && (styleData = normalizeStyleData(parentNode.data))) {
        extend(res, styleData);
      }
    }
    return res
  }

  /*  */

  const cssVarRE = /^--/;
  const importantRE = /\s*!important$/;
  const setProp = (el, name, val) => {
    /* istanbul ignore if */
    if (cssVarRE.test(name)) {
      el.style.setProperty(name, val);
    } else if (importantRE.test(val)) {
      el.style.setProperty(hyphenate(name), val.replace(importantRE, ''), 'important');
    } else {
      const normalizedName = normalize(name);
      if (Array.isArray(val)) {
        // Support values array created by autoprefixer, e.g.
        // {display: ["-webkit-box", "-ms-flexbox", "flex"]}
        // Set them one by one, and the browser will only set those it can recognize
        for (let i = 0, len = val.length; i < len; i++) {
          el.style[normalizedName] = val[i];
        }
      } else {
        el.style[normalizedName] = val;
      }
    }
  };

  const vendorNames = ['Webkit', 'Moz', 'ms'];

  let emptyStyle;
  const normalize = cached(function (prop) {
    emptyStyle = emptyStyle || document.createElement('div').style;
    prop = camelize(prop);
    if (prop !== 'filter' && (prop in emptyStyle)) {
      return prop
    }
    const capName = prop.charAt(0).toUpperCase() + prop.slice(1);
    for (let i = 0; i < vendorNames.length; i++) {
      const name = vendorNames[i] + capName;
      if (name in emptyStyle) {
        return name
      }
    }
  });

  function updateStyle (oldVnode, vnode) {
    const data = vnode.data;
    const oldData = oldVnode.data;

    if (isUndef(data.staticStyle) && isUndef(data.style) &&
      isUndef(oldData.staticStyle) && isUndef(oldData.style)
    ) {
      return
    }

    let cur, name;
    const el = vnode.elm;
    const oldStaticStyle = oldData.staticStyle;
    const oldStyleBinding = oldData.normalizedStyle || oldData.style || {};

    // if static style exists, stylebinding already merged into it when doing normalizeStyleData
    const oldStyle = oldStaticStyle || oldStyleBinding;

    const style = normalizeStyleBinding(vnode.data.style) || {};

    // store normalized style under a different key for next diff
    // make sure to clone it if it's reactive, since the user likely wants
    // to mutate it.
    vnode.data.normalizedStyle = isDef(style.__ob__)
      ? extend({}, style)
      : style;

    const newStyle = getStyle(vnode, true);

    for (name in oldStyle) {
      if (isUndef(newStyle[name])) {
        setProp(el, name, '');
      }
    }
    for (name in newStyle) {
      cur = newStyle[name];
      if (cur !== oldStyle[name]) {
        // ie9 setting to null has no effect, must use empty string
        setProp(el, name, cur == null ? '' : cur);
      }
    }
  }

  var style = {
    create: updateStyle,
    update: updateStyle
  };

  /*  */

  const whitespaceRE = /\s+/;

  /**
   * Add class with compatibility for SVG since classList is not supported on
   * SVG elements in IE
   */
  function addClass (el, cls) {
    /* istanbul ignore if */
    if (!cls || !(cls = cls.trim())) {
      return
    }

    /* istanbul ignore else */
    if (el.classList) {
      if (cls.indexOf(' ') > -1) {
        cls.split(whitespaceRE).forEach(c => el.classList.add(c));
      } else {
        el.classList.add(cls);
      }
    } else {
      const cur = ` ${el.getAttribute('class') || ''} `;
      if (cur.indexOf(' ' + cls + ' ') < 0) {
        el.setAttribute('class', (cur + cls).trim());
      }
    }
  }

  /**
   * Remove class with compatibility for SVG since classList is not supported on
   * SVG elements in IE
   */
  function removeClass (el, cls) {
    /* istanbul ignore if */
    if (!cls || !(cls = cls.trim())) {
      return
    }

    /* istanbul ignore else */
    if (el.classList) {
      if (cls.indexOf(' ') > -1) {
        cls.split(whitespaceRE).forEach(c => el.classList.remove(c));
      } else {
        el.classList.remove(cls);
      }
      if (!el.classList.length) {
        el.removeAttribute('class');
      }
    } else {
      let cur = ` ${el.getAttribute('class') || ''} `;
      const tar = ' ' + cls + ' ';
      while (cur.indexOf(tar) >= 0) {
        cur = cur.replace(tar, ' ');
      }
      cur = cur.trim();
      if (cur) {
        el.setAttribute('class', cur);
      } else {
        el.removeAttribute('class');
      }
    }
  }

  /*  */

  function resolveTransition (def$$1) {
    if (!def$$1) {
      return
    }
    /* istanbul ignore else */
    if (typeof def$$1 === 'object') {
      const res = {};
      if (def$$1.css !== false) {
        extend(res, autoCssTransition(def$$1.name || 'v'));
      }
      extend(res, def$$1);
      return res
    } else if (typeof def$$1 === 'string') {
      return autoCssTransition(def$$1)
    }
  }

  const autoCssTransition = cached(name => {
    return {
      enterClass: `${name}-enter`,
      enterToClass: `${name}-enter-to`,
      enterActiveClass: `${name}-enter-active`,
      leaveClass: `${name}-leave`,
      leaveToClass: `${name}-leave-to`,
      leaveActiveClass: `${name}-leave-active`
    }
  });

  const hasTransition = inBrowser && !isIE9;
  const TRANSITION = 'transition';
  const ANIMATION = 'animation';

  // Transition property/event sniffing
  let transitionProp = 'transition';
  let transitionEndEvent = 'transitionend';
  let animationProp = 'animation';
  let animationEndEvent = 'animationend';
  if (hasTransition) {
    /* istanbul ignore if */
    if (window.ontransitionend === undefined &&
      window.onwebkittransitionend !== undefined
    ) {
      transitionProp = 'WebkitTransition';
      transitionEndEvent = 'webkitTransitionEnd';
    }
    if (window.onanimationend === undefined &&
      window.onwebkitanimationend !== undefined
    ) {
      animationProp = 'WebkitAnimation';
      animationEndEvent = 'webkitAnimationEnd';
    }
  }

  // binding to window is necessary to make hot reload work in IE in strict mode
  const raf = inBrowser
    ? window.requestAnimationFrame
      ? window.requestAnimationFrame.bind(window)
      : setTimeout
    : /* istanbul ignore next */ fn => fn();

  function nextFrame (fn) {
    raf(() => {
      raf(fn);
    });
  }

  function addTransitionClass (el, cls) {
    const transitionClasses = el._transitionClasses || (el._transitionClasses = []);
    if (transitionClasses.indexOf(cls) < 0) {
      transitionClasses.push(cls);
      addClass(el, cls);
    }
  }

  function removeTransitionClass (el, cls) {
    if (el._transitionClasses) {
      remove(el._transitionClasses, cls);
    }
    removeClass(el, cls);
  }

  function whenTransitionEnds (
    el,
    expectedType,
    cb
  ) {
    const { type, timeout, propCount } = getTransitionInfo(el, expectedType);
    if (!type) return cb()
    const event = type === TRANSITION ? transitionEndEvent : animationEndEvent;
    let ended = 0;
    const end = () => {
      el.removeEventListener(event, onEnd);
      cb();
    };
    const onEnd = e => {
      if (e.target === el) {
        if (++ended >= propCount) {
          end();
        }
      }
    };
    setTimeout(() => {
      if (ended < propCount) {
        end();
      }
    }, timeout + 1);
    el.addEventListener(event, onEnd);
  }

  const transformRE = /\b(transform|all)(,|$)/;

  function getTransitionInfo (el, expectedType) {
    const styles = window.getComputedStyle(el);
    // JSDOM may return undefined for transition properties
    const transitionDelays = (styles[transitionProp + 'Delay'] || '').split(', ');
    const transitionDurations = (styles[transitionProp + 'Duration'] || '').split(', ');
    const transitionTimeout = getTimeout(transitionDelays, transitionDurations);
    const animationDelays = (styles[animationProp + 'Delay'] || '').split(', ');
    const animationDurations = (styles[animationProp + 'Duration'] || '').split(', ');
    const animationTimeout = getTimeout(animationDelays, animationDurations);

    let type;
    let timeout = 0;
    let propCount = 0;
    /* istanbul ignore if */
    if (expectedType === TRANSITION) {
      if (transitionTimeout > 0) {
        type = TRANSITION;
        timeout = transitionTimeout;
        propCount = transitionDurations.length;
      }
    } else if (expectedType === ANIMATION) {
      if (animationTimeout > 0) {
        type = ANIMATION;
        timeout = animationTimeout;
        propCount = animationDurations.length;
      }
    } else {
      timeout = Math.max(transitionTimeout, animationTimeout);
      type = timeout > 0
        ? transitionTimeout > animationTimeout
          ? TRANSITION
          : ANIMATION
        : null;
      propCount = type
        ? type === TRANSITION
          ? transitionDurations.length
          : animationDurations.length
        : 0;
    }
    const hasTransform =
      type === TRANSITION &&
      transformRE.test(styles[transitionProp + 'Property']);
    return {
      type,
      timeout,
      propCount,
      hasTransform
    }
  }

  function getTimeout (delays, durations) {
    /* istanbul ignore next */
    while (delays.length < durations.length) {
      delays = delays.concat(delays);
    }

    return Math.max.apply(null, durations.map((d, i) => {
      return toMs(d) + toMs(delays[i])
    }))
  }

  // Old versions of Chromium (below 61.0.3163.100) formats floating pointer numbers
  // in a locale-dependent way, using a comma instead of a dot.
  // If comma is not replaced with a dot, the input will be rounded down (i.e. acting
  // as a floor function) causing unexpected behaviors
  function toMs (s) {
    return Number(s.slice(0, -1).replace(',', '.')) * 1000
  }

  /*  */

  function enter (vnode, toggleDisplay) {
    const el = vnode.elm;

    // call leave callback now
    if (isDef(el._leaveCb)) {
      el._leaveCb.cancelled = true;
      el._leaveCb();
    }

    const data = resolveTransition(vnode.data.transition);
    if (isUndef(data)) {
      return
    }

    /* istanbul ignore if */
    if (isDef(el._enterCb) || el.nodeType !== 1) {
      return
    }

    const {
      css,
      type,
      enterClass,
      enterToClass,
      enterActiveClass,
      appearClass,
      appearToClass,
      appearActiveClass,
      beforeEnter,
      enter,
      afterEnter,
      enterCancelled,
      beforeAppear,
      appear,
      afterAppear,
      appearCancelled,
      duration
    } = data;

    // activeInstance will always be the <transition> component managing this
    // transition. One edge case to check is when the <transition> is placed
    // as the root node of a child component. In that case we need to check
    // <transition>'s parent for appear check.
    let context = activeInstance;
    let transitionNode = activeInstance.$vnode;
    while (transitionNode && transitionNode.parent) {
      context = transitionNode.context;
      transitionNode = transitionNode.parent;
    }

    const isAppear = !context._isMounted || !vnode.isRootInsert;

    if (isAppear && !appear && appear !== '') {
      return
    }

    const startClass = isAppear && appearClass
      ? appearClass
      : enterClass;
    const activeClass = isAppear && appearActiveClass
      ? appearActiveClass
      : enterActiveClass;
    const toClass = isAppear && appearToClass
      ? appearToClass
      : enterToClass;

    const beforeEnterHook = isAppear
      ? (beforeAppear || beforeEnter)
      : beforeEnter;
    const enterHook = isAppear
      ? (typeof appear === 'function' ? appear : enter)
      : enter;
    const afterEnterHook = isAppear
      ? (afterAppear || afterEnter)
      : afterEnter;
    const enterCancelledHook = isAppear
      ? (appearCancelled || enterCancelled)
      : enterCancelled;

    const explicitEnterDuration = toNumber(
      isObject(duration)
        ? duration.enter
        : duration
    );

    if (explicitEnterDuration != null) {
      checkDuration(explicitEnterDuration, 'enter', vnode);
    }

    const expectsCSS = css !== false && !isIE9;
    const userWantsControl = getHookArgumentsLength(enterHook);

    const cb = el._enterCb = once(() => {
      if (expectsCSS) {
        removeTransitionClass(el, toClass);
        removeTransitionClass(el, activeClass);
      }
      if (cb.cancelled) {
        if (expectsCSS) {
          removeTransitionClass(el, startClass);
        }
        enterCancelledHook && enterCancelledHook(el);
      } else {
        afterEnterHook && afterEnterHook(el);
      }
      el._enterCb = null;
    });

    if (!vnode.data.show) {
      // remove pending leave element on enter by injecting an insert hook
      mergeVNodeHook(vnode, 'insert', () => {
        const parent = el.parentNode;
        const pendingNode = parent && parent._pending && parent._pending[vnode.key];
        if (pendingNode &&
          pendingNode.tag === vnode.tag &&
          pendingNode.elm._leaveCb
        ) {
          pendingNode.elm._leaveCb();
        }
        enterHook && enterHook(el, cb);
      });
    }

    // start enter transition
    beforeEnterHook && beforeEnterHook(el);
    if (expectsCSS) {
      addTransitionClass(el, startClass);
      addTransitionClass(el, activeClass);
      nextFrame(() => {
        removeTransitionClass(el, startClass);
        if (!cb.cancelled) {
          addTransitionClass(el, toClass);
          if (!userWantsControl) {
            if (isValidDuration(explicitEnterDuration)) {
              setTimeout(cb, explicitEnterDuration);
            } else {
              whenTransitionEnds(el, type, cb);
            }
          }
        }
      });
    }

    if (vnode.data.show) {
      toggleDisplay && toggleDisplay();
      enterHook && enterHook(el, cb);
    }

    if (!expectsCSS && !userWantsControl) {
      cb();
    }
  }

  function leave (vnode, rm) {
    const el = vnode.elm;

    // call enter callback now
    if (isDef(el._enterCb)) {
      el._enterCb.cancelled = true;
      el._enterCb();
    }

    const data = resolveTransition(vnode.data.transition);
    if (isUndef(data) || el.nodeType !== 1) {
      return rm()
    }

    /* istanbul ignore if */
    if (isDef(el._leaveCb)) {
      return
    }

    const {
      css,
      type,
      leaveClass,
      leaveToClass,
      leaveActiveClass,
      beforeLeave,
      leave,
      afterLeave,
      leaveCancelled,
      delayLeave,
      duration
    } = data;

    const expectsCSS = css !== false && !isIE9;
    const userWantsControl = getHookArgumentsLength(leave);

    const explicitLeaveDuration = toNumber(
      isObject(duration)
        ? duration.leave
        : duration
    );

    if (isDef(explicitLeaveDuration)) {
      checkDuration(explicitLeaveDuration, 'leave', vnode);
    }

    const cb = el._leaveCb = once(() => {
      if (el.parentNode && el.parentNode._pending) {
        el.parentNode._pending[vnode.key] = null;
      }
      if (expectsCSS) {
        removeTransitionClass(el, leaveToClass);
        removeTransitionClass(el, leaveActiveClass);
      }
      if (cb.cancelled) {
        if (expectsCSS) {
          removeTransitionClass(el, leaveClass);
        }
        leaveCancelled && leaveCancelled(el);
      } else {
        rm();
        afterLeave && afterLeave(el);
      }
      el._leaveCb = null;
    });

    if (delayLeave) {
      delayLeave(performLeave);
    } else {
      performLeave();
    }

    function performLeave () {
      // the delayed leave may have already been cancelled
      if (cb.cancelled) {
        return
      }
      // record leaving element
      if (!vnode.data.show && el.parentNode) {
        (el.parentNode._pending || (el.parentNode._pending = {}))[(vnode.key)] = vnode;
      }
      beforeLeave && beforeLeave(el);
      if (expectsCSS) {
        addTransitionClass(el, leaveClass);
        addTransitionClass(el, leaveActiveClass);
        nextFrame(() => {
          removeTransitionClass(el, leaveClass);
          if (!cb.cancelled) {
            addTransitionClass(el, leaveToClass);
            if (!userWantsControl) {
              if (isValidDuration(explicitLeaveDuration)) {
                setTimeout(cb, explicitLeaveDuration);
              } else {
                whenTransitionEnds(el, type, cb);
              }
            }
          }
        });
      }
      leave && leave(el, cb);
      if (!expectsCSS && !userWantsControl) {
        cb();
      }
    }
  }

  // only used in dev mode
  function checkDuration (val, name, vnode) {
    if (typeof val !== 'number') {
      warn(
        `<transition> explicit ${name} duration is not a valid number - ` +
        `got ${JSON.stringify(val)}.`,
        vnode.context
      );
    } else if (isNaN(val)) {
      warn(
        `<transition> explicit ${name} duration is NaN - ` +
        'the duration expression might be incorrect.',
        vnode.context
      );
    }
  }

  function isValidDuration (val) {
    return typeof val === 'number' && !isNaN(val)
  }

  /**
   * Normalize a transition hook's argument length. The hook may be:
   * - a merged hook (invoker) with the original in .fns
   * - a wrapped component method (check ._length)
   * - a plain function (.length)
   */
  function getHookArgumentsLength (fn) {
    if (isUndef(fn)) {
      return false
    }
    const invokerFns = fn.fns;
    if (isDef(invokerFns)) {
      // invoker
      return getHookArgumentsLength(
        Array.isArray(invokerFns)
          ? invokerFns[0]
          : invokerFns
      )
    } else {
      return (fn._length || fn.length) > 1
    }
  }

  function _enter (_, vnode) {
    if (vnode.data.show !== true) {
      enter(vnode);
    }
  }

  var transition = inBrowser ? {
    create: _enter,
    activate: _enter,
    remove (vnode, rm) {
      /* istanbul ignore else */
      if (vnode.data.show !== true) {
        leave(vnode, rm);
      } else {
        rm();
      }
    }
  } : {};

  var platformModules = [
    attrs,
    klass,
    events,
    domProps,
    style,
    transition
  ];

  /*  */

  // the directive module should be applied last, after all
  // built-in modules have been applied.
  const modules = platformModules.concat(baseModules);

  const patch = createPatchFunction({ nodeOps, modules });

  /**
   * Not type checking this file because flow doesn't like attaching
   * properties to Elements.
   */

  /* istanbul ignore if */
  if (isIE9) {
    // http://www.matts411.com/post/internet-explorer-9-oninput/
    document.addEventListener('selectionchange', () => {
      const el = document.activeElement;
      if (el && el.vmodel) {
        trigger(el, 'input');
      }
    });
  }

  const directive = {
    inserted (el, binding, vnode, oldVnode) {
      if (vnode.tag === 'select') {
        // #6903
        if (oldVnode.elm && !oldVnode.elm._vOptions) {
          mergeVNodeHook(vnode, 'postpatch', () => {
            directive.componentUpdated(el, binding, vnode);
          });
        } else {
          setSelected(el, binding, vnode.context);
        }
        el._vOptions = [].map.call(el.options, getValue);
      } else if (vnode.tag === 'textarea' || isTextInputType(el.type)) {
        el._vModifiers = binding.modifiers;
        if (!binding.modifiers.lazy) {
          el.addEventListener('compositionstart', onCompositionStart);
          el.addEventListener('compositionend', onCompositionEnd);
          // Safari < 10.2 & UIWebView doesn't fire compositionend when
          // switching focus before confirming composition choice
          // this also fixes the issue where some browsers e.g. iOS Chrome
          // fires "change" instead of "input" on autocomplete.
          el.addEventListener('change', onCompositionEnd);
          /* istanbul ignore if */
          if (isIE9) {
            el.vmodel = true;
          }
        }
      }
    },

    componentUpdated (el, binding, vnode) {
      if (vnode.tag === 'select') {
        setSelected(el, binding, vnode.context);
        // in case the options rendered by v-for have changed,
        // it's possible that the value is out-of-sync with the rendered options.
        // detect such cases and filter out values that no longer has a matching
        // option in the DOM.
        const prevOptions = el._vOptions;
        const curOptions = el._vOptions = [].map.call(el.options, getValue);
        if (curOptions.some((o, i) => !looseEqual(o, prevOptions[i]))) {
          // trigger change event if
          // no matching option found for at least one value
          const needReset = el.multiple
            ? binding.value.some(v => hasNoMatchingOption(v, curOptions))
            : binding.value !== binding.oldValue && hasNoMatchingOption(binding.value, curOptions);
          if (needReset) {
            trigger(el, 'change');
          }
        }
      }
    }
  };

  function setSelected (el, binding, vm) {
    actuallySetSelected(el, binding, vm);
    /* istanbul ignore if */
    if (isIE || isEdge) {
      setTimeout(() => {
        actuallySetSelected(el, binding, vm);
      }, 0);
    }
  }

  function actuallySetSelected (el, binding, vm) {
    const value = binding.value;
    const isMultiple = el.multiple;
    if (isMultiple && !Array.isArray(value)) {
      warn(
        `<select multiple v-model="${binding.expression}"> ` +
        `expects an Array value for its binding, but got ${
        Object.prototype.toString.call(value).slice(8, -1)
      }`,
        vm
      );
      return
    }
    let selected, option;
    for (let i = 0, l = el.options.length; i < l; i++) {
      option = el.options[i];
      if (isMultiple) {
        selected = looseIndexOf(value, getValue(option)) > -1;
        if (option.selected !== selected) {
          option.selected = selected;
        }
      } else {
        if (looseEqual(getValue(option), value)) {
          if (el.selectedIndex !== i) {
            el.selectedIndex = i;
          }
          return
        }
      }
    }
    if (!isMultiple) {
      el.selectedIndex = -1;
    }
  }

  function hasNoMatchingOption (value, options) {
    return options.every(o => !looseEqual(o, value))
  }

  function getValue (option) {
    return '_value' in option
      ? option._value
      : option.value
  }

  function onCompositionStart (e) {
    e.target.composing = true;
  }

  function onCompositionEnd (e) {
    // prevent triggering an input event for no reason
    if (!e.target.composing) return
    e.target.composing = false;
    trigger(e.target, 'input');
  }

  function trigger (el, type) {
    const e = document.createEvent('HTMLEvents');
    e.initEvent(type, true, true);
    el.dispatchEvent(e);
  }

  /*  */

  // recursively search for possible transition defined inside the component root
  function locateNode (vnode) {
    return vnode.componentInstance && (!vnode.data || !vnode.data.transition)
      ? locateNode(vnode.componentInstance._vnode)
      : vnode
  }

  var show = {
    bind (el, { value }, vnode) {
      vnode = locateNode(vnode);
      const transition$$1 = vnode.data && vnode.data.transition;
      const originalDisplay = el.__vOriginalDisplay =
        el.style.display === 'none' ? '' : el.style.display;
      if (value && transition$$1) {
        vnode.data.show = true;
        enter(vnode, () => {
          el.style.display = originalDisplay;
        });
      } else {
        el.style.display = value ? originalDisplay : 'none';
      }
    },

    update (el, { value, oldValue }, vnode) {
      /* istanbul ignore if */
      if (!value === !oldValue) return
      vnode = locateNode(vnode);
      const transition$$1 = vnode.data && vnode.data.transition;
      if (transition$$1) {
        vnode.data.show = true;
        if (value) {
          enter(vnode, () => {
            el.style.display = el.__vOriginalDisplay;
          });
        } else {
          leave(vnode, () => {
            el.style.display = 'none';
          });
        }
      } else {
        el.style.display = value ? el.__vOriginalDisplay : 'none';
      }
    },

    unbind (
      el,
      binding,
      vnode,
      oldVnode,
      isDestroy
    ) {
      if (!isDestroy) {
        el.style.display = el.__vOriginalDisplay;
      }
    }
  };

  var platformDirectives = {
    model: directive,
    show
  };

  /*  */

  const transitionProps = {
    name: String,
    appear: Boolean,
    css: Boolean,
    mode: String,
    type: String,
    enterClass: String,
    leaveClass: String,
    enterToClass: String,
    leaveToClass: String,
    enterActiveClass: String,
    leaveActiveClass: String,
    appearClass: String,
    appearActiveClass: String,
    appearToClass: String,
    duration: [Number, String, Object]
  };

  // in case the child is also an abstract component, e.g. <keep-alive>
  // we want to recursively retrieve the real component to be rendered
  function getRealChild (vnode) {
    const compOptions = vnode && vnode.componentOptions;
    if (compOptions && compOptions.Ctor.options.abstract) {
      return getRealChild(getFirstComponentChild(compOptions.children))
    } else {
      return vnode
    }
  }

  function extractTransitionData (comp) {
    const data = {};
    const options = comp.$options;
    // props
    for (const key in options.propsData) {
      data[key] = comp[key];
    }
    // events.
    // extract listeners and pass them directly to the transition methods
    const listeners = options._parentListeners;
    for (const key in listeners) {
      data[camelize(key)] = listeners[key];
    }
    return data
  }

  function placeholder (h, rawChild) {
    if (/\d-keep-alive$/.test(rawChild.tag)) {
      return h('keep-alive', {
        props: rawChild.componentOptions.propsData
      })
    }
  }

  function hasParentTransition (vnode) {
    while ((vnode = vnode.parent)) {
      if (vnode.data.transition) {
        return true
      }
    }
  }

  function isSameChild (child, oldChild) {
    return oldChild.key === child.key && oldChild.tag === child.tag
  }

  const isNotTextNode = (c) => c.tag || isAsyncPlaceholder(c);

  const isVShowDirective = d => d.name === 'show';

  var Transition = {
    name: 'transition',
    props: transitionProps,
    abstract: true,

    render (h) {
      let children = this.$slots.default;
      if (!children) {
        return
      }

      // filter out text nodes (possible whitespaces)
      children = children.filter(isNotTextNode);
      /* istanbul ignore if */
      if (!children.length) {
        return
      }

      // warn multiple elements
      if (children.length > 1) {
        warn(
          '<transition> can only be used on a single element. Use ' +
          '<transition-group> for lists.',
          this.$parent
        );
      }

      const mode = this.mode;

      // warn invalid mode
      if (mode && mode !== 'in-out' && mode !== 'out-in'
      ) {
        warn(
          'invalid <transition> mode: ' + mode,
          this.$parent
        );
      }

      const rawChild = children[0];

      // if this is a component root node and the component's
      // parent container node also has transition, skip.
      if (hasParentTransition(this.$vnode)) {
        return rawChild
      }

      // apply transition data to child
      // use getRealChild() to ignore abstract components e.g. keep-alive
      const child = getRealChild(rawChild);
      /* istanbul ignore if */
      if (!child) {
        return rawChild
      }

      if (this._leaving) {
        return placeholder(h, rawChild)
      }

      // ensure a key that is unique to the vnode type and to this transition
      // component instance. This key will be used to remove pending leaving nodes
      // during entering.
      const id = `__transition-${this._uid}-`;
      child.key = child.key == null
        ? child.isComment
          ? id + 'comment'
          : id + child.tag
        : isPrimitive(child.key)
          ? (String(child.key).indexOf(id) === 0 ? child.key : id + child.key)
          : child.key;

      const data = (child.data || (child.data = {})).transition = extractTransitionData(this);
      const oldRawChild = this._vnode;
      const oldChild = getRealChild(oldRawChild);

      // mark v-show
      // so that the transition module can hand over the control to the directive
      if (child.data.directives && child.data.directives.some(isVShowDirective)) {
        child.data.show = true;
      }

      if (
        oldChild &&
        oldChild.data &&
        !isSameChild(child, oldChild) &&
        !isAsyncPlaceholder(oldChild) &&
        // #6687 component root is a comment node
        !(oldChild.componentInstance && oldChild.componentInstance._vnode.isComment)
      ) {
        // replace old child transition data with fresh one
        // important for dynamic transitions!
        const oldData = oldChild.data.transition = extend({}, data);
        // handle transition mode
        if (mode === 'out-in') {
          // return placeholder node and queue update when leave finishes
          this._leaving = true;
          mergeVNodeHook(oldData, 'afterLeave', () => {
            this._leaving = false;
            this.$forceUpdate();
          });
          return placeholder(h, rawChild)
        } else if (mode === 'in-out') {
          if (isAsyncPlaceholder(child)) {
            return oldRawChild
          }
          let delayedLeave;
          const performLeave = () => { delayedLeave(); };
          mergeVNodeHook(data, 'afterEnter', performLeave);
          mergeVNodeHook(data, 'enterCancelled', performLeave);
          mergeVNodeHook(oldData, 'delayLeave', leave => { delayedLeave = leave; });
        }
      }

      return rawChild
    }
  };

  /*  */

  const props = extend({
    tag: String,
    moveClass: String
  }, transitionProps);

  delete props.mode;

  var TransitionGroup = {
    props,

    beforeMount () {
      const update = this._update;
      this._update = (vnode, hydrating) => {
        const restoreActiveInstance = setActiveInstance(this);
        // force removing pass
        this.__patch__(
          this._vnode,
          this.kept,
          false, // hydrating
          true // removeOnly (!important, avoids unnecessary moves)
        );
        this._vnode = this.kept;
        restoreActiveInstance();
        update.call(this, vnode, hydrating);
      };
    },

    render (h) {
      const tag = this.tag || this.$vnode.data.tag || 'span';
      const map = Object.create(null);
      const prevChildren = this.prevChildren = this.children;
      const rawChildren = this.$slots.default || [];
      const children = this.children = [];
      const transitionData = extractTransitionData(this);

      for (let i = 0; i < rawChildren.length; i++) {
        const c = rawChildren[i];
        if (c.tag) {
          if (c.key != null && String(c.key).indexOf('__vlist') !== 0) {
            children.push(c);
            map[c.key] = c
            ;(c.data || (c.data = {})).transition = transitionData;
          } else {
            const opts = c.componentOptions;
            const name = opts ? (opts.Ctor.options.name || opts.tag || '') : c.tag;
            warn(`<transition-group> children must be keyed: <${name}>`);
          }
        }
      }

      if (prevChildren) {
        const kept = [];
        const removed = [];
        for (let i = 0; i < prevChildren.length; i++) {
          const c = prevChildren[i];
          c.data.transition = transitionData;
          c.data.pos = c.elm.getBoundingClientRect();
          if (map[c.key]) {
            kept.push(c);
          } else {
            removed.push(c);
          }
        }
        this.kept = h(tag, null, kept);
        this.removed = removed;
      }

      return h(tag, null, children)
    },

    updated () {
      const children = this.prevChildren;
      const moveClass = this.moveClass || ((this.name || 'v') + '-move');
      if (!children.length || !this.hasMove(children[0].elm, moveClass)) {
        return
      }

      // we divide the work into three loops to avoid mixing DOM reads and writes
      // in each iteration - which helps prevent layout thrashing.
      children.forEach(callPendingCbs);
      children.forEach(recordPosition);
      children.forEach(applyTranslation);

      // force reflow to put everything in position
      // assign to this to avoid being removed in tree-shaking
      // $flow-disable-line
      this._reflow = document.body.offsetHeight;

      children.forEach((c) => {
        if (c.data.moved) {
          const el = c.elm;
          const s = el.style;
          addTransitionClass(el, moveClass);
          s.transform = s.WebkitTransform = s.transitionDuration = '';
          el.addEventListener(transitionEndEvent, el._moveCb = function cb (e) {
            if (e && e.target !== el) {
              return
            }
            if (!e || /transform$/.test(e.propertyName)) {
              el.removeEventListener(transitionEndEvent, cb);
              el._moveCb = null;
              removeTransitionClass(el, moveClass);
            }
          });
        }
      });
    },

    methods: {
      hasMove (el, moveClass) {
        /* istanbul ignore if */
        if (!hasTransition) {
          return false
        }
        /* istanbul ignore if */
        if (this._hasMove) {
          return this._hasMove
        }
        // Detect whether an element with the move class applied has
        // CSS transitions. Since the element may be inside an entering
        // transition at this very moment, we make a clone of it and remove
        // all other transition classes applied to ensure only the move class
        // is applied.
        const clone = el.cloneNode();
        if (el._transitionClasses) {
          el._transitionClasses.forEach((cls) => { removeClass(clone, cls); });
        }
        addClass(clone, moveClass);
        clone.style.display = 'none';
        this.$el.appendChild(clone);
        const info = getTransitionInfo(clone);
        this.$el.removeChild(clone);
        return (this._hasMove = info.hasTransform)
      }
    }
  };

  function callPendingCbs (c) {
    /* istanbul ignore if */
    if (c.elm._moveCb) {
      c.elm._moveCb();
    }
    /* istanbul ignore if */
    if (c.elm._enterCb) {
      c.elm._enterCb();
    }
  }

  function recordPosition (c) {
    c.data.newPos = c.elm.getBoundingClientRect();
  }

  function applyTranslation (c) {
    const oldPos = c.data.pos;
    const newPos = c.data.newPos;
    const dx = oldPos.left - newPos.left;
    const dy = oldPos.top - newPos.top;
    if (dx || dy) {
      c.data.moved = true;
      const s = c.elm.style;
      s.transform = s.WebkitTransform = `translate(${dx}px,${dy}px)`;
      s.transitionDuration = '0s';
    }
  }

  var platformComponents = {
    Transition,
    TransitionGroup
  };

  /*  */

  // install platform specific utils
  Vue.config.mustUseProp = mustUseProp;
  Vue.config.isReservedTag = isReservedTag;
  Vue.config.isReservedAttr = isReservedAttr;
  Vue.config.getTagNamespace = getTagNamespace;
  Vue.config.isUnknownElement = isUnknownElement;

  // install platform runtime directives & components
  extend(Vue.options.directives, platformDirectives);
  extend(Vue.options.components, platformComponents);

  // install platform patch function
  Vue.prototype.__patch__ = inBrowser ? patch : noop;

  // public mount method
  Vue.prototype.$mount = function (
    el,
    hydrating
  ) {
    el = el && inBrowser ? query(el) : undefined;
    return mountComponent(this, el, hydrating)
  };

  // devtools global hook
  /* istanbul ignore next */
  if (inBrowser) {
    setTimeout(() => {
      if (config.devtools) {
        if (devtools) {
          devtools.emit('init', Vue);
        } else {
          console[console.info ? 'info' : 'log'](
            'Download the Vue Devtools extension for a better development experience:\n' +
            'https://github.com/vuejs/vue-devtools'
          );
        }
      }
      if (config.productionTip !== false &&
        typeof console !== 'undefined'
      ) {
        console[console.info ? 'info' : 'log'](
          `You are running Vue in development mode.\n` +
          `Make sure to turn on production mode when deploying for production.\n` +
          `See more tips at https://vuejs.org/guide/deployment.html`
        );
      }
    }, 0);
  }

  /*  */

  const defaultTagRE = /\{\{((?:.|\r?\n)+?)\}\}/g;
  const regexEscapeRE = /[-.*+?^${}()|[\]\/\\]/g;

  const buildRegex = cached(delimiters => {
    const open = delimiters[0].replace(regexEscapeRE, '\\$&');
    const close = delimiters[1].replace(regexEscapeRE, '\\$&');
    return new RegExp(open + '((?:.|\\n)+?)' + close, 'g')
  });



  function parseText (
    text,
    delimiters
  ) {
    const tagRE = delimiters ? buildRegex(delimiters) : defaultTagRE;
    if (!tagRE.test(text)) {
      return
    }
    const tokens = [];
    const rawTokens = [];
    let lastIndex = tagRE.lastIndex = 0;
    let match, index, tokenValue;
    while ((match = tagRE.exec(text))) {
      index = match.index;
      // push text token
      if (index > lastIndex) {
        rawTokens.push(tokenValue = text.slice(lastIndex, index));
        tokens.push(JSON.stringify(tokenValue));
      }
      // tag token
      const exp = parseFilters(match[1].trim());
      tokens.push(`_s(${exp})`);
      rawTokens.push({ '@binding': exp });
      lastIndex = index + match[0].length;
    }
    if (lastIndex < text.length) {
      rawTokens.push(tokenValue = text.slice(lastIndex));
      tokens.push(JSON.stringify(tokenValue));
    }
    return {
      expression: tokens.join('+'),
      tokens: rawTokens
    }
  }

  /*  */

  function transformNode (el, options) {
    const warn = options.warn || baseWarn;
    const staticClass = getAndRemoveAttr(el, 'class');
    if (staticClass) {
      const res = parseText(staticClass, options.delimiters);
      if (res) {
        warn(
          `class="${staticClass}": ` +
          'Interpolation inside attributes has been removed. ' +
          'Use v-bind or the colon shorthand instead. For example, ' +
          'instead of <div class="{{ val }}">, use <div :class="val">.',
          el.rawAttrsMap['class']
        );
      }
    }
    if (staticClass) {
      el.staticClass = JSON.stringify(staticClass);
    }
    const classBinding = getBindingAttr(el, 'class', false /* getStatic */);
    if (classBinding) {
      el.classBinding = classBinding;
    }
  }

  function genData (el) {
    let data = '';
    if (el.staticClass) {
      data += `staticClass:${el.staticClass},`;
    }
    if (el.classBinding) {
      data += `class:${el.classBinding},`;
    }
    return data
  }

  var klass$1 = {
    staticKeys: ['staticClass'],
    transformNode,
    genData
  };

  /*  */

  function transformNode$1 (el, options) {
    const warn = options.warn || baseWarn;
    const staticStyle = getAndRemoveAttr(el, 'style');
    if (staticStyle) {
      /* istanbul ignore if */
      {
        const res = parseText(staticStyle, options.delimiters);
        if (res) {
          warn(
            `style="${staticStyle}": ` +
            'Interpolation inside attributes has been removed. ' +
            'Use v-bind or the colon shorthand instead. For example, ' +
            'instead of <div style="{{ val }}">, use <div :style="val">.',
            el.rawAttrsMap['style']
          );
        }
      }
      el.staticStyle = JSON.stringify(parseStyleText(staticStyle));
    }

    const styleBinding = getBindingAttr(el, 'style', false /* getStatic */);
    if (styleBinding) {
      el.styleBinding = styleBinding;
    }
  }

  function genData$1 (el) {
    let data = '';
    if (el.staticStyle) {
      data += `staticStyle:${el.staticStyle},`;
    }
    if (el.styleBinding) {
      data += `style:(${el.styleBinding}),`;
    }
    return data
  }

  var style$1 = {
    staticKeys: ['staticStyle'],
    transformNode: transformNode$1,
    genData: genData$1
  };

  /*  */

  let decoder;

  var he = {
    decode (html) {
      decoder = decoder || document.createElement('div');
      decoder.innerHTML = html;
      return decoder.textContent
    }
  };

  /*  */

  const isUnaryTag = makeMap(
    'area,base,br,col,embed,frame,hr,img,input,isindex,keygen,' +
    'link,meta,param,source,track,wbr'
  );

  // Elements that you can, intentionally, leave open
  // (and which close themselves)
  const canBeLeftOpenTag = makeMap(
    'colgroup,dd,dt,li,options,p,td,tfoot,th,thead,tr,source'
  );

  // HTML5 tags https://html.spec.whatwg.org/multipage/indices.html#elements-3
  // Phrasing Content https://html.spec.whatwg.org/multipage/dom.html#phrasing-content
  const isNonPhrasingTag = makeMap(
    'address,article,aside,base,blockquote,body,caption,col,colgroup,dd,' +
    'details,dialog,div,dl,dt,fieldset,figcaption,figure,footer,form,' +
    'h1,h2,h3,h4,h5,h6,head,header,hgroup,hr,html,legend,li,menuitem,meta,' +
    'optgroup,option,param,rp,rt,source,style,summary,tbody,td,tfoot,th,thead,' +
    'title,tr,track'
  );

  /**
   * Not type-checking this file because it's mostly vendor code.
   */

  // Regular Expressions for parsing tags and attributes
  const attribute = /^\s*([^\s"'<>\/=]+)(?:\s*(=)\s*(?:"([^"]*)"+|'([^']*)'+|([^\s"'=<>`]+)))?/;
  const dynamicArgAttribute = /^\s*((?:v-[\w-]+:|@|:|#)\[[^=]+?\][^\s"'<>\/=]*)(?:\s*(=)\s*(?:"([^"]*)"+|'([^']*)'+|([^\s"'=<>`]+)))?/;
  const ncname = `[a-zA-Z_][\\-\\.0-9_a-zA-Z${unicodeRegExp.source}]*`;
  const qnameCapture = `((?:${ncname}\\:)?${ncname})`;
  const startTagOpen = new RegExp(`^<${qnameCapture}`);
  const startTagClose = /^\s*(\/?)>/;
  const endTag = new RegExp(`^<\\/${qnameCapture}[^>]*>`);
  const doctype = /^<!DOCTYPE [^>]+>/i;
  // #7298: escape - to avoid being passed as HTML comment when inlined in page
  const comment = /^<!\--/;
  const conditionalComment = /^<!\[/;

  // Special Elements (can contain anything)
  const isPlainTextElement = makeMap('script,style,textarea', true);
  const reCache = {};

  const decodingMap = {
    '&lt;': '<',
    '&gt;': '>',
    '&quot;': '"',
    '&amp;': '&',
    '&#10;': '\n',
    '&#9;': '\t',
    '&#39;': "'"
  };
  const encodedAttr = /&(?:lt|gt|quot|amp|#39);/g;
  const encodedAttrWithNewLines = /&(?:lt|gt|quot|amp|#39|#10|#9);/g;

  // #5992
  const isIgnoreNewlineTag = makeMap('pre,textarea', true);
  const shouldIgnoreFirstNewline = (tag, html) => tag && isIgnoreNewlineTag(tag) && html[0] === '\n';

  function decodeAttr (value, shouldDecodeNewlines) {
    const re = shouldDecodeNewlines ? encodedAttrWithNewLines : encodedAttr;
    return value.replace(re, match => decodingMap[match])
  }

  function parseHTML (html, options) {
    const stack = [];
    const expectHTML = options.expectHTML;
    const isUnaryTag$$1 = options.isUnaryTag || no;
    const canBeLeftOpenTag$$1 = options.canBeLeftOpenTag || no;
    let index = 0;
    let last, lastTag;
    while (html) {
      last = html;
      // Make sure we're not in a plaintext content element like script/style
      if (!lastTag || !isPlainTextElement(lastTag)) {
        let textEnd = html.indexOf('<');
        if (textEnd === 0) {
          // Comment:
          if (comment.test(html)) {
            const commentEnd = html.indexOf('-->');

            if (commentEnd >= 0) {
              if (options.shouldKeepComment) {
                options.comment(html.substring(4, commentEnd), index, index + commentEnd + 3);
              }
              advance(commentEnd + 3);
              continue
            }
          }

          // http://en.wikipedia.org/wiki/Conditional_comment#Downlevel-revealed_conditional_comment
          if (conditionalComment.test(html)) {
            const conditionalEnd = html.indexOf(']>');

            if (conditionalEnd >= 0) {
              advance(conditionalEnd + 2);
              continue
            }
          }

          // Doctype:
          const doctypeMatch = html.match(doctype);
          if (doctypeMatch) {
            advance(doctypeMatch[0].length);
            continue
          }

          // End tag:
          const endTagMatch = html.match(endTag);
          if (endTagMatch) {
            const curIndex = index;
            advance(endTagMatch[0].length);
            parseEndTag(endTagMatch[1], curIndex, index);
            continue
          }

          // Start tag:
          const startTagMatch = parseStartTag();
          if (startTagMatch) {
            handleStartTag(startTagMatch);
            if (shouldIgnoreFirstNewline(startTagMatch.tagName, html)) {
              advance(1);
            }
            continue
          }
        }

        let text, rest, next;
        if (textEnd >= 0) {
          rest = html.slice(textEnd);
          while (
            !endTag.test(rest) &&
            !startTagOpen.test(rest) &&
            !comment.test(rest) &&
            !conditionalComment.test(rest)
          ) {
            // < in plain text, be forgiving and treat it as text
            next = rest.indexOf('<', 1);
            if (next < 0) break
            textEnd += next;
            rest = html.slice(textEnd);
          }
          text = html.substring(0, textEnd);
        }

        if (textEnd < 0) {
          text = html;
        }

        if (text) {
          advance(text.length);
        }

        if (options.chars && text) {
          options.chars(text, index - text.length, index);
        }
      } else {
        let endTagLength = 0;
        const stackedTag = lastTag.toLowerCase();
        const reStackedTag = reCache[stackedTag] || (reCache[stackedTag] = new RegExp('([\\s\\S]*?)(</' + stackedTag + '[^>]*>)', 'i'));
        const rest = html.replace(reStackedTag, function (all, text, endTag) {
          endTagLength = endTag.length;
          if (!isPlainTextElement(stackedTag) && stackedTag !== 'noscript') {
            text = text
              .replace(/<!\--([\s\S]*?)-->/g, '$1') // #7298
              .replace(/<!\[CDATA\[([\s\S]*?)]]>/g, '$1');
          }
          if (shouldIgnoreFirstNewline(stackedTag, text)) {
            text = text.slice(1);
          }
          if (options.chars) {
            options.chars(text);
          }
          return ''
        });
        index += html.length - rest.length;
        html = rest;
        parseEndTag(stackedTag, index - endTagLength, index);
      }

      if (html === last) {
        options.chars && options.chars(html);
        if (!stack.length && options.warn) {
          options.warn(`Mal-formatted tag at end of template: "${html}"`, { start: index + html.length });
        }
        break
      }
    }

    // Clean up any remaining tags
    parseEndTag();

    function advance (n) {
      index += n;
      html = html.substring(n);
    }

    function parseStartTag () {
      const start = html.match(startTagOpen);
      if (start) {
        const match = {
          tagName: start[1],
          attrs: [],
          start: index
        };
        advance(start[0].length);
        let end, attr;
        while (!(end = html.match(startTagClose)) && (attr = html.match(dynamicArgAttribute) || html.match(attribute))) {
          attr.start = index;
          advance(attr[0].length);
          attr.end = index;
          match.attrs.push(attr);
        }
        if (end) {
          match.unarySlash = end[1];
          advance(end[0].length);
          match.end = index;
          return match
        }
      }
    }

    function handleStartTag (match) {
      const tagName = match.tagName;
      const unarySlash = match.unarySlash;

      if (expectHTML) {
        if (lastTag === 'p' && isNonPhrasingTag(tagName)) {
          parseEndTag(lastTag);
        }
        if (canBeLeftOpenTag$$1(tagName) && lastTag === tagName) {
          parseEndTag(tagName);
        }
      }

      const unary = isUnaryTag$$1(tagName) || !!unarySlash;

      const l = match.attrs.length;
      const attrs = new Array(l);
      for (let i = 0; i < l; i++) {
        const args = match.attrs[i];
        const value = args[3] || args[4] || args[5] || '';
        const shouldDecodeNewlines = tagName === 'a' && args[1] === 'href'
          ? options.shouldDecodeNewlinesForHref
          : options.shouldDecodeNewlines;
        attrs[i] = {
          name: args[1],
          value: decodeAttr(value, shouldDecodeNewlines)
        };
        if (options.outputSourceRange) {
          attrs[i].start = args.start + args[0].match(/^\s*/).length;
          attrs[i].end = args.end;
        }
      }

      if (!unary) {
        stack.push({ tag: tagName, lowerCasedTag: tagName.toLowerCase(), attrs: attrs, start: match.start, end: match.end });
        lastTag = tagName;
      }

      if (options.start) {
        options.start(tagName, attrs, unary, match.start, match.end);
      }
    }

    function parseEndTag (tagName, start, end) {
      let pos, lowerCasedTagName;
      if (start == null) start = index;
      if (end == null) end = index;

      // Find the closest opened tag of the same type
      if (tagName) {
        lowerCasedTagName = tagName.toLowerCase();
        for (pos = stack.length - 1; pos >= 0; pos--) {
          if (stack[pos].lowerCasedTag === lowerCasedTagName) {
            break
          }
        }
      } else {
        // If no tag name is provided, clean shop
        pos = 0;
      }

      if (pos >= 0) {
        // Close all the open elements, up the stack
        for (let i = stack.length - 1; i >= pos; i--) {
          if (i > pos || !tagName &&
            options.warn
          ) {
            options.warn(
              `tag <${stack[i].tag}> has no matching end tag.`,
              { start: stack[i].start, end: stack[i].end }
            );
          }
          if (options.end) {
            options.end(stack[i].tag, start, end);
          }
        }

        // Remove the open elements from the stack
        stack.length = pos;
        lastTag = pos && stack[pos - 1].tag;
      } else if (lowerCasedTagName === 'br') {
        if (options.start) {
          options.start(tagName, [], true, start, end);
        }
      } else if (lowerCasedTagName === 'p') {
        if (options.start) {
          options.start(tagName, [], false, start, end);
        }
        if (options.end) {
          options.end(tagName, start, end);
        }
      }
    }
  }

  /*  */

  const onRE = /^@|^v-on:/;
  const dirRE = /^v-|^@|^:|^#/;
  const forAliasRE = /([\s\S]*?)\s+(?:in|of)\s+([\s\S]*)/;
  const forIteratorRE = /,([^,\}\]]*)(?:,([^,\}\]]*))?$/;
  const stripParensRE = /^\(|\)$/g;
  const dynamicArgRE = /^\[.*\]$/;

  const argRE = /:(.*)$/;
  const bindRE = /^:|^\.|^v-bind:/;
  const modifierRE = /\.[^.\]]+(?=[^\]]*$)/g;

  const slotRE = /^v-slot(:|$)|^#/;

  const lineBreakRE = /[\r\n]/;
  const whitespaceRE$1 = /[ \f\t\r\n]+/g;

  const invalidAttributeRE = /[\s"'<>\/=]/;

  const decodeHTMLCached = cached(he.decode);

  const emptySlotScopeToken = `_empty_`;

  // configurable state
  let warn$2;
  let delimiters;
  let transforms;
  let preTransforms;
  let postTransforms;
  let platformIsPreTag;
  let platformMustUseProp;
  let platformGetTagNamespace;
  let maybeComponent;

  function createASTElement (
    tag,
    attrs,
    parent
  ) {
    return {
      type: 1,
      tag,
      attrsList: attrs,
      attrsMap: makeAttrsMap(attrs),
      rawAttrsMap: {},
      parent,
      children: []
    }
  }

  /**
   * Convert HTML string to AST.
   */
  function parse (
    template,
    options
  ) {
    warn$2 = options.warn || baseWarn;

    platformIsPreTag = options.isPreTag || no;
    platformMustUseProp = options.mustUseProp || no;
    platformGetTagNamespace = options.getTagNamespace || no;
    const isReservedTag = options.isReservedTag || no;
    maybeComponent = (el) => !!(
      el.component ||
      el.attrsMap[':is'] ||
      el.attrsMap['v-bind:is'] ||
      !(el.attrsMap.is ? isReservedTag(el.attrsMap.is) : isReservedTag(el.tag))
    );
    transforms = pluckModuleFunction(options.modules, 'transformNode');
    preTransforms = pluckModuleFunction(options.modules, 'preTransformNode');
    postTransforms = pluckModuleFunction(options.modules, 'postTransformNode');

    delimiters = options.delimiters;

    const stack = [];
    const preserveWhitespace = options.preserveWhitespace !== false;
    const whitespaceOption = options.whitespace;
    let root;
    let currentParent;
    let inVPre = false;
    let inPre = false;
    let warned = false;

    function warnOnce (msg, range) {
      if (!warned) {
        warned = true;
        warn$2(msg, range);
      }
    }

    function closeElement (element) {
      trimEndingWhitespace(element);
      if (!inVPre && !element.processed) {
        element = processElement(element, options);
      }
      // tree management
      if (!stack.length && element !== root) {
        // allow root elements with v-if, v-else-if and v-else
        if (root.if && (element.elseif || element.else)) {
          {
            checkRootConstraints(element);
          }
          addIfCondition(root, {
            exp: element.elseif,
            block: element
          });
        } else {
          warnOnce(
            `Component template should contain exactly one root element. ` +
            `If you are using v-if on multiple elements, ` +
            `use v-else-if to chain them instead.`,
            { start: element.start }
          );
        }
      }
      if (currentParent && !element.forbidden) {
        if (element.elseif || element.else) {
          processIfConditions(element, currentParent);
        } else {
          if (element.slotScope) {
            // scoped slot
            // keep it in the children list so that v-else(-if) conditions can
            // find it as the prev node.
            const name = element.slotTarget || '"default"'
            ;(currentParent.scopedSlots || (currentParent.scopedSlots = {}))[name] = element;
          }
          currentParent.children.push(element);
          element.parent = currentParent;
        }
      }

      // final children cleanup
      // filter out scoped slots
      element.children = element.children.filter(c => !(c).slotScope);
      // remove trailing whitespace node again
      trimEndingWhitespace(element);

      // check pre state
      if (element.pre) {
        inVPre = false;
      }
      if (platformIsPreTag(element.tag)) {
        inPre = false;
      }
      // apply post-transforms
      for (let i = 0; i < postTransforms.length; i++) {
        postTransforms[i](element, options);
      }
    }

    function trimEndingWhitespace (el) {
      // remove trailing whitespace node
      if (!inPre) {
        let lastNode;
        while (
          (lastNode = el.children[el.children.length - 1]) &&
          lastNode.type === 3 &&
          lastNode.text === ' '
        ) {
          el.children.pop();
        }
      }
    }

    function checkRootConstraints (el) {
      if (el.tag === 'slot' || el.tag === 'template') {
        warnOnce(
          `Cannot use <${el.tag}> as component root element because it may ` +
          'contain multiple nodes.',
          { start: el.start }
        );
      }
      if (el.attrsMap.hasOwnProperty('v-for')) {
        warnOnce(
          'Cannot use v-for on stateful component root element because ' +
          'it renders multiple elements.',
          el.rawAttrsMap['v-for']
        );
      }
    }

    parseHTML(template, {
      warn: warn$2,
      expectHTML: options.expectHTML,
      isUnaryTag: options.isUnaryTag,
      canBeLeftOpenTag: options.canBeLeftOpenTag,
      shouldDecodeNewlines: options.shouldDecodeNewlines,
      shouldDecodeNewlinesForHref: options.shouldDecodeNewlinesForHref,
      shouldKeepComment: options.comments,
      outputSourceRange: options.outputSourceRange,
      start (tag, attrs, unary, start, end) {
        // check namespace.
        // inherit parent ns if there is one
        const ns = (currentParent && currentParent.ns) || platformGetTagNamespace(tag);

        // handle IE svg bug
        /* istanbul ignore if */
        if (isIE && ns === 'svg') {
          attrs = guardIESVGBug(attrs);
        }

        let element = createASTElement(tag, attrs, currentParent);
        if (ns) {
          element.ns = ns;
        }

        {
          if (options.outputSourceRange) {
            element.start = start;
            element.end = end;
            element.rawAttrsMap = element.attrsList.reduce((cumulated, attr) => {
              cumulated[attr.name] = attr;
              return cumulated
            }, {});
          }
          attrs.forEach(attr => {
            if (invalidAttributeRE.test(attr.name)) {
              warn$2(
                `Invalid dynamic argument expression: attribute names cannot contain ` +
                `spaces, quotes, <, >, / or =.`,
                {
                  start: attr.start + attr.name.indexOf(`[`),
                  end: attr.start + attr.name.length
                }
              );
            }
          });
        }

        if (isForbiddenTag(element) && !isServerRendering()) {
          element.forbidden = true;
          warn$2(
            'Templates should only be responsible for mapping the state to the ' +
            'UI. Avoid placing tags with side-effects in your templates, such as ' +
            `<${tag}>` + ', as they will not be parsed.',
            { start: element.start }
          );
        }

        // apply pre-transforms
        for (let i = 0; i < preTransforms.length; i++) {
          element = preTransforms[i](element, options) || element;
        }

        if (!inVPre) {
          processPre(element);
          if (element.pre) {
            inVPre = true;
          }
        }
        if (platformIsPreTag(element.tag)) {
          inPre = true;
        }
        if (inVPre) {
          processRawAttrs(element);
        } else if (!element.processed) {
          // structural directives
          processFor(element);
          processIf(element);
          processOnce(element);
        }

        if (!root) {
          root = element;
          {
            checkRootConstraints(root);
          }
        }

        if (!unary) {
          currentParent = element;
          stack.push(element);
        } else {
          closeElement(element);
        }
      },

      end (tag, start, end) {
        const element = stack[stack.length - 1];
        // pop stack
        stack.length -= 1;
        currentParent = stack[stack.length - 1];
        if (options.outputSourceRange) {
          element.end = end;
        }
        closeElement(element);
      },

      chars (text, start, end) {
        if (!currentParent) {
          {
            if (text === template) {
              warnOnce(
                'Component template requires a root element, rather than just text.',
                { start }
              );
            } else if ((text = text.trim())) {
              warnOnce(
                `text "${text}" outside root element will be ignored.`,
                { start }
              );
            }
          }
          return
        }
        // IE textarea placeholder bug
        /* istanbul ignore if */
        if (isIE &&
          currentParent.tag === 'textarea' &&
          currentParent.attrsMap.placeholder === text
        ) {
          return
        }
        const children = currentParent.children;
        if (inPre || text.trim()) {
          text = isTextTag(currentParent) ? text : decodeHTMLCached(text);
        } else if (!children.length) {
          // remove the whitespace-only node right after an opening tag
          text = '';
        } else if (whitespaceOption) {
          if (whitespaceOption === 'condense') {
            // in condense mode, remove the whitespace node if it contains
            // line break, otherwise condense to a single space
            text = lineBreakRE.test(text) ? '' : ' ';
          } else {
            text = ' ';
          }
        } else {
          text = preserveWhitespace ? ' ' : '';
        }
        if (text) {
          if (!inPre && whitespaceOption === 'condense') {
            // condense consecutive whitespaces into single space
            text = text.replace(whitespaceRE$1, ' ');
          }
          let res;
          let child;
          if (!inVPre && text !== ' ' && (res = parseText(text, delimiters))) {
            child = {
              type: 2,
              expression: res.expression,
              tokens: res.tokens,
              text
            };
          } else if (text !== ' ' || !children.length || children[children.length - 1].text !== ' ') {
            child = {
              type: 3,
              text
            };
          }
          if (child) {
            if (options.outputSourceRange) {
              child.start = start;
              child.end = end;
            }
            children.push(child);
          }
        }
      },
      comment (text, start, end) {
        // adding anything as a sibling to the root node is forbidden
        // comments should still be allowed, but ignored
        if (currentParent) {
          const child = {
            type: 3,
            text,
            isComment: true
          };
          if (options.outputSourceRange) {
            child.start = start;
            child.end = end;
          }
          currentParent.children.push(child);
        }
      }
    });
    return root
  }

  function processPre (el) {
    if (getAndRemoveAttr(el, 'v-pre') != null) {
      el.pre = true;
    }
  }

  function processRawAttrs (el) {
    const list = el.attrsList;
    const len = list.length;
    if (len) {
      const attrs = el.attrs = new Array(len);
      for (let i = 0; i < len; i++) {
        attrs[i] = {
          name: list[i].name,
          value: JSON.stringify(list[i].value)
        };
        if (list[i].start != null) {
          attrs[i].start = list[i].start;
          attrs[i].end = list[i].end;
        }
      }
    } else if (!el.pre) {
      // non root node in pre blocks with no attributes
      el.plain = true;
    }
  }

  function processElement (
    element,
    options
  ) {
    processKey(element);

    // determine whether this is a plain element after
    // removing structural attributes
    element.plain = (
      !element.key &&
      !element.scopedSlots &&
      !element.attrsList.length
    );

    processRef(element);
    processSlotContent(element);
    processSlotOutlet(element);
    processComponent(element);
    for (let i = 0; i < transforms.length; i++) {
      element = transforms[i](element, options) || element;
    }
    processAttrs(element);
    return element
  }

  function processKey (el) {
    const exp = getBindingAttr(el, 'key');
    if (exp) {
      {
        if (el.tag === 'template') {
          warn$2(
            `<template> cannot be keyed. Place the key on real elements instead.`,
            getRawBindingAttr(el, 'key')
          );
        }
        if (el.for) {
          const iterator = el.iterator2 || el.iterator1;
          const parent = el.parent;
          if (iterator && iterator === exp && parent && parent.tag === 'transition-group') {
            warn$2(
              `Do not use v-for index as key on <transition-group> children, ` +
              `this is the same as not using keys.`,
              getRawBindingAttr(el, 'key'),
              true /* tip */
            );
          }
        }
      }
      el.key = exp;
    }
  }

  function processRef (el) {
    const ref = getBindingAttr(el, 'ref');
    if (ref) {
      el.ref = ref;
      el.refInFor = checkInFor(el);
    }
  }

  function processFor (el) {
    let exp;
    if ((exp = getAndRemoveAttr(el, 'v-for'))) {
      const res = parseFor(exp);
      if (res) {
        extend(el, res);
      } else {
        warn$2(
          `Invalid v-for expression: ${exp}`,
          el.rawAttrsMap['v-for']
        );
      }
    }
  }



  function parseFor (exp) {
    const inMatch = exp.match(forAliasRE);
    if (!inMatch) return
    const res = {};
    res.for = inMatch[2].trim();
    const alias = inMatch[1].trim().replace(stripParensRE, '');
    const iteratorMatch = alias.match(forIteratorRE);
    if (iteratorMatch) {
      res.alias = alias.replace(forIteratorRE, '').trim();
      res.iterator1 = iteratorMatch[1].trim();
      if (iteratorMatch[2]) {
        res.iterator2 = iteratorMatch[2].trim();
      }
    } else {
      res.alias = alias;
    }
    return res
  }

  function processIf (el) {
    const exp = getAndRemoveAttr(el, 'v-if');
    if (exp) {
      el.if = exp;
      addIfCondition(el, {
        exp: exp,
        block: el
      });
    } else {
      if (getAndRemoveAttr(el, 'v-else') != null) {
        el.else = true;
      }
      const elseif = getAndRemoveAttr(el, 'v-else-if');
      if (elseif) {
        el.elseif = elseif;
      }
    }
  }

  function processIfConditions (el, parent) {
    const prev = findPrevElement(parent.children);
    if (prev && prev.if) {
      addIfCondition(prev, {
        exp: el.elseif,
        block: el
      });
    } else {
      warn$2(
        `v-${el.elseif ? ('else-if="' + el.elseif + '"') : 'else'} ` +
        `used on element <${el.tag}> without corresponding v-if.`,
        el.rawAttrsMap[el.elseif ? 'v-else-if' : 'v-else']
      );
    }
  }

  function findPrevElement (children) {
    let i = children.length;
    while (i--) {
      if (children[i].type === 1) {
        return children[i]
      } else {
        if (children[i].text !== ' ') {
          warn$2(
            `text "${children[i].text.trim()}" between v-if and v-else(-if) ` +
            `will be ignored.`,
            children[i]
          );
        }
        children.pop();
      }
    }
  }

  function addIfCondition (el, condition) {
    if (!el.ifConditions) {
      el.ifConditions = [];
    }
    el.ifConditions.push(condition);
  }

  function processOnce (el) {
    const once$$1 = getAndRemoveAttr(el, 'v-once');
    if (once$$1 != null) {
      el.once = true;
    }
  }

  // handle content being passed to a component as slot,
  // e.g. <template slot="xxx">, <div slot-scope="xxx">
  function processSlotContent (el) {
    let slotScope;
    if (el.tag === 'template') {
      slotScope = getAndRemoveAttr(el, 'scope');
      /* istanbul ignore if */
      if (slotScope) {
        warn$2(
          `the "scope" attribute for scoped slots have been deprecated and ` +
          `replaced by "slot-scope" since 2.5. The new "slot-scope" attribute ` +
          `can also be used on plain elements in addition to <template> to ` +
          `denote scoped slots.`,
          el.rawAttrsMap['scope'],
          true
        );
      }
      el.slotScope = slotScope || getAndRemoveAttr(el, 'slot-scope');
    } else if ((slotScope = getAndRemoveAttr(el, 'slot-scope'))) {
      /* istanbul ignore if */
      if (el.attrsMap['v-for']) {
        warn$2(
          `Ambiguous combined usage of slot-scope and v-for on <${el.tag}> ` +
          `(v-for takes higher priority). Use a wrapper <template> for the ` +
          `scoped slot to make it clearer.`,
          el.rawAttrsMap['slot-scope'],
          true
        );
      }
      el.slotScope = slotScope;
    }

    // slot="xxx"
    const slotTarget = getBindingAttr(el, 'slot');
    if (slotTarget) {
      el.slotTarget = slotTarget === '""' ? '"default"' : slotTarget;
      el.slotTargetDynamic = !!(el.attrsMap[':slot'] || el.attrsMap['v-bind:slot']);
      // preserve slot as an attribute for native shadow DOM compat
      // only for non-scoped slots.
      if (el.tag !== 'template' && !el.slotScope) {
        addAttr(el, 'slot', slotTarget, getRawBindingAttr(el, 'slot'));
      }
    }

    // 2.6 v-slot syntax
    {
      if (el.tag === 'template') {
        // v-slot on <template>
        const slotBinding = getAndRemoveAttrByRegex(el, slotRE);
        if (slotBinding) {
          {
            if (el.slotTarget || el.slotScope) {
              warn$2(
                `Unexpected mixed usage of different slot syntaxes.`,
                el
              );
            }
            if (el.parent && !maybeComponent(el.parent)) {
              warn$2(
                `<template v-slot> can only appear at the root level inside ` +
                `the receiving component`,
                el
              );
            }
          }
          const { name, dynamic } = getSlotName(slotBinding);
          el.slotTarget = name;
          el.slotTargetDynamic = dynamic;
          el.slotScope = slotBinding.value || emptySlotScopeToken; // force it into a scoped slot for perf
        }
      } else {
        // v-slot on component, denotes default slot
        const slotBinding = getAndRemoveAttrByRegex(el, slotRE);
        if (slotBinding) {
          {
            if (!maybeComponent(el)) {
              warn$2(
                `v-slot can only be used on components or <template>.`,
                slotBinding
              );
            }
            if (el.slotScope || el.slotTarget) {
              warn$2(
                `Unexpected mixed usage of different slot syntaxes.`,
                el
              );
            }
            if (el.scopedSlots) {
              warn$2(
                `To avoid scope ambiguity, the default slot should also use ` +
                `<template> syntax when there are other named slots.`,
                slotBinding
              );
            }
          }
          // add the component's children to its default slot
          const slots = el.scopedSlots || (el.scopedSlots = {});
          const { name, dynamic } = getSlotName(slotBinding);
          const slotContainer = slots[name] = createASTElement('template', [], el);
          slotContainer.slotTarget = name;
          slotContainer.slotTargetDynamic = dynamic;
          slotContainer.children = el.children.filter((c) => {
            if (!c.slotScope) {
              c.parent = slotContainer;
              return true
            }
          });
          slotContainer.slotScope = slotBinding.value || emptySlotScopeToken;
          // remove children as they are returned from scopedSlots now
          el.children = [];
          // mark el non-plain so data gets generated
          el.plain = false;
        }
      }
    }
  }

  function getSlotName (binding) {
    let name = binding.name.replace(slotRE, '');
    if (!name) {
      if (binding.name[0] !== '#') {
        name = 'default';
      } else {
        warn$2(
          `v-slot shorthand syntax requires a slot name.`,
          binding
        );
      }
    }
    return dynamicArgRE.test(name)
      // dynamic [name]
      ? { name: name.slice(1, -1), dynamic: true }
      // static name
      : { name: `"${name}"`, dynamic: false }
  }

  // handle <slot/> outlets
  function processSlotOutlet (el) {
    if (el.tag === 'slot') {
      el.slotName = getBindingAttr(el, 'name');
      if (el.key) {
        warn$2(
          `\`key\` does not work on <slot> because slots are abstract outlets ` +
          `and can possibly expand into multiple elements. ` +
          `Use the key on a wrapping element instead.`,
          getRawBindingAttr(el, 'key')
        );
      }
    }
  }

  function processComponent (el) {
    let binding;
    if ((binding = getBindingAttr(el, 'is'))) {
      el.component = binding;
    }
    if (getAndRemoveAttr(el, 'inline-template') != null) {
      el.inlineTemplate = true;
    }
  }

  function processAttrs (el) {
    const list = el.attrsList;
    let i, l, name, rawName, value, modifiers, syncGen, isDynamic;
    for (i = 0, l = list.length; i < l; i++) {
      name = rawName = list[i].name;
      value = list[i].value;
      if (dirRE.test(name)) {
        // mark element as dynamic
        el.hasBindings = true;
        // modifiers
        modifiers = parseModifiers(name.replace(dirRE, ''));
        // support .foo shorthand syntax for the .prop modifier
        if (modifiers) {
          name = name.replace(modifierRE, '');
        }
        if (bindRE.test(name)) { // v-bind
          name = name.replace(bindRE, '');
          value = parseFilters(value);
          isDynamic = dynamicArgRE.test(name);
          if (isDynamic) {
            name = name.slice(1, -1);
          }
          if (
            value.trim().length === 0
          ) {
            warn$2(
              `The value for a v-bind expression cannot be empty. Found in "v-bind:${name}"`
            );
          }
          if (modifiers) {
            if (modifiers.prop && !isDynamic) {
              name = camelize(name);
              if (name === 'innerHtml') name = 'innerHTML';
            }
            if (modifiers.camel && !isDynamic) {
              name = camelize(name);
            }
            if (modifiers.sync) {
              syncGen = genAssignmentCode(value, `$event`);
              if (!isDynamic) {
                addHandler(
                  el,
                  `update:${camelize(name)}`,
                  syncGen,
                  null,
                  false,
                  warn$2,
                  list[i]
                );
                if (hyphenate(name) !== camelize(name)) {
                  addHandler(
                    el,
                    `update:${hyphenate(name)}`,
                    syncGen,
                    null,
                    false,
                    warn$2,
                    list[i]
                  );
                }
              } else {
                // handler w/ dynamic event name
                addHandler(
                  el,
                  `"update:"+(${name})`,
                  syncGen,
                  null,
                  false,
                  warn$2,
                  list[i],
                  true // dynamic
                );
              }
            }
          }
          if ((modifiers && modifiers.prop) || (
            !el.component && platformMustUseProp(el.tag, el.attrsMap.type, name)
          )) {
            addProp(el, name, value, list[i], isDynamic);
          } else {
            addAttr(el, name, value, list[i], isDynamic);
          }
        } else if (onRE.test(name)) { // v-on
          name = name.replace(onRE, '');
          isDynamic = dynamicArgRE.test(name);
          if (isDynamic) {
            name = name.slice(1, -1);
          }
          addHandler(el, name, value, modifiers, false, warn$2, list[i], isDynamic);
        } else { // normal directives
          name = name.replace(dirRE, '');
          // parse arg
          const argMatch = name.match(argRE);
          let arg = argMatch && argMatch[1];
          isDynamic = false;
          if (arg) {
            name = name.slice(0, -(arg.length + 1));
            if (dynamicArgRE.test(arg)) {
              arg = arg.slice(1, -1);
              isDynamic = true;
            }
          }
          addDirective(el, name, rawName, value, arg, isDynamic, modifiers, list[i]);
          if (name === 'model') {
            checkForAliasModel(el, value);
          }
        }
      } else {
        // literal attribute
        {
          const res = parseText(value, delimiters);
          if (res) {
            warn$2(
              `${name}="${value}": ` +
              'Interpolation inside attributes has been removed. ' +
              'Use v-bind or the colon shorthand instead. For example, ' +
              'instead of <div id="{{ val }}">, use <div :id="val">.',
              list[i]
            );
          }
        }
        addAttr(el, name, JSON.stringify(value), list[i]);
        // #6887 firefox doesn't update muted state if set via attribute
        // even immediately after element creation
        if (!el.component &&
            name === 'muted' &&
            platformMustUseProp(el.tag, el.attrsMap.type, name)) {
          addProp(el, name, 'true', list[i]);
        }
      }
    }
  }

  function checkInFor (el) {
    let parent = el;
    while (parent) {
      if (parent.for !== undefined) {
        return true
      }
      parent = parent.parent;
    }
    return false
  }

  function parseModifiers (name) {
    const match = name.match(modifierRE);
    if (match) {
      const ret = {};
      match.forEach(m => { ret[m.slice(1)] = true; });
      return ret
    }
  }

  function makeAttrsMap (attrs) {
    const map = {};
    for (let i = 0, l = attrs.length; i < l; i++) {
      if (
        map[attrs[i].name] && !isIE && !isEdge
      ) {
        warn$2('duplicate attribute: ' + attrs[i].name, attrs[i]);
      }
      map[attrs[i].name] = attrs[i].value;
    }
    return map
  }

  // for script (e.g. type="x/template") or style, do not decode content
  function isTextTag (el) {
    return el.tag === 'script' || el.tag === 'style'
  }

  function isForbiddenTag (el) {
    return (
      el.tag === 'style' ||
      (el.tag === 'script' && (
        !el.attrsMap.type ||
        el.attrsMap.type === 'text/javascript'
      ))
    )
  }

  const ieNSBug = /^xmlns:NS\d+/;
  const ieNSPrefix = /^NS\d+:/;

  /* istanbul ignore next */
  function guardIESVGBug (attrs) {
    const res = [];
    for (let i = 0; i < attrs.length; i++) {
      const attr = attrs[i];
      if (!ieNSBug.test(attr.name)) {
        attr.name = attr.name.replace(ieNSPrefix, '');
        res.push(attr);
      }
    }
    return res
  }

  function checkForAliasModel (el, value) {
    let _el = el;
    while (_el) {
      if (_el.for && _el.alias === value) {
        warn$2(
          `<${el.tag} v-model="${value}">: ` +
          `You are binding v-model directly to a v-for iteration alias. ` +
          `This will not be able to modify the v-for source array because ` +
          `writing to the alias is like modifying a function local variable. ` +
          `Consider using an array of objects and use v-model on an object property instead.`,
          el.rawAttrsMap['v-model']
        );
      }
      _el = _el.parent;
    }
  }

  /*  */

  function preTransformNode (el, options) {
    if (el.tag === 'input') {
      const map = el.attrsMap;
      if (!map['v-model']) {
        return
      }

      let typeBinding;
      if (map[':type'] || map['v-bind:type']) {
        typeBinding = getBindingAttr(el, 'type');
      }
      if (!map.type && !typeBinding && map['v-bind']) {
        typeBinding = `(${map['v-bind']}).type`;
      }

      if (typeBinding) {
        const ifCondition = getAndRemoveAttr(el, 'v-if', true);
        const ifConditionExtra = ifCondition ? `&&(${ifCondition})` : ``;
        const hasElse = getAndRemoveAttr(el, 'v-else', true) != null;
        const elseIfCondition = getAndRemoveAttr(el, 'v-else-if', true);
        // 1. checkbox
        const branch0 = cloneASTElement(el);
        // process for on the main node
        processFor(branch0);
        addRawAttr(branch0, 'type', 'checkbox');
        processElement(branch0, options);
        branch0.processed = true; // prevent it from double-processed
        branch0.if = `(${typeBinding})==='checkbox'` + ifConditionExtra;
        addIfCondition(branch0, {
          exp: branch0.if,
          block: branch0
        });
        // 2. add radio else-if condition
        const branch1 = cloneASTElement(el);
        getAndRemoveAttr(branch1, 'v-for', true);
        addRawAttr(branch1, 'type', 'radio');
        processElement(branch1, options);
        addIfCondition(branch0, {
          exp: `(${typeBinding})==='radio'` + ifConditionExtra,
          block: branch1
        });
        // 3. other
        const branch2 = cloneASTElement(el);
        getAndRemoveAttr(branch2, 'v-for', true);
        addRawAttr(branch2, ':type', typeBinding);
        processElement(branch2, options);
        addIfCondition(branch0, {
          exp: ifCondition,
          block: branch2
        });

        if (hasElse) {
          branch0.else = true;
        } else if (elseIfCondition) {
          branch0.elseif = elseIfCondition;
        }

        return branch0
      }
    }
  }

  function cloneASTElement (el) {
    return createASTElement(el.tag, el.attrsList.slice(), el.parent)
  }

  var model$1 = {
    preTransformNode
  };

  var modules$1 = [
    klass$1,
    style$1,
    model$1
  ];

  /*  */

  function text (el, dir) {
    if (dir.value) {
      addProp(el, 'textContent', `_s(${dir.value})`, dir);
    }
  }

  /*  */

  function html (el, dir) {
    if (dir.value) {
      addProp(el, 'innerHTML', `_s(${dir.value})`, dir);
    }
  }

  var directives$1 = {
    model,
    text,
    html
  };

  /*  */

  const baseOptions = {
    expectHTML: true,
    modules: modules$1,
    directives: directives$1,
    isPreTag,
    isUnaryTag,
    mustUseProp,
    canBeLeftOpenTag,
    isReservedTag,
    getTagNamespace,
    staticKeys: genStaticKeys(modules$1)
  };

  /*  */

  let isStaticKey;
  let isPlatformReservedTag;

  const genStaticKeysCached = cached(genStaticKeys$1);

  /**
   * Goal of the optimizer: walk the generated template AST tree
   * and detect sub-trees that are purely static, i.e. parts of
   * the DOM that never needs to change.
   *
   * Once we detect these sub-trees, we can:
   *
   * 1. Hoist them into constants, so that we no longer need to
   *    create fresh nodes for them on each re-render;
   * 2. Completely skip them in the patching process.
   */
  function optimize (root, options) {
    if (!root) return
    isStaticKey = genStaticKeysCached(options.staticKeys || '');
    isPlatformReservedTag = options.isReservedTag || no;
    // first pass: mark all non-static nodes.
    markStatic$1(root);
    // second pass: mark static roots.
    markStaticRoots(root, false);
  }

  function genStaticKeys$1 (keys) {
    return makeMap(
      'type,tag,attrsList,attrsMap,plain,parent,children,attrs,start,end,rawAttrsMap' +
      (keys ? ',' + keys : '')
    )
  }

  function markStatic$1 (node) {
    node.static = isStatic(node);
    if (node.type === 1) {
      // do not make component slot content static. this avoids
      // 1. components not able to mutate slot nodes
      // 2. static slot content fails for hot-reloading
      if (
        !isPlatformReservedTag(node.tag) &&
        node.tag !== 'slot' &&
        node.attrsMap['inline-template'] == null
      ) {
        return
      }
      for (let i = 0, l = node.children.length; i < l; i++) {
        const child = node.children[i];
        markStatic$1(child);
        if (!child.static) {
          node.static = false;
        }
      }
      if (node.ifConditions) {
        for (let i = 1, l = node.ifConditions.length; i < l; i++) {
          const block = node.ifConditions[i].block;
          markStatic$1(block);
          if (!block.static) {
            node.static = false;
          }
        }
      }
    }
  }

  function markStaticRoots (node, isInFor) {
    if (node.type === 1) {
      if (node.static || node.once) {
        node.staticInFor = isInFor;
      }
      // For a node to qualify as a static root, it should have children that
      // are not just static text. Otherwise the cost of hoisting out will
      // outweigh the benefits and it's better off to just always render it fresh.
      if (node.static && node.children.length && !(
        node.children.length === 1 &&
        node.children[0].type === 3
      )) {
        node.staticRoot = true;
        return
      } else {
        node.staticRoot = false;
      }
      if (node.children) {
        for (let i = 0, l = node.children.length; i < l; i++) {
          markStaticRoots(node.children[i], isInFor || !!node.for);
        }
      }
      if (node.ifConditions) {
        for (let i = 1, l = node.ifConditions.length; i < l; i++) {
          markStaticRoots(node.ifConditions[i].block, isInFor);
        }
      }
    }
  }

  function isStatic (node) {
    if (node.type === 2) { // expression
      return false
    }
    if (node.type === 3) { // text
      return true
    }
    return !!(node.pre || (
      !node.hasBindings && // no dynamic bindings
      !node.if && !node.for && // not v-if or v-for or v-else
      !isBuiltInTag(node.tag) && // not a built-in
      isPlatformReservedTag(node.tag) && // not a component
      !isDirectChildOfTemplateFor(node) &&
      Object.keys(node).every(isStaticKey)
    ))
  }

  function isDirectChildOfTemplateFor (node) {
    while (node.parent) {
      node = node.parent;
      if (node.tag !== 'template') {
        return false
      }
      if (node.for) {
        return true
      }
    }
    return false
  }

  /*  */

  const fnExpRE = /^([\w$_]+|\([^)]*?\))\s*=>|^function(?:\s+[\w$]+)?\s*\(/;
  const fnInvokeRE = /\([^)]*?\);*$/;
  const simplePathRE = /^[A-Za-z_$][\w$]*(?:\.[A-Za-z_$][\w$]*|\['[^']*?']|\["[^"]*?"]|\[\d+]|\[[A-Za-z_$][\w$]*])*$/;

  // KeyboardEvent.keyCode aliases
  const keyCodes = {
    esc: 27,
    tab: 9,
    enter: 13,
    space: 32,
    up: 38,
    left: 37,
    right: 39,
    down: 40,
    'delete': [8, 46]
  };

  // KeyboardEvent.key aliases
  const keyNames = {
    // #7880: IE11 and Edge use `Esc` for Escape key name.
    esc: ['Esc', 'Escape'],
    tab: 'Tab',
    enter: 'Enter',
    // #9112: IE11 uses `Spacebar` for Space key name.
    space: [' ', 'Spacebar'],
    // #7806: IE11 uses key names without `Arrow` prefix for arrow keys.
    up: ['Up', 'ArrowUp'],
    left: ['Left', 'ArrowLeft'],
    right: ['Right', 'ArrowRight'],
    down: ['Down', 'ArrowDown'],
    // #9112: IE11 uses `Del` for Delete key name.
    'delete': ['Backspace', 'Delete', 'Del']
  };

  // #4868: modifiers that prevent the execution of the listener
  // need to explicitly return null so that we can determine whether to remove
  // the listener for .once
  const genGuard = condition => `if(${condition})return null;`;

  const modifierCode = {
    stop: '$event.stopPropagation();',
    prevent: '$event.preventDefault();',
    self: genGuard(`$event.target !== $event.currentTarget`),
    ctrl: genGuard(`!$event.ctrlKey`),
    shift: genGuard(`!$event.shiftKey`),
    alt: genGuard(`!$event.altKey`),
    meta: genGuard(`!$event.metaKey`),
    left: genGuard(`'button' in $event && $event.button !== 0`),
    middle: genGuard(`'button' in $event && $event.button !== 1`),
    right: genGuard(`'button' in $event && $event.button !== 2`)
  };

  function genHandlers (
    events,
    isNative
  ) {
    const prefix = isNative ? 'nativeOn:' : 'on:';
    let staticHandlers = ``;
    let dynamicHandlers = ``;
    for (const name in events) {
      const handlerCode = genHandler(events[name]);
      if (events[name] && events[name].dynamic) {
        dynamicHandlers += `${name},${handlerCode},`;
      } else {
        staticHandlers += `"${name}":${handlerCode},`;
      }
    }
    staticHandlers = `{${staticHandlers.slice(0, -1)}}`;
    if (dynamicHandlers) {
      return prefix + `_d(${staticHandlers},[${dynamicHandlers.slice(0, -1)}])`
    } else {
      return prefix + staticHandlers
    }
  }

  function genHandler (handler) {
    if (!handler) {
      return 'function(){}'
    }

    if (Array.isArray(handler)) {
      return `[${handler.map(handler => genHandler(handler)).join(',')}]`
    }

    const isMethodPath = simplePathRE.test(handler.value);
    const isFunctionExpression = fnExpRE.test(handler.value);
    const isFunctionInvocation = simplePathRE.test(handler.value.replace(fnInvokeRE, ''));

    if (!handler.modifiers) {
      if (isMethodPath || isFunctionExpression) {
        return handler.value
      }
      return `function($event){${
      isFunctionInvocation ? `return ${handler.value}` : handler.value
    }}` // inline statement
    } else {
      let code = '';
      let genModifierCode = '';
      const keys = [];
      for (const key in handler.modifiers) {
        if (modifierCode[key]) {
          genModifierCode += modifierCode[key];
          // left/right
          if (keyCodes[key]) {
            keys.push(key);
          }
        } else if (key === 'exact') {
          const modifiers = (handler.modifiers);
          genModifierCode += genGuard(
            ['ctrl', 'shift', 'alt', 'meta']
              .filter(keyModifier => !modifiers[keyModifier])
              .map(keyModifier => `$event.${keyModifier}Key`)
              .join('||')
          );
        } else {
          keys.push(key);
        }
      }
      if (keys.length) {
        code += genKeyFilter(keys);
      }
      // Make sure modifiers like prevent and stop get executed after key filtering
      if (genModifierCode) {
        code += genModifierCode;
      }
      const handlerCode = isMethodPath
        ? `return ${handler.value}.apply(null, arguments)`
        : isFunctionExpression
          ? `return (${handler.value}).apply(null, arguments)`
          : isFunctionInvocation
            ? `return ${handler.value}`
            : handler.value;
      return `function($event){${code}${handlerCode}}`
    }
  }

  function genKeyFilter (keys) {
    return (
      // make sure the key filters only apply to KeyboardEvents
      // #9441: can't use 'keyCode' in $event because Chrome autofill fires fake
      // key events that do not have keyCode property...
      `if(!$event.type.indexOf('key')&&` +
      `${keys.map(genFilterCode).join('&&')})return null;`
    )
  }

  function genFilterCode (key) {
    const keyVal = parseInt(key, 10);
    if (keyVal) {
      return `$event.keyCode!==${keyVal}`
    }
    const keyCode = keyCodes[key];
    const keyName = keyNames[key];
    return (
      `_k($event.keyCode,` +
      `${JSON.stringify(key)},` +
      `${JSON.stringify(keyCode)},` +
      `$event.key,` +
      `${JSON.stringify(keyName)}` +
      `)`
    )
  }

  /*  */

  function on (el, dir) {
    if (dir.modifiers) {
      warn(`v-on without argument does not support modifiers.`);
    }
    el.wrapListeners = (code) => `_g(${code},${dir.value})`;
  }

  /*  */

  function bind$1 (el, dir) {
    el.wrapData = (code) => {
      return `_b(${code},'${el.tag}',${dir.value},${
      dir.modifiers && dir.modifiers.prop ? 'true' : 'false'
    }${
      dir.modifiers && dir.modifiers.sync ? ',true' : ''
    })`
    };
  }

  /*  */

  var baseDirectives = {
    on,
    bind: bind$1,
    cloak: noop
  };

  /*  */





  class CodegenState {
    
    
    
    
    
    
    
    
    

    constructor (options) {
      this.options = options;
      this.warn = options.warn || baseWarn;
      this.transforms = pluckModuleFunction(options.modules, 'transformCode');
      this.dataGenFns = pluckModuleFunction(options.modules, 'genData');
      this.directives = extend(extend({}, baseDirectives), options.directives);
      const isReservedTag = options.isReservedTag || no;
      this.maybeComponent = (el) => !!el.component || !isReservedTag(el.tag);
      this.onceId = 0;
      this.staticRenderFns = [];
      this.pre = false;
    }
  }



  function generate (
    ast,
    options
  ) {
    const state = new CodegenState(options);
    // fix #11483, Root level <script> tags should not be rendered.
    const code = ast ? (ast.tag === 'script' ? 'null' : genElement(ast, state)) : '_c("div")';
    return {
      render: `with(this){return ${code}}`,
      staticRenderFns: state.staticRenderFns
    }
  }

  function genElement (el, state) {
    if (el.parent) {
      el.pre = el.pre || el.parent.pre;
    }

    if (el.staticRoot && !el.staticProcessed) {
      return genStatic(el, state)
    } else if (el.once && !el.onceProcessed) {
      return genOnce(el, state)
    } else if (el.for && !el.forProcessed) {
      return genFor(el, state)
    } else if (el.if && !el.ifProcessed) {
      return genIf(el, state)
    } else if (el.tag === 'template' && !el.slotTarget && !state.pre) {
      return genChildren(el, state) || 'void 0'
    } else if (el.tag === 'slot') {
      return genSlot(el, state)
    } else {
      // component or element
      let code;
      if (el.component) {
        code = genComponent(el.component, el, state);
      } else {
        let data;
        if (!el.plain || (el.pre && state.maybeComponent(el))) {
          data = genData$2(el, state);
        }

        const children = el.inlineTemplate ? null : genChildren(el, state, true);
        code = `_c('${el.tag}'${
        data ? `,${data}` : '' // data
      }${
        children ? `,${children}` : '' // children
      })`;
      }
      // module transforms
      for (let i = 0; i < state.transforms.length; i++) {
        code = state.transforms[i](el, code);
      }
      return code
    }
  }

  // hoist static sub-trees out
  function genStatic (el, state) {
    el.staticProcessed = true;
    // Some elements (templates) need to behave differently inside of a v-pre
    // node.  All pre nodes are static roots, so we can use this as a location to
    // wrap a state change and reset it upon exiting the pre node.
    const originalPreState = state.pre;
    if (el.pre) {
      state.pre = el.pre;
    }
    state.staticRenderFns.push(`with(this){return ${genElement(el, state)}}`);
    state.pre = originalPreState;
    return `_m(${
    state.staticRenderFns.length - 1
  }${
    el.staticInFor ? ',true' : ''
  })`
  }

  // v-once
  function genOnce (el, state) {
    el.onceProcessed = true;
    if (el.if && !el.ifProcessed) {
      return genIf(el, state)
    } else if (el.staticInFor) {
      let key = '';
      let parent = el.parent;
      while (parent) {
        if (parent.for) {
          key = parent.key;
          break
        }
        parent = parent.parent;
      }
      if (!key) {
        state.warn(
          `v-once can only be used inside v-for that is keyed. `,
          el.rawAttrsMap['v-once']
        );
        return genElement(el, state)
      }
      return `_o(${genElement(el, state)},${state.onceId++},${key})`
    } else {
      return genStatic(el, state)
    }
  }

  function genIf (
    el,
    state,
    altGen,
    altEmpty
  ) {
    el.ifProcessed = true; // avoid recursion
    return genIfConditions(el.ifConditions.slice(), state, altGen, altEmpty)
  }

  function genIfConditions (
    conditions,
    state,
    altGen,
    altEmpty
  ) {
    if (!conditions.length) {
      return altEmpty || '_e()'
    }

    const condition = conditions.shift();
    if (condition.exp) {
      return `(${condition.exp})?${
      genTernaryExp(condition.block)
    }:${
      genIfConditions(conditions, state, altGen, altEmpty)
    }`
    } else {
      return `${genTernaryExp(condition.block)}`
    }

    // v-if with v-once should generate code like (a)?_m(0):_m(1)
    function genTernaryExp (el) {
      return altGen
        ? altGen(el, state)
        : el.once
          ? genOnce(el, state)
          : genElement(el, state)
    }
  }

  function genFor (
    el,
    state,
    altGen,
    altHelper
  ) {
    const exp = el.for;
    const alias = el.alias;
    const iterator1 = el.iterator1 ? `,${el.iterator1}` : '';
    const iterator2 = el.iterator2 ? `,${el.iterator2}` : '';

    if (state.maybeComponent(el) &&
      el.tag !== 'slot' &&
      el.tag !== 'template' &&
      !el.key
    ) {
      state.warn(
        `<${el.tag} v-for="${alias} in ${exp}">: component lists rendered with ` +
        `v-for should have explicit keys. ` +
        `See https://vuejs.org/guide/list.html#key for more info.`,
        el.rawAttrsMap['v-for'],
        true /* tip */
      );
    }

    el.forProcessed = true; // avoid recursion
    return `${altHelper || '_l'}((${exp}),` +
      `function(${alias}${iterator1}${iterator2}){` +
        `return ${(altGen || genElement)(el, state)}` +
      '})'
  }

  function genData$2 (el, state) {
    let data = '{';

    // directives first.
    // directives may mutate the el's other properties before they are generated.
    const dirs = genDirectives(el, state);
    if (dirs) data += dirs + ',';

    // key
    if (el.key) {
      data += `key:${el.key},`;
    }
    // ref
    if (el.ref) {
      data += `ref:${el.ref},`;
    }
    if (el.refInFor) {
      data += `refInFor:true,`;
    }
    // pre
    if (el.pre) {
      data += `pre:true,`;
    }
    // record original tag name for components using "is" attribute
    if (el.component) {
      data += `tag:"${el.tag}",`;
    }
    // module data generation functions
    for (let i = 0; i < state.dataGenFns.length; i++) {
      data += state.dataGenFns[i](el);
    }
    // attributes
    if (el.attrs) {
      data += `attrs:${genProps(el.attrs)},`;
    }
    // DOM props
    if (el.props) {
      data += `domProps:${genProps(el.props)},`;
    }
    // event handlers
    if (el.events) {
      data += `${genHandlers(el.events, false)},`;
    }
    if (el.nativeEvents) {
      data += `${genHandlers(el.nativeEvents, true)},`;
    }
    // slot target
    // only for non-scoped slots
    if (el.slotTarget && !el.slotScope) {
      data += `slot:${el.slotTarget},`;
    }
    // scoped slots
    if (el.scopedSlots) {
      data += `${genScopedSlots(el, el.scopedSlots, state)},`;
    }
    // component v-model
    if (el.model) {
      data += `model:{value:${
      el.model.value
    },callback:${
      el.model.callback
    },expression:${
      el.model.expression
    }},`;
    }
    // inline-template
    if (el.inlineTemplate) {
      const inlineTemplate = genInlineTemplate(el, state);
      if (inlineTemplate) {
        data += `${inlineTemplate},`;
      }
    }
    data = data.replace(/,$/, '') + '}';
    // v-bind dynamic argument wrap
    // v-bind with dynamic arguments must be applied using the same v-bind object
    // merge helper so that class/style/mustUseProp attrs are handled correctly.
    if (el.dynamicAttrs) {
      data = `_b(${data},"${el.tag}",${genProps(el.dynamicAttrs)})`;
    }
    // v-bind data wrap
    if (el.wrapData) {
      data = el.wrapData(data);
    }
    // v-on data wrap
    if (el.wrapListeners) {
      data = el.wrapListeners(data);
    }
    return data
  }

  function genDirectives (el, state) {
    const dirs = el.directives;
    if (!dirs) return
    let res = 'directives:[';
    let hasRuntime = false;
    let i, l, dir, needRuntime;
    for (i = 0, l = dirs.length; i < l; i++) {
      dir = dirs[i];
      needRuntime = true;
      const gen = state.directives[dir.name];
      if (gen) {
        // compile-time directive that manipulates AST.
        // returns true if it also needs a runtime counterpart.
        needRuntime = !!gen(el, dir, state.warn);
      }
      if (needRuntime) {
        hasRuntime = true;
        res += `{name:"${dir.name}",rawName:"${dir.rawName}"${
        dir.value ? `,value:(${dir.value}),expression:${JSON.stringify(dir.value)}` : ''
      }${
        dir.arg ? `,arg:${dir.isDynamicArg ? dir.arg : `"${dir.arg}"`}` : ''
      }${
        dir.modifiers ? `,modifiers:${JSON.stringify(dir.modifiers)}` : ''
      }},`;
      }
    }
    if (hasRuntime) {
      return res.slice(0, -1) + ']'
    }
  }

  function genInlineTemplate (el, state) {
    const ast = el.children[0];
    if (el.children.length !== 1 || ast.type !== 1) {
      state.warn(
        'Inline-template components must have exactly one child element.',
        { start: el.start }
      );
    }
    if (ast && ast.type === 1) {
      const inlineRenderFns = generate(ast, state.options);
      return `inlineTemplate:{render:function(){${
      inlineRenderFns.render
    }},staticRenderFns:[${
      inlineRenderFns.staticRenderFns.map(code => `function(){${code}}`).join(',')
    }]}`
    }
  }

  function genScopedSlots (
    el,
    slots,
    state
  ) {
    // by default scoped slots are considered "stable", this allows child
    // components with only scoped slots to skip forced updates from parent.
    // but in some cases we have to bail-out of this optimization
    // for example if the slot contains dynamic names, has v-if or v-for on them...
    let needsForceUpdate = el.for || Object.keys(slots).some(key => {
      const slot = slots[key];
      return (
        slot.slotTargetDynamic ||
        slot.if ||
        slot.for ||
        containsSlotChild(slot) // is passing down slot from parent which may be dynamic
      )
    });

    // #9534: if a component with scoped slots is inside a conditional branch,
    // it's possible for the same component to be reused but with different
    // compiled slot content. To avoid that, we generate a unique key based on
    // the generated code of all the slot contents.
    let needsKey = !!el.if;

    // OR when it is inside another scoped slot or v-for (the reactivity may be
    // disconnected due to the intermediate scope variable)
    // #9438, #9506
    // TODO: this can be further optimized by properly analyzing in-scope bindings
    // and skip force updating ones that do not actually use scope variables.
    if (!needsForceUpdate) {
      let parent = el.parent;
      while (parent) {
        if (
          (parent.slotScope && parent.slotScope !== emptySlotScopeToken) ||
          parent.for
        ) {
          needsForceUpdate = true;
          break
        }
        if (parent.if) {
          needsKey = true;
        }
        parent = parent.parent;
      }
    }

    const generatedSlots = Object.keys(slots)
      .map(key => genScopedSlot(slots[key], state))
      .join(',');

    return `scopedSlots:_u([${generatedSlots}]${
    needsForceUpdate ? `,null,true` : ``
  }${
    !needsForceUpdate && needsKey ? `,null,false,${hash(generatedSlots)}` : ``
  })`
  }

  function hash(str) {
    let hash = 5381;
    let i = str.length;
    while(i) {
      hash = (hash * 33) ^ str.charCodeAt(--i);
    }
    return hash >>> 0
  }

  function containsSlotChild (el) {
    if (el.type === 1) {
      if (el.tag === 'slot') {
        return true
      }
      return el.children.some(containsSlotChild)
    }
    return false
  }

  function genScopedSlot (
    el,
    state
  ) {
    const isLegacySyntax = el.attrsMap['slot-scope'];
    if (el.if && !el.ifProcessed && !isLegacySyntax) {
      return genIf(el, state, genScopedSlot, `null`)
    }
    if (el.for && !el.forProcessed) {
      return genFor(el, state, genScopedSlot)
    }
    const slotScope = el.slotScope === emptySlotScopeToken
      ? ``
      : String(el.slotScope);
    const fn = `function(${slotScope}){` +
      `return ${el.tag === 'template'
      ? el.if && isLegacySyntax
        ? `(${el.if})?${genChildren(el, state) || 'undefined'}:undefined`
        : genChildren(el, state) || 'undefined'
      : genElement(el, state)
    }}`;
    // reverse proxy v-slot without scope on this.$slots
    const reverseProxy = slotScope ? `` : `,proxy:true`;
    return `{key:${el.slotTarget || `"default"`},fn:${fn}${reverseProxy}}`
  }

  function genChildren (
    el,
    state,
    checkSkip,
    altGenElement,
    altGenNode
  ) {
    const children = el.children;
    if (children.length) {
      const el = children[0];
      // optimize single v-for
      if (children.length === 1 &&
        el.for &&
        el.tag !== 'template' &&
        el.tag !== 'slot'
      ) {
        const normalizationType = checkSkip
          ? state.maybeComponent(el) ? `,1` : `,0`
          : ``;
        return `${(altGenElement || genElement)(el, state)}${normalizationType}`
      }
      const normalizationType = checkSkip
        ? getNormalizationType(children, state.maybeComponent)
        : 0;
      const gen = altGenNode || genNode;
      return `[${children.map(c => gen(c, state)).join(',')}]${
      normalizationType ? `,${normalizationType}` : ''
    }`
    }
  }

  // determine the normalization needed for the children array.
  // 0: no normalization needed
  // 1: simple normalization needed (possible 1-level deep nested array)
  // 2: full normalization needed
  function getNormalizationType (
    children,
    maybeComponent
  ) {
    let res = 0;
    for (let i = 0; i < children.length; i++) {
      const el = children[i];
      if (el.type !== 1) {
        continue
      }
      if (needsNormalization(el) ||
          (el.ifConditions && el.ifConditions.some(c => needsNormalization(c.block)))) {
        res = 2;
        break
      }
      if (maybeComponent(el) ||
          (el.ifConditions && el.ifConditions.some(c => maybeComponent(c.block)))) {
        res = 1;
      }
    }
    return res
  }

  function needsNormalization (el) {
    return el.for !== undefined || el.tag === 'template' || el.tag === 'slot'
  }

  function genNode (node, state) {
    if (node.type === 1) {
      return genElement(node, state)
    } else if (node.type === 3 && node.isComment) {
      return genComment(node)
    } else {
      return genText(node)
    }
  }

  function genText (text) {
    return `_v(${text.type === 2
    ? text.expression // no need for () because already wrapped in _s()
    : transformSpecialNewlines(JSON.stringify(text.text))
  })`
  }

  function genComment (comment) {
    return `_e(${JSON.stringify(comment.text)})`
  }

  function genSlot (el, state) {
    const slotName = el.slotName || '"default"';
    const children = genChildren(el, state);
    let res = `_t(${slotName}${children ? `,function(){return ${children}}` : ''}`;
    const attrs = el.attrs || el.dynamicAttrs
      ? genProps((el.attrs || []).concat(el.dynamicAttrs || []).map(attr => ({
          // slot props are camelized
          name: camelize(attr.name),
          value: attr.value,
          dynamic: attr.dynamic
        })))
      : null;
    const bind$$1 = el.attrsMap['v-bind'];
    if ((attrs || bind$$1) && !children) {
      res += `,null`;
    }
    if (attrs) {
      res += `,${attrs}`;
    }
    if (bind$$1) {
      res += `${attrs ? '' : ',null'},${bind$$1}`;
    }
    return res + ')'
  }

  // componentName is el.component, take it as argument to shun flow's pessimistic refinement
  function genComponent (
    componentName,
    el,
    state
  ) {
    const children = el.inlineTemplate ? null : genChildren(el, state, true);
    return `_c(${componentName},${genData$2(el, state)}${
    children ? `,${children}` : ''
  })`
  }

  function genProps (props) {
    let staticProps = ``;
    let dynamicProps = ``;
    for (let i = 0; i < props.length; i++) {
      const prop = props[i];
      const value = transformSpecialNewlines(prop.value);
      if (prop.dynamic) {
        dynamicProps += `${prop.name},${value},`;
      } else {
        staticProps += `"${prop.name}":${value},`;
      }
    }
    staticProps = `{${staticProps.slice(0, -1)}}`;
    if (dynamicProps) {
      return `_d(${staticProps},[${dynamicProps.slice(0, -1)}])`
    } else {
      return staticProps
    }
  }

  // #3895, #4268
  function transformSpecialNewlines (text) {
    return text
      .replace(/\u2028/g, '\\u2028')
      .replace(/\u2029/g, '\\u2029')
  }

  /*  */



  // these keywords should not appear inside expressions, but operators like
  // typeof, instanceof and in are allowed
  const prohibitedKeywordRE = new RegExp('\\b' + (
    'do,if,for,let,new,try,var,case,else,with,await,break,catch,class,const,' +
    'super,throw,while,yield,delete,export,import,return,switch,default,' +
    'extends,finally,continue,debugger,function,arguments'
  ).split(',').join('\\b|\\b') + '\\b');

  // these unary operators should not be used as property/method names
  const unaryOperatorsRE = new RegExp('\\b' + (
    'delete,typeof,void'
  ).split(',').join('\\s*\\([^\\)]*\\)|\\b') + '\\s*\\([^\\)]*\\)');

  // strip strings in expressions
  const stripStringRE = /'(?:[^'\\]|\\.)*'|"(?:[^"\\]|\\.)*"|`(?:[^`\\]|\\.)*\$\{|\}(?:[^`\\]|\\.)*`|`(?:[^`\\]|\\.)*`/g;

  // detect problematic expressions in a template
  function detectErrors (ast, warn) {
    if (ast) {
      checkNode(ast, warn);
    }
  }

  function checkNode (node, warn) {
    if (node.type === 1) {
      for (const name in node.attrsMap) {
        if (dirRE.test(name)) {
          const value = node.attrsMap[name];
          if (value) {
            const range = node.rawAttrsMap[name];
            if (name === 'v-for') {
              checkFor(node, `v-for="${value}"`, warn, range);
            } else if (name === 'v-slot' || name[0] === '#') {
              checkFunctionParameterExpression(value, `${name}="${value}"`, warn, range);
            } else if (onRE.test(name)) {
              checkEvent(value, `${name}="${value}"`, warn, range);
            } else {
              checkExpression(value, `${name}="${value}"`, warn, range);
            }
          }
        }
      }
      if (node.children) {
        for (let i = 0; i < node.children.length; i++) {
          checkNode(node.children[i], warn);
        }
      }
    } else if (node.type === 2) {
      checkExpression(node.expression, node.text, warn, node);
    }
  }

  function checkEvent (exp, text, warn, range) {
    const stripped = exp.replace(stripStringRE, '');
    const keywordMatch = stripped.match(unaryOperatorsRE);
    if (keywordMatch && stripped.charAt(keywordMatch.index - 1) !== '$') {
      warn(
        `avoid using JavaScript unary operator as property name: ` +
        `"${keywordMatch[0]}" in expression ${text.trim()}`,
        range
      );
    }
    checkExpression(exp, text, warn, range);
  }

  function checkFor (node, text, warn, range) {
    checkExpression(node.for || '', text, warn, range);
    checkIdentifier(node.alias, 'v-for alias', text, warn, range);
    checkIdentifier(node.iterator1, 'v-for iterator', text, warn, range);
    checkIdentifier(node.iterator2, 'v-for iterator', text, warn, range);
  }

  function checkIdentifier (
    ident,
    type,
    text,
    warn,
    range
  ) {
    if (typeof ident === 'string') {
      try {
        new Function(`var ${ident}=_`);
      } catch (e) {
        warn(`invalid ${type} "${ident}" in expression: ${text.trim()}`, range);
      }
    }
  }

  function checkExpression (exp, text, warn, range) {
    try {
      new Function(`return ${exp}`);
    } catch (e) {
      const keywordMatch = exp.replace(stripStringRE, '').match(prohibitedKeywordRE);
      if (keywordMatch) {
        warn(
          `avoid using JavaScript keyword as property name: ` +
          `"${keywordMatch[0]}"\n  Raw expression: ${text.trim()}`,
          range
        );
      } else {
        warn(
          `invalid expression: ${e.message} in\n\n` +
          `    ${exp}\n\n` +
          `  Raw expression: ${text.trim()}\n`,
          range
        );
      }
    }
  }

  function checkFunctionParameterExpression (exp, text, warn, range) {
    try {
      new Function(exp, '');
    } catch (e) {
      warn(
        `invalid function parameter expression: ${e.message} in\n\n` +
        `    ${exp}\n\n` +
        `  Raw expression: ${text.trim()}\n`,
        range
      );
    }
  }

  /*  */

  const range = 2;

  function generateCodeFrame (
    source,
    start = 0,
    end = source.length
  ) {
    const lines = source.split(/\r?\n/);
    let count = 0;
    const res = [];
    for (let i = 0; i < lines.length; i++) {
      count += lines[i].length + 1;
      if (count >= start) {
        for (let j = i - range; j <= i + range || end > count; j++) {
          if (j < 0 || j >= lines.length) continue
          res.push(`${j + 1}${repeat(` `, 3 - String(j + 1).length)}|  ${lines[j]}`);
          const lineLength = lines[j].length;
          if (j === i) {
            // push underline
            const pad = start - (count - lineLength) + 1;
            const length = end > count ? lineLength - pad : end - start;
            res.push(`   |  ` + repeat(` `, pad) + repeat(`^`, length));
          } else if (j > i) {
            if (end > count) {
              const length = Math.min(end - count, lineLength);
              res.push(`   |  ` + repeat(`^`, length));
            }
            count += lineLength + 1;
          }
        }
        break
      }
    }
    return res.join('\n')
  }

  function repeat (str, n) {
    let result = '';
    if (n > 0) {
      while (true) { // eslint-disable-line
        if (n & 1) result += str;
        n >>>= 1;
        if (n <= 0) break
        str += str;
      }
    }
    return result
  }

  /*  */



  function createFunction (code, errors) {
    try {
      return new Function(code)
    } catch (err) {
      errors.push({ err, code });
      return noop
    }
  }

  function createCompileToFunctionFn (compile) {
    const cache = Object.create(null);

    return function compileToFunctions (
      template,
      options,
      vm
    ) {
      options = extend({}, options);
      const warn$$1 = options.warn || warn;
      delete options.warn;

      /* istanbul ignore if */
      {
        // detect possible CSP restriction
        try {
          new Function('return 1');
        } catch (e) {
          if (e.toString().match(/unsafe-eval|CSP/)) {
            warn$$1(
              'It seems you are using the standalone build of Vue.js in an ' +
              'environment with Content Security Policy that prohibits unsafe-eval. ' +
              'The template compiler cannot work in this environment. Consider ' +
              'relaxing the policy to allow unsafe-eval or pre-compiling your ' +
              'templates into render functions.'
            );
          }
        }
      }

      // check cache
      const key = options.delimiters
        ? String(options.delimiters) + template
        : template;
      if (cache[key]) {
        return cache[key]
      }

      // compile
      const compiled = compile(template, options);

      // check compilation errors/tips
      {
        if (compiled.errors && compiled.errors.length) {
          if (options.outputSourceRange) {
            compiled.errors.forEach(e => {
              warn$$1(
                `Error compiling template:\n\n${e.msg}\n\n` +
                generateCodeFrame(template, e.start, e.end),
                vm
              );
            });
          } else {
            warn$$1(
              `Error compiling template:\n\n${template}\n\n` +
              compiled.errors.map(e => `- ${e}`).join('\n') + '\n',
              vm
            );
          }
        }
        if (compiled.tips && compiled.tips.length) {
          if (options.outputSourceRange) {
            compiled.tips.forEach(e => tip(e.msg, vm));
          } else {
            compiled.tips.forEach(msg => tip(msg, vm));
          }
        }
      }

      // turn code into functions
      const res = {};
      const fnGenErrors = [];
      res.render = createFunction(compiled.render, fnGenErrors);
      res.staticRenderFns = compiled.staticRenderFns.map(code => {
        return createFunction(code, fnGenErrors)
      });

      // check function generation errors.
      // this should only happen if there is a bug in the compiler itself.
      // mostly for codegen development use
      /* istanbul ignore if */
      {
        if ((!compiled.errors || !compiled.errors.length) && fnGenErrors.length) {
          warn$$1(
            `Failed to generate render function:\n\n` +
            fnGenErrors.map(({ err, code }) => `${err.toString()} in\n\n${code}\n`).join('\n'),
            vm
          );
        }
      }

      return (cache[key] = res)
    }
  }

  /*  */

  function createCompilerCreator (baseCompile) {
    return function createCompiler (baseOptions) {
      function compile (
        template,
        options
      ) {
        const finalOptions = Object.create(baseOptions);
        const errors = [];
        const tips = [];

        let warn = (msg, range, tip) => {
          (tip ? tips : errors).push(msg);
        };

        if (options) {
          if (options.outputSourceRange) {
            // $flow-disable-line
            const leadingSpaceLength = template.match(/^\s*/)[0].length;

            warn = (msg, range, tip) => {
              const data = { msg };
              if (range) {
                if (range.start != null) {
                  data.start = range.start + leadingSpaceLength;
                }
                if (range.end != null) {
                  data.end = range.end + leadingSpaceLength;
                }
              }
              (tip ? tips : errors).push(data);
            };
          }
          // merge custom modules
          if (options.modules) {
            finalOptions.modules =
              (baseOptions.modules || []).concat(options.modules);
          }
          // merge custom directives
          if (options.directives) {
            finalOptions.directives = extend(
              Object.create(baseOptions.directives || null),
              options.directives
            );
          }
          // copy other options
          for (const key in options) {
            if (key !== 'modules' && key !== 'directives') {
              finalOptions[key] = options[key];
            }
          }
        }

        finalOptions.warn = warn;

        const compiled = baseCompile(template.trim(), finalOptions);
        {
          detectErrors(compiled.ast, warn);
        }
        compiled.errors = errors;
        compiled.tips = tips;
        return compiled
      }

      return {
        compile,
        compileToFunctions: createCompileToFunctionFn(compile)
      }
    }
  }

  /*  */

  // `createCompilerCreator` allows creating compilers that use alternative
  // parser/optimizer/codegen, e.g the SSR optimizing compiler.
  // Here we just export a default compiler using the default parts.
  const createCompiler = createCompilerCreator(function baseCompile (
    template,
    options
  ) {
    const ast = parse(template.trim(), options);
    if (options.optimize !== false) {
      optimize(ast, options);
    }
    const code = generate(ast, options);
    return {
      ast,
      render: code.render,
      staticRenderFns: code.staticRenderFns
    }
  });

  /*  */

  const { compile, compileToFunctions } = createCompiler(baseOptions);

  /*  */

  // check whether current browser encodes a char inside attribute values
  let div;
  function getShouldDecode (href) {
    div = div || document.createElement('div');
    div.innerHTML = href ? `<a href="\n"/>` : `<div a="\n"/>`;
    return div.innerHTML.indexOf('&#10;') > 0
  }

  // #3663: IE encodes newlines inside attribute values while other browsers don't
  const shouldDecodeNewlines = inBrowser ? getShouldDecode(false) : false;
  // #6828: chrome encodes content in a[href]
  const shouldDecodeNewlinesForHref = inBrowser ? getShouldDecode(true) : false;

  /*  */

  const idToTemplate = cached(id => {
    const el = query(id);
    return el && el.innerHTML
  });

  const mount = Vue.prototype.$mount;
  Vue.prototype.$mount = function (
    el,
    hydrating
  ) {
    el = el && query(el);

    /* istanbul ignore if */
    if (el === document.body || el === document.documentElement) {
      warn(
        `Do not mount Vue to <html> or <body> - mount to normal elements instead.`
      );
      return this
    }

    const options = this.$options;
    // resolve template/el and convert to render function
    if (!options.render) {
      let template = options.template;
      if (template) {
        if (typeof template === 'string') {
          if (template.charAt(0) === '#') {
            template = idToTemplate(template);
            /* istanbul ignore if */
            if (!template) {
              warn(
                `Template element not found or is empty: ${options.template}`,
                this
              );
            }
          }
        } else if (template.nodeType) {
          template = template.innerHTML;
        } else {
          {
            warn('invalid template option:' + template, this);
          }
          return this
        }
      } else if (el) {
        template = getOuterHTML(el);
      }
      if (template) {
        /* istanbul ignore if */
        if (config.performance && mark) {
          mark('compile');
        }

        const { render, staticRenderFns } = compileToFunctions(template, {
          outputSourceRange: "development" !== 'production',
          shouldDecodeNewlines,
          shouldDecodeNewlinesForHref,
          delimiters: options.delimiters,
          comments: options.comments
        }, this);
        options.render = render;
        options.staticRenderFns = staticRenderFns;

        /* istanbul ignore if */
        if (config.performance && mark) {
          mark('compile end');
          measure(`vue ${this._name} compile`, 'compile', 'compile end');
        }
      }
    }
    return mount.call(this, el, hydrating)
  };

  /**
   * Get outerHTML of elements, taking care
   * of SVG elements in IE as well.
   */
  function getOuterHTML (el) {
    if (el.outerHTML) {
      return el.outerHTML
    } else {
      const container = document.createElement('div');
      container.appendChild(el.cloneNode(true));
      return container.innerHTML
    }
  }

  Vue.compile = compileToFunctions;

  // ===========================================================================
  function main(staticPrefix, url, prefix, timestamp, logoUrl, navbarBackground, navbarColor, navbarLightButtons, locale, allLocales, i18nStrings, logoHomeUrl) {
    PywbI18N.init(locale, i18nStrings);
    new CDXLoader(staticPrefix, url, prefix, timestamp, logoUrl, navbarBackground, navbarColor, navbarLightButtons, allLocales, logoHomeUrl);
  }

  // ===========================================================================
  class CDXLoader {
    constructor(staticPrefix, url, prefix, timestamp, logoUrl, navbarBackground, navbarColor, navbarLightButtons, allLocales, logoHomeUrl) {
      this.loadingSpinner = null;
      this.loaded = false;
      this.opts = {};
      this.prefix = prefix;
      this.staticPrefix = staticPrefix;
      this.logoUrl = logoUrl;
      this.logoHomeUrl = logoHomeUrl;
      this.navbarBackground = navbarBackground;
      this.navbarColor = navbarColor;
      this.navbarLightButtons = navbarLightButtons;
      this.timestamp = timestamp;

      this.isReplay = (timestamp !== undefined);

      setTimeout(() => {
        if (!this.loaded) {
          this.loadingSpinner = new LoadingSpinner({text: PywbI18N.instance?.getText('Loading...'), isSmall: !!timestamp}); // bootstrap loading-spinner EARLY ON
          this.loadingSpinner.setOn();
        }
      }, 500);

      if (this.isReplay) {
        window.WBBanner = new VueBannerWrapper(this, url, timestamp);
      }

      let queryURL;

      // query form *?=url...
      if (window.location.href.indexOf("*?") > 0) {
        queryURL = window.location.href.replace("*?", "cdx?") + "&output=json";
        url = new URL(queryURL).searchParams.get("url");

      // otherwise, traditional calendar form /*/<url>
      } else if (url) {
        const params = new URLSearchParams();
        params.set("url", url);
        params.set("output", "json");
        queryURL = prefix + "cdx?" + params.toString();

      // otherwise, an error since no URL
      } else {
        throw new Error("No query URL specified");
      }

      const logoImg = this.staticPrefix + "/" + (this.logoUrl ? this.logoUrl : "pywb-logo-sm.png");

      this.app = this.initApp({logoImg, logoHomeUrl, navbarBackground, navbarColor, navbarLightButtons, url, allLocales, timestamp});

      this.loadCDX(queryURL).then((cdxList) => {
        this.setAppData(cdxList, url, this.timestamp);
      });
    }

    initApp(config = {}) {
      const app = new Vue(__vue_component__);

      app.$set(app, "config", {...app.config, ...config, prefix: this.prefix});

      app.$mount("#app");

      app.$on("show-snapshot", (snapshot) => this.loadSnapshot(snapshot));
      app.$on("data-set-and-render-completed", () => {
        if (this.loadingSpinner) {
          this.loadingSpinner.setOff(); // only turn off loading-spinner AFTER app has told us it is DONE DONE
        }
        this.loaded = true;
      });

      return app;
    }

    async updateSnapshot(url, timestamp) {
      const params = new URLSearchParams();
      params.set("url", url);
      params.set("output", "json");
      const queryURL = this.prefix + "cdx?" + params.toString();

      const cdxList = await this.loadCDX(queryURL);

      this.setAppData(cdxList, url, timestamp);
    }

    async updateTimestamp(url, timestamp) {
      this.timestamp = timestamp;

      if (this.cdxLoading) {
        return;
      }

      this.app.setSnapshot({url, timestamp});
    }

    setAppData(cdxList, url, timestamp) {
      this.app.setData(new PywbData(cdxList));

      this.app.initBannerState(this.isReplay);

      // if set on initial load, may not have timestamp yet
      // will be updated later
      if (timestamp) {
        this.updateTimestamp(url, timestamp);
      }
    }

    async loadCDX(queryURL) {
      //  this.loadingSpinner.setOn(); // start loading-spinner when CDX loading begins
      this.cdxLoading = true;
      const queryWorker = new Worker(this.staticPrefix + "/queryWorker.js");

      const p = new Promise((resolve) => {
        const cdxList = [];

        queryWorker.addEventListener("message", (event) => {
          const data = event.data;
          switch (data.type) {
          case "cdxRecord":
            cdxList.push(data.record);
            break;

          case "finished":
            this.cdxLoading = false;
            resolve(cdxList);
            break;
          }
        });
      });

      queryWorker.postMessage({
        type: "query",
        queryURL
      });

      const results = await p;

      queryWorker.terminate();
      //delete queryWorker;

      return results;
    }

    loadSnapshot(snapshot, reloadIFrame=true) {
      if (!this.isReplay) {
        window.location.href = this.prefix + snapshot.id + "/" + snapshot.url;
      } else if (window.cframe) {
        const ts = snapshot.id + "";
        if (ts !== this.timestamp) {
          window.cframe.load_url(snapshot.url, ts, reloadIFrame);
        }
      }
    }
  }


  // ===========================================================================
  class VueBannerWrapper
  {
    constructor(loader, url, ts) {
      this.loading = true;
      this.lastSurt = this.getSurt(url);
      this.lastTs = ts;
      this.loader = loader;
    }

    init() {}

    stillIndicatesLoading() {
      return this.loading;
    }

    updateCaptureInfo(/*url, ts, is_live*/) {
      this.loading = false;
    }

    onMessage(event) {
      const type = event.data.wb_type;

      if (type === "load" || type === "replace-url") {
        const surt = this.getSurt(event.data.url);

        if (event.data.title) {
          this.loader.app.updateTitle(event.data.title);
        }

        if (surt !== this.lastSurt) {
          this.loader.updateSnapshot(event.data.url, event.data.ts);
          this.lastSurt = surt;
        } else if (event.data.ts !== this.lastTs) {
          this.loader.updateTimestamp(event.data.url, event.data.ts);
          this.lastTs = event.data.ts;
        }
      }
    }

    getSurt(url) {
      try {
        if (!url.startsWith("https:") && !url.startsWith("http:")) {
          return url;
        }
        url = url.replace(/^(https?:\/\/)www\d*\./, "$1");
        const urlObj = new URL(url.toLowerCase());

        const hostParts = urlObj.hostname.split(".").reverse();
        let surt = hostParts.join(",");
        if (urlObj.port) {
          surt += ":" + urlObj.port;
        }
        surt += ")";
        surt += urlObj.pathname;
        if (urlObj.search) {
          urlObj.searchParams.sort();
          surt += urlObj.search;
        }
        return surt;
      } catch (e) {
        return url;
      }
    }
  }

  exports.main = main;

  Object.defineProperty(exports, '__esModule', { value: true });

  return exports;

}({}));
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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