"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CouchChanges = void 0;
const buffer = require("buffer");
const events_1 = require("events");
const https_1 = require("https");
const url_1 = require("url");
const zlib_1 = require("zlib");
/**
 * The maximum length of a response body we can accept that we will be able to
 * successfully parse out. There is a limit in node on how big a string instance
 * can be, and `JSON.parse` only operates on string instances. If we get too
 * many unicode code-points, we will simply not be able to parse the result out,
 * and will thow an error with an appropriate error message.
 */
const MAX_DATA_LENGTH = Math.min(
// NOTE: MAX_STRING_LENGTH is expressed in UTF-16 code units. We expect to receive JSON
// which ultimately is UTF-8-encoded text. An UTF-16 code unit is 32 bits, but this
// maps to between 8 and 32 bits in UTF-8... Consequently, we can't allow more than
// MAX_StRING_LENGTH / 4 if we want to be certain we can stringify the buffer.
Math.floor(buffer.constants.MAX_STRING_LENGTH / 4), 
// This is the maximum length of a Buffer instance. We cannot technically accept any
// more data than this... so we won't even be trying to.
buffer.constants.MAX_LENGTH);
/**
 * A utility class that helps with traversing CouchDB database changes streams
 * in a promise-based, page-by-page manner.
 */
class CouchChanges extends events_1.EventEmitter {
    /**
     * @param baseUrl  the CouchDB endpoint URL.
     * @param database the name of the database for which changes are fetched.
     */
    constructor(baseUrl, database) {
        super();
        // Setting up for keep-alive connections.
        this.agent = new https_1.Agent({
            keepAlive: true,
            keepAliveMsecs: 5000,
            maxSockets: 4,
            timeout: 60000,
        });
        this.baseUrl = new url_1.URL(database, baseUrl);
    }
    /**
     * @returns summary informations about the database.
     */
    async info() {
        return await this.https('get', this.baseUrl);
    }
    /**
     * Obtains a batch of changes from the database.
     *
     * @param since     the sequence value since when history should be fetched.
     * @param batchSize the maximum amount of changes to return in a single page.
     *
     * @returns a page of changes.
     */
    async changes(since, opts) {
        var _a;
        const batchSize = (_a = opts === null || opts === void 0 ? void 0 : opts.batchSize) !== null && _a !== void 0 ? _a : 100;
        const changesUrl = new url_1.URL('_changes', this.baseUrl);
        changesUrl.searchParams.set('include_docs', 'true');
        changesUrl.searchParams.set('limit', batchSize.toFixed());
        changesUrl.searchParams.set('selector', '_filter');
        changesUrl.searchParams.set('seq_interval', batchSize.toFixed());
        changesUrl.searchParams.set('since', since.toString());
        changesUrl.searchParams.set('timeout', '20000' /* ms */);
        const filter = { name: { $gt: null } };
        try {
            return await this.https('post', changesUrl, filter);
        }
        catch (err) {
            if (err instanceof ResponseTooLargeError && batchSize > 1) {
                // The response was too large, try again, but with a smaller batch size
                const smallerBatchSize = Math.max(1, Math.floor(batchSize / 2));
                console.log(`Batch of ${batchSize} from ${since} was too large... Trying again with batch size of ${smallerBatchSize}`);
                return this.changes(since, { ...opts, batchSize: smallerBatchSize });
            }
            // Else simply forward the error out again...
            return Promise.reject(err);
        }
    }
    /**
     * Makes an HTTPs request using the provided method, url, and optionally payload. This function
     * properly handles input that is received with `Content-Type: gzip` and automatically retries
     * typical transient errors (HTTP 5XX, ECONNRESET, etc...) with linear back-off and no maximum
     * retry count (this is used in Lambda functions, which de-facto caps the amount of attempts
     * that will be made due to the function time out).
     *
     * @param method the HTTP method used for the request (e.g: 'get', 'post', ...).
     * @param url    the URL to request.
     * @param body   an optional HTTP request payload, which will be JSON-encoded.
     *
     * @param attempt the request attempt number (used to determine back-off / retry).
     *
     * @returns the JSON-decoded response body.
     */
    https(method, url, body, attempt = 1) {
        return new Promise((ok, ko) => {
            const retry = () => setTimeout(() => {
                console.log(`Retrying ${method.toUpperCase()} ${url}`);
                this.https(method, url, body, attempt + 1).then(ok, ko);
            }, Math.min(500 * attempt, 5000));
            const headers = {
                'Accept': 'application/json',
                'Accept-Encoding': 'gzip',
            };
            if (body) {
                headers['Content-Type'] = 'application/json';
            }
            console.log(`Request: ${method.toUpperCase()} ${url}`);
            const req = https_1.request(url, {
                agent: this.agent,
                headers,
                method,
                port: 443,
                servername: url.hostname,
            }, (res) => {
                if (res.statusCode == null) {
                    const error = new Error(`[FATAL] Request failed: ${method.toUpperCase()} ${url}`);
                    Error.captureStackTrace(error);
                    return ko(error);
                }
                console.log(`Response: ${method.toUpperCase()} ${url} => HTTP ${res.statusCode} (${res.statusMessage})`);
                // Transient (server) errors:
                if (res.statusCode >= 500 && res.statusCode < 600) {
                    console.error(`[RETRYABLE] HTTP ${res.statusCode} (${res.statusMessage}) - ${method.toUpperCase()} ${url}`);
                    // Call again after a short back-off
                    return retry();
                }
                // Permanent (client) errors:
                if (res.statusCode >= 400 && res.statusCode < 500) {
                    const error = new Error(`[FATAL] HTTP ${res.statusCode} (${res.statusMessage}) - ${method.toUpperCase()} ${url}`);
                    Error.captureStackTrace(error);
                    return ko(error);
                }
                res.once('error', (err) => {
                    // Don't call the `close` handler - we are reporting failure or retrying in a new request here...
                    res.removeAllListeners('close');
                    if (err.code === 'ECONNRESET') {
                        // Transient networking problem?
                        console.error(`[RETRYABLE] ${err.code} - ${method.toUpperCase()} ${url}`);
                        retry();
                    }
                    else {
                        ko(err);
                    }
                });
                // We'll increase the buffer size by at least this much if the buffer we currently have is
                // not large enough. CouchDB typically sends response in `chunked` encoding, so we don't
                // know the full length of the response in advance... We hence try to avoid doing too many
                // expensive copies if the response is very large (it can be!).
                const bufferIncrements = 1048576; /* 1MiB */
                let data = Buffer.alloc(typeof res.headers['content-length'] === 'string'
                    ? Number.parseInt(res.headers['content-length'])
                    : bufferIncrements);
                let dataLength = 0;
                res.on('data', (chunk) => {
                    const chunkBuffer = chunk = Buffer.from(chunk);
                    // Check if we still have capacity to accept that data without going too large to parse...
                    if (dataLength + chunkBuffer.length > MAX_DATA_LENGTH) {
                        console.log(`Response too large (> ${MAX_DATA_LENGTH}), aborting: ${method.toUpperCase()} ${url}`);
                        // We won't be able to stringify this... no point in continuing. Calling destroy with
                        // an error will cause the `error` event to be emitted, then the `close` event will be
                        // emitted. Any outstanding data will be dropped, the socket will be destroyed.
                        req.destroy(new ResponseTooLargeError(method, url));
                        return;
                    }
                    if (dataLength + chunkBuffer.length > data.length) {
                        // Buffer is not large enough, extend it to fit new data...
                        const existing = data;
                        data = Buffer.alloc(Math.min(data.length + Math.max(chunkBuffer.length, bufferIncrements), buffer.constants.MAX_LENGTH));
                        existing.copy(data);
                    }
                    chunkBuffer.copy(data, dataLength);
                    dataLength += chunkBuffer.length;
                });
                res.once('close', () => {
                    // Ensure buffer is trimmed to correct length.
                    data = data.subarray(0, dataLength);
                    try {
                        if (res.headers['content-encoding'] === 'gzip') {
                            data = zlib_1.gunzipSync(data);
                        }
                        console.log(`Response: ${method.toUpperCase()} ${url} => ${dataLength} bytes`);
                        try {
                            ok(JSON.parse(data.toString('utf-8')));
                        }
                        catch (err) {
                            if (err.code === 'ERR_STRING_TOO_LONG') {
                                ko(new ResponseTooLargeError(method, url, err));
                            }
                            else {
                                ko(err);
                            }
                        }
                    }
                    catch (error) {
                        if (error.code === 'Z_BUF_ERROR') {
                            // Truncated payload... Connection cut too early?
                            console.error(`[RETRYABLE] Z_BUF_ERROR (${error.message}) - ${method.toUpperCase()} ${url}`);
                            retry();
                        }
                        else {
                            ko(error);
                        }
                    }
                });
            });
            req.end(body && JSON.stringify(body, null, 2));
        });
    }
}
exports.CouchChanges = CouchChanges;
class ResponseTooLargeError extends Error {
    constructor(method, url, cause) {
        super(`The response to ${method.toUpperCase()} ${url} is too large, and cannot be JSON.parse'd.`);
        this.cause = cause;
        this.name = ResponseTooLargeError.name;
        Error.captureStackTrace(this, ResponseTooLargeError);
    }
}
//# sourceMappingURL=data:application/json;base64,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