"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                            ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const cwd = process.cwd();
        try {
            process.chdir(tmpdir);
            console.log(`Preparing package install at ${tmpdir}`);
            const installResult = await shell_out_lambda_shared_1.shellOutWithOutput('npm', 'install', tarball, '--ignore-scripts', '--no-bin-links', '--no-save', '--include=dev', '--no-package-lock', '--json');
            if (installResult.exitCode !== 0) {
                if (installResult.signal) {
                    throw new Error(`"npm install" was killed by signal ${installResult.signal}`);
                }
                const toThrow = new Error(`"npm install" failed with exit code ${installResult.exitCode}`);
                toThrow.name = 'jsii-docgen.NpmError';
                try {
                    const json = JSON.parse(installResult.stdout.toString('utf-8'));
                    console.error(`Installation result: ${JSON.stringify(json, null, 2)}`);
                    if (json.error.code) {
                        toThrow.name += `.${json.error.code}`;
                    }
                }
                catch (error) {
                    console.error(`Installation result: ${installResult.stdout.toString('utf-8')}`);
                }
                throw toThrow;
            }
        }
        finally {
            process.chdir(cwd);
        }
        const packageDir = path.join(tmpdir, 'node_modules', packageName);
        console.log(`Generating documentation from ${packageDir}...`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            if (event.languages && !event.languages[language.toString()]) {
                console.log(`Skipping language ${language} as it was not requested!`);
                continue;
            }
            const isSupported = language === language_1.DocumentationLanguage.TYPESCRIPT || assembly.targets[language.targetName];
            if (!isSupported) {
                console.error(`Package ${packageName}@${packageVersion} does not support ${language}, skipping!`);
                console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
                for (const submodule of [undefined, ...submodules]) {
                    const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language, submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                    const response = await uploadFile(event.bucket, key, event.assembly.versionId);
                    created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                }
                continue;
            }
            const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                metrics.setDimensions();
                metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                const uploads = new Map();
                const docs = await docgen.Documentation.forProject(packageDir, {
                    assembliesDir: tmpdir,
                    language: docgen.Language.fromString(lang.name),
                });
                function renderAndDispatch(submodule) {
                    console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                    const page = Buffer.from(docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render());
                    metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                    const { buffer: body, contentEncoding } = compressContent(page);
                    metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                    const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule));
                    console.log(`Uploading ${key}`);
                    const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                    uploads.set(key, upload);
                }
                renderAndDispatch();
                for (const submodule of submodules) {
                    renderAndDispatch(submodule);
                }
                for (const [key, upload] of uploads.entries()) {
                    const response = await upload;
                    created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                    console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
                }
            });
            await generateDocs(language);
        }
        return created;
    });
}
exports.handler = handler;
function compressContent(buffer) {
    if (buffer.length < 1024) {
        return { buffer };
    }
    const gz = zlib_1.gzipSync(buffer, { level: 9 });
    // If it did not compress well, we'll keep the un-compressed original...
    if (gz.length >= buffer.length) {
        return { buffer };
    }
    return { buffer: gz, contentEncoding: 'gzip' };
}
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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