"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnsupportedLanguageError = exports.DocumentationLanguage = void 0;
/**
 * Supported languages to generate documentation in.
 */
class DocumentationLanguage {
    constructor(name, targetName = name) {
        this.name = name;
        this.targetName = targetName;
    }
    /**
     * Transform a literal string to the `DocumentationLanguage` object.
     *
     * Throws an `UnsupportedLanguageError` if the language is not supported.
     */
    static fromString(lang) {
        switch (lang) {
            case DocumentationLanguage.TYPESCRIPT.toString():
                return DocumentationLanguage.TYPESCRIPT;
            case DocumentationLanguage.PYTHON.toString():
                return DocumentationLanguage.PYTHON;
            case DocumentationLanguage.JAVA.toString():
                return DocumentationLanguage.JAVA;
            case DocumentationLanguage.CSHARP.toString():
            case DocumentationLanguage.CSHARP.targetName:
                return DocumentationLanguage.CSHARP;
            default:
                throw new UnsupportedLanguageError(lang, DocumentationLanguage.ALL);
        }
    }
    toString() {
        return this.name;
    }
}
exports.DocumentationLanguage = DocumentationLanguage;
/**
 * TypeScript.
 */
DocumentationLanguage.TYPESCRIPT = new DocumentationLanguage('typescript');
/**
 * Python.
 */
DocumentationLanguage.PYTHON = new DocumentationLanguage('python');
/**
 * Java.
 */
DocumentationLanguage.JAVA = new DocumentationLanguage('java');
/**
 * CSharp.
 */
DocumentationLanguage.CSHARP = new DocumentationLanguage('csharp', 'dotnet');
/**
 * All supported languages.
 */
DocumentationLanguage.ALL = [
    DocumentationLanguage.TYPESCRIPT,
    DocumentationLanguage.PYTHON,
    DocumentationLanguage.JAVA,
    DocumentationLanguage.CSHARP,
];
class UnsupportedLanguageError extends Error {
    constructor(lang, supported) {
        super(`Unsupported language: ${lang}. Supported languages are: [${supported}]`);
    }
}
exports.UnsupportedLanguageError = UnsupportedLanguageError;
//# sourceMappingURL=data:application/json;base64,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