"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageStats = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const package_stats_1 = require("./package-stats");
/**
 * Builds or re-builds the `stats.json` object in the designated bucket.
 */
class PackageStats extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.bucket = props.bucket;
        this.statsKey = props.objectKey;
        this.handler = new package_stats_1.PackageStats(this, 'Default', {
            description: `Creates the stats.json object in ${props.bucket.bucketName}`,
            environment: {
                CATALOG_BUCKET_NAME: this.bucket.bucketName,
                CATALOG_OBJECT_KEY: constants_1.CATALOG_KEY,
                STATS_BUCKET_NAME: this.bucket.bucketName,
                STATS_OBJECT_KEY: props.objectKey,
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 256,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
        });
        const updatePeriod = (_b = props.updatePeriod) !== null && _b !== void 0 ? _b : core_1.Duration.days(1);
        const rule = new events.Rule(this, 'Rule', {
            schedule: events.Schedule.rate(updatePeriod),
        });
        rule.addTarget(new targets.LambdaFunction(this.handler));
        this.bucket.grantReadWrite(this.handler);
        const failureAlarm = this.handler.metricErrors().createAlarm(scope, 'PackageStats/Failures', {
            alarmName: `${scope.node.path}/PackageStats/Failures`,
            alarmDescription: [
                'The package stats function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.handler)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        props.monitoring.addLowSeverityAlarm('PackageStats Failures', failureAlarm);
    }
    metricPackagesCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "RegisteredPackagesWithStats" /* REGISTERED_PACKAGES_WITH_STATS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.PackageStats = PackageStats;
//# sourceMappingURL=data:application/json;base64,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