"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inventory = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const canary_1 = require("./canary");
const constants_1 = require("./constants");
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
class Inventory extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rate = (_a = props.scheduleRate) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5);
        this.canary = new canary_1.Canary(this, 'Resource', {
            description: '[ConstructHub/Inventory] A canary that periodically inspects the list of indexed packages',
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: props.bucket.bucketName,
            },
            logRetention: props.logRetention,
            memorySize: 10240,
            timeout: rate,
        });
        const grant = props.bucket.grantRead(this.canary);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(rate),
            targets: [new aws_events_targets_1.LambdaFunction(this.canary)],
        }).node.addDependency(grant);
        props.monitoring.addHighSeverityAlarm('Inventory Canary is not Running', this.canary.metricInvocations({ period: core_1.Duration.minutes(5) }).createAlarm(this, 'Not Running', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The inventory canary is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        props.monitoring.addHighSeverityAlarm('Inventory Canary is failing', this.canary.metricErrors({ period: core_1.Duration.minutes(5) }).createAlarm(this, 'Failures', {
            alarmName: `${this.node.path}/Failures`,
            alarmDescription: [
                'The inventory canary is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
        }));
    }
    get function() {
        return this.canary;
    }
    metricMissingPackageMetadataCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingAssemblyCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageCount" /* PACKAGE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageMajorCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingPackageTarballCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricSubmoduleCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "SubmoduleCount" /* SUBMODULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricUnknownObjectCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages for which all versions are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions for which all versions are missing a
     * documnetation artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that are missing a documnetation
     * artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that have at least one version for which there is
     * available documentation in the provided `DocumentationLanguage`.
     */
    metricSupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that have at least one version for
     * which there is available documentation in the provided
     * `DocumentationLanguage`.
     */
    metricSupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that have available documentation in the
     * provided `DocumentationLanguage`.
     */
    metricSupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that have available documentation
     * in the provided `DocumentationLanguage`.
     */
    metricSupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.Inventory = Inventory;
//# sourceMappingURL=data:application/json;base64,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