# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['csql',
 'csql._',
 'csql._.input',
 'csql._.models',
 'csql._.persist',
 'csql._.renderer',
 'csql.contrib',
 'csql.contrib.persist',
 'csql.contrib.render',
 'csql.contrib.render.param',
 'csql.render']

package_data = \
{'': ['*']}

extras_require = \
{'docs': ['pandas>=1.3.4,<2.0.0',
          'Sphinx>=4.3.1,<5.0.0',
          'sphinx-autobuild>=2021.3.14,<2022.0.0',
          'sphinx-rtd-theme>=1.0.0,<2.0.0',
          'sphinx-external-toc>=0.2.3,<0.3.0',
          'myst-parser[linkify]>=0.16.1,<0.17.0',
          'snowflake-connector-python>=2.7.2,<3.0.0'],
 'notebooks': ['openpyxl>=3.0.9,<4.0.0',
               'duckdb>=0.3.1,<0.4.0',
               'ipykernel>=6.6.0,<7.0.0'],
 'pandas': ['pandas>=1.3.4,<2.0.0']}

setup_kwargs = {
    'name': 'csql',
    'version': '0.11.0',
    'description': 'Simple library for writing composeable SQL queries',
    'long_description': '\n# csql - Composeable SQL\n\n<!-- (intro)= -->\n\n**csql** is a Python library to help you write more manageable SQL queries. You can write your queries as small, self-contained chunks, preview the results without pulling a whole result-set down from your database, then refer to them in future queries.\n\nThere are also useful features for handling database parameters properly.\n\nThe intended use-case is for data analysis and exploration.\n\n[![PyPI version](https://badge.fury.io/py/csql.svg)](https://pypi.org/project/csql/)\n\n**Full documentation** is available at https://csql.readthedocs.io/en/latest/api.html .\n\n## Example\n\n```py\nfrom csql import Q, Parameters\nimport pandas as pd\nfrom datetime import date\n\ncon = your_database_connection()\n```\n\nStart with a straightforward query:\n```py\np = Parameters(\n\tcreated_on = date(2020,1,1)\n)\nq1 = Q(f"""\nselect\n\tcustomers.id,\n\tfirst(customers.name) as name,\n\tfirst(created_on) as created_on,\n\tsum(sales.value) as sales\nfrom\n\tcustomers\n\tjoin sales on customers.id = sales.customer_id\nwhere created_on > {p[\'created_on\']}}\ngroup by customers.id\n""")\n\nprint(q1.preview_pd(con))\n```\n\n|  | id | name | created_on | sales |\n|--|----|------|------------|-------|\n|1 |111 |John Smith | 2020-02-05 | 32.0 |\n|2 |112 |Amy Zhang | 2020-05-01 | 101.5 |\n|3 |115 |Tran Thanh | 2020-03-02 | 100000.0 |\n\n\nThe preview will pull down 10 rows to a) sanity-check the result of what you\'ve just written, and b) validate your sql.\n\n-----\n\nNow, try building some new queries *that build on your previous queries*:\n```py\nq2 = Q(f"""\nselect\n\tntile(100) over (order by sales)\n\t\tas ntile_100,\n\tname,\n\tsales\nfrom {q1}\n""")\n\nprint(q2.preview_pd(con))\n```\n\n|  | ntile_100 | name | sales |\n|--|-----------|------|-------|\n| 1|29| John Smith| 32.0 |\n| 2|50|Amy Zhang | 101.5 |\n| 3|99|Tran Thanh | 100000.0 |\n\n-----\n\n```py\nq3 = Q(f"""\nselect\n\tntile_100,\n\tmin(sales),\n\tmax(sales)\nfrom {q2}\ngroup by ntile_100\norder by ntile_100\n""")\n\n# this time, we\'ll pull the whole result instead of just previewing:\nresult = pd.read_sql(**q3.pd(), con=con)\nprint(result)\n```\n|  | ntile_100 | min(sales) | max(sales) |\n|--|-----------|----------|--------------|\n| 28| 29 | 25 | 33.3 |\n| 49| 50 | 98 | 120 |\n| 98| 99 | 5004 | 100000.0 |\n\n-----\n\n## Cool! But, how does it work?\n\nThe basic idea is to turn your queries into a CTE by keeping track of what builds on top of what. For example, for the last query shown, `q3`, what actually gets sent to the database is:\n\n```sql\nwith _subQuery0 as (\n\tselect\n\t\tcustomers.id,\n\t\tfirst(customers.name) as name,\n\t\tfirst(created_on) as created_on,\n\t\tsum(sales.value) as sales\n\tfrom\n\t\tcustomers\n\t\tjoin sales on customers.id = sales.customer_id\n\twhere created_on > :1\n\tgroup by customers.id\n),\n_subQuery1 as (\n\tselect\n\t\tntile(100) over (order by sales)\n\t\t\tas ntile_100,\n\t\tname,\n\t\tsales\n\tfrom _subQuery0\n)\nselect\n\tntile_100,\n\tmin(sales),\n\tmax(sales)\nfrom _subQuery1\ngroup by ntile_100\norder by ntile_100\n```\n\nwhich is exactly the sort of unmaintainable and undebuggable monstrosity that this library is designed to help you avoid.\n\n\n<!-- (end-intro)= -->\n\n## Design Notes\n\nI am perhaps overly optimistic about this, but currently I think this should work with most SQL dialects. It doesn\'t attempt to parse your SQL, uses CTEs which are widely supported, and passes numeric style parameters.\nIt\'s also not actually tied to `pandas` at all - `.pd()` is just a convenience method to build a dict you can splat into pd.read_sql.\n\n## Easy Parameters\n\n<!-- (params)= -->\n\nUsing proper SQL prepared statements is great to do, but can be annoying to maintain. Additionally, it can be incredibly\nannoying when you are trying to use a list from Python:\n\n```py\ncon = my_connection()\nids_i_want = [1, 2, 3]\nwith con.cursor() as c:\n\t# uh oh, you can\'t do this\n\tc.execute(\'select * from customers where id in :1\', (ids_i_want,))\n\n\t# you need to do something like this instead\n\tc.execute(\'select * from customers where id in (:1, :2, :3), (ids_i_want[0], ids_i_want[1], ids_i_want[2],))\n```\n\n`csql` makes this much easier - you can embed your parameters naturally with string interpolation, and they will still be\nsent as proper parameterized statements.\n\n```py\np = Parameters(\n\tids_i_want = [1, 2, 3],\n\tname = \'Jarrad\'\n)\n\nget_customers = Q(f\'\'\'\n\tselect * from customers\n\twhere\n\t\tids in {p[\'ids_i_want\']}\n\t\tor name = {p[\'name\']}\n\'\'\')\n\nwith con.cursor() as c:\n\tc.execute(*get_customers.db)\n```\n\nThat final statement is actually equivalent to:\n\n```py\nwith con.cursor() as c:\n\tc.execute(\'\'\'\n\t\tselect * from customers\n\t\twhere\n\t\t\tids in (:1, :2, :3)\n\t\t\tor name = :4\n\t\'\'\', [1, 2, 3, \'Jarrad\'])\n```\n\n\n\n<!-- (end-params)= -->\n\n\n## Changing Parameter Values\n\n<!-- (reparam)= -->\n\nParameters aren\'t super useful if they are set in stone, but `csql` wants you\nto give values at the query definition time! How can you pass different values later?\n\nThis is achieved by passing `newParams` to {meth}`csql.Query.build`:\n\n```py\np = Parameters(\n  start=datetime.now() - timedelta(days=3),\n  end=datetime.now()\n)\nq = Q(f\'select count(*) from events where start <= date and date < end\')\npd.read_sql(**q.pd, con=con)\n# 42 # 3 days ago to now, as per `p`.\nnewParams = {\'start\': date(2010,1,1)}\npd.read_sql(**q.build(newParams=newParams).pd, con=con)\n# 42000 # 2010 to now, with new value for `start` provided.\n```\n\n<!-- (end-reparam)= -->\n\n## SQL Dialects\n\n<!-- (sql-dialects)= -->\n\nDifferent dialects can be specified at render time, or as the default dialect of your Queries. Currently the only things dialects control are parameter rendering and limits, but I expect to see some scope creep around here...\nDialects are instances of {class}`csql.dialect.SQLDialect` and can be found in {mod}`csql.dialect`. The default dialect is {class}`csql.dialect.DefaultDialect`, which uses a numeric parameter renderer. You can specify your own prefered dialect per-query:\n\n```py\nq = csql.Q(\n\tf"select 1 from thinger",\n\tdialect=csql.dialect.DuckDB\n)\n```\n\nIf you want to set a default, use `functools.partial` like so:\n\n```py\nimport functools\nQ = functools.partial(csql.Q, dialect=csql.dialect.DuckDB)\nq = Q(f"select 1 from thinger")\n```\n\n### Inferred Dialects\n\nIf a query `q2` references a previous query `q1`, and `q1` has a dialect specified, then `q2` will use `q1`\'s dialect by default.\n\n```py\nq1 = csql.Q(\'select 1 from thinger\', dialect=csql.dialect.Snowflake)\nq2 = csql.Q(\'select count(*) from {q1})\nassert q2.default_dialect == csql.dialect.Snowflake\n```\n\nIf you reference multiple queries with conflicting dialects, you\'ll get an error. Normally this is because you\'ve actually\nforgotten to specify something somewhere. If you\'re doing this on purpose, override by setting `dialect=` to `Q` manually.\n\n### DIY Dialects \n\nYou can construct your own dialects:\n\n```py\nimport csql.dialect\nMyDialect = csql.dialect.SQLDialect(\n  paramstyle=csql.dialect.ParamStyle.qmark\n)\n```\n\nThere are presets for some common databases (see below), and I\'m very happy to accept PRs for any\nothers.\n\n\n<!-- (end-sql-dialects)= -->\n\n\nHow to use Caching\n==================\n\n<!-- (persist)= -->\n\nOnce you have a few queries chained together, you may\nstart to get annoyed by how long one or two big things at the start\ntake, and wonder if there\'s a way to stop them being executed each time.\n\nFor example,\n\n```py\nq1 = Q(f\'select id, date, rank() over (partition by name order by date) as rank from customers\')\nq2 = Q(f\'select date, count(*) from {q1}\')\nprint(q2.preview_pd(con))\n# takes 2 mins becuase q1 is so slow\nprint(q2.preview_pd(con))\n# same thing again, also takes 2 mins\nq3 = Q(f\'select max(date) from {q2}\')\nprint(q3.preview_pd(con))\n# also takes 2 mins because q1 is so slow\n```\n\nThe solution is to use {meth}`csql.Query.persist` on the slow query you want to re-use.\nAbove, we could either do this on `q1` or `q2`, depending on what works best with\nour database. I\'ll demonstrate `q2`:\n\n```py\nq1 = Q(f\'select id, date, rank() over (partition by name order by date) as rank from customers\')\ncache = TempTableCacher(con)\nq2 = Q(f\'select date, count(*) from {q1}\').persist(cache) # <--- !!\nprint(q2.preview_pd(con))\n# still takes 2 mins\nprint(q2.preview_pd(con))\n# now this is fast!\nq3 = Q(f\'select max(date) from {q2}\')\nprint(q3.preview_pd(con))\n# now this is fast as well!\n```\n\nThe only general builtin caching method is {class}`csql.contrib.persist.TempTableCacher`, however it\'s straightforward\nto write your own. You may want to also see {mod}`csql.contrib.persist` as there is a Snowflake-specific example in there as well.\n\n<!-- (end-persist)= -->\n',
    'author': 'Jarrad Whitaker',
    'author_email': 'akdor1154@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/akdor1154/python-csql',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
