import dataclasses
from typing import Optional, List
from enum import Enum
from dataclasses import dataclass

from cloudrail.knowledge.context.azure.resources.constants.azure_resource_type import AzureResourceType
from cloudrail.knowledge.context.azure.resources.azure_resource import AzureResource
from cloudrail.knowledge.context.azure.resources.monitor.azure_monitor_diagnostic_setting import AzureMonitorDiagnosticSetting
from cloudrail.knowledge.context.azure.resources.i_monitor_settings import IMonitorSettings


class IoTHubSkuName(str, Enum):
    B1 = 'B1'
    B2 = 'B2'
    B3 = 'B3'
    F1 = 'F1'
    S1 = 'S1'
    S2 = 'S2'
    S3 = 'S3'


class IoTHubEndpointType(str, Enum):
    STORAGE_CONTAINER = 'AzureIotHub.StorageContainer'
    SERVICE_BUS_QUEUE = 'AzureIotHub.ServiceBusQueue'
    SERVICE_BUS_TOPIC = 'AzureIotHub.ServiceBusTopic'
    EVENT_HUB = 'AzureIotHub.EventHub'


class IoTHubEndpointEncoding(str, Enum):
    AVRO = 'avro'
    AVRO_DEFLATE = 'avrodeflate'


class IoTHubRouteSource(str, Enum):
    NONE = 'None'
    DEVICE_MESSAGES = 'DeviceMessages'
    INVALID = 'Invalid'
    TWIN_CHANGE_EVENTS = 'TwinChangeEvents'
    DEVICE_LIFECYCLE_EVENTS = 'DeviceLifecycleEvents'
    DEVICE_CONNECTION_STATE_EVENTS = 'DeviceConnectionStateEvents'
    DEVICE_JOB_LIFECYCLE_EVENTS = 'DeviceJobLifecycleEvents'


class IoTHubIpFilterRuleAction(str, Enum):
    ACCEPT = 'Accept'
    REJECT = 'Reject'


@dataclass
class IoTHubSku:
    """
        Attributes:
            name: The name of the SKU
            capacity: The number of provisioned IoT Hub units.
    """
    name: IoTHubSkuName
    capacity: int


@dataclass
class IoTHubEndpoint:
    """
        Attributes:
            type: The type of endpoint.
            connection_string: The connection string for the endpoint.
            name: The name of the endpoint.
            batch_frequency_in_seconds: Time interval at which blobs are written to storage.
            max_chunk_size_in_bytes: Maximum number of bytes for each blob written to storage.
            container_name: The name of storage container in the storage account.
            encoding: Encoding that is used to serialize messages to blobs.
            file_name_format: File name format for the blob.
            resource_group_name: The resource group in which the endpoint will be created.
    """
    type: IoTHubEndpointType
    connection_string: str
    name: str
    batch_frequency_in_seconds: int
    max_chunk_size_in_bytes: int
    container_name: Optional[str]
    encoding: IoTHubEndpointEncoding
    file_name_format: Optional[str]
    resource_group_name: Optional[str]


@dataclass
class IoTHubFallbackRoute:
    """
        Attributes:
            source: The source that the routing rule is to be applied to.
            condition: The condition that is evaluated to apply the routing rule.
            endpoint_names: The endpoints to which messages that satisfy the condition are routed.
            enabled: Whether the fallback route is enabled
    """
    source: Optional[IoTHubRouteSource]
    condition: str
    endpoint_names: List[str]
    enabled: bool


@dataclass
class IoTHubFileUpload:
    """
        Attributes:
            connection_string: The connection string for the Azure Storage account to which files are uploaded.
            container_name: The name of the root container where you upload files.
            sas_ttl: The period of time for which the SAS URI generated by IoT Hub for file upload is valid, specified as an ISO 8601 timespan duration.
            notifications: Used to specify whether file notifications are sent to IoT Hub on upload.
            lock_duration: The lock duration for the file upload notifications queue, specified as an ISO 8601 timespan duration.
            default_ttl: The period of time for which a file upload notification message is available to consume before it is expired by the IoT hub, specified as an ISO 8601 timespan duration.
            max_delivery_count: The number of times the IoT hub attempts to deliver a file upload notification message.
    """
    connection_string: str
    container_name: str
    sas_ttl: str
    notifications: bool
    lock_duration: str
    default_ttl: str
    max_delivery_count: int


@dataclass
class IoTHubIpFilterRule:
    """
        Attributes:
            name: The name of the filter.
            ip_mask: The IP address range in CIDR notation.
            action: The desired action for requests captured by this rule.
    """
    name: str
    ip_mask: str
    action: IoTHubIpFilterRuleAction


@dataclass
class IoTHubRoute:
    """
        Attributes:
            name: The name of the route
            source: The source that the routing rule is to be applied to.
            condition: The condition that is evaluated to apply the routing rule.
            endpoint_names: The list of endpoints to which messages that satisfy the condition are routed.
            enabled: Whether a route is enabled.
    """
    name: str
    source: IoTHubRouteSource
    condition: str
    endpoint_names: List[str]
    enabled: bool


@dataclass
class IoTHubEnrichment:
    """
        Attributes:
            key: The key of the enrichment.
            value: The value of the enrichment.
            endpoint_names: The list of endpoints which will be enriched.
    """
    key: str
    value: str
    endpoint_names: List[str]


class AzureIoTHub(AzureResource, IMonitorSettings):
    """
        Attributes:
            name: The name of the IoT Hub.
            sku: The SKU configuration of IoT Hub.
            event_hub_partition_count: The number of device-to-cloud partitions used by backing event hubs.
            event_hub_retention_in_days: The event hub retention to use in days.
            endpoint_list: List of endpoints configuration.
            fallback_route: A fallback route for messages that don't match any of the supplied routes.
            file_upload: A file upload configurations.
            ip_filter_rule_list: A list of IP filter rules.
            route_list: A list of routes for message delivering.
            enrichment_list: A list of enrichment key/value pairs for endpoints.
            public_network_access_enabled: If the IoT Hub resource is accessible from a public network.
            min_tls_version: Specifies the minimum TLS version to support for this hub. The only valid value is 1.2.
    """

    def __init__(self,
                 name: str,
                 sku: IoTHubSku,
                 event_hub_partition_count: int,
                 event_hub_retention_in_days: int,
                 endpoint_list: List[IoTHubEndpoint],
                 fallback_route: IoTHubFallbackRoute,
                 file_upload: Optional[IoTHubFileUpload],
                 ip_filter_rule_list: Optional[List[IoTHubIpFilterRule]],
                 route_list: Optional[List[IoTHubRoute]],
                 enrichment_list: Optional[List[IoTHubEnrichment]],
                 public_network_access_enabled: bool,
                 min_tls_version: str):
        super().__init__(AzureResourceType.AZURERM_IOT_HUB)
        self.name: str = name
        self.sku: IoTHubSku = sku
        self.event_hub_partition_count: int = event_hub_partition_count
        self.event_hub_retention_in_days: int = event_hub_retention_in_days
        self.ip_filter_rule_list: Optional[List[IoTHubIpFilterRule]] = ip_filter_rule_list
        self.public_network_access_enabled: bool = public_network_access_enabled
        self.min_tls_version: str = min_tls_version
        self.endpoint_list: List[IoTHubEndpoint] = endpoint_list
        self.fallback_route: IoTHubFallbackRoute = fallback_route
        self.file_upload: Optional[IoTHubFileUpload] = file_upload
        self.route_list: Optional[List[IoTHubRoute]] = route_list
        self.enrichment_list: Optional[List[IoTHubEnrichment]] = enrichment_list
        self.monitor_diagnostic_settings: List[AzureMonitorDiagnosticSetting] = []

    def get_cloud_resource_url(self) -> Optional[str]:
        return f'https://portal.azure.com/#@{self.tenant_id}/resource{self._id}/overview'

    @property
    def is_tagable(self) -> bool:
        return True

    def get_keys(self) -> List[str]:
        return [self.get_id()]

    def get_name(self) -> str:
        return self.name

    def get_type(self, is_plural: bool = False) -> str:
        return 'IoT Hub' + ('s' if is_plural else '')

    def to_drift_detection_object(self) -> dict:
        return {'tags': self.tags,
                'event_hub_partition_count': self.event_hub_partition_count,
                'event_hub_retention_in_days': self.event_hub_retention_in_days,
                'public_network_access_enabled': self.public_network_access_enabled,
                'ip_filter_rule_list': [dataclasses.asdict(rule) for rule in self.ip_filter_rule_list],
                'endpoint_list': [dataclasses.asdict(endpoint) for endpoint in self.endpoint_list],
                'fallback_route': dataclasses.asdict(self.fallback_route),
                'file_upload': self.file_upload and dataclasses.asdict(self.file_upload),
                'route_list': [dataclasses.asdict(route) for route in self.route_list],
                'enrichment_list': [dataclasses.asdict(enrichment) for enrichment in self.enrichment_list],
                'min_tls_version': self.min_tls_version}

    def get_monitor_settings(self) -> List[AzureMonitorDiagnosticSetting]:
        return self.monitor_diagnostic_settings
