from typing import Optional, List
from dataclasses import dataclass
from cloudrail.knowledge.context.azure.resources.azure_resource import AzureResource
from cloudrail.knowledge.context.azure.resources.constants.azure_resource_type import AzureResourceType


@dataclass
class AzureMsSqlServerVulnerabilityAssessmentRecurringScans:
    """
        Attributes:
            enabled: If recurring scans is enabled or disabled.
            email_subscription_admins: Boolean flag which specifies if the schedule scan notification will be sent to the subscription administrators.
            emails: Specifies an array of e-mail addresses to which the scan notification is sent.
    """
    enabled: bool
    email_subscription_admins: bool
    emails: Optional[List[str]]


class AzureMsSqlServerVulnerabilityAssessment(AzureResource):
    """
        Attributes:
            server_security_alert_policy_id: The ID of the security alert policy of the MS SQL Server.
            storage_container_path: A blob storage container path to hold the scan results.
            storage_account_access_key: Specifies the identifier key of the storage account for vulnerability assessment scan results.
            storage_container_sas_key: A shared access signature (SAS Key) that has write access to the blob container specified in storage_container_path parameter.
            recurring_scans_settings: The recurring scans settings.
    """
    def __init__(self,
                 server_security_alert_policy_id: str,
                 storage_container_path: Optional[str],
                 storage_account_access_key: Optional[str],
                 storage_container_sas_key: Optional[str],
                 recurring_scans_settings: Optional[AzureMsSqlServerVulnerabilityAssessmentRecurringScans]):
        super().__init__(AzureResourceType.AZURERM_MSSQL_SERVER_VULNERABILITY_ASSESSMENT)
        self.server_security_alert_policy_id: str = server_security_alert_policy_id
        self.storage_container_path: Optional[str] = storage_container_path
        self.storage_account_access_key: Optional[str] = storage_account_access_key
        self.storage_container_sas_key: Optional[str] = storage_container_sas_key
        self.recurring_scans_settings: Optional[AzureMsSqlServerVulnerabilityAssessmentRecurringScans] = recurring_scans_settings

    def get_keys(self) -> List[str]:
        return [self.get_id()]

    def get_name(self) -> Optional[str]:
        return f'SQL server vulnerability assessment of server {self.server_security_alert_policy_id}'

    def get_cloud_resource_url(self) -> Optional[str]:
        return f'https://portal.azure.com/#@{self.tenant_id}/resource{self._id}/overview'

    def get_type(self, is_plural: bool = False) -> str:
        return 'Azure SQL server vulnerability assesment ' + ('s' if is_plural else '')

    @property
    def is_tagable(self) -> bool:
        return False

    @staticmethod
    def is_standalone() -> bool:
        return False

    def to_drift_detection_object(self) -> dict:
        return {'storage_container_path': self.storage_container_path,
                'recurring_scans_settings': self.recurring_scans_settings}

    @property
    def emails_sent_to_admins(self) -> bool:
        return self.recurring_scans_settings and \
            self.recurring_scans_settings.email_subscription_admins and \
                self.recurring_scans_settings.enabled and \
                    self.recurring_scans_settings.emails
