"""This module provides various methods and utilities to turn TIM into HTML & SVG."""

# TODO: The HTML and SVG implementations are completely independent at the moment,
#       which is pretty annoying to maintain. It would be great to consolidate them
#       at some point.

from __future__ import annotations

from html import escape
from typing import Iterator

from .colors import Color
from .widgets import Widget
from .terminal import get_terminal
from .parser import Token, TokenType, StyledText, tim

MARGIN = 15
BODY_MARGIN = 70
CHAR_WIDTH = 0.62
CHAR_HEIGHT = 1.15
FONT_SIZE = 15

FONT_WIDTH = FONT_SIZE * CHAR_WIDTH
FONT_HEIGHT = FONT_SIZE * CHAR_HEIGHT * 1.1

HTML_FORMAT = """\
<html>
    <head>
        <style>
            body {{
                --ptg-background: {background};
                --ptg-foreground: {foreground};
                color: var(--ptg-foreground);
                background-color: var(--ptg-background);
            }}
            a {{
                text-decoration: none;
                color: inherit;
            }}
            code {{
                font-size: {font_size}px;
                font-family: Menlo, 'DejaVu Sans Mono', consolas, 'Courier New', monospace;
                line-height: 1.2em;
            }}
            .ptg-position {{
                position: absolute;
            }}
{styles}
        </style>
    </head>
    <body>
        <pre class="ptg">
            <code>
{content}
            </code>
        </pre>
    </body>
</html>"""

SVG_MARGIN_LEFT = 50
TEXT_MARGIN_LEFT = 20

TEXT_MARGIN_TOP = 35
SVG_MARGIN_TOP = 20

SVG_FORMAT = f"""\
<svg width="{{total_width}}" height="{{total_height}}"
    viewBox="0 0 {{total_width}} {{total_height}}" xmlns="http://www.w3.org/2000/svg">
    <!-- Generated by PyTermGUI -->
    <style type="text/css">
        text {{{{
            font-size: {FONT_SIZE}px;
            font-family: Menlo, 'DejaVu Sans Mono', consolas, 'Courier New', monospace;
            alignment-baseline: text-after-edge;
        }}}}

        .{{prefix}}-title {{{{
            /*font-family: 'arial';*/
            fill: #94999A;
            font-size: 13px;
            font-weight: bold;
        }}}}
{{stylesheet}}
    </style>
    <rect x="{SVG_MARGIN_LEFT}" y="{SVG_MARGIN_TOP}"
        rx="9px" ry="9px" stroke-width="1px" stroke-linejoin="round"
        width="{{terminal_width}}" height="{{terminal_height}}" fill="{{background}}" />
    <circle cx="{SVG_MARGIN_LEFT+15}" cy="{SVG_MARGIN_TOP + 15}" r="6" fill="#ff6159"/>
    <circle cx="{SVG_MARGIN_LEFT+35}" cy="{SVG_MARGIN_TOP + 15}" r="6" fill="#ffbd2e"/>
    <circle cx="{SVG_MARGIN_LEFT+55}" cy="{SVG_MARGIN_TOP + 15}" r="6" fill="#28c941"/>
    <text x="{{title_x}}" y="{{title_y}}" text-anchor="middle"
        class="{{prefix}}-title">{{title}}</text>
{{code}}
</svg>"""

_STYLE_TO_CSS = {
    "bold": "font-weight: bold",
    "italic": "font-style: italic",
    "dim": "opacity: 0.7",
    "underline": "text-decoration: underline",
    "strikethrough": "text-decoration: line-through",
    "overline": "text-decoration: overline",
}


__all__ = ["token_to_css", "to_html"]


def _get_cls(prefix: str | None, index: int) -> str:
    """Constructs a class identifier with the given prefix and index."""

    return "ptg" + ("-" + prefix if prefix is not None else "") + str(index)


def _generate_stylesheet(document_styles: list[list[str]], prefix: str | None) -> str:
    """Generates a '\\n' joined CSS stylesheet from the given styles."""

    stylesheet = ""
    for i, styles in enumerate(document_styles):
        stylesheet += "\n." + _get_cls(prefix, i) + " {" + "; ".join(styles) + "}"

    return stylesheet


def _generate_index_in(lst: list[list[str]], item: list[str]) -> int:
    """Returns the given item's index in the list, len(lst) if not found."""

    index = len(lst)

    if item in lst:
        return lst.index(item)

    return index


# Note: This whole routine will be massively refactored in an upcoming update,
#       once StyledText has a bit of a better way of managing style attributes.
#       Until then we must ignore some linting issues :(.
def _get_spans(  # pylint: disable=too-many-locals
    line: str,
    vertical_offset: float,
    horizontal_offset: float,
    include_background: bool,
) -> Iterator[tuple[str, list[str]]]:
    """Creates `span` elements from the given line, yields them with their styles.

    Args:
        line: The ANSI line of text to use.

    Yields:
        Tuples of the span text (more on that later), and a list of CSS styles applied
        to it.  The span text is in the format `<span{}>content</span>`, and it doesn't
        yet have the styles formatted into it.
    """

    def _adjust_pos(
        position: int, scale: float, offset: float, digits: int = 2
    ) -> float:
        """Adjusts a given position for the HTML canvas' scale."""

        return round(position * scale + offset / FONT_SIZE, digits)

    position = None

    for styled in tim.get_styled_plains(line):
        styles = []
        if include_background:
            styles.append("background-color: var(--ptg-background)")

        has_link = False
        has_inverse = False

        for token in sorted(
            styled.tokens, key=lambda token: token.ttype is TokenType.COLOR
        ):
            if token.ttype is TokenType.PLAIN:
                continue

            if token.ttype is TokenType.POSITION:
                assert isinstance(token.data, str)

                if token.data != position:
                    # Yield closer if there is already an active positioner
                    if position is not None:
                        yield "</div>", []

                    position = token.data
                    split = tuple(map(int, position.split(",")))

                    adjusted = (
                        _adjust_pos(split[0], CHAR_WIDTH, horizontal_offset),
                        _adjust_pos(split[1], CHAR_HEIGHT, vertical_offset),
                    )

                    yield (
                        "<div class='ptg-position'"
                        + f" style='left: {adjusted[0]}em; top: {adjusted[1]}em'>"
                    ), []

            elif token.ttype is TokenType.LINK:
                has_link = True
                yield f"<a href='{token.data}'>", []

            elif token.ttype is TokenType.STYLE and token.name == "inverse":
                has_inverse = True

                # Add default inverted colors, in case the text doesn't have any
                # color applied.
                styles.append("color: var(--ptg-background);")
                styles.append("background-color: var(--ptg-foreground)")

                continue

            css = token_to_css(token, has_inverse)
            if css is not None and css not in styles:
                styles.append(css)

        escaped = (
            escape(styled.plain)
            .replace("{", "{{")
            .replace("}", "}}")
            .replace(" ", "&#160;")
        )

        if len(styles) == 0:
            yield f"<span>{escaped}</span>", []
            continue

        tag = "<span{}>" + escaped + "</span>"
        tag += "</a>" if has_link else ""

        yield tag, styles


def token_to_css(token: Token, invert: bool = False) -> str:
    """Finds the CSS representation of a token.

    Args:
        token: The token to represent.
        invert: If set, the role of background & foreground colors
            are flipped.
    """

    if token.ttype is TokenType.COLOR:
        color = token.data
        assert isinstance(color, Color)

        style = "color:" + color.hex

        if invert:
            color.background = not color.background

        if color.background:
            style = "background-" + style

        return style

    if token.ttype is TokenType.STYLE and token.name in _STYLE_TO_CSS:
        return _STYLE_TO_CSS[token.name]

    return ""


# We take this many arguments for future proofing and customization, not much we can
# do about it.
def to_html(  # pylint: disable=too-many-arguments, too-many-locals
    obj: Widget | StyledText | str,
    prefix: str | None = None,
    inline_styles: bool = False,
    include_background: bool = True,
    vertical_offset: float = 0.0,
    horizontal_offset: float = 0.0,
    formatter: str = HTML_FORMAT,
    joiner: str = "\n",
) -> str:
    """Creates a static HTML representation of the given object.

    Note that the output HTML will not be very attractive or easy to read. This is
    because these files probably aren't meant to be read by a human anyways, so file
    sizes are more important.

    If you do care about the visual style of the output, you can run it through some
    prettifiers to get the result you are looking for.

    Args:
        obj: The object to represent. Takes either a Widget or some markup text.
        prefix: The prefix included in the generated classes, e.g. instead of `ptg-0`,
            you would get `ptg-my-prefix-0`.
        inline_styles: If set, styles will be set for each span using the inline `style`
            argument, otherwise a full style section is constructed.
        include_background: Whether to include the terminal's background color in the
            output.
    """

    document_styles: list[list[str]] = []

    if isinstance(obj, Widget):
        data = obj.get_lines()

    else:
        data = obj.splitlines()

    lines = []
    for dataline in data:
        line = ""

        for span, styles in _get_spans(
            dataline, vertical_offset, horizontal_offset, include_background
        ):
            index = _generate_index_in(document_styles, styles)
            if index == len(document_styles):
                document_styles.append(styles)

            if inline_styles:
                stylesheet = ";".join(styles)
                line += span.format(f" styles='{stylesheet}'")

            else:
                line += span.format(" class='" + _get_cls(prefix, index) + "'")

        # Close any previously not closed divs
        line += "</div>" * (line.count("<div") - line.count("</div"))
        lines.append(line)

    stylesheet = ""
    if not inline_styles:
        stylesheet = _generate_stylesheet(document_styles, prefix)

    document = formatter.format(
        foreground=Color.get_default_foreground().hex,
        background=Color.get_default_background().hex if include_background else "",
        content=joiner.join(lines),
        styles=stylesheet,
        font_size=FONT_SIZE,
    )

    return document


def _escape_text(text: str) -> str:
    """Escapes HTML and replaces ' ' with &nbsp;."""

    return escape(text).replace(" ", "&#160;")


def _handle_tokens_svg(
    text: StyledText, default_fore: str
) -> tuple[tuple[int, int] | None, str | None, list[str]]:
    """Builds CSS styles that apply to the text."""

    default = f"fill:{default_fore}"
    styles = [default]
    back = pos = None

    for token in text.tokens:
        if token.ttype is TokenType.POSITION:
            assert isinstance(token.data, str)
            mapped = tuple(map(int, token.data.split(",")))
            pos = mapped[0], mapped[1]
            continue

        if token.ttype is TokenType.COLOR:
            color = token.data
            assert isinstance(color, Color)

            if color.background:
                back = color.hex
                continue

            styles.remove(default)
            styles.append(f"fill:{color.hex}")
            continue

        css = token_to_css(token)

        if css != "":
            styles.append(css)

    return pos, back, styles


def _slugify(text: str) -> str:
    """Turns the given text into a slugified form."""

    return text.replace(" ", "-").replace("_", "-")


def _make_tag(tagname: str, content: str = "", **attrs) -> str:
    """Creates a tag."""

    tag = f"<{tagname} "

    for key, value in attrs.items():
        if key == "raw":
            tag += " " + value
            continue

        tag += f"{_slugify(key)}='{value}' "

    tag += f">{content}</{tagname}>"

    return tag


def to_svg(  # pylint: disable=too-many-locals
    obj: Widget | StyledText | str,
    prefix: str | None = None,
    inline_styles: bool = False,
    title: str = "PyTermGUI",
    formatter: str = SVG_FORMAT,
) -> str:
    """Creates an SVG screenshot of the given object.

    This screenshot tries to mimick what the Kitty terminal looks like on MacOS,
    complete with the menu buttons and drop shadow. The `title` argument will be
    displayed in the window's top bar.

    Args:
        obj: The object to represent. Takes either a Widget or some markup text.
        prefix: The prefix included in the generated classes, e.g. instead of `ptg-0`,
            you would get `ptg-my-prefix-0`.
        inline_styles: If set, styles will be set for each span using the inline `style`
            argument, otherwise a full style section is constructed.
        title: A string to display in the top bar of the fake terminal.
        formatter: The formatting string to use. Inspect `pytermgui.exporters.SVG_FORMAT`
            to see all of its arguments.
    """

    def _is_block(text: str) -> bool:
        """Determines whether the given text only contains block characters.

        These characters reside in the unicode range of 9600-9631, which is what we test
        against.
        """

        return all(9600 <= ord(char) <= 9631 for char in text)

    if prefix is None:
        prefix = "ptg"

    terminal = get_terminal()
    default_fore = Color.get_default_foreground().hex
    default_back = Color.get_default_background().hex

    text = ""

    lines = 1
    cursor_x = cursor_y = 0.0
    document_styles: list[list[str]] = []

    # We manually set all text to have an alignment-baseline of
    # text-after-edge to avoid block characters rendering in the
    # wrong place (not at the top of their "box"), but with that
    # our background rects will be rendered in the wrong place too,
    # so this is used to offset that.
    baseline_offset = 0.17 * FONT_HEIGHT

    if isinstance(obj, Widget):
        obj = "\n".join(obj.get_lines())

    for plain in tim.get_styled_plains(obj):
        should_newline = False

        pos, back, styles = _handle_tokens_svg(plain, default_fore)

        index = _generate_index_in(document_styles, styles)

        if index == len(document_styles):
            document_styles.append(styles)

        style_attr = (
            f"style='{';'.join(styles)}'"
            if inline_styles
            else f"class='{_get_cls(prefix, index)}'"
        )

        # Manual positioning
        if pos is not None:
            cursor_x = pos[0] * FONT_WIDTH - 10
            cursor_y = pos[1] * FONT_HEIGHT - 15

        for line in plain.plain.splitlines():
            text_len = len(line) * FONT_WIDTH

            if should_newline:
                cursor_y += FONT_HEIGHT
                cursor_x = 0

                lines += 1
                if lines > terminal.height:
                    break

            text += _make_tag(
                "rect",
                x=cursor_x,
                y=cursor_y - (baseline_offset if not _is_block(line) else 0),
                fill=back or default_back,
                width=text_len * 1.02,
                height=FONT_HEIGHT,
            )

            text += _make_tag(
                "text",
                _escape_text(line),
                x=cursor_x,
                y=cursor_y + FONT_SIZE,
                textLength=text_len,
                raw=style_attr,
            )

            cursor_x += text_len
            should_newline = True

        if lines > terminal.height:
            break

        if plain.plain.endswith("\n"):
            cursor_y += FONT_HEIGHT
            cursor_x = 0

            lines += 1

    output = (
        _make_tag(
            "g",
            text,
            transform=(
                f"translate({TEXT_MARGIN_LEFT + SVG_MARGIN_LEFT}, "
                + f"{TEXT_MARGIN_TOP + SVG_MARGIN_TOP})"
            ),
        )
        + "\n"
    )

    stylesheet = "" if inline_styles else _generate_stylesheet(document_styles, prefix)

    terminal_width = terminal.width * FONT_WIDTH + 2 * TEXT_MARGIN_LEFT
    terminal_height = terminal.height * FONT_HEIGHT + 2 * TEXT_MARGIN_TOP

    return formatter.format(
        # Dimensions
        total_width=terminal_width + 2 * SVG_MARGIN_LEFT,
        total_height=terminal_height + 2 * SVG_MARGIN_TOP,
        terminal_width=terminal_width * 1.02,
        terminal_height=terminal_height - 15,
        # Styles
        background=default_back,
        stylesheet=stylesheet,
        # Title information
        title=title,
        title_x=terminal_width // 2 + 30,
        title_y=SVG_MARGIN_TOP + FONT_HEIGHT,
        # Code
        code=output,
        prefix=prefix,
    )
