/*
 * Decompiled with CFR 0.152.
 */
package net.librec.math.structure;

import java.io.Serializable;
import java.util.Arrays;
import net.librec.common.LibrecException;
import net.librec.math.algorithm.Randoms;
import net.librec.math.algorithm.SVD;
import net.librec.math.structure.DataMatrix;
import net.librec.math.structure.DenseVector;
import net.librec.math.structure.MatrixEntry;
import net.librec.math.structure.SparseMatrix;
import net.librec.math.structure.SparseVector;
import net.librec.math.structure.VectorEntry;
import net.librec.util.StringUtil;

public class DenseMatrix
implements DataMatrix,
Serializable {
    private static final long serialVersionUID = -2069621030647530185L;
    public int numRows;
    public int numColumns;
    public int topN;
    public double[][] data;

    public DenseMatrix(int numRows, int numColumns) {
        this.numRows = numRows;
        this.numColumns = numColumns;
        this.data = new double[numRows][numColumns];
    }

    public DenseMatrix(int numRows, int numColumns, int topN) {
        this.numRows = numRows;
        this.numColumns = numColumns;
        this.topN = topN;
        this.data = new double[numRows][numColumns];
    }

    public DenseMatrix(double[][] array) {
        this(array.length, array[0].length);
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                this.data[i][j] = array[i][j];
            }
        }
    }

    public DenseMatrix(double[][] array, int numRows, int numColumns) {
        this.numRows = numRows;
        this.numColumns = numColumns;
        this.data = array;
    }

    public DenseMatrix(DenseMatrix mat) {
        this(mat.data);
    }

    public DenseMatrix clone() {
        return new DenseMatrix(this);
    }

    public static DenseMatrix eye(int dim) {
        DenseMatrix mat = new DenseMatrix(dim, dim);
        for (int i = 0; i < mat.numRows; ++i) {
            mat.set(i, i, 1.0);
        }
        return mat;
    }

    public void init(double mean, double sigma) {
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                this.data[i][j] = Randoms.gaussian(mean, sigma);
            }
        }
    }

    public void init(double range) {
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                this.data[i][j] = Randoms.uniform(0.0, range);
            }
        }
    }

    public void init() {
        this.init(1.0);
    }

    public int numRows() {
        return this.numRows;
    }

    public int numColumns() {
        return this.numColumns;
    }

    public DenseVector row(int rowId) {
        return this.row(rowId, true);
    }

    public DenseVector row(int rowId, boolean deep) {
        return new DenseVector(this.data[rowId], deep);
    }

    public DenseMatrix getSubMatrix(int rowStart, int rowEnd, int colStart, int colEnd) {
        if (rowStart >= rowEnd || colStart >= colEnd) {
            return null;
        }
        int r = rowEnd - rowStart + 1;
        int c = colEnd - colStart + 1;
        double[][] d = new double[r][c];
        for (int i = rowStart; i <= rowEnd; ++i) {
            for (int j = colStart; j <= colEnd; ++j) {
                double a;
                d[i - rowStart][j - colStart] = a = this.data[i][j];
            }
        }
        return new DenseMatrix(d);
    }

    public DenseVector column(int column) {
        DenseVector vec = new DenseVector(this.numRows);
        for (int i = 0; i < this.numRows; ++i) {
            vec.set(i, this.data[i][column]);
        }
        return vec;
    }

    public double columnMean(int column) {
        double sum = 0.0;
        for (int i = 0; i < this.numRows; ++i) {
            sum += this.data[i][column];
        }
        return sum / (double)this.numRows;
    }

    public double norm() {
        double res = 0.0;
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                res += this.data[i][j] * this.data[i][j];
            }
        }
        return Math.sqrt(res);
    }

    public static double rowMult(DenseMatrix m, int mrow, DenseMatrix n, int nrow) {
        assert (m.numColumns == n.numColumns);
        double res = 0.0;
        int k = m.numColumns;
        for (int j = 0; j < k; ++j) {
            res += m.get(mrow, j) * n.get(nrow, j);
        }
        return res;
    }

    public static double colMult(DenseMatrix m, int mcol, DenseMatrix n, int ncol) {
        assert (m.numRows == n.numRows);
        double res = 0.0;
        int k = m.numRows;
        for (int j = 0; j < k; ++j) {
            res += m.get(j, mcol) * n.get(j, ncol);
        }
        return res;
    }

    public static double product(DenseMatrix m, int mrow, DenseMatrix n, int ncol) throws LibrecException {
        if (m.numColumns != n.numRows) {
            throw new LibrecException("m.numColumns should equal to n.numRows");
        }
        double res = 0.0;
        for (int j = 0; j < m.numColumns; ++j) {
            res += m.get(mrow, j) * n.get(j, ncol);
        }
        return res;
    }

    public static DenseMatrix kroneckerProduct(DenseMatrix M, DenseMatrix N) {
        DenseMatrix res = new DenseMatrix(M.numRows * N.numRows, M.numColumns * N.numColumns);
        for (int i = 0; i < M.numRows; ++i) {
            for (int j = 0; j < M.numColumns; ++j) {
                double Mij = M.get(i, j);
                for (int ni = 0; ni < N.numRows; ++ni) {
                    for (int nj = 0; nj < N.numColumns; ++nj) {
                        int row = i * N.numRows + ni;
                        int col = j * N.numColumns + nj;
                        res.set(row, col, Mij * N.get(ni, nj));
                    }
                }
            }
        }
        return res;
    }

    public static DenseMatrix khatriRaoProduct(DenseMatrix M, DenseMatrix N) throws Exception {
        if (M.numColumns != N.numColumns) {
            throw new Exception("The number of columns of two matrices is not equal!");
        }
        DenseMatrix res = new DenseMatrix(M.numRows * N.numRows, M.numColumns);
        for (int j = 0; j < M.numColumns; ++j) {
            for (int i = 0; i < M.numRows; ++i) {
                double Mij = M.get(i, j);
                for (int ni = 0; ni < N.numRows; ++ni) {
                    int row = ni + i * N.numRows;
                    res.set(row, j, Mij * N.get(ni, j));
                }
            }
        }
        return res;
    }

    public static DenseMatrix hadamardProduct(DenseMatrix M, DenseMatrix N) throws Exception {
        if (M.numRows != N.numRows || M.numColumns != N.numColumns) {
            throw new Exception("The dimensions of two matrices are not consistent!");
        }
        DenseMatrix res = new DenseMatrix(M.numRows, M.numColumns);
        for (int i = 0; i < M.numRows; ++i) {
            for (int j = 0; j < M.numColumns; ++j) {
                res.set(i, j, M.get(i, j) * N.get(i, j));
            }
        }
        return res;
    }

    public DenseMatrix transMult() {
        DenseMatrix res = new DenseMatrix(this.numColumns, this.numColumns);
        for (int i = 0; i < this.numColumns; ++i) {
            for (int k = 0; k < this.numColumns; ++k) {
                double val = 0.0;
                for (int j = 0; j < this.numRows; ++j) {
                    val += this.get(j, i) * this.get(j, k);
                }
                res.set(i, k, val);
            }
        }
        return res;
    }

    public DenseMatrix mult(DenseMatrix mat) throws LibrecException {
        if (this.numColumns != mat.numRows) {
            throw new LibrecException("this.numColumns should equal to mat.numRows");
        }
        DenseMatrix res = new DenseMatrix(this.numRows, mat.numColumns);
        for (int i = 0; i < res.numRows; ++i) {
            for (int j = 0; j < res.numColumns; ++j) {
                double product = 0.0;
                for (int k = 0; k < this.numColumns; ++k) {
                    product += this.data[i][k] * mat.data[k][j];
                }
                res.set(i, j, product);
            }
        }
        return res;
    }

    public DenseMatrix mult(SparseMatrix mat) throws LibrecException {
        if (this.numColumns != mat.numRows) {
            throw new LibrecException("numColumns should equal to numRows");
        }
        DenseMatrix res = new DenseMatrix(this.numRows, mat.numColumns);
        for (int j = 0; j < res.numColumns; ++j) {
            SparseVector col = mat.column(j);
            for (int i = 0; i < res.numRows; ++i) {
                double product = 0.0;
                for (VectorEntry ve : col) {
                    product += this.data[i][ve.index()] * ve.get();
                }
                res.set(i, j, product);
            }
        }
        return res;
    }

    public DenseVector mult(DenseVector vec) throws LibrecException {
        if (this.numColumns != vec.size) {
            throw new LibrecException("this.numColumns should equal to vec.size");
        }
        DenseVector res = new DenseVector(this.numRows);
        for (int i = 0; i < this.numRows; ++i) {
            res.set(i, this.row(i, false).inner(vec));
        }
        return res;
    }

    public DenseVector mult(SparseVector vec) {
        DenseVector res = new DenseVector(this.numRows);
        for (int i = 0; i < this.numRows; ++i) {
            double product = 0.0;
            for (VectorEntry ve : vec) {
                product += this.data[i][ve.index()] * ve.get();
            }
            res.set(i, product);
        }
        return res;
    }

    public static DenseMatrix mult(SparseMatrix sm, DenseMatrix dm) throws LibrecException {
        if (sm.numColumns != dm.numRows) {
            throw new LibrecException("sm.numColumns should equal to dm.numRows");
        }
        DenseMatrix res = new DenseMatrix(sm.numRows, dm.numColumns);
        for (int i = 0; i < res.numRows; ++i) {
            SparseVector row = sm.row(i);
            for (int j = 0; j < res.numColumns; ++j) {
                double product = 0.0;
                for (int k : row.getIndex()) {
                    product += row.get(k) * dm.data[k][j];
                }
                res.set(i, j, product);
            }
        }
        return res;
    }

    @Override
    public double get(int row, int column) {
        return this.data[row][column];
    }

    @Override
    public void set(int row, int column, double val) {
        if (this.topN >= 0) {
            this.data[row][column] = val;
        }
    }

    public void setAll(double val) {
        for (int row = 0; row < this.numRows; ++row) {
            for (int col = 0; col < this.numColumns; ++col) {
                this.data[row][col] = val;
            }
        }
    }

    public double sumOfRow(int row) {
        double res = 0.0;
        for (int col = 0; col < this.numColumns; ++col) {
            res += this.data[row][col];
        }
        return res;
    }

    public double sumOfColumn(int col) {
        double res = 0.0;
        for (int row = 0; row < this.numRows; ++row) {
            res += this.data[row][col];
        }
        return res;
    }

    public double sum() {
        double res = 0.0;
        for (int row = 0; row < this.numRows; ++row) {
            for (int col = 0; col < this.numColumns; ++col) {
                res += this.data[row][col];
            }
        }
        return res;
    }

    public DenseMatrix scale(double val) {
        DenseMatrix mat = new DenseMatrix(this.numRows, this.numColumns);
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                mat.data[i][j] = this.data[i][j] * val;
            }
        }
        return mat;
    }

    public DenseMatrix scaleEqual(double val) {
        for (int i = 0; i < this.numRows; ++i) {
            int j = 0;
            while (j < this.numColumns) {
                double[] dArray = this.data[i];
                int n = j++;
                dArray[n] = dArray[n] * val;
            }
        }
        return this;
    }

    public void add(int row, int column, double val) {
        double[] dArray = this.data[row];
        int n = column;
        dArray[n] = dArray[n] + val;
    }

    public DenseMatrix add(DenseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                res.data[i][j] = this.data[i][j] + mat.data[i][j];
            }
        }
        return res;
    }

    public DenseMatrix addEqual(DenseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                double[] dArray = this.data[i];
                int n = j;
                dArray[n] = dArray[n] + mat.data[i][j];
            }
        }
        return this;
    }

    public DenseMatrix add(SparseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        DenseMatrix res = this.clone();
        for (MatrixEntry me : mat) {
            res.add(me.row(), me.column(), me.get());
        }
        return res;
    }

    public DenseMatrix addEqual(SparseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        for (MatrixEntry me : mat) {
            double[] dArray = this.data[me.row()];
            int n = me.column();
            dArray[n] = dArray[n] + me.get();
        }
        return this;
    }

    public DenseMatrix add(double val) {
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                res.data[i][j] = this.data[i][j] + val;
            }
        }
        return res;
    }

    public DenseMatrix addEqual(double val) {
        for (int i = 0; i < this.numRows; ++i) {
            int j = 0;
            while (j < this.numColumns) {
                double[] dArray = this.data[i];
                int n = j++;
                dArray[n] = dArray[n] + val;
            }
        }
        return this;
    }

    public DenseMatrix minus(DenseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                res.data[i][j] = this.data[i][j] - mat.data[i][j];
            }
        }
        return res;
    }

    public DenseMatrix minusEqual(DenseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                double[] dArray = this.data[i];
                int n = j;
                dArray[n] = dArray[n] - mat.data[i][j];
            }
        }
        return this;
    }

    public DenseMatrix minus(SparseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        DenseMatrix res = this.clone();
        for (MatrixEntry me : mat) {
            res.add(me.row(), me.column(), -me.get());
        }
        return res;
    }

    public DenseMatrix minusEqual(SparseMatrix mat) throws LibrecException {
        if (this.numRows != mat.numRows) {
            throw new LibrecException("numRows should be equal");
        }
        if (this.numColumns != mat.numColumns) {
            throw new LibrecException("numColumns should be equal");
        }
        for (MatrixEntry me : mat) {
            double[] dArray = this.data[me.row()];
            int n = me.column();
            dArray[n] = dArray[n] - me.get();
        }
        return this;
    }

    public DenseMatrix minus(double val) {
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        for (int i = 0; i < this.numRows; ++i) {
            for (int j = 0; j < this.numColumns; ++j) {
                res.data[i][j] = this.data[i][j] - val;
            }
        }
        return res;
    }

    public DenseMatrix minusEqual(double val) {
        for (int i = 0; i < this.numRows; ++i) {
            int j = 0;
            while (j < this.numColumns) {
                double[] dArray = this.data[i];
                int n = j++;
                dArray[n] = dArray[n] - val;
            }
        }
        return this;
    }

    public DenseMatrix cholesky() {
        if (this.numRows != this.numColumns) {
            throw new RuntimeException("Matrix is not square");
        }
        int n = this.numRows;
        DenseMatrix L = new DenseMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j <= i; ++j) {
                double sum = 0.0;
                for (int k = 0; k < j; ++k) {
                    sum += L.get(i, k) * L.get(j, k);
                }
                double val = i == j ? Math.sqrt(this.data[i][i] - sum) : (this.data[i][j] - sum) / L.get(j, j);
                L.set(i, j, val);
            }
            if (!Double.isNaN(L.get(i, i))) continue;
            return null;
        }
        return L.transpose();
    }

    public DenseMatrix transpose() {
        DenseMatrix mat = new DenseMatrix(this.numColumns, this.numRows);
        for (int i = 0; i < mat.numRows; ++i) {
            for (int j = 0; j < mat.numColumns; ++j) {
                mat.set(i, j, this.data[j][i]);
            }
        }
        return mat;
    }

    public DenseMatrix cov() {
        DenseMatrix mat = new DenseMatrix(this.numColumns, this.numColumns);
        for (int i = 0; i < this.numColumns; ++i) {
            DenseVector xi = this.column(i);
            xi = xi.minus(xi.mean());
            mat.set(i, i, xi.inner(xi) / (double)(xi.size - 1));
            for (int j = i + 1; j < this.numColumns; ++j) {
                DenseVector yi = this.column(j);
                double val = xi.inner(yi.minus(yi.mean())) / (double)(xi.size - 1);
                mat.set(i, j, val);
                mat.set(j, i, val);
            }
        }
        return mat;
    }

    public DenseMatrix inverse() {
        int j;
        int i;
        int k;
        if (this.numRows != this.numColumns) {
            throw new RuntimeException("Only square matrix can do inversion");
        }
        int n = this.numRows;
        DenseMatrix mat = new DenseMatrix(this);
        if (n == 1) {
            mat.set(0, 0, 1.0 / mat.get(0, 0));
            return mat;
        }
        int[] row = new int[n];
        int[] col = new int[n];
        double[] temp = new double[n];
        for (k = 0; k < n; ++k) {
            row[k] = k;
            col[k] = k;
        }
        for (k = 0; k < n; ++k) {
            int j2;
            double pivot = mat.get(row[k], col[k]);
            int I_pivot = k;
            int J_pivot = k;
            for (i = k; i < n; ++i) {
                for (j2 = k; j2 < n; ++j2) {
                    double abs_pivot = Math.abs(pivot);
                    if (!(Math.abs(mat.get(row[i], col[j2])) > abs_pivot)) continue;
                    I_pivot = i;
                    J_pivot = j2;
                    pivot = mat.get(row[i], col[j2]);
                }
            }
            if (Math.abs(pivot) < 1.0E-10) {
                throw new RuntimeException("Matrix is singular !");
            }
            int hold = row[k];
            row[k] = row[I_pivot];
            row[I_pivot] = hold;
            hold = col[k];
            col[k] = col[J_pivot];
            col[J_pivot] = hold;
            mat.set(row[k], col[k], 1.0 / pivot);
            for (j = 0; j < n; ++j) {
                if (j == k) continue;
                mat.set(row[k], col[j], mat.get(row[k], col[j]) * mat.get(row[k], col[k]));
            }
            for (i = 0; i < n; ++i) {
                if (k == i) continue;
                for (j2 = 0; j2 < n; ++j2) {
                    if (k == j2) continue;
                    double val = mat.get(row[i], col[j2]) - mat.get(row[i], col[k]) * mat.get(row[k], col[j2]);
                    mat.set(row[i], col[j2], val);
                }
                mat.set(row[i], col[k], -mat.get(row[i], col[k]) * mat.get(row[k], col[k]));
            }
        }
        for (int j3 = 0; j3 < n; ++j3) {
            for (i = 0; i < n; ++i) {
                temp[col[i]] = mat.get(row[i], j3);
            }
            for (i = 0; i < n; ++i) {
                mat.set(i, j3, temp[i]);
            }
        }
        for (int i2 = 0; i2 < n; ++i2) {
            for (j = 0; j < n; ++j) {
                temp[row[j]] = mat.get(i2, col[j]);
            }
            for (j = 0; j < n; ++j) {
                mat.set(i2, j, temp[j]);
            }
        }
        return mat;
    }

    public DenseMatrix inv() {
        if (this.numRows != this.numColumns) {
            throw new RuntimeException("Dimensions disagree");
        }
        int n = this.numRows;
        DenseMatrix mat = DenseMatrix.eye(n);
        if (n == 1) {
            mat.set(0, 0, 1.0 / this.get(0, 0));
            return mat;
        }
        DenseMatrix b = new DenseMatrix(this);
        for (int i = 0; i < n; ++i) {
            double mag2;
            int j;
            double mag = 0.0;
            int pivot = -1;
            for (j = i; j < n; ++j) {
                mag2 = Math.abs(b.get(j, i));
                if (!(mag2 > mag)) continue;
                mag = mag2;
                pivot = j;
            }
            if (pivot == -1 || mag == 0.0) {
                return mat;
            }
            if (pivot != i) {
                int j2;
                for (j2 = i; j2 < n; ++j2) {
                    double temp = b.get(i, j2);
                    b.set(i, j2, b.get(pivot, j2));
                    b.set(pivot, j2, temp);
                }
                for (j2 = 0; j2 < n; ++j2) {
                    double temp = mat.get(i, j2);
                    mat.set(i, j2, mat.get(pivot, j2));
                    mat.set(pivot, j2, temp);
                }
            }
            mag = b.get(i, i);
            for (j = i; j < n; ++j) {
                b.set(i, j, b.get(i, j) / mag);
            }
            for (j = 0; j < n; ++j) {
                mat.set(i, j, mat.get(i, j) / mag);
            }
            for (int k = 0; k < n; ++k) {
                int j3;
                if (k == i) continue;
                mag2 = b.get(k, i);
                for (j3 = i; j3 < n; ++j3) {
                    b.set(k, j3, b.get(k, j3) - mag2 * b.get(i, j3));
                }
                for (j3 = 0; j3 < n; ++j3) {
                    mat.set(k, j3, mat.get(k, j3) - mag2 * mat.get(i, j3));
                }
            }
        }
        return mat;
    }

    public DenseMatrix pinv() throws LibrecException {
        if (this.numRows < this.numColumns) {
            DenseMatrix res = this.transpose().pinv();
            if (res != null) {
                res = res.transpose();
            }
            return res;
        }
        SVD svd = this.svd();
        DenseMatrix U = svd.getU();
        DenseMatrix S = svd.getS();
        DenseMatrix V = svd.getV();
        DenseMatrix SPlus = S.clone();
        for (int i = 0; i < SPlus.numRows; ++i) {
            double val = SPlus.get(i, i);
            if (val == 0.0) continue;
            SPlus.set(i, i, 1.0 / val);
        }
        return V.mult(SPlus).mult(U.transpose());
    }

    public SVD svd() {
        return new SVD(this);
    }

    public void setRow(int row, double val) {
        Arrays.fill(this.data[row], val);
    }

    public void setRow(int row, DenseVector vals) {
        for (int j = 0; j < this.numColumns; ++j) {
            this.data[row][j] = vals.data[j];
        }
    }

    public void clear() {
        for (int i = 0; i < this.numRows; ++i) {
            this.setRow(i, 0.0);
        }
    }

    public String toString() {
        return StringUtil.toString(this.data);
    }

    public double[][] getData() {
        return this.data;
    }

    @Override
    public int size() {
        return this.numRows * this.numColumns;
    }
}

