import sys
import pytest
import numpy as np
from numpy.testing import assert_almost_equal, assert_equal
from scipy.linalg import orth
from mvlearn.embed import GCCA


################## Helper Functions ####################
def generate_data(n=10, elbows=3, seed=1):
    """
    Generate data matrix with a specific number of elbows on scree plot
    """
    np.random.seed(seed)
    x = np.random.binomial(1, 0.6, (n ** 2)).reshape(n, n)
    xorth = orth(x)
    d = np.zeros(xorth.shape[0])
    for i in range(0, len(d), int(len(d) / (elbows + 1))):
        d[:i] += 10
    A = xorth.T.dot(np.diag(d)).dot(xorth)
    return A, d

def _get_Xs(n_views=2):
    np.random.seed(0)
    n_obs = 4
    n_features = 6
    X = np.random.normal(0, 1, size=(n_views, n_obs, n_features))
    return X

def _compute_dissimilarity(arr):
    n = len(arr)
    out = np.zeros((n, n))
    for i in range(n):
        out[i] = np.linalg.norm(arr - arr[i])
    return out

################## Test Functionality ####################
def test_fit_transform():
    n = 2
    Xs = _get_Xs(n)

    projs = GCCA(fraction_var=0.9).fit_transform(Xs)
    dists = _compute_dissimilarity(projs)

    # Checks up to 7 decimal points
    assert_almost_equal(np.zeros((n, n)), dists)

def test_fit_view_idx():
    n = 2
    Xs = _get_Xs(n)

    gcca = GCCA(fraction_var=0.9).fit(Xs)
    projs = [gcca.transform(Xs[i], view_idx=i) for i in range(n)]

    dists = _compute_dissimilarity(projs)

    # Checks up to 7 decimal points
    assert_almost_equal(np.zeros((n, n)), dists)

def test_fit_tall():
    n = 2
    Xs = _get_Xs(n)

    projs = GCCA(fraction_var=0.9, tall=True).fit_transform(Xs)
    dists = _compute_dissimilarity(projs)

    # Checks up to 7 decimal points
    assert_almost_equal(np.zeros((n, n)), dists)

def test_fit_n_components():
    n = 2
    Xs = _get_Xs(n)

    projs = GCCA(n_components=3).fit_transform(Xs)
    dists = _compute_dissimilarity(projs)

    # Checks up to 7 decimal points
    assert_almost_equal(np.zeros((n, n)), dists)

def test_fit_sv_tolerance():
    n = 2
    Xs = _get_Xs(n)

    projs = GCCA(sv_tolerance=1).fit_transform(Xs)
    dists = _compute_dissimilarity(projs)

    # Checks up to 7 decimal points
    assert_almost_equal(np.zeros((n, n)), dists)

def test_fit_elbows():
    X, _ = generate_data(10, 3)
    Xs = [X, X]

    gcca = GCCA(n_elbows=2)
    _ = gcca.fit_transform(Xs)

    assert_equal(gcca.ranks_[0], 4)

def test_max_ranks():
    X, _ = generate_data(10, 3)
    Xs = [X, X]

    gcca = GCCA(n_elbows=2, max_rank=True)
    projs = gcca.fit_transform(Xs)

    assert_equal(gcca.ranks_[0], 4)

def test_n_jobs():
    X, _ = generate_data(10, 3)
    Xs = [X, X]
    gcca = GCCA(n_elbows=2, n_jobs=-1)
    _ = gcca.fit_transform(Xs)

def test_partial_equals_full():
    X1, _ = generate_data(10, 3, seed=1)
    X2 = X1*np.random.normal(0,1,X1.shape)
    Xs = [X1, X2]
    gcca = GCCA()
    projs_full = gcca.fit(Xs).transform(Xs)
    projs_partial = gcca.partial_fit(Xs, reset=True).transform(Xs)
    assert_almost_equal(np.abs(projs_full), np.abs(projs_partial))

def test_partial_multistep():
    X1, _ = generate_data(10, 3, seed=1)
    X2 = X1*np.random.normal(0,1,X1.shape)
    Xs = [X1, X2]
    gcca = GCCA()
    projs_full = gcca.partial_fit(Xs).transform(Xs)
    projs_partial = gcca.partial_fit(
        Xs[0], reset=True, multiview_step=False
        ).partial_fit(
            Xs[1], reset=False, multiview_step=True
        ).transform(Xs)
    assert_almost_equal(np.abs(projs_full), np.abs(projs_partial))

################## Test Error/Warning Calls ####################
test_mat = np.array([[1, 2], [3, 4]])
mat_good = np.ones((2, 4, 2))
Xs = np.random.normal(0, 1, size=(2, 4, 6))
@pytest.mark.parametrize(
    "Xs,params,err",
    [
        ({"Xs": [[]]}, {}, ValueError),  # Empty input
        ({"Xs": test_mat}, {}, ValueError),  # Single matrix input
        ({"Xs": mat_good}, {"fraction_var": "fail"}, TypeError),
        ({"Xs": mat_good}, {"fraction_var": -1}, ValueError),
        ({"Xs": mat_good}, {"n_components": "fail"}, TypeError),
        ({"Xs": mat_good}, {"n_components": -1}, ValueError),
        ({"Xs": mat_good}, {"sv_tolerance": "fail"}, TypeError),
        ({"Xs": mat_good}, {"sv_tolerance": -1}, ValueError),
        ({"Xs": mat_good}, {"n_components": mat_good.shape[1]}, ValueError)
    ],
)
def test_bad_inputs(Xs, params, err):
    with pytest.raises(err):
        np.random.seed(1)
        GCCA(**params).fit(**Xs)

def test_no_fit(Xs={"Xs": mat_good}, err=RuntimeError):
    with pytest.raises(err):
        np.random.seed(1)
        GCCA().transform(**Xs)

def test_multiview_step():
    X1, _ = generate_data(10, 3, seed=1)
    gcca = GCCA()
    with pytest.warns(UserWarning):
        projs_partial = gcca.partial_fit(X1)