# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['returns',
 'returns._internal',
 'returns._internal.futures',
 'returns._internal.pipeline',
 'returns.context',
 'returns.contrib',
 'returns.contrib.hypothesis',
 'returns.contrib.mypy',
 'returns.contrib.mypy._features',
 'returns.contrib.mypy._structures',
 'returns.contrib.mypy._typeops',
 'returns.contrib.pytest',
 'returns.interfaces',
 'returns.interfaces.specific',
 'returns.methods',
 'returns.pointfree',
 'returns.primitives']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=3.7,<4.0']

entry_points = \
{'hypothesis': ['_ = returns.contrib.hypothesis._entrypoint'],
 'pytest11': ['returns = returns.contrib.pytest.plugin']}

setup_kwargs = {
    'name': 'returns',
    'version': '0.16.0',
    'description': 'Make your functions return something meaningful, typed, and safe!',
    'long_description': '[![Returns logo](https://raw.githubusercontent.com/dry-python/brand/master/logo/returns_white-outline.png)](https://github.com/dry-python/returns)\n\n-----\n\n[![Build Status](https://github.com/dry-python/returns/workflows/test/badge.svg?branch=master&event=push)](https://github.com/dry-python/returns/actions?query=workflow%3Atest)\n[![codecov](https://codecov.io/gh/dry-python/returns/branch/master/graph/badge.svg)](https://codecov.io/gh/dry-python/returns)\n[![Documentation Status](https://readthedocs.org/projects/returns/badge/?version=latest)](https://returns.readthedocs.io/en/latest/?badge=latest)\n[![Python Version](https://img.shields.io/pypi/pyversions/returns.svg)](https://pypi.org/project/returns/)\n[![wemake-python-styleguide](https://img.shields.io/badge/style-wemake-000000.svg)](https://github.com/wemake-services/wemake-python-styleguide)\n[![Telegram chat](https://img.shields.io/badge/chat-join-blue?logo=telegram)](https://t.me/drypython)\n\n-----\n\nMake your functions return something meaningful, typed, and safe!\n\n\n## Features\n\n- Brings functional programming to Python land\n- Provides a bunch of primitives to write declarative business logic\n- Enforces better architecture\n- Fully typed with annotations and checked with `mypy`, [PEP561 compatible](https://www.python.org/dev/peps/pep-0561/)\n- Adds emulated Higher Kinded Types support\n- Provides type-safe interfaces to create your own data-types with enforced laws\n- Has a bunch of helpers for better composition\n- Pythonic and pleasant to write and to read 🐍\n- Support functions and coroutines, framework agnostic\n- Easy to start: has lots of docs, tests, and tutorials\n\n[Quickstart](https://returns.readthedocs.io/en/latest/pages/quickstart.html) right now!\n\n\n## Installation\n\n```bash\npip install returns\n```\n\nYou are also required to [configure](https://returns.readthedocs.io/en/latest/pages/container.html#type-safety)\n`mypy` correctly and install our plugin\nto fix [this existing issue](https://github.com/python/mypy/issues/3157):\n\n```ini\n# In setup.cfg or mypy.ini:\n[mypy]\nplugins =\n  returns.contrib.mypy.returns_plugin\n```\n\nWe also recommend to use the same `mypy` settings [we use](https://github.com/wemake-services/wemake-python-styleguide/blob/master/styles/mypy.toml).\n\nMake sure you know how to get started, [check out our docs](https://returns.readthedocs.io/en/latest/)!\n[Try our demo](https://repl.it/@sobolevn/returns#ex.py).\n\n\n## Contents\n\n- [Maybe container](#maybe-container) that allows you to write `None`-free code\n- [RequiresContext container](#requirescontext-container) that allows you to use typed functional dependency injection\n- [Result container](#result-container) that let\'s you to get rid of exceptions\n- [IO container](#io-container) and [IOResult](#troublesome-io) that marks all impure operations and structures them\n- [Future container](#future-container) and [FutureResult](#async-code-without-exceptions) to work with `async` code\n- [Write your own container!](https://returns.readthedocs.io/en/latest/pages/create-your-own-container.html) You would still have all the features for your own types (including full existing code reuse and type-safety)\n\n\n## Maybe container\n\n`None` is called the [worst mistake in the history of Computer Science](https://www.infoq.com/presentations/Null-References-The-Billion-Dollar-Mistake-Tony-Hoare/).\n\nSo, what can we do to check for `None` in our programs?\nYou can use builtin [Optional](https://mypy.readthedocs.io/en/stable/kinds_of_types.html#optional-types-and-the-none-type) type\nand write a lot of `if some is not None:` conditions.\nBut, **having `null` checks here and there makes your code unreadable**.\n\n```python\nuser: Optional[User]\ndiscount_program: Optional[\'DiscountProgram\'] = None\n\nif user is not None:\n     balance = user.get_balance()\n     if balance is not None:\n         credit = balance.credit_amount()\n         if credit is not None and credit > 0:\n             discount_program = choose_discount(credit)\n```\n\nOr you can use\n[Maybe](https://returns.readthedocs.io/en/latest/pages/maybe.html) container!\nIt consists of `Some` and `Nothing` types,\nrepresenting existing state and empty (instead of `None`) state respectively.\n\n```python\nfrom typing import Optional\nfrom returns.maybe import Maybe, maybe\n\n@maybe  # decorator to convert existing Optional[int] to Maybe[int]\ndef bad_function() -> Optional[int]:\n    ...\n\nmaybe_number: Maybe[float] = bad_function().bind_optional(\n    lambda number: number / 2,\n)\n# => Maybe will return Some[float] only if there\'s a non-None value\n#    Otherwise, will return Nothing\n```\n\nYou can be sure that `.bind_optional()` method won\'t be called for `Nothing`.\nForget about `None`-related errors forever!\n\nWe can also bind a `Optional`-returning function over a container.\nTo achieve this, we are going to use `.bind_optional` method.\n\nAnd that\'s how your initial refactored code will look like:\n\n```python\nuser: Optional[User]\n\n# Type hint here is optional, it only helps the reader here:\ndiscount_program: Maybe[\'DiscountProgram\'] = Maybe.from_optional(\n    user,\n).bind_optional(  # This won\'t be called if `user is None`\n    lambda real_user: real_user.get_balance(),\n).bind_optional(  # This won\'t be called if `real_user.get_balance()` is None\n    lambda balance: balance.credit_amount(),\n).bind_optional(  # And so on!\n    lambda credit: choose_discount(credit) if credit > 0 else None,\n)\n```\n\nMuch better, isn\'t it?\n\n\n## RequiresContext container\n\nMany developers do use some kind of dependency injection in Python.\nAnd usually it is based on the idea\nthat there\'s some kind of a container and assembly process.\n\nFunctional approach is much simpler!\n\nImagine that you have a `django` based game, where you award users with points for each guessed letter in a word (unguessed letters are marked as `\'.\'`):\n\n```python\nfrom django.http import HttpRequest, HttpResponse\nfrom words_app.logic import calculate_points\n\ndef view(request: HttpRequest) -> HttpResponse:\n    user_word: str = request.POST[\'word\']  # just an example\n    points = calculate_points(user_word)\n    ...  # later you show the result to user somehow\n\n# Somewhere in your `words_app/logic.py`:\n\ndef calculate_points(word: str) -> int:\n    guessed_letters_count = len([letter for letter in word if letter != \'.\'])\n    return _award_points_for_letters(guessed_letters_count)\n\ndef _award_points_for_letters(guessed: int) -> int:\n    return 0 if guessed < 5 else guessed  # minimum 6 points possible!\n```\n\nAwesome! It works, users are happy, your logic is pure and awesome.\nBut, later you decide to make the game more fun:\nlet\'s make the minimal accountable letters threshold\nconfigurable for an extra challenge.\n\nYou can just do it directly:\n\n```python\ndef _award_points_for_letters(guessed: int, threshold: int) -> int:\n    return 0 if guessed < threshold else guessed\n```\n\nThe problem is that `_award_points_for_letters` is deeply nested.\nAnd then you have to pass `threshold` through the whole callstack,\nincluding `calculate_points` and all other functions that might be on the way.\nAll of them will have to accept `threshold` as a parameter!\nThis is not useful at all!\nLarge code bases will struggle a lot from this change.\n\nOk, you can directly use `django.settings` (or similar)\nin your `_award_points_for_letters` function.\nAnd **ruin your pure logic with framework specific details**. That\'s ugly!\n\nOr you can use `RequiresContext` container. Let\'s see how our code changes:\n\n```python\nfrom django.conf import settings\nfrom django.http import HttpRequest, HttpResponse\nfrom words_app.logic import calculate_points\n\ndef view(request: HttpRequest) -> HttpResponse:\n    user_word: str = request.POST[\'word\']  # just an example\n    points = calculate_points(user_words)(settings)  # passing the dependencies\n    ...  # later you show the result to user somehow\n\n# Somewhere in your `words_app/logic.py`:\n\nfrom typing_extensions import Protocol\nfrom returns.context import RequiresContext\n\nclass _Deps(Protocol):  # we rely on abstractions, not direct values or types\n    WORD_THRESHOLD: int\n\ndef calculate_points(word: str) -> RequiresContext[int, _Deps]:\n    guessed_letters_count = len([letter for letter in word if letter != \'.\'])\n    return _award_points_for_letters(guessed_letters_count)\n\ndef _award_points_for_letters(guessed: int) -> RequiresContext[int, _Deps]:\n    return RequiresContext(\n        lambda deps: 0 if guessed < deps.WORD_THRESHOLD else guessed,\n    )\n```\n\nAnd now you can pass your dependencies in a really direct and explicit way.\nAnd have the type-safety to check what you pass to cover your back.\nCheck out [RequiresContext](https://returns.readthedocs.io/en/latest/pages/context.html) docs for more. There you will learn how to make `\'.\'` also configurable.\n\nWe also have [RequiresContextResult](https://returns.readthedocs.io/en/latest/pages/context.html#requirescontextresult-container)\nfor context-related operations that might fail. And also [RequiresContextIOResult](https://returns.readthedocs.io/en/latest/pages/context.html#requirescontextioresult-container) and [RequiresContextFutureResult](https://returns.readthedocs.io/en/latest/pages/context.html#requirescontextfutureresult-container).\n\n\n## Result container\n\nPlease, make sure that you are also aware of\n[Railway Oriented Programming](https://fsharpforfunandprofit.com/rop/).\n\n### Straight-forward approach\n\nConsider this code that you can find in **any** `python` project.\n\n```python\nimport requests\n\ndef fetch_user_profile(user_id: int) -> \'UserProfile\':\n    """Fetches UserProfile dict from foreign API."""\n    response = requests.get(\'/api/users/{0}\'.format(user_id))\n    response.raise_for_status()\n    return response.json()\n```\n\nSeems legit, does it not?\nIt also seems like a pretty straightforward code to test.\nAll you need is to mock `requests.get` to return the structure you need.\n\nBut, there are hidden problems in this tiny code sample\nthat are almost impossible to spot at the first glance.\n\n### Hidden problems\n\nLet\'s have a look at the exact same code,\nbut with the all hidden problems explained.\n\n```python\nimport requests\n\ndef fetch_user_profile(user_id: int) -> \'UserProfile\':\n    """Fetches UserProfile dict from foreign API."""\n    response = requests.get(\'/api/users/{0}\'.format(user_id))\n\n    # What if we try to find user that does not exist?\n    # Or network will go down? Or the server will return 500?\n    # In this case the next line will fail with an exception.\n    # We need to handle all possible errors in this function\n    # and do not return corrupt data to consumers.\n    response.raise_for_status()\n\n    # What if we have received invalid JSON?\n    # Next line will raise an exception!\n    return response.json()\n```\n\nNow, all (probably all?) problems are clear.\nHow can we be sure that this function will be safe\nto use inside our complex business logic?\n\nWe really cannot be sure!\nWe will have to create **lots** of `try` and `except` cases\njust to catch the expected exceptions. Our code will become complex and unreadable with all this mess!\n\nOr we can go with the top level `except Exception:` case\nto catch literally everything.\nAnd this way we would end up with catching unwanted ones.\nThis approach can hide serious problems from us for a long time.\n\n### Pipe example\n\n```python\nimport requests\nfrom returns.result import Result, safe\nfrom returns.pipeline import flow\nfrom returns.pointfree import bind\n\ndef fetch_user_profile(user_id: int) -> Result[\'UserProfile\', Exception]:\n    """Fetches `UserProfile` TypedDict from foreign API."""\n    return flow(\n        user_id,\n        _make_request,\n        bind(_parse_json),\n    )\n\n@safe\ndef _make_request(user_id: int) -> requests.Response:\n    # TODO: we are not yet done with this example, read more about `IO`:\n    response = requests.get(\'/api/users/{0}\'.format(user_id))\n    response.raise_for_status()\n    return response\n\n@safe\ndef _parse_json(response: requests.Response) -> \'UserProfile\':\n    return response.json()\n```\n\nNow we have a clean and a safe and declarative way\nto express our business needs:\n\n- We start from making a request, that might fail at any moment,\n- Then parsing the response if the request was successful,\n- And then return the result.\n\nNow, instead of returning regular values\nwe return values wrapped inside a special container\nthanks to the\n[@safe](https://returns.readthedocs.io/en/latest/pages/result.html#safe)\ndecorator. It will return [Success[YourType] or Failure[Exception]](https://returns.readthedocs.io/en/latest/pages/result.html).\nAnd will never throw exception at us!\n\nWe also use [flow](https://returns.readthedocs.io/en/latest/pages/pipeline.html#flow)\nand [bind](https://returns.readthedocs.io/en/latest/pages/pointfree.html#bind)\nfunctions for handy and declarative composition.\n\nThis way we can be sure that our code won\'t break in\nrandom places due to some implicit exception.\nNow we control all parts and are prepared for the explicit errors.\n\nWe are not yet done with this example,\nlet\'s continue to improve it in the next chapter.\n\n\n## IO container\n\nLet\'s look at our example from another angle.\nAll its functions look like regular ones:\nit is impossible to tell whether they are [pure](https://en.wikipedia.org/wiki/Pure_function)\nor impure from the first sight.\n\nIt leads to a very important consequence:\n*we start to mix pure and impure code together*.\nWe should not do that!\n\nWhen these two concepts are mixed\nwe suffer really bad when testing or reusing it.\nAlmost everything should be pure by default.\nAnd we should explicitly mark impure parts of the program.\n\nThat\'s why we have created `IO` container\nto mark impure functions that never fail.\n\nThese impure functions use `random`, current datetime, environment, or console:\n\n```python\nimport random\nimport datetime as dt\n\nfrom returns.io import IO\n\ndef get_random_number() -> IO[int]:  # or use `@impure` decorator\n    return IO(random.randint(1, 10))  # isn\'t pure, because random\n\nnow: Callable[[], IO[dt.datetime]] = impure(dt.datetime.now)\n\n@impure\ndef return_and_show_next_number(previous: int) -> int:\n    next_number = previous + 1\n    print(next_number)  # isn\'t pure, because does IO\n    return next_number\n```\n\nNow we can clearly see which functions are pure and which ones are impure.\nThis helps us a lot in building large applications, unit testing you code,\nand composing business logic together.\n\n### Troublesome IO\n\nAs it was already said, we use `IO` when we handle functions that do not fail.\n\nWhat if our function can fail and is impure?\nLike `requests.get()` we had earlier in our example.\n\nThen we have to use a special `IOResult` type instead of a regular `Result`.\nLet\'s find the difference:\n\n- Our `_parse_json` function always returns\n  the same result (hopefully) for the same input:\n  you can either parse valid `json` or fail on invalid one.\n  That\'s why we return pure `Result`, there\'s no `IO` inside\n- Our `_make_request` function is impure and can fail.\n  Try to send two similar requests with and without internet connection.\n  The result will be different for the same input.\n  That\'s why we must use `IOResult` here: it can fail and has `IO`\n\nSo, in order to fulfill our requirement and separate pure code from impure one,\nwe have to refactor our example.\n\n### Explicit IO\n\nLet\'s make our [IO](https://returns.readthedocs.io/en/latest/pages/io.html)\nexplicit!\n\n```python\nimport requests\nfrom returns.io import IOResult, impure_safe\nfrom returns.result import safe\nfrom returns.pipeline import flow\nfrom returns.pointfree import bind_result\n\ndef fetch_user_profile(user_id: int) -> IOResult[\'UserProfile\', Exception]:\n    """Fetches `UserProfile` TypedDict from foreign API."""\n    return flow(\n        user_id,\n        _make_request,\n        # before: def (Response) -> UserProfile\n        # after safe: def (Response) -> ResultE[UserProfile]\n        # after bind_result: def (IOResultE[Response]) -> IOResultE[UserProfile]\n        bind_result(_parse_json),\n    )\n\n@impure_safe\ndef _make_request(user_id: int) -> requests.Response:\n    response = requests.get(\'/api/users/{0}\'.format(user_id))\n    response.raise_for_status()\n    return response\n\n@safe\ndef _parse_json(response: requests.Response) -> \'UserProfile\':\n    return response.json()\n```\n\nAnd later we can use [unsafe_perform_io](https://returns.readthedocs.io/en/latest/pages/io.html#unsafe-perform-io)\nsomewhere at the top level of our program to get the pure (or "real") value.\n\nAs a result of this refactoring session, we know everything about our code:\n\n- Which parts can fail,\n- Which parts are impure,\n- How to compose them in a smart, readable, and typesafe manner.\n\n\n## Future container\n\nThere are several issues with `async` code in Python:\n\n1. You cannot call `async` function from a sync one\n2. Any unexpectedly thrown exception can ruin your whole event loop\n3. Ugly composition with lots of `await` statements\n\n`Future` and `FutureResult` containers solve these issues!\n\n### Mixing sync and async code\n\nThe main feature of [Future](https://returns.readthedocs.io/en/latest/pages/future.html)\nis that it allows to run async code\nwhile maintaining sync context. Let\'s see an example.\n\nLet\'s say we have two functions,\nthe `first` one returns a number and the `second` one increments it:\n\n```python\nasync def first() -> int:\n    return 1\n\ndef second():  # How can we call `first()` from here?\n    return first() + 1  # Boom! Don\'t do this. We illustrate a problem here.\n```\n\nIf we try to just run `first()`, we will just create an unawaited coroutine.\nIt won\'t return the value we want.\n\nBut, if we would try to run `await first()`,\nthen we would need to change `second` to be `async`.\nAnd sometimes it is not possible for various reasons.\n\nHowever, with `Future` we can "pretend" to call async code from sync code:\n\n```python\nfrom returns.future import Future\n\ndef second() -> Future[int]:\n    return Future(first()).map(lambda num: num + 1)\n```\n\nWithout touching our `first` async function\nor making `second` async we have achieved our goal.\nNow, our async value is incremented inside a sync function.\n\nHowever, `Future` still requires to be executed inside a proper eventloop:\n\n```python\nimport anyio  # or asyncio, or any other lib\n\n# We can then pass our `Future` to any library: asyncio, trio, curio.\n# And use any event loop: regular, uvloop, even a custom one, etc\nassert anyio.run(second().awaitable) == 2\n```\n\nAs you can see `Future` allows you\nto work with async functions from a sync context.\nAnd to mix these two realms together.\nUse raw `Future` for operations that cannot fail or raise exceptions.\nPretty much the same logic we had with our `IO` container.\n\n### Async code without exceptions\n\nWe have already covered how [`Result`](#result-container) works\nfor both pure and impure code.\nThe main idea is: we don\'t raise exceptions, we return them.\nIt is **especially** critical in async code,\nbecause a single exception can ruin\nall our coroutines running in a single eventloop.\n\nWe have a handy combination of `Future` and `Result` containers: `FutureResult`.\nAgain, this is exactly like `IOResult`, but for impure async code.\nUse it when your `Future` might have problems:\nlike HTTP requests or filesystem operations.\n\nYou can easily turn any wild throwing coroutine into a calm `FutureResult`:\n\n```python\nimport anyio\nfrom returns.future import future_safe\nfrom returns.io import IOFailure\n\n@future_safe\nasync def raising():\n    raise ValueError(\'Not so fast!\')\n\nioresult = anyio.run(raising.awaitable)  # all `Future`s return IO containers\nassert ioresult == IOFailure(ValueError(\'Not so fast!\'))  # True\n```\n\nUsing `FutureResult` will keep your code safe from exceptions.\nYou can always `await` or execute inside an eventloop any `FutureResult`\nto get sync `IOResult` instance to work with it in a sync manner.\n\n### Better async composition\n\nPreviously, you had to do quite a lot of `await`ing while writing `async` code:\n\n```python\nasync def fetch_user(user_id: int) -> \'User\':\n    ...\n\nasync def get_user_permissions(user: \'User\') -> \'Permissions\':\n    ...\n\nasync def ensure_allowed(permissions: \'Permissions\') -> bool:\n    ...\n\nasync def main(user_id: int) -> bool:\n    # Also, don\'t forget to handle all possible errors with `try / except`!\n    user = await fetch_user(user_id)  # We will await each time we use a coro!\n    permissions = await get_user_permissions(user)\n    return await ensure_allowed(permissions)\n```\n\nSome people are ok with it, but some people don\'t like this imperative style.\nThe problem is that there was no choice.\n\nBut now, you can do the same thing in functional style!\nWith the help of `Future` and `FutureResult` containers:\n\n```python\nimport anyio\nfrom returns.future import FutureResultE, future_safe\nfrom returns.io import IOSuccess, IOFailure\n\n@future_safe\nasync def fetch_user(user_id: int) -> \'User\':\n    ...\n\n@future_safe\nasync def get_user_permissions(user: \'User\') -> \'Permissions\':\n    ...\n\n@future_safe\nasync def ensure_allowed(permissions: \'Permissions\') -> bool:\n    ...\n\ndef main(user_id: int) -> FutureResultE[bool]:\n    # We can now turn `main` into a sync function, it does not `await` at all.\n    # We also don\'t care about exceptions anymore, they are already handled.\n    return fetch_user(user_id).bind(get_user_permissions).bind(ensure_allowed)\n\ncorrect_user_id: int  # has required permissions\nbanned_user_id: int  # does not have required permissions\nwrong_user_id: int  # does not exist\n\n# We can have correct business results:\nassert anyio.run(main(correct_user_id).awaitable) == IOSuccess(True)\nassert anyio.run(main(banned_user_id).awaitable) == IOSuccess(False)\n\n# Or we can have errors along the way:\nassert anyio.run(main(wrong_user_id).awaitable) == IOFailure(\n    UserDoesNotExistError(...),\n)\n```\n\nOr even something really fancy:\n\n```python\nfrom returns.pointfree import bind\nfrom returns.pipeline import flow\n\ndef main(user_id: int) -> FutureResultE[bool]:\n    return flow(\n        fetch_user(user_id),\n        bind(get_user_permissions),\n        bind(ensure_allowed),\n    )\n```\n\nLater we can also refactor our logical functions to be sync\nand to return `FutureResult`.\n\nLovely, isn\'t it?\n\n\n## More!\n\nWant more?\n[Go to the docs!](https://returns.readthedocs.io)\nOr [try our demo](https://repl.it/@sobolevn/returns#ex.py).\nOr read these articles:\n\n- [Python exceptions considered an anti-pattern](https://sobolevn.me/2019/02/python-exceptions-considered-an-antipattern)\n- [Enforcing Single Responsibility Principle in Python](https://sobolevn.me/2019/03/enforcing-srp)\n- [Typed functional Dependency Injection in Python](https://sobolevn.me/2020/02/typed-functional-dependency-injection)\n- [How Async Should Have Been](https://sobolevn.me/2020/06/how-async-should-have-been)\n- [Higher Kinded Types in Python](https://sobolevn.me/2020/10/higher-kinded-types-in-python)\n- [Make Tests a Part of Your App](https://sobolevn.me/2021/02/make-tests-a-part-of-your-app)\n\nDo you have an article to submit? Feel free to open a pull request!\n\n<p align="center">&mdash; ⭐️ &mdash;</p>\n<p align="center"><i>Drylabs maintains dry-python and helps those who want to use it inside their organizations.</i></p>\n<p align="center"><i>Read more at <a href="https://drylabs.io">drylabs.io</a></i></p>\n',
    'author': 'sobolevn',
    'author_email': 'mail@sobolevn.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://returns.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
