"""Automation models for event scheduling and task automation."""

import uuid
from django.db import models
from django.db.models import Q
from django.core.exceptions import ValidationError
from .base import UUIDModel
from .company import Company
from .organization import Organization, OrganizationUser
from .user import CompanyUser


class Automation(UUIDModel):
    """
    Automatización principal basada en el patrón Trigger-Action.

    Una automatización define un conjunto de acciones que se ejecutan cuando
    se activan uno o más triggers (disparadores).
    """

    STATUS_CHOICES = [
        ('active', 'Active'),
        ('paused', 'Paused'),
        ('archived', 'Archived'),
        ('draft', 'Draft'),
    ]

    EXECUTION_MODE_CHOICES = [
        ('sequential', 'Sequential'),
        ('parallel', 'Parallel'),  # Futuro
    ]

    ON_ERROR_CHOICES = [
        ('stop', 'Stop'),
        ('continue', 'Continue'),
        ('retry', 'Retry'),
    ]

    # FKs - Al menos uno debe estar presente
    company = models.ForeignKey(
        Company,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        db_index=True,
        help_text="Company específica (opcional si se define organization)"
    )
    organization = models.ForeignKey(
        Organization,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        db_index=True,
        help_text="Organization para automations compartidas (opcional si se define company)"
    )

    # Creator - only ONE of these should be filled (XOR)
    created_by_user = models.ForeignKey(
        CompanyUser,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='created_automations',
        help_text="User creator (company-level)"
    )
    created_by_org_user = models.ForeignKey(
        OrganizationUser,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='created_automations',
        help_text="OrganizationUser creator (org-level)"
    )

    # Info básica
    name = models.CharField(max_length=255)
    description = models.TextField(blank=True)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='draft')

    # Configuración de ejecución
    execution_mode = models.CharField(
        max_length=20,
        choices=EXECUTION_MODE_CHOICES,
        default='sequential'
    )
    max_executions = models.IntegerField(
        null=True,
        blank=True,
        help_text="Máximo número de ejecuciones (null = ilimitado)"
    )
    execution_count = models.IntegerField(default=0)

    # Manejo de errores
    on_error = models.CharField(
        max_length=20,
        choices=ON_ERROR_CHOICES,
        default='stop'
    )
    retry_count = models.IntegerField(default=3)
    retry_delay_seconds = models.IntegerField(default=60)

    # Tracking de ejecución
    last_executed_at = models.DateTimeField(null=True, blank=True)
    next_execution_at = models.DateTimeField(null=True, blank=True, db_index=True)

    # Metadata
    tags = models.JSONField(default=list, blank=True)
    metadata = models.JSONField(default=dict, blank=True)

    class Meta:
        db_table = '"automations"."automations"'
        app_label = 'constec_db'
        indexes = [
            models.Index(fields=['company', 'status']),
            models.Index(fields=['organization', 'status']),
            models.Index(fields=['status', 'next_execution_at']),
        ]
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.name} ({self.status})"

    @property
    def creator(self):
        """Helper to access creator regardless of type."""
        return self.created_by_user or self.created_by_org_user


class Trigger(UUIDModel):
    """
    Trigger (disparador) que inicia la ejecución de una automatización.

    Tipos soportados:
    - schedule: Programado por fecha/hora
    - webhook: Disparado por HTTP POST
    - manual: Ejecución manual del usuario
    - event: Disparado por evento del sistema (futuro)
    """

    TRIGGER_TYPE_CHOICES = [
        ('schedule', 'Schedule'),
        ('webhook', 'Webhook'),
        ('manual', 'Manual'),
        ('event', 'Event'),
    ]

    # FKs
    automation = models.ForeignKey(
        Automation,
        on_delete=models.CASCADE,
        related_name='triggers'
    )

    # Info básica
    trigger_type = models.CharField(max_length=20, choices=TRIGGER_TYPE_CHOICES)
    name = models.CharField(max_length=255)
    description = models.TextField(blank=True)
    is_enabled = models.BooleanField(default=True)

    # Configuración tipo-específica
    config = models.JSONField(
        default=dict,
        help_text="Configuración específica del tipo de trigger (schedule, webhook, etc.)"
    )

    # Tracking
    last_triggered_at = models.DateTimeField(null=True, blank=True)
    trigger_count = models.IntegerField(default=0)
    priority = models.IntegerField(default=50, help_text="0-100, mayor = más prioridad")

    class Meta:
        db_table = '"automations"."triggers"'
        app_label = 'constec_db'
        indexes = [
            models.Index(fields=['automation', 'is_enabled']),
            models.Index(fields=['trigger_type', 'is_enabled']),
        ]
        ordering = ['-priority', 'created_at']

    def __str__(self):
        return f"{self.trigger_type}: {self.name}"


class Action(UUIDModel):
    """
    Acción que se ejecuta como parte de una automatización.

    Tipos soportados:
    - notification: Enviar notificación (email, SMS, WhatsApp, push)
    - webhook: Ejecutar HTTP request
    - batch_task: Ejecutar tarea batch (futuro)
    - flow_execution: Ejecutar workflow LangGraph (futuro)
    """

    ACTION_TYPE_CHOICES = [
        ('notification', 'Notification'),
        ('webhook', 'Webhook'),
        ('batch_task', 'Batch Task'),
        ('flow_execution', 'Flow Execution'),
    ]

    ON_ERROR_CHOICES = [
        ('stop', 'Stop'),
        ('continue', 'Continue'),
        ('retry', 'Retry'),
    ]

    # FKs
    automation = models.ForeignKey(
        Automation,
        on_delete=models.CASCADE,
        related_name='actions'
    )

    # Info básica
    action_type = models.CharField(max_length=20, choices=ACTION_TYPE_CHOICES)
    name = models.CharField(max_length=255)
    description = models.TextField(blank=True)
    is_enabled = models.BooleanField(default=True)
    order = models.IntegerField(default=0, help_text="Orden de ejecución")

    # Configuración tipo-específica
    config = models.JSONField(
        default=dict,
        help_text="Configuración específica del tipo de acción"
    )

    # Manejo de errores (override de automation)
    on_error = models.CharField(
        max_length=20,
        choices=ON_ERROR_CHOICES,
        null=True,
        blank=True,
        help_text="Override del on_error de la automation"
    )
    retry_count = models.IntegerField(
        null=True,
        blank=True,
        help_text="Override del retry_count de la automation"
    )

    # Ejecución condicional (futuro)
    condition = models.JSONField(
        null=True,
        blank=True,
        help_text="Condición JSONLogic para ejecutar esta acción"
    )

    # Tracking
    last_executed_at = models.DateTimeField(null=True, blank=True)
    execution_count = models.IntegerField(default=0)
    success_count = models.IntegerField(default=0)
    failure_count = models.IntegerField(default=0)

    class Meta:
        db_table = '"automations"."actions"'
        app_label = 'constec_db'
        indexes = [
            models.Index(fields=['automation', 'order']),
            models.Index(fields=['action_type', 'is_enabled']),
        ]
        ordering = ['order', 'created_at']

    def __str__(self):
        return f"{self.action_type}: {self.name}"


class ExecutionLog(UUIDModel):
    """
    Log de ejecución de una automatización.

    Registra cada ejecución con su estado, duración, errores y detalles
    de cada acción ejecutada.
    """

    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('running', 'Running'),
        ('completed', 'Completed'),
        ('failed', 'Failed'),
        ('partial', 'Partial'),  # Algunas acciones fallaron
    ]

    TRIGGERED_BY_CHOICES = [
        ('schedule', 'Schedule'),
        ('webhook', 'Webhook'),
        ('manual', 'Manual'),
        ('system', 'System'),
    ]

    # FKs
    automation = models.ForeignKey(
        Automation,
        on_delete=models.CASCADE,
        related_name='execution_logs'
    )
    trigger = models.ForeignKey(
        Trigger,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='execution_logs'
    )
    triggered_by_user = models.ForeignKey(
        CompanyUser,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='triggered_executions',
        help_text="User who triggered execution (company-level)"
    )
    triggered_by_org_user = models.ForeignKey(
        OrganizationUser,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='triggered_executions',
        help_text="OrganizationUser who triggered execution (org-level)"
    )

    # Estado
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    triggered_by = models.CharField(max_length=20, choices=TRIGGERED_BY_CHOICES)

    # Timing
    started_at = models.DateTimeField(auto_now_add=True)
    completed_at = models.DateTimeField(null=True, blank=True)
    duration_seconds = models.IntegerField(null=True, blank=True)

    # Contexto de ejecución
    execution_context = models.JSONField(
        default=dict,
        help_text="Variables y contexto disponible durante la ejecución"
    )

    # Estadísticas de acciones
    actions_total = models.IntegerField(default=0)
    actions_succeeded = models.IntegerField(default=0)
    actions_failed = models.IntegerField(default=0)
    actions_skipped = models.IntegerField(default=0)

    # Log detallado de cada acción
    action_logs = models.JSONField(
        default=list,
        help_text="Array de logs detallados de cada acción ejecutada"
    )

    # Errores
    error_message = models.TextField(blank=True)
    error_traceback = models.TextField(blank=True)

    class Meta:
        db_table = '"automations"."execution_logs"'
        app_label = 'constec_db'
        indexes = [
            models.Index(fields=['automation', 'started_at']),
            models.Index(fields=['status', 'started_at']),
            models.Index(fields=['triggered_by', 'started_at']),
        ]
        ordering = ['-started_at']

    def __str__(self):
        return f"{self.automation.name} - {self.status} ({self.started_at})"

    @property
    def trigger_user(self):
        """Helper to access trigger user regardless of type."""
        return self.triggered_by_user or self.triggered_by_org_user


class NotificationTemplate(UUIDModel):
    """
    Template para notificaciones reutilizables.

    Soporta variables en formato {{variable_name}} que se reemplazan
    en tiempo de ejecución.
    """

    CHANNEL_CHOICES = [
        ('email', 'Email'),
        ('whatsapp', 'WhatsApp'),
        ('sms', 'SMS'),
        ('push', 'Push Notification'),
    ]

    # FKs - Al menos uno debe estar presente
    company = models.ForeignKey(
        Company,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        db_index=True,
        help_text="Company específica (opcional si se define organization)"
    )
    organization = models.ForeignKey(
        Organization,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        db_index=True,
        help_text="Organization para templates compartidos (opcional si se define company)"
    )

    # Creator - only ONE of these should be filled (XOR)
    created_by_user = models.ForeignKey(
        CompanyUser,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='created_notification_templates',
        help_text="User creator (company-level)"
    )
    created_by_org_user = models.ForeignKey(
        OrganizationUser,
        on_delete=models.CASCADE,
        null=True,
        blank=True,
        related_name='created_notification_templates_org',
        help_text="OrganizationUser creator (org-level)"
    )

    # Info básica
    name = models.CharField(max_length=255)
    description = models.TextField(blank=True)
    channel = models.CharField(max_length=20, choices=CHANNEL_CHOICES)

    # Contenido
    subject = models.CharField(
        max_length=255,
        blank=True,
        help_text="Asunto (solo para email)"
    )
    body = models.TextField(help_text="Cuerpo del mensaje con variables {{variable}}")

    # Variables esperadas
    variables = models.JSONField(
        default=list,
        help_text="Array de nombres de variables esperadas ['customer_name', 'amount']"
    )

    # is_active inherited from UUIDModel

    class Meta:
        db_table = '"automations"."notification_templates"'
        app_label = 'constec_db'
        indexes = [
            models.Index(fields=['company', 'channel', 'is_active']),
            models.Index(fields=['organization', 'channel', 'is_active']),
        ]
        ordering = ['name']

    def __str__(self):
        return f"{self.channel}: {self.name}"

    @property
    def creator(self):
        """Helper to access creator regardless of type."""
        return self.created_by_user or self.created_by_org_user
