"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const dotenv = require("dotenv");
const extensions = require("../../extensions");
const resultDotEnv = dotenv.config({ path: __dirname + '/../../../dotenv/cf-authentication-by-oauth2/.env' });
if (resultDotEnv.error) {
    throw resultDotEnv.error;
}
const app = new cdk.App();
const stack = new cdk.Stack(app, 'cf-authentication-by-oauth2-demo');
// create the cloudfront distribution with extension(s)
const OAuth2AuthorizationCodeGrant = new extensions.OAuth2AuthorizationCodeGrant(stack, 'OAuth2AuthorizationCodeGrant', {
    clientId: process.env.CLIENT_ID,
    clientSecret: process.env.CLIENT_SECRET,
    clientDomain: process.env.CLIENT_DOMAIN,
    clientPublicKey: new Buffer(process.env.CLIENT_PUBLIC_KEY).toString('base64'),
    callbackPath: process.env.CALLBACK_PATH,
    jwtArgorithm: process.env.JWT_ARGORITHM,
    authorizeUrl: process.env.AUTHORIZE_URL,
    authorizeParams: new Buffer(process.env.AUTHORIZE_PARAMS).toString('base64'),
    debugEnable: process.env.DEBUG_ENABLE,
});
// create Demo S3 Bucket.
const bucket = new s3.Bucket(stack, 'demoBucket', {
    autoDeleteObjects: true,
    removalPolicy: cdk.RemovalPolicy.DESTROY,
    websiteIndexDocument: 'index.html',
    websiteErrorDocument: 'index.html',
});
// create index.html in the demo folder
fs.writeFileSync(path.join(__dirname, 'index.html'), '<h1>Hello CloudFront Extension (OAuth2 Authentication) with CDK!!!</h1><p>You have logged in. Enjoy your private content.</p>');
// Put demo Object to Bucket.
new aws_s3_deployment_1.BucketDeployment(stack, 'BucketDeployment', {
    sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
    destinationBucket: bucket,
    retainOnDelete: false,
});
// CloudFront OriginAccessIdentity for Bucket
const originAccessIdentity = new cf.OriginAccessIdentity(stack, 'OriginAccessIdentity', {
    comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
});
// CloudfrontWebDistribution
const cloudfrontWebDistribution = new cf.CloudFrontWebDistribution(stack, 'CloudFrontWebDistribution', {
    originConfigs: [
        {
            s3OriginSource: {
                originAccessIdentity,
                s3BucketSource: bucket,
            },
            behaviors: [{
                    isDefaultBehavior: true,
                    lambdaFunctionAssociations: [OAuth2AuthorizationCodeGrant],
                }],
        },
    ],
    priceClass: cf.PriceClass.PRICE_CLASS_ALL,
});
new cdk.CfnOutput(stack, 'distributionDomainName', {
    value: cloudfrontWebDistribution.distributionDomainName,
});
//# sourceMappingURL=data:application/json;base64,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