"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const cf = require("@aws-cdk/aws-cloudfront");
const aws_cloudfront_origins_1 = require("@aws-cdk/aws-cloudfront-origins");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create a cloudfront distribution with an extension (L@E)
    const convertQueryStringProsp = { args: ['language', 'name'] };
    const convertQueryString = new extensions.ConvertQueryString(stack, 'ConvertQueryStringDemo', convertQueryStringProsp);
    // A CloudFront distribution
    const cloudFrontDistribution = new cf.Distribution(stack, 'CloudFrontDistribution', {
        defaultBehavior: {
            origin: new aws_cloudfront_origins_1.HttpOrigin('postman-echo.com', {
                httpPort: 80,
                originPath: '/get',
                originSslProtocols: [cf.OriginSslPolicy.TLS_V1],
                keepaliveTimeout: cdk.Duration.seconds(10),
                protocolPolicy: cf.OriginProtocolPolicy.HTTP_ONLY,
                readTimeout: cdk.Duration.seconds(10),
            }),
            edgeLambdas: [convertQueryString],
            cachePolicy: new cf.CachePolicy(stack, 'DefaultCachePolicy', {
                cachePolicyName: 'ConvertQueryString-Cache-Policy',
                queryStringBehavior: cf.CacheQueryStringBehavior.all(),
            }),
            originRequestPolicy: new cf.OriginRequestPolicy(stack, 'RequestPolicy', {
                originRequestPolicyName: 'ConvertQueryString-Request-Policy',
                queryStringBehavior: cf.OriginRequestQueryStringBehavior.all(),
                headerBehavior: cf.OriginRequestHeaderBehavior.all(),
                comment: 'just for demonstration.',
            }),
        },
        comment: 'The CloudFront distribution based on the custom origin',
        priceClass: cf.PriceClass.PRICE_CLASS_200,
    });
    new cdk.CfnOutput(stack, 'DistributionDomainName', {
        value: cloudFrontDistribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Comment: 'The CloudFront distribution based on the custom origin',
            DefaultCacheBehavior: {
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-request',
                        LambdaFunctionARN: {
                            Ref: 'ConvertQueryStringFuncCurrentVersion4FB275862a9b84221e5ba9190684389f5c63a7be',
                        },
                    },
                ],
                ViewerProtocolPolicy: 'allow-all',
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        HTTPPort: 80,
                        OriginKeepaliveTimeout: 10,
                        OriginProtocolPolicy: 'http-only',
                        OriginReadTimeout: 10,
                        OriginSSLProtocols: [
                            'TLSv1',
                        ],
                    },
                    DomainName: 'postman-echo.com',
                    Id: 'demostackCloudFrontDistributionOrigin15405BC3B',
                    OriginPath: '/get',
                },
            ],
            PriceClass: 'PriceClass_200',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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