#  telectron - Telegram MTProto API Client Library for Python
#  Copyright (C) 2017-2021 Dan <https://github.com/delivrance>
#
#  This file is part of telectron.
#
#  telectron is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  telectron is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public License
#  along with telectron.  If not, see <http://www.gnu.org/licenses/>.

from typing import Dict

import telectron
from telectron import raw
from telectron import types
from ..object import Object


class ChatInviteLink(Object):
    """An invite link for a chat.

    Parameters:
        invite_link (``str``):
            The invite link. If the link was created by another chat administrator, then the second part of the
            link will be replaced with "...".

        date (``int``):
            The date in Unix timestamp when the link was created.

        is_primary (``bool``):
            True, if the link is primary.

        is_revoked (``bool``):
            True, if the link is revoked.

        creator (:obj:`~telectron.types.User`, *optional*):
            Creator of the link.

        expire_date (``int``, *optional*):
            Point in time (Unix timestamp) when the link will expire or has been expired.

        member_limit (``int``, *optional*):
            Maximum number of users that can be members of the chat simultaneously after joining the chat via this
            invite link; 1-99999.

        member_count (``int``, *optional*):
            Number of users that joined via this link and are currently member of the chat.
    """

    def __init__(
        self, *,
        invite_link: str,
        creator: "types.User",
        date: int,
        is_primary: bool = None,
        is_revoked: bool = None,
        start_date: int = None,
        expire_date: int = None,
        member_limit: int = None,
        member_count: int = None
    ):
        super().__init__()

        self.invite_link = invite_link
        self.creator = creator
        self.date = date
        self.is_primary = is_primary
        self.is_revoked = is_revoked
        self.start_date = start_date
        self.expire_date = expire_date
        self.member_limit = member_limit
        self.member_count = member_count

    @staticmethod
    def _parse(
        client: "telectron.Client",
        invite: "raw.types.ChatInviteExported",
        users: Dict[int, "raw.types.User"] = None
    ) -> "ChatInviteLink":
        creator = (
            types.User._parse(client, users[invite.admin_id])
            if users is not None
            else None
        )

        return ChatInviteLink(
            invite_link=invite.link,
            creator=creator,
            date=invite.date,
            is_primary=invite.permanent,
            is_revoked=invite.revoked,
            expire_date=invite.expire_date,
            member_limit=invite.usage_limit,
            member_count=invite.usage
        )
