"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = void 0;
const child_process_1 = require("child_process");
const console = require("console");
const os = require("os");
const path = require("path");
const process = require("process");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const jsii_rosetta_1 = require("jsii-rosetta");
const transliterate_1 = require("jsii-rosetta/lib/commands/transliterate");
const shared_1 = require("../shared");
const clients = new Map();
const PACKAGE_KEY_REGEX = new RegExp(`^${shared_1.constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${shared_1.constants.PACKAGE_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
async function handler(event, context) {
    var _a, _b;
    console.log(JSON.stringify(event, null, 2));
    const created = new Array();
    for (const record of event.Records) {
        // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
        // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
        // that would not replace `@` in the position where it is in the keys... So we have to work on
        // the URI components instead.
        const inputKey = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
        const [, packageName, packageVersion] = (_a = inputKey.match(PACKAGE_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${inputKey}". It was expected to match ${PACKAGE_KEY_REGEX}!`);
        }
        const client = (clients.has(record.awsRegion)
            ? clients
            : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
        console.log(`Source Bucket:  ${record.s3.bucket.name}`);
        console.log(`Source Key:     ${inputKey}`);
        console.log(`Source Version: ${record.s3.object.versionId}`);
        const object = await client.getObject({
            Bucket: record.s3.bucket.name,
            Key: inputKey,
            VersionId: record.s3.object.versionId,
        }).promise();
        const workdir = await fs.mkdtemp(path.join(os.tmpdir(), 'workdir'));
        try {
            const tarball = path.join(workdir, `${packageName.replace('@', '').replace('/', '-')}-${packageVersion}.tgz`);
            await fs.writeFile(tarball, object.Body);
            await new Promise((ok, ko) => {
                // --ignore-scripts disables lifecycle hooks, in order to prevent execution of arbitrary code
                // --no-bin-links ensures npm does not insert anything in $PATH
                const npmInstall = child_process_1.spawn('npm', ['install', '--ignore-scripts', '--no-bin-links', '--no-save', tarball], {
                    cwd: workdir,
                    env: {
                        ...process.env,
                        HOME: os.tmpdir(),
                    },
                    stdio: ['ignore', 'inherit', 'inherit'],
                });
                npmInstall.once('error', ko);
                npmInstall.once('close', (code, signal) => {
                    if (code === 0) {
                        ok();
                    }
                    else {
                        ko(`"npm install" command ${code != null ? `exited with code ${code}` : `was terminated by signal ${signal}`}`);
                    }
                });
            });
            const packageDir = path.join(workdir, 'node_modules', ...packageName.split('/'));
            await transliterate_1.transliterateAssembly([packageDir], [jsii_rosetta_1.TargetLanguage.PYTHON]);
            // Payload object key => packages/[<@scope>/]<name>/v<version>/package.tgz
            // Output object key  => packages/[<@scope>/]<name>/v<version>/assembly-python.json
            const key = inputKey.replace(/\/[^/]+$/, shared_1.constants.assemblyKeySuffix(jsii_rosetta_1.TargetLanguage.PYTHON));
            const response = await client.putObject({
                Bucket: record.s3.bucket.name,
                Key: key,
                Body: await fs.readFile(path.join(packageDir, '.jsii.python')),
                ContentType: 'text/json',
                Metadata: {
                    'Origin-Version-Id': (_b = record.s3.object.versionId) !== null && _b !== void 0 ? _b : 'N/A',
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise();
            created.push({
                bucket: record.s3.bucket.name,
                key,
                versionId: response.VersionId,
            });
        }
        finally {
            await fs.remove(workdir);
        }
    }
    return created;
}
exports.handler = handler;
/**
 * Visible for testing. Clears the caches so that the next execution runs clean.
 */
function reset() {
    clients.clear();
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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