"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const ingestion_1 = require("./ingestion");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: 'Ingests new package versions into the Construct Hub',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
            },
            memorySize: 10240,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantWrite(handler);
        handler.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        this.grantPrincipal = handler.grantPrincipal;
    }
}
exports.Ingestion = Ingestion;
//# sourceMappingURL=data:application/json;base64,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