"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.STAGED_KEY_PREFIX = exports.FAILED_KEY_PREFIX = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const shared_1 = require("../shared");
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const MARKER_FILE_NAME = 'couchdb-last-transaction-id';
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
exports.FAILED_KEY_PREFIX = 'failed/';
exports.STAGED_KEY_PREFIX = 'staged/';
/**
 * This function triggers on a fixed schedule and reads a stream of changes frm npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * Currently we don't handle the function execution timeout, and accept that the last batch processed might be processed again,
 * relying on the idempotency on the consumer side.
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = shared_1.requireEnv('QUEUE_URL');
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', async (batch) => {
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                // Filter out all elements that don't have a "name" in the document, as
                // these are schemas, which are not relevant to our business here.
                batch = batch.filter((item) => item.doc.name);
                console.log(`Identified ${batch.length} package update element(s)`);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                // Process all remaining updates
                await Promise.all(versionInfos.map(processUpdatedVersion));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                if (context.getRemainingTimeInMillis() >= 120000 /* 2 minutes */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
        })
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await shared_1.aws.s3().getObject({
                Bucket: stagingBucket,
                Key: MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }) {
        var _a;
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${exports.STAGED_KEY_PREFIX}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await shared_1.aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing ${infos.name}@${infos.version}: ${err}`);
            await putObject(`${exports.FAILED_KEY_PREFIX}${seq}`, JSON.stringify(infos, null, 2), {
                ContentType: 'text/json',
                Metadata: {
                    // User-defined metadata is limited to 2KB in size, in total. So we
                    // cap the error text to 1KB maximum, allowing up to 1KB for other
                    // attributes (which should be sufficient).
                    'Error': `${(_a = err.stack) !== null && _a !== void 0 ? _a : err}`.substring(0, 1024),
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return shared_1.aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes) {
    const result = new Array();
    for (const change of changes) {
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${JSON.stringify(change, null, 2)}`);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${JSON.stringify(change, null, 2)}`);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isRelevantPackageVersion(infos)) {
                    result.push({ infos, modified, seq: change.seq });
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isRelevantPackageVersion(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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