"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const events_1 = require("events");
const fs_1 = require("fs");
const path = require("path");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const jsii_rosetta_1 = require("jsii-rosetta");
const shared_1 = require("../../../backend/shared");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-rosetta/lib/commands/transliterate');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    transliterator_lambda_1.reset();
    done();
});
test('scoped package', async () => {
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `${shared_1.constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${shared_1.constants.PACKAGE_KEY_SUFFIX}`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/'));
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', `@${packageScope}`, packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', `@${packageScope}`, packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key
        // The key is not URI-encoded when it's sent to the S3 SDK...
        .split('/').map((comp) => decodeURIComponent(comp)).join('/')
        .replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
test('unscoped package', async () => {
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `${shared_1.constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${shared_1.constants.PACKAGE_KEY_SUFFIX}`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key);
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key.replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
class MockChildProcess extends events_1.EventEmitter {
    constructor(spawnfile, spawnargs, promise) {
        super();
        this.spawnfile = spawnfile;
        this.spawnargs = spawnargs;
        this.stdin = null;
        this.stdout = null;
        this.stderr = null;
        this.stdio = [this.stdin, this.stdout, this.stderr, null, null];
        this.exitCode = 0;
        this.killed = false;
        this.signalCode = null;
        this.pid = -1; // Obviously fake
        promise.then(() => this.emit('close', this.exitCode, this.signalCode), (err) => this.emit('error', err));
    }
    get connected() {
        throw new Error('Not Implemented');
    }
    disconnect() {
        throw new Error('Not Implemented');
    }
    kill() {
        throw new Error('Not Implemented');
    }
    ref() {
        throw new Error('Not Implemented');
    }
    send() {
        throw new Error('Not Implemented');
    }
    unref() {
        throw new Error('Not Implemented');
    }
}
//# sourceMappingURL=data:application/json;base64,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