"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
const shared_1 = require("../../../backend/shared");
let mockBucketName;
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    shared_1.aws.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    done();
});
test('no indexed packages', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(shared_1.constants.CATALOG_KEY);
        }
        catch (e) {
            return cb(e);
        }
        return cb(new NoSuchKeyError());
    });
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(shared_1.constants.STORAGE_KEY_PREFIX);
            expect(req.ContinuationToken).toBeUndefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, {});
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(shared_1.constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('initial build', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        const matches = new RegExp(`^${shared_1.constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    const mockFirstPage = [
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${shared_1.constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}name/v1.2.3${shared_1.constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}name/v1.2.3${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
    ];
    const mockSecondPage = [
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${shared_1.constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${shared_1.constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(shared_1.constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(shared_1.constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('incremental build', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        const matches = new RegExp(`^${shared_1.constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else if (req.Key === shared_1.constants.CATALOG_KEY) {
            return cb(null, {
                Body: JSON.stringify({
                    packages: [
                        {
                            description: 'Package @scope/package, version 2.3.4',
                            languages: { foo: 'bar' },
                            major: 2,
                            name: '@scope/package',
                            version: '2.3.4',
                        },
                        {
                            description: 'Package name, version 1.0.0',
                            languages: { foo: 'bar' },
                            major: 1,
                            name: 'name',
                            version: '1.0.0',
                        },
                        {
                            description: 'Package name, version 2.0.0-pre.10',
                            languages: { foo: 'bar' },
                            major: 2,
                            name: 'name',
                            version: '2.0.0-pre.10',
                        },
                    ],
                    updatedAt: new Date().toISOString(),
                }, null, 2),
            });
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    const mockFirstPage = [
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}name/v1.2.3${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
    ];
    const mockSecondPage = [
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}@scope/package/v2.0.5${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${shared_1.constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre.1${shared_1.constants.PACKAGE_KEY_SUFFIX}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(shared_1.constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(shared_1.constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '4');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 2.3.4',
                    languages: { foo: 'bar' },
                    major: 2,
                    name: '@scope/package',
                    version: '2.3.4',
                },
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre.10',
                    languages: { foo: 'bar' },
                    major: 2,
                    name: 'name',
                    version: '2.0.0-pre.10',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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