# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nnfs']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.23.1,<2.0.0', 'tqdm>=4.64.0,<5.0.0']

setup_kwargs = {
    'name': 'nnfs-implementation',
    'version': '0.0.1',
    'description': 'Impelementation of Neural Networks from Scratch based on this book: https://nnfs.io/',
    'long_description': '# Neural Network from Scratch Implementation\n\nIn this repository, with only `numpy` as only dependency, I have implemented a neural network from scratch based on this marvelous book:\n[nnfs.io](https://nnfs.io/).\n\n![Data](./pics/Data.png)\n![Boundaries](./pics/Boundaries.png)\n\n## Installation\n\n```bash\npython3.9 -m pip install nnfs-implementation;\n```\nP.S.: This package is compatible with python3.9+.\n\n## Examples\nThere is [a module](nnfs/examples.py) with some examples in the repository, but since this package is use the `nnfs` name internally,\n**it will have a name collision** with the original [nnfs package](https://pypi.org/project/nnfs/) which provides the toy datasets, and you cannot run it. although a glance on it doesn\'t hurt.\nAlso, There is this notebook [VanillaNN.ipynb](VanillaNN.ipynb) which is the full implementation will all the examples and some nice plots at the bottom. I highly recommend you to take a look at it.\n\n## Instructions\nA simple multi-class classifier can be implemented with the following code:\n\n```python\nmodel = Model(loss=SoftmaxLoss(), optimizer=Adam(), metric=Accuracy())\nmodel.add(Layer(28 * 28, 512))\nmodel.add(ReLU())\nmodel.add(Layer(512, 10))\nmodel.fit(train_images, train_labels, epochs=10, batch_size=512)\nvalidation_accuracy = Accuracy.evaluate(test_labels, model.predict(test_images))\nprint(f"Validation Accuracy: {validation_accuracy:.2%}")\n```\n\n### Model\nFor defining the main model, you should use this pattern:\n```python\nfrom nnfs.loss import SoftmaxLoss\nfrom nnfs.metrics import Accuracy\nfrom nnfs.model import Model\nfrom nnfs.optimizers import Adam\n\nmodel = Model(loss=SoftmaxLoss(), optimizer=Adam(), metric=Accuracy())\n```\n\nYou have to define the loss function, the optimizer, and the metric. For any of these components you can change the default values like this:\n\n```python\nmodel = Model(loss=SoftmaxLoss(), optimizer=Adam(learning_rate=0.001), metric=Accuracy())\n```\n\n### Loss Criterion\nThe loss criterion is the function that will be used to calculate the loss of the model. \nnnfs-implementation provides four loss criterion: (but none of them support multi-label classification):\n\n```python\nfrom nnfs.loss import BinaryLoss, CategoricalLoss, MSELoss, SoftmaxLoss\n```\n\n#### BinaryLoss\nBinary cross-entropy loss, you must use a single neuron with a sigmoid activation function for the last layer. (CLASSIFICATION)\n\n#### CategoricalLoss\nCategorical cross-entropy loss, you can use it with combination of multiple neurons with a softmax activation function for the last layer. \n**BE AWARE that backpropagation of softmax and CategoricalLoss is inefficient and expensive, so it\'s HIGHLY recommended to use SoftmaxLoss instead.** (CLASSIFICATION)\n\n#### MSELoss\nMean squared error loss. (REGRESSION)\n\n#### SoftmaxLoss\nSoftmax activation layer combined with categorical cross-entropy loss.\nThe derivatives of combining these two layers is computationally a lot cheaper than processing each one separately;\nso we merge them into one layer. **THERE IS NO NEED TO USE A SOFTMAX AT LAST LAYER ANYMORE.** (CLASSIFICATION)\n\n### Optimizers\nWe can take advantage of two optimizers; Momentum and Adam. Adam is superior to Momentum in most cases.\n\n```python\nfrom nnfs.optimizers import Adam, Momentum\n\nadam = Adam(learning_rate=0.01, decay=1e-5, beta_1=0.9, beta_2=0.999)  # Default Values\nmomentum = Momentum(learning_rate=0.01, decay=1e-5, beta=0.1) # Default Values\nmodel = Model(loss=SoftmaxLoss(), optimizer=adam, metric=Accuracy())\n```\n\nYou must play around with the parameters of the optimizer to find the best one for your model.\ndecay is the learning rate decay.\n\n### Metrics\nWe have 5 metrics which we use during training and evaluation phase.\n```python\nfrom nnfs.metrics import Accuracy, ExplainedVariance, FScore, Precision, Recall\n\nmodel = Model(loss=SoftmaxLoss(), optimizer=adam, metric=Accuracy())\nvalidation_metric = ExplainedVariance.evaluate(y_test, model.predict(X_test))\n```\nExplainedVariance is for regression tasks, all the others are for classifications.\n**Precision, Recall and FScore are calculated by Macro averaging.**\n\n### Layers\nWe provide FullyConnectedLayer and DropoutLayer.\n\n```python\nfrom nnfs.layer import Dropout, Layer\n\nmodel.add(Layer(64, 128)) # FullyConnectedLayer with 64 inputs and 128 outputs.\nmodel.add(Dropout(0.2)) # DropoutLayer with 20% dropout.\nmodel.add(Layer(2, 64, w_l2=5e-4, b_l2=5e-4)) # FullyConnectedLayer with L2 regularization. (weight and bias)\nmodel.add(Layer(2, 64, w_l2=5e-4, b_l2=5e-4, w_l1=5e-4, b_l1=5e-4)) # FullyConnectedLayer with L1 and L2 regularization. (weight and bias)\n```\n\n### Activation Functions\nnnfs-implementation provides six activation functions:\n\n```python\nfrom nnfs.activations import LeakyReLU, Linear, ReLU, Sigmoid, Softmax, TanH\n\nmodel.add(ReLU())\nmodel.add(LeakyReLU(negative_slope=0.1))\n```\nLinear should be used for regression in the last layer. \n\n### Training\n```python\nmodel.fit(X_train, y_train, epochs=1_000, batch_size=None, shuffle=True)  # Default Values\nmodel.fit(X_train, y_train, epochs=20, batch_size=512, shuffle=True) \n```\nIf batch_size is None, then the whole dataset will be used.\n\n### Prediction\n```python\ny_pred = model.predict(X_test)\ny_pred_proba = model.predict_proba(X_test)\n```\n\n### Loading & Saving\nTo load a model, simply use:\n```python\nfrom nnfs.model import Model\n\nmodel = Model.load("model.pkl")\n```\nand to save the model:\n```python\nmodel.save("model.pkl", cleanup=True)\n```\ncleanup option will remove some unnecessary attributes that leads to dramatically reduced saved model size.\ndefault value is True.\n\n\n## Caution\n**This project is only for demonstration purpose.** It works on numpy arrays (CPU) so you will face a poor performance.\n\n\n## Contributing\nPull requests are welcome. For major changes, please open an issue first to discuss what you would like to change.\n\nContact me: <OSS@Mahyar24.com> :)\n\n## License\n[GNU GPLv3](https://choosealicense.com/licenses/gpl-3.0/)\n',
    'author': 'Mahyar Mahdavi',
    'author_email': 'Mahyar@Mahyar24.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mahyar24/NeuralNetFromScratch/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
