"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    var _a;
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const prefix = (_a = options.tagPrefix) !== null && _a !== void 0 ? _a : '';
    const bumpFile = path_1.join(cwd, options.bumpFile);
    const changelogFile = path_1.join(cwd, options.changelog);
    const releaseTagFile = path_1.join(cwd, options.releaseTagFile);
    await fs_extra_1.mkdirp(path_1.dirname(bumpFile));
    await fs_extra_1.mkdirp(path_1.dirname(changelogFile));
    await fs_extra_1.mkdirp(path_1.dirname(releaseTagFile));
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({ cwd, major, prerelease, prefix });
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // check if the latest commit already has a version tag
    const currentTags = util_1.execCapture('git tag --points-at HEAD', { cwd }).toString('utf8').split('\n');
    logging.info(`Tags listed on current commit: ${currentTags}`);
    let skipBump = false;
    if (currentTags.includes(latestTag)) {
        logging.info('Skipping bump...');
        skipBump = true;
        // delete the existing tag (locally)
        // if we don't do this, standard-version generates an empty changelog
        util_1.exec(`git tag --delete ${latestTag}`, { cwd });
    }
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, '.versionrc.json');
    await generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, options.versionrcOptions);
    const cmd = ['npx', 'standard-version@^9'];
    if (isFirstRelease) {
        cmd.push('--first-release');
    }
    util_1.exec(cmd.join(' '), { cwd });
    // add the tag back if it was previously removed
    if (currentTags.includes(latestTag)) {
        util_1.exec(`git tag ${latestTag}`, { cwd });
    }
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_extra_1.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_extra_1.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, 'utf8'));
}
function generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, configOptions) {
    return fs_extra_1.writeFile(rcfile, JSON.stringify({
        ...{
            packageFiles: [{
                    filename: versionFile,
                    type: 'json',
                }],
            bumpFiles: [{
                    filename: versionFile,
                    type: 'json',
                }],
            commitAll: false,
            infile: changelogFile,
            prerelease: prerelease,
            header: '',
            skip: {
                commit: true,
                tag: true,
                bump: skipBump,
            },
            ...configOptions,
        },
    }, undefined, 2));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    const prefixFilter = major ? `${prefix}v${major}.*` : `${prefix}v*`;
    const listGitTags = [
        'git',
        '-c "versionsort.suffix=-"',
        'tag',
        '--sort="-version:refname"',
        '--list',
        `"${prefixFilter}"`,
    ].join(' ');
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString('utf8');
    let tags = stdout === null || stdout === void 0 ? void 0 : stdout.split('\n');
    // if "pre" is set, filter versions that end with "-PRE.ddd".
    if (prerelease) {
        tags = tags.filter(x => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    }
    tags = tags.filter(x => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major !== null && major !== void 0 ? major : 0}.0.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith('v')) {
        latestVersion = latestVersion.substr(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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