"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
/**
 * (experimental) Base project.
 *
 * @experimental
 */
class Project {
    /**
     * @experimental
     */
    constructor(options) {
        var _b, _c, _d;
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.initProject = render_options_1.resolveInitProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this.projenCommand = (_b = options.projenCommand) !== null && _b !== void 0 ? _b : 'npx projen';
        this.outdir = this.determineOutdir(options.outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        (_c = this.parent) === null || _c === void 0 ? void 0 : _c._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated('/.projen/**'); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        this.gitignore = new ignore_file_1.IgnoreFile(this, '.gitignore');
        this.gitignore.exclude('node_modules/'); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
            description: 'Synthesize project files',
        });
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = (_d = options.projenrcJson) !== null && _d !== void 0 ? _d : false;
        if (projenrcJson) {
            new projenrc_json_1.Projenrc(this, options.projenrcJsonOptions);
        }
        new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
            omitEmpty: true,
            obj: () => ({
                // replace `\` with `/` to ensure paths match across platforms
                files: this.files.filter(f => f.readonly).map(f => f.path.replace(/\\/g, '/')),
            }),
        });
    }
    /**
     * (experimental) Returns all the components within this project.
     *
     * @experimental
     */
    get components() {
        return [...this._components];
    }
    /**
     * (experimental) All files in this project.
     *
     * @experimental
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components.filter(isFile).sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * (experimental) Adds a new task to this project.
     *
     * This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add.
     * @param props Task properties.
     * @experimental
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * (experimental) Removes a task from a project.
     *
     * @param name The name of the task to remove.
     * @returns The `Task` that was removed, otherwise `undefined`.
     * @experimental
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    /**
     * @experimental
     */
    get buildTask() { return this.projectBuild.buildTask; }
    /**
     * @experimental
     */
    get compileTask() { return this.projectBuild.compileTask; }
    /**
     * @experimental
     */
    get testTask() { return this.projectBuild.testTask; }
    /**
     * @experimental
     */
    get preCompileTask() { return this.projectBuild.preCompileTask; }
    /**
     * @experimental
     */
    get postCompileTask() { return this.projectBuild.postCompileTask; }
    /**
     * @experimental
     */
    get packageTask() { return this.projectBuild.packageTask; }
    /**
     * (experimental) Finds a file at the specified relative path within this project and all its subprojects.
     *
     * @param filePath The file path.
     * @returns a `FileBase` or undefined if there is no file in that path
     * @experimental
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath) ? filePath : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * (deprecated) Finds a json file by name.
     *
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (experimental) Finds an object file (like JsonFile, YamlFile, etc.) by name.
     *
     * @param filePath The file path.
     * @experimental
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * (deprecated) Prints a "tip" message during synthesis.
     *
     * @param message The message.
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * (experimental) Exclude the matching files from pre-synth cleanup.
     *
     * Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match.
     * @experimental
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * (experimental) Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required.
     * @experimental
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * (experimental) Exclude these files from the bundled package.
     *
     * Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude.
     * @experimental
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * (experimental) Adds a .gitignore pattern.
     *
     * @param pattern The glob pattern to ignore.
     * @experimental
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * (experimental) Consider a set of files as "generated".
     *
     * This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     * @experimental
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * (experimental) Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     *
     * @experimental
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug('Synthesizing project...');
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + '/**');
        }
        // delete all generated files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        this.logger.debug('Synthesis complete');
    }
    /**
     * (experimental) Called before all components are synthesized.
     *
     * @experimental
     */
    preSynthesize() { }
    /**
     * (experimental) Called after all components are synthesized.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
    /**
     * Resolves the project's output directory.
     */
    determineOutdir(outdirOption) {
        if (this.parent && outdirOption && path.isAbsolute(outdirOption)) {
            throw new Error('"outdir" must be a relative path');
        }
        // if this is a subproject, it is relative to the parent
        if (this.parent) {
            if (!outdirOption) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            return path.resolve(this.parent.outdir, outdirOption);
        }
        // if this is running inside a test and outdir is not explicitly set
        // use a temp directory (unless cwd is aleady under tmp)
        if (common_1.IS_TEST_RUN && !outdirOption) {
            const realCwd = fs_1.realpathSync(process.cwd());
            const realTmp = fs_1.realpathSync(os_1.tmpdir());
            if (realCwd.startsWith(realTmp)) {
                return path.resolve(realCwd, outdirOption !== null && outdirOption !== void 0 ? outdirOption : '.');
            }
            return fs_1.mkdtempSync(path.join(os_1.tmpdir(), 'projen.'));
        }
        return path.resolve(outdirOption !== null && outdirOption !== void 0 ? outdirOption : '.');
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.46.3" };
/**
 * (experimental) The name of the default task (the task executed when `projen` is run without arguments).
 *
 * Normally
 * this task should synthesize the project files.
 *
 * @experimental
 */
Project.DEFAULT_TASK = 'default';
/**
 * (deprecated) Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    ProjectType["UNKNOWN"] = "unknown";
    ProjectType["LIB"] = "lib";
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkJBQStDO0FBQy9DLDJCQUE0QjtBQUM1Qiw2QkFBNkI7QUFDN0IsdUNBQW1EO0FBQ25ELHFDQUF1RDtBQUV2RCxpREFBOEM7QUFDOUMsaUNBQWtDO0FBQ2xDLG1EQUFvRDtBQUNwRCwrQ0FBMkM7QUFFM0MsZ0VBQWlFO0FBQ2pFLGlDQUFrQztBQUNsQyxxQ0FBaUQ7QUFDakQsK0NBQTJDO0FBRTNDLG1EQUErRDtBQUMvRCxtREFBNEQ7QUFFNUQsbUNBQWdDO0FBQ2hDLGlDQUFrQzs7Ozs7O0FBMkJsQyxNQUFhLE9BQU87Ozs7SUFrRGxCLFlBQVksT0FBdUI7O1FBTGxCLGdCQUFXLEdBQUcsSUFBSSxLQUFLLEVBQWEsQ0FBQztRQUNyQyxnQkFBVyxHQUFHLElBQUksS0FBSyxFQUFXLENBQUM7UUFDbkMsU0FBSSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFJMUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxtQ0FBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUUvQyxJQUFJLENBQUMsSUFBSSxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUM7UUFDekIsSUFBSSxDQUFDLE1BQU0sR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQzdCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxFQUFFLENBQUM7UUFDN0IsSUFBSSxDQUFDLGFBQWEsU0FBRyxPQUFPLENBQUMsYUFBYSxtQ0FBSSxZQUFZLENBQUM7UUFFM0QsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFFbEQsMkVBQTJFO1FBQzNFLE1BQUEsSUFBSSxDQUFDLE1BQU0sMENBQUUsY0FBYyxDQUFDLElBQUksRUFBRTtRQUVsQywyRUFBMkU7UUFFM0UsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLGlDQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLDhEQUE4RDtRQUNyRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyw4Q0FBOEM7UUFFckcsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLHdCQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsa0NBQWtDO1FBQzNFLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRXRELHNFQUFzRTtRQUN0RSw2Q0FBNkM7UUFDN0MsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLGFBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUU7WUFDMUQsV0FBVyxFQUFFLDBCQUEwQjtTQUN4QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksNEJBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUzQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksMkJBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVuQyxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFaEQsTUFBTSxZQUFZLFNBQUcsT0FBTyxDQUFDLFlBQVksbUNBQUksS0FBSyxDQUFDO1FBQ25ELElBQUksWUFBWSxFQUFFO1lBQ2hCLElBQUksd0JBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLG1CQUFtQixDQUFDLENBQUM7U0FDakQ7UUFFRCxJQUFJLGVBQVEsQ0FBQyxJQUFJLEVBQUUsdUJBQWEsRUFBRTtZQUNoQyxTQUFTLEVBQUUsSUFBSTtZQUNmLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO2dCQUNWLDhEQUE4RDtnQkFDOUQsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQzthQUMvRSxDQUFDO1NBQ0gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7O0lBR0QsSUFBVyxVQUFVO1FBQ25CLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUMvQixDQUFDOzs7Ozs7SUFHRCxJQUFXLEtBQUs7UUFDZCxNQUFNLE1BQU0sR0FBRyxDQUFDLENBQVksRUFBaUIsRUFBRSxDQUFDLENBQUMsWUFBWSxlQUFRLENBQUM7UUFDdEUsT0FBTyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUMxRixDQUFDOzs7Ozs7Ozs7OztJQUdNLE9BQU8sQ0FBQyxJQUFZLEVBQUUsUUFBcUIsRUFBRztRQUNuRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN6QyxDQUFDOzs7Ozs7OztJQUdNLFVBQVUsQ0FBQyxJQUFZO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDckMsQ0FBQzs7OztJQUVELElBQVcsU0FBUyxLQUFLLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDOzs7O0lBQzlELElBQVcsV0FBVyxLQUFLLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDOzs7O0lBQ2xFLElBQVcsUUFBUSxLQUFLLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDOzs7O0lBQzVELElBQVcsY0FBYyxLQUFLLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDOzs7O0lBQ3hFLElBQVcsZUFBZSxLQUFLLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDOzs7O0lBQzFFLElBQVcsV0FBVyxLQUFLLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDOzs7Ozs7OztJQUczRCxXQUFXLENBQUMsUUFBZ0I7UUFDakMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDNUYsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQzdCLElBQUksUUFBUSxLQUFLLElBQUksQ0FBQyxZQUFZLEVBQUU7Z0JBQ2xDLE9BQU8sSUFBSSxDQUFDO2FBQ2I7U0FDRjtRQUVELEtBQUssTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNwQyxNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3pDLElBQUksSUFBSSxFQUFFO2dCQUNSLE9BQU8sSUFBSSxDQUFDO2FBQ2I7U0FDRjtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Ozs7Ozs7SUFHTSxlQUFlLENBQUMsUUFBZ0I7UUFDckMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDVCxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELElBQUksQ0FBQyxDQUFDLElBQUksWUFBWSxlQUFRLENBQUMsRUFBRTtZQUMvQixNQUFNLElBQUksS0FBSyxDQUFDLGNBQWMsUUFBUSxtQ0FBbUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQ25HO1FBRUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDOzs7Ozs7O0lBR00saUJBQWlCLENBQUMsUUFBZ0I7UUFDdkMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ1QsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxJQUFJLENBQUMsQ0FBQyxJQUFJLFlBQVksd0JBQVUsQ0FBQyxFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsY0FBYyxRQUFRLHFDQUFxQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7U0FDckc7UUFFRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsT0FBZTtRQUMzQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUMxQixDQUFDOzs7Ozs7Ozs7O0lBR00scUJBQXFCLENBQUMsR0FBRyxLQUFlO1FBQzdDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQztJQUN6QyxDQUFDOzs7Ozs7Ozs7SUFHTSxjQUFjLENBQUMsSUFBVTtRQUM5QixPQUFPLGNBQWMsdUJBQWMsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDckQsQ0FBQzs7Ozs7Ozs7OztJQUdNLGdCQUFnQixDQUFDLFFBQWdCO1FBQ3RDLHNDQUFzQztJQUN4QyxDQUFDOzs7Ozs7O0lBR00sWUFBWSxDQUFDLE9BQWU7UUFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQzs7Ozs7Ozs7Ozs7SUFHTSxpQkFBaUIsQ0FBQyxLQUFhO1FBQ3BDLHNDQUFzQztJQUN4QyxDQUFDOzs7Ozs7Ozs7Ozs7O0lBR00sS0FBSztRQUNWLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUM7UUFDM0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUU3QyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7UUFFckIsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ25DLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztTQUN0QjtRQUVELCtFQUErRTtRQUMvRSxnRkFBZ0Y7UUFDaEYsS0FBSyxNQUFNLFVBQVUsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ3pDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxVQUFVLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxDQUFDO1NBQ3ZEO1FBRUQsbUVBQW1FO1FBQ25FLGlCQUFPLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBRXpDLEtBQUssTUFBTSxVQUFVLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUN6QyxVQUFVLENBQUMsS0FBSyxFQUFFLENBQUM7U0FDcEI7UUFFRCxLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDbkMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1NBQ25CO1FBRUQsSUFBSSxDQUFDLGVBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLEVBQUU7WUFDOUMsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO2dCQUNuQyxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7YUFDdkI7WUFFRCxxQkFBcUI7WUFDckIsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO1NBQ3ZCO1FBRUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUMxQyxDQUFDOzs7Ozs7SUFHTSxhQUFhLEtBQUksQ0FBQzs7Ozs7Ozs7SUFHbEIsY0FBYyxLQUFJLENBQUM7SUFFMUI7OztPQUdHO0lBQ0ksYUFBYSxDQUFDLFNBQW9CO1FBQ3ZDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNILGNBQWMsQ0FBQyxVQUFtQjtRQUNoQyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssSUFBSSxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztTQUNuRTtRQUVELG1DQUFtQztRQUNuQyxLQUFLLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDaEMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsS0FBSyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDOUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7YUFDdkY7U0FDRjtRQUVELElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7T0FFRztJQUNLLGVBQWUsQ0FBQyxZQUFxQjtRQUMzQyxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksWUFBWSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDaEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsd0RBQXdEO1FBQ3hELElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNmLElBQUksQ0FBQyxZQUFZLEVBQUU7Z0JBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMsNENBQTRDLENBQUMsQ0FBQzthQUMvRDtZQUVELE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxZQUFZLENBQUMsQ0FBQztTQUN2RDtRQUVELG9FQUFvRTtRQUNwRSx3REFBd0Q7UUFDeEQsSUFBSSxvQkFBVyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ2hDLE1BQU0sT0FBTyxHQUFHLGlCQUFZLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDNUMsTUFBTSxPQUFPLEdBQUcsaUJBQVksQ0FBQyxXQUFNLEVBQUUsQ0FBQyxDQUFDO1lBRXZDLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsRUFBRTtnQkFDL0IsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxZQUFZLGFBQVosWUFBWSxjQUFaLFlBQVksR0FBSSxHQUFHLENBQUMsQ0FBQzthQUNuRDtZQUVELE9BQU8sZ0JBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQU0sRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUM7U0FDcEQ7UUFFRCxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxhQUFaLFlBQVksY0FBWixZQUFZLEdBQUksR0FBRyxDQUFDLENBQUM7SUFDM0MsQ0FBQzs7QUE1VEgsMEJBNlRDOzs7Ozs7Ozs7OztBQTNUd0Isb0JBQVksR0FBRyxTQUFTLENBQUM7Ozs7OztBQStUbEQsSUFBWSxXQVNYO0FBVEQsV0FBWSxXQUFXO0lBRXJCLGtDQUFtQixDQUFBO0lBR25CLDBCQUFXLENBQUE7SUFHWCwwQkFBVyxDQUFBO0FBQ2IsQ0FBQyxFQVRXLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBU3RCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgbWtkdGVtcFN5bmMsIHJlYWxwYXRoU3luYyB9IGZyb20gJ2ZzJztcbmltcG9ydCB7IHRtcGRpciB9IGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBjbGVhbnVwLCBGSUxFX01BTklGRVNUIH0gZnJvbSAnLi9jbGVhbnVwJztcbmltcG9ydCB7IElTX1RFU1RfUlVOLCBQUk9KRU5fVkVSU0lPTiB9IGZyb20gJy4vY29tbW9uJztcbmltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gJy4vY29tcG9uZW50JztcbmltcG9ydCB7IERlcGVuZGVuY2llcyB9IGZyb20gJy4vZGVwZW5kZW5jaWVzJztcbmltcG9ydCB7IEZpbGVCYXNlIH0gZnJvbSAnLi9maWxlJztcbmltcG9ydCB7IEdpdEF0dHJpYnV0ZXNGaWxlIH0gZnJvbSAnLi9naXRhdHRyaWJ1dGVzJztcbmltcG9ydCB7IElnbm9yZUZpbGUgfSBmcm9tICcuL2lnbm9yZS1maWxlJztcbmltcG9ydCAqIGFzIGludmVudG9yeSBmcm9tICcuL2ludmVudG9yeSc7XG5pbXBvcnQgeyByZXNvbHZlSW5pdFByb2plY3QgfSBmcm9tICcuL2phdmFzY3JpcHQvcmVuZGVyLW9wdGlvbnMnO1xuaW1wb3J0IHsgSnNvbkZpbGUgfSBmcm9tICcuL2pzb24nO1xuaW1wb3J0IHsgTG9nZ2VyLCBMb2dnZXJPcHRpb25zIH0gZnJvbSAnLi9sb2dnZXInO1xuaW1wb3J0IHsgT2JqZWN0RmlsZSB9IGZyb20gJy4vb2JqZWN0LWZpbGUnO1xuaW1wb3J0IHsgSW5pdFByb2plY3RPcHRpb25IaW50cyB9IGZyb20gJy4vb3B0aW9uLWhpbnRzJztcbmltcG9ydCB7IFByb2plY3RCdWlsZCBhcyBQcm9qZWN0QnVpbGQgfSBmcm9tICcuL3Byb2plY3QtYnVpbGQnO1xuaW1wb3J0IHsgUHJvamVucmMsIFByb2plbnJjT3B0aW9ucyB9IGZyb20gJy4vcHJvamVucmMtanNvbic7XG5pbXBvcnQgeyBUYXNrLCBUYXNrT3B0aW9ucyB9IGZyb20gJy4vdGFzayc7XG5pbXBvcnQgeyBUYXNrcyB9IGZyb20gJy4vdGFza3MnO1xuaW1wb3J0IHsgaXNUcnV0aHkgfSBmcm9tICcuL3V0aWwnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVjdE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBhcmVudD86IFByb2plY3Q7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvdXRkaXI/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsb2dnaW5nPzogTG9nZ2VyT3B0aW9ucztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb2plbnJjSnNvbj86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcm9qZW5yY0pzb25PcHRpb25zPzogUHJvamVucmNPcHRpb25zO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJvamVuQ29tbWFuZD86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBQcm9qZWN0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBERUZBVUxUX1RBU0sgPSAnZGVmYXVsdCc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGdpdGlnbm9yZTogSWdub3JlRmlsZTtcblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBnaXRhdHRyaWJ1dGVzOiBHaXRBdHRyaWJ1dGVzRmlsZTtcblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcGFyZW50PzogUHJvamVjdDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG91dGRpcjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcm9vdDogUHJvamVjdDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdGFza3M6IFRhc2tzO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZGVwczogRGVwZW5kZW5jaWVzO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbG9nZ2VyOiBMb2dnZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGluaXRQcm9qZWN0PzogSW5pdFByb2plY3Q7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHByb2plbkNvbW1hbmQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdFRhc2s6IFRhc2s7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHByb2plY3RCdWlsZDogUHJvamVjdEJ1aWxkO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2NvbXBvbmVudHMgPSBuZXcgQXJyYXk8Q29tcG9uZW50PigpO1xuICBwcml2YXRlIHJlYWRvbmx5IHN1YnByb2plY3RzID0gbmV3IEFycmF5PFByb2plY3Q+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgdGlwcyA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgZXhjbHVkZUZyb21DbGVhbnVwOiBzdHJpbmdbXTtcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBQcm9qZWN0T3B0aW9ucykge1xuICAgIHRoaXMuaW5pdFByb2plY3QgPSByZXNvbHZlSW5pdFByb2plY3Qob3B0aW9ucyk7XG5cbiAgICB0aGlzLm5hbWUgPSBvcHRpb25zLm5hbWU7XG4gICAgdGhpcy5wYXJlbnQgPSBvcHRpb25zLnBhcmVudDtcbiAgICB0aGlzLmV4Y2x1ZGVGcm9tQ2xlYW51cCA9IFtdO1xuICAgIHRoaXMucHJvamVuQ29tbWFuZCA9IG9wdGlvbnMucHJvamVuQ29tbWFuZCA/PyAnbnB4IHByb2plbic7XG5cbiAgICB0aGlzLm91dGRpciA9IHRoaXMuZGV0ZXJtaW5lT3V0ZGlyKG9wdGlvbnMub3V0ZGlyKTtcbiAgICB0aGlzLnJvb3QgPSB0aGlzLnBhcmVudCA/IHRoaXMucGFyZW50LnJvb3QgOiB0aGlzO1xuXG4gICAgLy8gbXVzdCBoYXBwZW4gYWZ0ZXIgdGhpcy5vdXRkaXIsIHRoaXMucGFyZW50IGFuZCB0aGlzLnJvb3QgYXJlIGluaXRpYWxpemVkXG4gICAgdGhpcy5wYXJlbnQ/Ll9hZGRTdWJQcm9qZWN0KHRoaXMpO1xuXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cbiAgICB0aGlzLmdpdGF0dHJpYnV0ZXMgPSBuZXcgR2l0QXR0cmlidXRlc0ZpbGUodGhpcyk7XG4gICAgdGhpcy5hbm5vdGF0ZUdlbmVyYXRlZCgnLy5wcm9qZW4vKionKTsgLy8gY29udGVudHMgIG9mIHRoZSAucHJvamVuLyBkaXJlY3RvcnkgYXJlIGdlbmVyYXRlZCBieSBwcm9qZW5cbiAgICB0aGlzLmFubm90YXRlR2VuZXJhdGVkKGAvJHt0aGlzLmdpdGF0dHJpYnV0ZXMucGF0aH1gKTsgLy8gdGhlIC5naXRhdHRyaWJ1dGVzIGZpbGUgaXRzZWxmIGlzIGdlbmVyYXRlZFxuXG4gICAgdGhpcy5naXRpZ25vcmUgPSBuZXcgSWdub3JlRmlsZSh0aGlzLCAnLmdpdGlnbm9yZScpO1xuICAgIHRoaXMuZ2l0aWdub3JlLmV4Y2x1ZGUoJ25vZGVfbW9kdWxlcy8nKTsgLy8gY3JlYXRlZCBieSBydW5uaW5nIGBucHggcHJvamVuYFxuICAgIHRoaXMuZ2l0aWdub3JlLmluY2x1ZGUoYC8ke3RoaXMuZ2l0YXR0cmlidXRlcy5wYXRofWApO1xuXG4gICAgLy8gb2ggbm86IHRhc2tzIGRlcGVuZHMgb24gZ2l0aWdub3JlIHNvIGl0IGhhcyB0byBiZSBpbml0aWFsaXplZCBhZnRlclxuICAgIC8vIHNtZWxscyBsaWtlIGRlcCBpbmplY3Rpb25uIGJ1dCBnb2QgZm9yYmlkLlxuICAgIHRoaXMudGFza3MgPSBuZXcgVGFza3ModGhpcyk7XG5cbiAgICB0aGlzLmRlZmF1bHRUYXNrID0gdGhpcy50YXNrcy5hZGRUYXNrKFByb2plY3QuREVGQVVMVF9UQVNLLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ1N5bnRoZXNpemUgcHJvamVjdCBmaWxlcycsXG4gICAgfSk7XG5cbiAgICB0aGlzLnByb2plY3RCdWlsZCA9IG5ldyBQcm9qZWN0QnVpbGQodGhpcyk7XG5cbiAgICB0aGlzLmRlcHMgPSBuZXcgRGVwZW5kZW5jaWVzKHRoaXMpO1xuXG4gICAgdGhpcy5sb2dnZXIgPSBuZXcgTG9nZ2VyKHRoaXMsIG9wdGlvbnMubG9nZ2luZyk7XG5cbiAgICBjb25zdCBwcm9qZW5yY0pzb24gPSBvcHRpb25zLnByb2plbnJjSnNvbiA/PyBmYWxzZTtcbiAgICBpZiAocHJvamVucmNKc29uKSB7XG4gICAgICBuZXcgUHJvamVucmModGhpcywgb3B0aW9ucy5wcm9qZW5yY0pzb25PcHRpb25zKTtcbiAgICB9XG5cbiAgICBuZXcgSnNvbkZpbGUodGhpcywgRklMRV9NQU5JRkVTVCwge1xuICAgICAgb21pdEVtcHR5OiB0cnVlLFxuICAgICAgb2JqOiAoKSA9PiAoe1xuICAgICAgICAvLyByZXBsYWNlIGBcXGAgd2l0aCBgL2AgdG8gZW5zdXJlIHBhdGhzIG1hdGNoIGFjcm9zcyBwbGF0Zm9ybXNcbiAgICAgICAgZmlsZXM6IHRoaXMuZmlsZXMuZmlsdGVyKGYgPT4gZi5yZWFkb25seSkubWFwKGYgPT4gZi5wYXRoLnJlcGxhY2UoL1xcXFwvZywgJy8nKSksXG4gICAgICB9KSxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBnZXQgY29tcG9uZW50cygpIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX2NvbXBvbmVudHNdO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldCBmaWxlcygpOiBGaWxlQmFzZVtdIHtcbiAgICBjb25zdCBpc0ZpbGUgPSAoYzogQ29tcG9uZW50KTogYyBpcyBGaWxlQmFzZSA9PiBjIGluc3RhbmNlb2YgRmlsZUJhc2U7XG4gICAgcmV0dXJuIHRoaXMuX2NvbXBvbmVudHMuZmlsdGVyKGlzRmlsZSkuc29ydCgoZjEsIGYyKSA9PiBmMS5wYXRoLmxvY2FsZUNvbXBhcmUoZjIucGF0aCkpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFRhc2sobmFtZTogc3RyaW5nLCBwcm9wczogVGFza09wdGlvbnMgPSB7IH0pIHtcbiAgICByZXR1cm4gdGhpcy50YXNrcy5hZGRUYXNrKG5hbWUsIHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZW1vdmVUYXNrKG5hbWU6IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLnRhc2tzLnJlbW92ZVRhc2sobmFtZSk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGJ1aWxkVGFzaygpIHsgcmV0dXJuIHRoaXMucHJvamVjdEJ1aWxkLmJ1aWxkVGFzazsgfVxuICBwdWJsaWMgZ2V0IGNvbXBpbGVUYXNrKCkgeyByZXR1cm4gdGhpcy5wcm9qZWN0QnVpbGQuY29tcGlsZVRhc2s7IH1cbiAgcHVibGljIGdldCB0ZXN0VGFzaygpIHsgcmV0dXJuIHRoaXMucHJvamVjdEJ1aWxkLnRlc3RUYXNrOyB9XG4gIHB1YmxpYyBnZXQgcHJlQ29tcGlsZVRhc2soKSB7IHJldHVybiB0aGlzLnByb2plY3RCdWlsZC5wcmVDb21waWxlVGFzazsgfVxuICBwdWJsaWMgZ2V0IHBvc3RDb21waWxlVGFzaygpIHsgcmV0dXJuIHRoaXMucHJvamVjdEJ1aWxkLnBvc3RDb21waWxlVGFzazsgfVxuICBwdWJsaWMgZ2V0IHBhY2thZ2VUYXNrKCkgeyByZXR1cm4gdGhpcy5wcm9qZWN0QnVpbGQucGFja2FnZVRhc2s7IH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHRyeUZpbmRGaWxlKGZpbGVQYXRoOiBzdHJpbmcpOiBGaWxlQmFzZSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgYWJzb2x1dGUgPSBwYXRoLmlzQWJzb2x1dGUoZmlsZVBhdGgpID8gZmlsZVBhdGggOiBwYXRoLnJlc29sdmUodGhpcy5vdXRkaXIsIGZpbGVQYXRoKTtcbiAgICBmb3IgKGNvbnN0IGZpbGUgb2YgdGhpcy5maWxlcykge1xuICAgICAgaWYgKGFic29sdXRlID09PSBmaWxlLmFic29sdXRlUGF0aCkge1xuICAgICAgICByZXR1cm4gZmlsZTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGNoaWxkIG9mIHRoaXMuc3VicHJvamVjdHMpIHtcbiAgICAgIGNvbnN0IGZpbGUgPSBjaGlsZC50cnlGaW5kRmlsZShhYnNvbHV0ZSk7XG4gICAgICBpZiAoZmlsZSkge1xuICAgICAgICByZXR1cm4gZmlsZTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgdHJ5RmluZEpzb25GaWxlKGZpbGVQYXRoOiBzdHJpbmcpOiBKc29uRmlsZSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgZmlsZSA9IHRoaXMudHJ5RmluZE9iamVjdEZpbGUoZmlsZVBhdGgpO1xuICAgIGlmICghZmlsZSkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBpZiAoIShmaWxlIGluc3RhbmNlb2YgSnNvbkZpbGUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGZvdW5kIGZpbGUgJHtmaWxlUGF0aH0gYnV0IGl0IGlzIG5vdCBhIEpzb25GaWxlLiBnb3Q6ICR7ZmlsZS5jb25zdHJ1Y3Rvci5uYW1lfWApO1xuICAgIH1cblxuICAgIHJldHVybiBmaWxlO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgdHJ5RmluZE9iamVjdEZpbGUoZmlsZVBhdGg6IHN0cmluZyk6IE9iamVjdEZpbGUgfCB1bmRlZmluZWQge1xuICAgIGNvbnN0IGZpbGUgPSB0aGlzLnRyeUZpbmRGaWxlKGZpbGVQYXRoKTtcbiAgICBpZiAoIWZpbGUpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgaWYgKCEoZmlsZSBpbnN0YW5jZW9mIE9iamVjdEZpbGUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGZvdW5kIGZpbGUgJHtmaWxlUGF0aH0gYnV0IGl0IGlzIG5vdCBhIE9iamVjdEZpbGUuIGdvdDogJHtmaWxlLmNvbnN0cnVjdG9yLm5hbWV9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGZpbGU7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRUaXAobWVzc2FnZTogc3RyaW5nKSB7XG4gICAgdGhpcy50aXBzLnB1c2gobWVzc2FnZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkRXhjbHVkZUZyb21DbGVhbnVwKC4uLmdsb2JzOiBzdHJpbmdbXSkge1xuICAgIHRoaXMuZXhjbHVkZUZyb21DbGVhbnVwLnB1c2goLi4uZ2xvYnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJ1blRhc2tDb21tYW5kKHRhc2s6IFRhc2spIHtcbiAgICByZXR1cm4gYG5weCBwcm9qZW5AJHtQUk9KRU5fVkVSU0lPTn0gJHt0YXNrLm5hbWV9YDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFBhY2thZ2VJZ25vcmUoX3BhdHRlcm46IHN0cmluZykge1xuICAgIC8vIG5vdGhpbmcgdG8gZG8gYXQgdGhlIGFic3RyYWN0IGxldmVsXG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkR2l0SWdub3JlKHBhdHRlcm46IHN0cmluZykge1xuICAgIHRoaXMuZ2l0aWdub3JlLmFkZFBhdHRlcm5zKHBhdHRlcm4pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbm5vdGF0ZUdlbmVyYXRlZChfZ2xvYjogc3RyaW5nKTogdm9pZCB7XG4gICAgLy8gbm90aGluZyB0byBkbyBhdCB0aGUgYWJzdHJhY3QgbGV2ZWxcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3ludGgoKTogdm9pZCB7XG4gICAgY29uc3Qgb3V0ZGlyID0gdGhpcy5vdXRkaXI7XG4gICAgdGhpcy5sb2dnZXIuZGVidWcoJ1N5bnRoZXNpemluZyBwcm9qZWN0Li4uJyk7XG5cbiAgICB0aGlzLnByZVN5bnRoZXNpemUoKTtcblxuICAgIGZvciAoY29uc3QgY29tcCBvZiB0aGlzLl9jb21wb25lbnRzKSB7XG4gICAgICBjb21wLnByZVN5bnRoZXNpemUoKTtcbiAgICB9XG5cbiAgICAvLyB3ZSBleGNsdWRlIGFsbCBzdWJwcm9qZWN0IGRpcmVjdG9yaWVzIHRvIGVuc3VyZSB0aGF0IHdoZW4gc3VicHJvamVjdC5zeW50aCgpXG4gICAgLy8gZ2V0cyBjYWxsZWQgYmVsb3cgYWZ0ZXIgY2xlYW51cCgpLCBzdWJwcm9qZWN0IGdlbmVyYXRlZCBmaWxlcyBhcmUgbGVmdCBpbnRhY3RcbiAgICBmb3IgKGNvbnN0IHN1YnByb2plY3Qgb2YgdGhpcy5zdWJwcm9qZWN0cykge1xuICAgICAgdGhpcy5hZGRFeGNsdWRlRnJvbUNsZWFudXAoc3VicHJvamVjdC5vdXRkaXIgKyAnLyoqJyk7XG4gICAgfVxuXG4gICAgLy8gZGVsZXRlIGFsbCBnZW5lcmF0ZWQgZmlsZXMgYmVmb3JlIHdlIHN0YXJ0IHN5bnRoZXNpemluZyBuZXcgb25lc1xuICAgIGNsZWFudXAob3V0ZGlyLCB0aGlzLmV4Y2x1ZGVGcm9tQ2xlYW51cCk7XG5cbiAgICBmb3IgKGNvbnN0IHN1YnByb2plY3Qgb2YgdGhpcy5zdWJwcm9qZWN0cykge1xuICAgICAgc3VicHJvamVjdC5zeW50aCgpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgY29tcCBvZiB0aGlzLl9jb21wb25lbnRzKSB7XG4gICAgICBjb21wLnN5bnRoZXNpemUoKTtcbiAgICB9XG5cbiAgICBpZiAoIWlzVHJ1dGh5KHByb2Nlc3MuZW52LlBST0pFTl9ESVNBQkxFX1BPU1QpKSB7XG4gICAgICBmb3IgKGNvbnN0IGNvbXAgb2YgdGhpcy5fY29tcG9uZW50cykge1xuICAgICAgICBjb21wLnBvc3RTeW50aGVzaXplKCk7XG4gICAgICB9XG5cbiAgICAgIC8vIHByb2plY3QtbGV2ZWwgaG9va1xuICAgICAgdGhpcy5wb3N0U3ludGhlc2l6ZSgpO1xuICAgIH1cblxuICAgIHRoaXMubG9nZ2VyLmRlYnVnKCdTeW50aGVzaXMgY29tcGxldGUnKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcHJlU3ludGhlc2l6ZSgpIHt9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHBvc3RTeW50aGVzaXplKCkge31cblxuICAvKipcbiAgICogQWRkcyBhIGNvbXBvbmVudCB0byB0aGUgcHJvamVjdC5cbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2FkZENvbXBvbmVudChjb21wb25lbnQ6IENvbXBvbmVudCkge1xuICAgIHRoaXMuX2NvbXBvbmVudHMucHVzaChjb21wb25lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzdWItcHJvamVjdCB0byB0aGlzIHByb2plY3QuXG4gICAqXG4gICAqIFRoaXMgaXMgYXV0b21hdGljYWxseSBjYWxsZWQgd2hlbiBhIG5ldyBwcm9qZWN0IGlzIGNyZWF0ZWQgd2l0aCBgcGFyZW50YFxuICAgKiBwb2ludGluZyB0byB0aGlzIHByb2plY3QsIHNvIHRoZXJlIGlzIG5vIHJlYWwgbmVlZCB0byBjYWxsIHRoaXMgbWFudWFsbHkuXG4gICAqXG4gICAqIEBwYXJhbSBzdWItcHJvamVjdCBUaGUgY2hpbGQgcHJvamVjdCB0byBhZGQuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgX2FkZFN1YlByb2plY3Qoc3VicHJvamVjdDogUHJvamVjdCkge1xuICAgIGlmIChzdWJwcm9qZWN0LnBhcmVudCAhPT0gdGhpcykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdcInBhcmVudFwiIG9mIGNoaWxkIHByb2plY3QgbXVzdCBiZSB0aGlzIHByb2plY3QnKTtcbiAgICB9XG5cbiAgICAvLyBjaGVjayB0aGF0IGBvdXRkaXJgIGlzIGV4Y2x1c2l2ZVxuICAgIGZvciAoY29uc3QgcCBvZiB0aGlzLnN1YnByb2plY3RzKSB7XG4gICAgICBpZiAocGF0aC5yZXNvbHZlKHAub3V0ZGlyKSA9PT0gcGF0aC5yZXNvbHZlKHN1YnByb2plY3Qub3V0ZGlyKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYHRoZXJlIGlzIGFscmVhZHkgYSBzdWItcHJvamVjdCB3aXRoIFwib3V0ZGlyXCI6ICR7c3VicHJvamVjdC5vdXRkaXJ9YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdGhpcy5zdWJwcm9qZWN0cy5wdXNoKHN1YnByb2plY3QpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlc29sdmVzIHRoZSBwcm9qZWN0J3Mgb3V0cHV0IGRpcmVjdG9yeS5cbiAgICovXG4gIHByaXZhdGUgZGV0ZXJtaW5lT3V0ZGlyKG91dGRpck9wdGlvbj86IHN0cmluZykge1xuICAgIGlmICh0aGlzLnBhcmVudCAmJiBvdXRkaXJPcHRpb24gJiYgcGF0aC5pc0Fic29sdXRlKG91dGRpck9wdGlvbikpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignXCJvdXRkaXJcIiBtdXN0IGJlIGEgcmVsYXRpdmUgcGF0aCcpO1xuICAgIH1cblxuICAgIC8vIGlmIHRoaXMgaXMgYSBzdWJwcm9qZWN0LCBpdCBpcyByZWxhdGl2ZSB0byB0aGUgcGFyZW50XG4gICAgaWYgKHRoaXMucGFyZW50KSB7XG4gICAgICBpZiAoIW91dGRpck9wdGlvbikge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wib3V0ZGlyXCIgbXVzdCBiZSBzcGVjaWZpZWQgZm9yIHN1YnByb2plY3RzJyk7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiBwYXRoLnJlc29sdmUodGhpcy5wYXJlbnQub3V0ZGlyLCBvdXRkaXJPcHRpb24pO1xuICAgIH1cblxuICAgIC8vIGlmIHRoaXMgaXMgcnVubmluZyBpbnNpZGUgYSB0ZXN0IGFuZCBvdXRkaXIgaXMgbm90IGV4cGxpY2l0bHkgc2V0XG4gICAgLy8gdXNlIGEgdGVtcCBkaXJlY3RvcnkgKHVubGVzcyBjd2QgaXMgYWxlYWR5IHVuZGVyIHRtcClcbiAgICBpZiAoSVNfVEVTVF9SVU4gJiYgIW91dGRpck9wdGlvbikge1xuICAgICAgY29uc3QgcmVhbEN3ZCA9IHJlYWxwYXRoU3luYyhwcm9jZXNzLmN3ZCgpKTtcbiAgICAgIGNvbnN0IHJlYWxUbXAgPSByZWFscGF0aFN5bmModG1wZGlyKCkpO1xuXG4gICAgICBpZiAocmVhbEN3ZC5zdGFydHNXaXRoKHJlYWxUbXApKSB7XG4gICAgICAgIHJldHVybiBwYXRoLnJlc29sdmUocmVhbEN3ZCwgb3V0ZGlyT3B0aW9uID8/ICcuJyk7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiBta2R0ZW1wU3luYyhwYXRoLmpvaW4odG1wZGlyKCksICdwcm9qZW4uJykpO1xuICAgIH1cblxuICAgIHJldHVybiBwYXRoLnJlc29sdmUob3V0ZGlyT3B0aW9uID8/ICcuJyk7XG4gIH1cbn1cblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIFByb2plY3RUeXBlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFVOS05PV04gPSAndW5rbm93bicsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgTElCID0gJ2xpYicsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQVBQID0gJ2FwcCdcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSW5pdFByb2plY3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZnFuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXJnczogUmVjb3JkPHN0cmluZywgYW55PjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0eXBlOiBpbnZlbnRvcnkuUHJvamVjdFR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb21tZW50czogSW5pdFByb2plY3RPcHRpb25IaW50cztcbn1cbiJdfQ==