"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJavaScriptOptions = exports.resolveInitProject = exports.renderProjenInitOptions = void 0;
const inventory = require("../inventory");
const option_hints_1 = require("../option-hints");
const PROJEN_NEW = '__new__';
const TAB = makePadding(2);
/**
 * Renders options as if the project was created via `projen new` (embeds the __new__ field).
 */
function renderProjenInitOptions(fqn, args, comments = option_hints_1.InitProjectOptionHints.NONE) {
    return {
        ...args,
        [PROJEN_NEW]: { fqn, args, comments },
    };
}
exports.renderProjenInitOptions = renderProjenInitOptions;
function resolveInitProject(opts) {
    const f = opts[PROJEN_NEW];
    if (!f) {
        return undefined;
    }
    const type = inventory.resolveProjectType(f.fqn);
    if (!type) {
        throw new Error(`unable to resolve project type for ${f.fqn}`);
    }
    return {
        args: f.args,
        fqn: f.fqn,
        type: type,
        comments: f.comments,
    };
}
exports.resolveInitProject = resolveInitProject;
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 *
 * Returns the printed output and a set of required imports as an object
 * in the form { options, imports }.
 */
function renderJavaScriptOptions(opts) {
    var _a, _b, _c, _d;
    const renders = {};
    const optionsWithDefaults = [];
    const allImports = new Set();
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        if (opts.args[optionName] !== undefined) {
            const arg = opts.args[optionName];
            const { js, imports } = renderArgAsJavaScript(arg, option);
            for (const importStr of imports) {
                allImports.add(importStr);
            }
            renders[optionName] = `${optionName}: ${js},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            renders[optionName] = `// ${optionName}: ${defaultValue},`;
        }
    }
    const bootstrap = (_c = opts.bootstrap) !== null && _c !== void 0 ? _c : false;
    if (bootstrap) {
        for (const arg of ((_d = opts.omitFromBootstrap) !== null && _d !== void 0 ? _d : [])) {
            delete opts.args[arg];
        }
        renders[PROJEN_NEW] = `${PROJEN_NEW}: ${JSON.stringify({ args: opts.args, fqn: opts.type.fqn, comments: opts.comments })},`;
        optionsWithDefaults.push(PROJEN_NEW);
    }
    // generate rendering
    const result = [];
    result.push('{');
    // render options with defaults
    optionsWithDefaults.sort();
    for (const optionName of optionsWithDefaults) {
        result.push(`${TAB}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults as comments
    if (opts.comments === option_hints_1.InitProjectOptionHints.ALL) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined);
        result.push(...renderCommentedOptionsByModule(renders, options));
    }
    else if (opts.comments === option_hints_1.InitProjectOptionHints.FEATURED) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined && opt.featured);
        result.push(...renderCommentedOptionsInOrder(renders, options));
    }
    else if (opts.comments === option_hints_1.InitProjectOptionHints.NONE) {
        // don't render any extra options
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return { renderedOptions: result.join('\n'), imports: allImports };
}
exports.renderJavaScriptOptions = renderJavaScriptOptions;
function renderCommentedOptionsByModule(renders, options) {
    var _a;
    const optionsByModule = {};
    for (const option of options) {
        const parentModule = option.parent;
        optionsByModule[parentModule] = (_a = optionsByModule[parentModule]) !== null && _a !== void 0 ? _a : [];
        optionsByModule[parentModule].push(option);
    }
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const [moduleName, optionGroup] of Object.entries(optionsByModule).sort()) {
        result.push(`${TAB}/* ${moduleName} */`);
        for (const option of optionGroup) {
            const paramRender = renders[option.name];
            const docstring = option.docs || 'No documentation found.';
            result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
        }
        result.push('');
    }
    return result;
}
function renderCommentedOptionsInOrder(renders, options) {
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const option of options) {
        const paramRender = renders[option.name];
        const docstring = option.docs || 'No documentation found.';
        result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
    }
    return result;
}
/**
 * Renders a value as a JavaScript value, converting strings to enums where
 * appropriate. The type must be JSON-like (string, number, boolean, array,
 * enum, or JSON object).
 *
 * Returns a JavaScript expression as a string, and the names of any
 * necessary imports.
 */
function renderArgAsJavaScript(arg, option) {
    if (option.kind === 'enum') {
        if (!option.fqn) {
            throw new Error(`fqn field is missing from enum option ${option.name}`);
        }
        const parts = option.fqn.split('.'); // -> ['projen', 'web', 'MyEnum']
        const enumChoice = String(arg).toUpperCase().replace(/-/g, '_'); // custom-value -> CUSTOM_VALUE
        const js = `${parts.slice(1).join('.')}.${enumChoice}`; // -> web.MyEnum.CUSTOM_VALUE
        const importName = parts[1]; // -> web
        return { js, imports: [importName] };
    }
    else if (option.jsonLike) {
        return { js: JSON.stringify(arg), imports: [] };
    }
    else {
        throw new Error(`Unexpected option ${option.name} - cannot render a value for this option because it does not have a JSON-like type.`);
    }
}
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
//# sourceMappingURL=data:application/json;base64,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