"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultNpmToken = exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const url_1 = require("url");
const fs_extra_1 = require("fs-extra");
const semver = require("semver");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const util_1 = require("../util");
const UNLICENSED = 'UNLICENSED';
const DEFAULT_NPM_REGISTRY_URL = 'https://registry.npmjs.org/';
const GITHUB_PACKAGES_REGISTRY = 'npm.pkg.github.com';
const DEFAULT_NPM_TOKEN_SECRET = 'NPM_TOKEN';
const DEFAULT_GITHUB_TOKEN_SECRET = 'GITHUB_TOKEN';
/**
 * (experimental) Represents the npm `package.json` file.
 *
 * @experimental
 */
class NodePackage extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = (_b = options.packageName) !== null && _b !== void 0 ? _b : project.name;
        this.peerDependencyOptions = (_c = options.peerDependencyOptions) !== null && _c !== void 0 ? _c : {};
        this.allowLibraryDependencies = (_d = options.allowLibraryDependencies) !== null && _d !== void 0 ? _d : true;
        this.packageManager = (_e = options.packageManager) !== null && _e !== void 0 ? _e : NodePackageManager.YARN;
        this.entrypoint = (_f = options.entrypoint) !== null && _f !== void 0 ? _f : 'lib/index.js';
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.processDeps(options);
        const prev = (_g = this.readPackageJson()) !== null && _g !== void 0 ? _g : {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository ? undefined : {
                type: 'git',
                url: options.repository,
                directory: options.repositoryDirectory,
            },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== '' ? this.entrypoint : undefined,
            license: () => { var _b; return (_b = this.license) !== null && _b !== void 0 ? _b : UNLICENSED; },
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev === null || prev === void 0 ? void 0 : prev.version),
            bugs: (options.bugsEmail || options.bugsUrl) ? {
                email: options.bugsEmail,
                url: options.bugsUrl,
            } : undefined,
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries((_h = options.scripts) !== null && _h !== void 0 ? _h : {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, 'package.json', {
            obj: this.manifest,
            readonly: false,
        });
        this.addKeywords(...(_j = options.keywords) !== null && _j !== void 0 ? _j : []);
        this.addBin((_k = options.bin) !== null && _k !== void 0 ? _k : {});
        // automatically add all executable files under "bin"
        if ((_l = options.autoDetectBin) !== null && _l !== void 0 ? _l : true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.addNodeEngine();
        // license
        if ((_m = options.licensed) !== null && _m !== void 0 ? _m : true) {
            this.license = (_o = options.license) !== null && _o !== void 0 ? _o : 'Apache-2.0';
        }
    }
    /**
     * (experimental) Defines normal dependencies.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * (experimental) Defines development/test dependencies.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * (experimental) Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(',')}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * (experimental) Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install.
     * @experimental
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(',')}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * (experimental) Adds an `engines` requirement to your package.
     *
     * @param engine The engine (e.g. `node`).
     * @param version The semantic version requirement (e.g. `^10`).
     * @experimental
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * (experimental) Adds keywords to package.json (deduplicated).
     *
     * @param keywords The keywords to add.
     * @experimental
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    /**
     * @experimental
     */
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * (experimental) Override the contents of an npm package.json script.
     *
     * @param name The script name.
     * @param command The command to execute.
     * @experimental
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * (experimental) Removes the npm script (always successful).
     *
     * @param name The name of the script.
     * @experimental
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * (deprecated) Indicates if a script by the given name is defined.
     *
     * @param name The name of the script.
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * (experimental) Directly set fields in `package.json`.
     *
     * @param name field name.
     * @param value field value.
     * @experimental
     * @escape true
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * (experimental) Sets the package version.
     *
     * @param version Package version.
     * @experimental
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * (experimental) Returns the command to execute in order to install all dependencies (always frozen).
     *
     * @experimental
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * (experimental) Renders `yarn install` or `npm install` with lockfile update (not frozen).
     *
     * @experimental
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * (experimental) Render a package manager specific command to upgrade all requested dependencies.
     *
     * @experimental
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .map(d => d.name)
                    .filter(d => include ? include.includes(d) : true)
                    .filter(d => !exclude.includes(d))
                    .join(' ')}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                lazy = upgradePackages('yarn upgrade');
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages('npm update');
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages('pnpm update');
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    /**
     * (experimental) Called after synthesis.
     *
     * Order is *not* guaranteed.
     *
     * @experimental
     */
    postSynthesize() {
        super.postSynthesize();
        const outdir = this.project.outdir;
        util_1.exec(this.renderInstallCommand(this.isAutomatedBuild), { cwd: outdir });
        this.resolveDepsAndWritePackageJson(outdir);
    }
    /**
     * (experimental) The command which executes "projen".
     *
     * @experimental
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_1.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return '0.0.0';
        }
        return currVersion !== null && currVersion !== void 0 ? currVersion : '0.0.0';
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return util_1.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl !== null && npmRegistryUrl !== void 0 ? npmRegistryUrl : DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = (_b = options.npmAccess) !== null && _b !== void 0 ? _b : defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const isAwsCodeArtifact = release_1.isAwsCodeArtifactRegistry(npmRegistryUrl);
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
        }
        else {
            if (((_c = options.codeArtifactOptions) === null || _c === void 0 ? void 0 : _c.accessKeyIdSecret) || ((_d = options.codeArtifactOptions) === null || _d === void 0 ? void 0 : _d.secretAccessKeySecret) || ((_e = options.codeArtifactOptions) === null || _e === void 0 ? void 0 : _e.roleToAssume)) {
                throw new Error('codeArtifactOptions must only be specified when publishing AWS CodeArtifact.');
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact) {
            codeArtifactOptions = {
                accessKeyIdSecret: (_g = (_f = options.codeArtifactOptions) === null || _f === void 0 ? void 0 : _f.accessKeyIdSecret) !== null && _g !== void 0 ? _g : 'AWS_ACCESS_KEY_ID',
                secretAccessKeySecret: (_j = (_h = options.codeArtifactOptions) === null || _h === void 0 ? void 0 : _h.secretAccessKeySecret) !== null && _j !== void 0 ? _j : 'AWS_SECRET_ACCESS_KEY',
                roleToAssume: (_k = options.codeArtifactOptions) === null || _k === void 0 ? void 0 : _k.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
        };
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = '';
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine('node', nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == '/') {
            return '';
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    'yarn install',
                    '--check-files',
                    ...frozen ? ['--frozen-lockfile'] : [],
                ].join(' ');
            case NodePackageManager.NPM:
                return frozen
                    ? 'npm ci'
                    : 'npm install';
            case NodePackageManager.PNPM:
                return [
                    'pnpm i',
                    ...frozen ? ['--frozen-lockfile'] : [],
                ].join(' ');
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        var _b, _c, _d, _e;
        this.addDeps(...(_b = options.deps) !== null && _b !== void 0 ? _b : []);
        this.addDevDeps(...(_c = options.devDeps) !== null && _c !== void 0 ? _c : []);
        this.addPeerDeps(...(_d = options.peerDeps) !== null && _d !== void 0 ? _d : []);
        this.addBundledDeps(...(_e = options.bundledDeps) !== null && _e !== void 0 ? _e : []);
    }
    renderDependencies() {
        var _b, _c, _d;
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        const pinned = (_b = this.peerDependencyOptions.pinnedDevDependency) !== null && _b !== void 0 ? _b : true;
        if (pinned) {
            for (const dep of this.project.deps.all.filter(d => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // skip if we already have a runtime dependency on this peer
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME)) {
                    continue;
                }
                if (dep.version) {
                    const ver = (_c = semver.minVersion(dep.version)) === null || _c === void 0 ? void 0 : _c.version;
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += '@' + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            const version = (_d = dep.version) !== null && _d !== void 0 ? _d : '*';
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(dep.name);
                    if (this.project.deps.all.find(d => d.name === dep.name && d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${dep.name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[dep.name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[dep.name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[dep.name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[dep.name] = version;
                    break;
            }
        }
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = devDependencies;
        this.manifest.peerDependencies = peerDependencies;
        this.manifest.dependencies = dependencies;
        this.manifest.bundledDependencies = bundledDependencies;
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== '*' || !currentVersion || currentVersion === '*') {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current !== null && current !== void 0 ? current : {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    resolveDepsAndWritePackageJson(outdir) {
        const root = path_1.join(outdir, 'package.json');
        const pkg = fs_extra_1.readJsonSync(root);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current !== null && current !== void 0 ? current : {};
            user = user !== null && user !== void 0 ? user : {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === '*') {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, { paths: [outdir] });
                        const module = fs_extra_1.readJsonSync(modulePath);
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return util_1.sorted(result);
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error('assertion failed');
        }
        pkg.dependencies = resolveDeps(pkg.dependencies, rendered.dependencies);
        pkg.devDependencies = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        pkg.peerDependencies = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        util_1.writeFile(root, JSON.stringify(pkg, undefined, 2));
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL ? this.npmRegistryUrl : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName) ? this.npmAccess : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_1.sorted(kwds.sort());
    }
    renderEngines() {
        return util_1.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = 'bin';
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_extra_1.existsSync(bindir)) {
            for (const file of fs_extra_1.readdirSync(bindir)) {
                try {
                    fs_extra_1.accessSync(path_1.join(bindir, file), fs_extra_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, '/');
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        var _b;
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: (_b = options.authorOrganization) !== null && _b !== void 0 ? _b : false,
            };
        }
        else {
            if (options.authorEmail || options.authorUrl || options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_1.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        for (const task of this.project.tasks.all.sort((x, y) => x.name.localeCompare(y.name))) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, 'package.json');
        if (!fs_extra_1.existsSync(file)) {
            return undefined;
        }
        return fs_extra_1.readJsonSync(file);
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.46.3" };
/**
 * (experimental) The node package manager to use.
 *
 * @experimental
 */
var NodePackageManager;
(function (NodePackageManager) {
    NodePackageManager["YARN"] = "yarn";
    NodePackageManager["NPM"] = "npm";
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * (experimental) Npm package access level.
 *
 * @experimental
 */
var NpmAccess;
(function (NpmAccess) {
    NpmAccess["PUBLIC"] = "public";
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes('@');
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if (release_1.isAwsCodeArtifactRegistry(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return npmToken !== null && npmToken !== void 0 ? npmToken : (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET);
}
exports.defaultNpmToken = defaultNpmToken;
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN) {
        return 'yarn.lock';
    }
    else if (packageManager === NodePackageManager.NPM) {
        return 'package-lock.json';
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return 'pnpm-lock.yaml';
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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