"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * (experimental) Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 *
 * @experimental
 */
class Projenrc extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, pom, options = {}) {
        var _b, _c, _d;
        super(project);
        const projenVersion = (_b = options.projenVersion) !== null && _b !== void 0 ? _b : common_1.PROJEN_VERSION;
        this.className = (_c = options.className) !== null && _c !== void 0 ? _c : 'projenrc';
        this.testScope = (_d = options.testScope) !== null && _d !== void 0 ? _d : true;
        const depType = this.testScope ? dependencies_1.DependencyType.TEST : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : '';
        const compileGoal = this.testScope ? 'compiler:testCompile' : 'compiler:compile';
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin('org.codehaus.mojo/exec-maven-plugin@3.0.0');
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask.exec(`mvn ${compileGoal} --quiet`);
        project.defaultTask.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    generateProjenrc() {
        var _b, _c, _d;
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = (_d = (_c = (_b = jsiiType.initializer) === null || _b === void 0 ? void 0 : _b.parameters) === null || _c === void 0 ? void 0 : _c[0].type) === null || _d === void 0 ? void 0 : _d.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn('cannot determine jsii type for project options');
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        const dir = this.testScope ? 'src/test/java' : 'src/main/java';
        const split = this.className.split('.');
        let javaClass, javaPackage;
        if (split.length === 1) {
            javaClass = split[0];
            javaPackage = [];
        }
        else {
            javaPackage = split.slice(0, split.length - 2);
            javaClass = split[split.length - 1];
        }
        const javaFile = path_1.join(this.project.outdir, dir, ...javaPackage, javaClass + '.java');
        // skip if file exists
        if (fs_extra_1.existsSync(javaFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = '') => lines.push(' '.repeat(indent * 4) + line);
        const openBlock = (line = '') => { emit(line + ' {'); indent++; };
        const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
            }
        }
        if (javaPackage.length > 0) {
            emit(`package ${javaPackage.join('.')};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiType)};`);
        emit(`import ${javaTarget.package}.${toJavaFullTypeName(jsiiOptionsType)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${javaClass}`);
        openBlock('public static void main(String[] args)');
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit('project.synth();');
        closeBlock();
        closeBlock();
        fs_extra_1.mkdirpSync(path_1.dirname(javaFile));
        fs_extra_1.writeFileSync(javaFile, lines.join('\n'));
        this.project.logger.info(`Project definition file was created at ${javaFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.46.3" };
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: '', imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push('.build()');
    const renderedOptions = lines.join(`\n${' '.repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === 'string' && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split('.');
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, '_');
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter(x => x).join('.');
}
//# sourceMappingURL=data:application/json;base64,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