"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * (experimental) Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 * @experimental
 */
class GithubWorkflow extends component_1.Component {
    /**
     * @experimental
     */
    constructor(github, name, options = {}) {
        super(github.project);
        this.events = {};
        this.jobs = {};
        this.name = name;
        this.concurrency = options.concurrency;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${name.toLocaleLowerCase()}.yml`, {
                obj: () => this.renderWorkflow(),
            });
        }
    }
    /**
     * (experimental) Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     * @experimental
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * (experimental) Adds a single job to the workflow.
     *
     * @param id The job name (unique within the workflow).
     * @param job The job specification.
     * @experimental
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * (experimental) Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     * @experimental
     */
    addJobs(jobs) {
        // verify that job has a "permissions" statement to ensure workflow can
        // operate in repos with default tokens set to readonly
        for (const [id, job] of Object.entries(jobs)) {
            if (!job.permissions) {
                throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
            }
        }
        // verify that job has a "runsOn" statement to ensure a worker can be selected appropriately
        for (const [id, job] of Object.entries(jobs)) {
            if (job.runsOn.length === 0) {
                throw new Error(`${id}: at least one runner selector labels must be provided in "runsOn" to ensure a runner instance can be selected`);
            }
        }
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency,
            jobs: renderJobs(this.jobs),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.46.3" };
function snakeCaseKeys(obj) {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
function renderJobs(jobs) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = util_1.kebabCaseKeys(_resolve_1.resolve(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            'name': job.name,
            'needs': arrayOrScalar(job.needs),
            'runs-on': arrayOrScalar(job.runsOn),
            'permissions': util_1.kebabCaseKeys(job.permissions),
            'environment': job.environment,
            'concurrency': job.concurrency,
            'outputs': renderJobOutputs(job.outputs),
            'env': job.env,
            'defaults': util_1.kebabCaseKeys(job.defaults),
            'if': job.if,
            'steps': steps,
            'timeout-minutes': job.timeoutMinutes,
            'strategy': renderJobStrategy(job.strategy),
            'continue-on-error': job.continueOnError,
            'container': job.container,
            'services': job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        var _b;
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            'max-parallel': strategy.maxParallel,
            'fail-fast': strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries((_b = strategy.matrix.domain) !== null && _b !== void 0 ? _b : {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
}
function arrayOrScalar(arr) {
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({ uses: 'actions/setup-java@v2', with: { 'distribution': 'temurin', 'java-version': tools.java.version } });
    }
    if (tools.node) {
        steps.push({ uses: 'actions/setup-node@v2', with: { 'node-version': tools.node.version } });
    }
    if (tools.python) {
        steps.push({ uses: 'actions/setup-python@v2', with: { 'python-version': tools.python.version } });
    }
    if (tools.go) {
        steps.push({ uses: 'actions/setup-go@v2', with: { 'go-version': tools.go.version } });
    }
    if (tools.dotnet) {
        steps.push({ uses: 'actions/setup-dotnet@v1', with: { 'dotnet-version': tools.dotnet.version } });
    }
    return steps;
}
//# sourceMappingURL=data:application/json;base64,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