"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyType = exports.Dependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const common_1 = require("./common");
const component_1 = require("./component");
const json_1 = require("./json");
/**
 * (experimental) The `Dependencies` component is responsible to track the list of dependencies a project has, and then used by project types as the model for rendering project-specific dependency manifests such as the dependencies section `package.json` files.
 *
 * To add a dependency you can use a project-type specific API such as
 * `nodeProject.addDeps()` or use the generic API of `project.deps`:
 *
 * @experimental
 */
class Dependencies extends component_1.Component {
    /**
     * (experimental) Adds a dependencies component to the project.
     *
     * @param project The parent project.
     * @experimental
     */
    constructor(project) {
        super(project);
        this._deps = new Array();
        // this is not really required at the moment, but actually quite useful as a
        // checked-in source of truth for dependencies and will potentially be
        // valuable in the future for CLI tools.
        new json_1.JsonFile(project, Dependencies.MANIFEST_FILE, {
            omitEmpty: true,
            obj: () => this.toJson(),
        });
    }
    /**
     * (experimental) Returns the coordinates of a dependency spec.
     *
     * Given `foo@^3.4.0` returns `{ name: "foo", version: "^3.4.0" }`.
     * Given `bar@npm:@bar/legacy` returns `{ name: "bar", version: "npm:@bar/legacy" }`.
     *
     * @experimental
     */
    static parseDependency(spec) {
        const scope = spec.startsWith('@');
        if (scope) {
            spec = spec.substr(1);
        }
        const [module, ...version] = spec.split('@');
        const name = scope ? `@${module}` : module;
        if (version.length == 0) {
            return { name };
        }
        else {
            return { name, version: version === null || version === void 0 ? void 0 : version.join('@') };
        }
    }
    /**
     * (experimental) A copy of all dependencies recorded for this project.
     *
     * The list is sorted by type->name->version
     *
     * @experimental
     */
    get all() {
        return [...this._deps].sort(compareDeps).map(normalizeDep);
    }
    /**
     * (experimental) Returns a dependency by name.
     *
     * Fails if there is no dependency defined by that name or if `type` is not
     * provided and there is more then one dependency type for this dependency.
     *
     * @param name The name of the dependency.
     * @param type The dependency type.
     * @returns a copy (cannot be modified)
     * @experimental
     */
    getDependency(name, type) {
        const dep = this.tryGetDependency(name, type);
        if (!dep) {
            const msg = type
                ? `there is no ${type} dependency defined on "${name}"`
                : `there is no dependency defined on "${name}"`;
            throw new Error(msg);
        }
        return dep;
    }
    /**
     * (experimental) Returns a dependency by name.
     *
     * Returns `undefined` if there is no dependency defined by that name or if
     * `type` is not provided and there is more then one dependency type for this
     * dependency.
     *
     * @param name The name of the dependency.
     * @param type The dependency type.
     * @returns a copy (cannot be modified) or undefined if there is no match
     * @experimental
     */
    tryGetDependency(name, type) {
        const idx = this.tryGetDependencyIndex(name, type);
        if (idx === -1) {
            return undefined;
        }
        return {
            ...normalizeDep(this._deps[idx]),
        };
    }
    /**
     * (experimental) Adds a dependency to this project.
     *
     * @param spec The dependency spec in the format `MODULE[@VERSION]` where `MODULE` is the package-manager-specific module name and `VERSION` is an optional semantic version requirement (e.g. `^3.4.0`).
     * @param type The type of the dependency.
     * @experimental
     */
    addDependency(spec, type, metadata = {}) {
        this.project.logger.debug(`${type}-dep ${spec}`);
        const dep = {
            ...Dependencies.parseDependency(spec),
            type,
            metadata,
        };
        const existingDepIndex = this.tryGetDependencyIndex(dep.name, type);
        if (existingDepIndex !== -1) {
            this.project.logger.debug(`updating existing ${dep.type}-dep ${dep.name} with more specific version/metadata`);
            this._deps[existingDepIndex] = dep;
        }
        else {
            this._deps.push(dep);
        }
        return dep;
    }
    /**
     * (experimental) Removes a dependency.
     *
     * @param name The name of the module to remove (without the version).
     * @param type The dependency type.
     * @experimental
     */
    removeDependency(name, type) {
        const removeIndex = this.tryGetDependencyIndex(name, type);
        if (removeIndex === -1) {
            return;
        }
        this._deps.splice(removeIndex, 1);
    }
    tryGetDependencyIndex(name, type) {
        const deps = this._deps.filter(d => d.name === name);
        if (deps.length === 0) {
            return -1; // not found
        }
        if (!type) {
            if (deps.length > 1) {
                throw new Error(`"${name}" is defined for multiple dependency types: ${deps.map(d => d.type).join(',')}. Please specify dependency type`);
            }
            type = deps[0].type;
        }
        return this._deps.findIndex(dep => dep.name === name && dep.type === type);
    }
    toJson() {
        if (this._deps.length === 0) {
            return undefined;
        }
        return {
            dependencies: this._deps.sort(compareDeps).map(normalizeDep),
        };
    }
}
exports.Dependencies = Dependencies;
_a = JSII_RTTI_SYMBOL_1;
Dependencies[_a] = { fqn: "projen.Dependencies", version: "0.46.3" };
/**
 * (experimental) The project-relative path of the deps manifest file.
 *
 * @experimental
 */
Dependencies.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, 'deps.json');
function normalizeDep(d) {
    const obj = {};
    for (const [k, v] of Object.entries(d)) {
        if (v == undefined) {
            continue;
        }
        if (typeof (v) === 'object' && Object.keys(v).length === 0) {
            continue;
        }
        if (Array.isArray(v) && v.length === 0) {
            continue;
        }
        obj[k] = v;
    }
    return obj;
}
function compareDeps(d1, d2) {
    return specOf(d1).localeCompare(specOf(d2));
    function specOf(dep) {
        let spec = dep.type + ':' + dep.name;
        if (dep.version) {
            spec += '@' + dep.version;
        }
        return spec;
    }
}
/**
 * (experimental) Type of dependency.
 *
 * @experimental
 */
var DependencyType;
(function (DependencyType) {
    DependencyType["RUNTIME"] = "runtime";
    DependencyType["PEER"] = "peer";
    DependencyType["BUNDLED"] = "bundled";
    DependencyType["BUILD"] = "build";
    DependencyType["TEST"] = "test";
    DependencyType["DEVENV"] = "devenv";
})(DependencyType = exports.DependencyType || (exports.DependencyType = {}));
//# sourceMappingURL=data:application/json;base64,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