"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const yargs = require("yargs");
const inventory = require("../../inventory");
const option_hints_1 = require("../../option-hints");
const projects_1 = require("../../projects");
const util_1 = require("../../util");
const macros_1 = require("../macros");
const util_2 = require("../util");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE-NAME] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE-NAME', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")' });
        args.option('git', { type: 'boolean', default: true, desc: 'Run `git init` and create an initial commit (use --no-git to disable)' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(false);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.simpleType !== 'string' && option.simpleType !== 'number' && option.simpleType !== 'boolean' && option.kind !== 'enum') {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        const argType = option.kind === 'enum' ? 'string' : option.simpleType;
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: argType,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => initProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return initProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(cwd, value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function initProjectFromModule(baseDir, spec, args) {
    var _a, _b;
    const projenVersion = (_a = args.projenVersion) !== null && _a !== void 0 ? _a : 'latest';
    const installCommand = util_2.renderInstallCommand(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        util_1.exec(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`npm ls --prefix=${baseDir} --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = util_2.installPackage(baseDir, spec);
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [
            baseDir,
        ],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter(x => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_b = type.options) !== null && _b !== void 0 ? _b : []) {
        if (option.simpleType !== 'string' && option.simpleType !== 'number' && option.simpleType !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            if (option.simpleType === 'number') {
                args[option.name] = parseInt(args[option.name]);
                args[option.switch] = args[option.name];
            }
            else if (option.simpleType === 'boolean') {
                const raw = args[option.name];
                const safe = typeof raw === 'string' ? util_1.isTruthy(raw) : raw;
                args[option.name] = safe;
                args[option.switch] = safe;
            }
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    args.devDeps = [spec];
    args['dev-deps'] = [spec];
    await initProject(baseDir, type, args);
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function initProject(baseDir, type, args) {
    var _a;
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    projects_1.Projects.createProject({
        dir: (_a = props.outdir) !== null && _a !== void 0 ? _a : baseDir,
        projectFqn: type.fqn,
        projectOptions: props,
        optionHints: args.comments ? option_hints_1.InitProjectOptionHints.FEATURED : option_hints_1.InitProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (fs.existsSync(path.join(baseDir, 'package.json')) && args.post) {
        util_1.exec('npm run eslint --if-present', { cwd: baseDir });
    }
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        git('init');
        git('add .');
        git('commit --allow-empty -m "chore: project created with projen"');
        git('branch -M main');
    }
}
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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