"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const BRANCH_REF = '${{ github.event.pull_request.head.ref }}';
const REPO_REF = '${{ github.event.pull_request.head.repo.full_name }}';
const PRIMARY_JOB_ID = 'build';
const SELF_MUTATION_STEP = 'self_mutation';
const SELF_MUTATION_COMMIT = 'self_mutation_commit';
/**
 * @experimental
 */
class BuildWorkflow extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g;
        super(project);
        const github = github_1.GitHub.of(project);
        if (!github) {
            throw new Error('BuildWorkflow is currently only supported for GitHub projects');
        }
        this.github = github;
        this.preBuildSteps = (_b = options.preBuildSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.mutableBuilds = (_d = options.mutableBuild) !== null && _d !== void 0 ? _d : true;
        this.gitIdentity = (_e = options.gitIdentity) !== null && _e !== void 0 ? _e : constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory = options.artifactsDirectory;
        const antitamper = (_f = options.antitamper) !== null && _f !== void 0 ? _f : true;
        // disable anti-tamper if build workflow is mutable
        this.antitamperSteps = ((_g = !this.mutableBuilds) !== null && _g !== void 0 ? _g : antitamper) ? [{
                // anti-tamper check (fails if there were changes to committed files)
                // this will identify any non-committed files generated during build (e.g. test snapshots)
                name: 'Anti-tamper check',
                run: 'git diff --ignore-space-at-eol --exit-code',
            }] : [];
        this.workflow = new github_1.GithubWorkflow(github, 'build');
        this.workflow.on({
            pullRequest: {},
            workflowDispatch: {},
        });
        this.workflow.addJob(PRIMARY_JOB_ID, {
            runsOn: ['ubuntu-latest'],
            container: options.containerImage ? { image: options.containerImage } : undefined,
            env: {
                CI: 'true',
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            outputs: {
                [SELF_MUTATION_COMMIT]: {
                    stepId: SELF_MUTATION_STEP,
                    outputName: SELF_MUTATION_COMMIT,
                },
            },
            steps: (() => this.renderSteps()),
        });
        this.primaryJobId = PRIMARY_JOB_ID;
        this._buildJobIds = [PRIMARY_JOB_ID];
        this.workflow.addJob('update-status', {
            runsOn: ['ubuntu-latest'],
            permissions: {
                checks: workflows_model_1.JobPermission.WRITE,
                actions: workflows_model_1.JobPermission.WRITE,
            },
            needs: (() => this._buildJobIds),
            if: `\${{ needs.${this.primaryJobId}.outputs.${SELF_MUTATION_COMMIT} }}`,
            steps: [
                // if we pushed changes, we need to manually update the status check so
                // that the PR will be green (we won't get here for forks with updates
                // because the push would have failed).
                {
                    name: 'Update status check (if changed)',
                    run: [
                        'gh api',
                        '-X POST',
                        `/repos/${REPO_REF}/check-runs`,
                        `-F name="${PRIMARY_JOB_ID}"`,
                        `-F head_sha="\${{ needs.${this.primaryJobId}.outputs.${SELF_MUTATION_COMMIT} }}"`,
                        '-F status="completed"',
                        '-F conclusion="success"',
                    ].join(' '),
                    env: {
                        GITHUB_TOKEN: '${{ secrets.GITHUB_TOKEN }}',
                    },
                },
                // if we pushed changes, we need to mark the current commit as failed, so
                // that GitHub auto-merge does not risk merging this commit before the
                // event for the new commit has registered.
                {
                    name: 'Cancel workflow (if changed)',
                    run: [
                        'gh api',
                        '-X POST',
                        `/repos/${REPO_REF}/actions/runs/\${{ github.run_id }}/cancel`,
                    ].join(' '),
                    env: {
                        GITHUB_TOKEN: '${{ secrets.GITHUB_TOKEN }}',
                    },
                },
            ],
        });
    }
    /**
     * (experimental) Returns a list of job IDs that are part of the build.
     *
     * @experimental
     */
    get buildJobIds() {
        return [...this._buildJobIds];
    }
    /**
     * (experimental) Adds steps that are executed after the build.
     *
     * @param steps The job steps.
     * @experimental
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * (experimental) Adds another job to the build workflow which is executed after the build job succeeded.
     *
     * @param id The id of the new job.
     * @param job The job specification.
     * @experimental
     */
    addPostBuildJob(id, job) {
        const steps = [];
        if (this.artifactsDirectory) {
            const artfiactName = 'build-artifact';
            // add a step at the end of the build workflow which will upload the
            // artifact so we can download it in each post-build job (do it once).
            if (!this.uploadArtitactSteps) {
                this.uploadArtitactSteps = [{
                        name: 'Upload artifact',
                        uses: 'actions/upload-artifact@v2.1.1',
                        // Setting to always will ensure that this step will run even if
                        // the previous ones have failed (e.g. coverage report, internal logs, etc)
                        if: 'always()',
                        with: {
                            name: artfiactName,
                            path: this.artifactsDirectory,
                        },
                    }];
            }
            steps.push({
                name: 'Download build artifacts',
                uses: 'actions/download-artifact@v2',
                with: {
                    name: artfiactName,
                    path: this.artifactsDirectory,
                },
            });
        }
        steps.push(...job.steps);
        this.workflow.addJob(id, {
            needs: [this.primaryJobId],
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._buildJobIds.push(id);
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderSteps() {
        const steps = new Array();
        steps.push({
            name: 'Checkout',
            uses: 'actions/checkout@v2',
            with: this.mutableBuilds ? {
                ref: BRANCH_REF,
                repository: REPO_REF,
            } : undefined,
        });
        steps.push(constants_1.setGitIdentityStep(this.gitIdentity));
        steps.push(...this.antitamperSteps);
        steps.push(...this.preBuildSteps);
        steps.push({
            name: this.buildTask.name,
            run: this.github.project.runTaskCommand(this.buildTask),
        });
        steps.push(...this.postBuildSteps);
        steps.push(...this.antitamperSteps);
        steps.push({
            name: 'Self mutation',
            id: SELF_MUTATION_STEP,
            run: [
                'if ! git diff --exit-code; then',
                '  git add .',
                '  git commit -m "chore: self mutation"',
                `  git push origin HEAD:${BRANCH_REF}`,
                `  echo "::set-output name=${SELF_MUTATION_COMMIT}::$(git rev-parse HEAD)"`,
                'fi',
            ].join('\n'),
        });
        if (this.uploadArtitactSteps) {
            steps.push(...this.uploadArtitactSteps);
        }
        return steps;
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.46.3" };
//# sourceMappingURL=data:application/json;base64,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