"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const file_1 = require("../file");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
/**
 * (experimental) Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @experimental
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * (experimental) Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use.
     * @param options Options.
     * @experimental
     */
    constructor(project, options) {
        var _c, _d, _e;
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error('No bundler found. Please add a Bundler component to your project.');
        }
        const runtime = (_c = options.runtime) !== null && _c !== void 0 ? _c : LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint === null || eslint === void 0 ? void 0 : eslint.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} extension`);
        }
        const basePath = path_1.join(path_1.dirname(entrypoint), path_1.basename(entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = (_d = options.constructFile) !== null && _d !== void 0 ? _d : `${basePath}-function.ts`;
        if (path_1.extname(constructFile) !== '.ts') {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = (_e = options.constructName) !== null && _e !== void 0 ? _e : case_1.pascal(path_1.basename(basePath)) + 'Function';
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ['aws-sdk'],
            ...options.bundlingOptions,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path_1.join(project.outdir, bundle.outfile);
        const constructAbs = path_1.join(project.outdir, constructFile);
        const relativeOutfile = path_1.relative(path_1.dirname(constructAbs), path_1.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        src.line(`// ${file_1.FileBase.PROJEN_MARKER}`);
        src.line('import * as path from \'path\';');
        if (cdkDeps.cdkMajorVersion === 1) {
            src.line('import * as lambda from \'@aws-cdk/aws-lambda\';');
            src.line('import { Construct } from \'@aws-cdk/core\';');
            cdkDeps.addV1Dependencies('@aws-cdk/aws-lambda');
            cdkDeps.addV1Dependencies('@aws-cdk/core');
        }
        else {
            src.line('import * as lambda from \'aws-cdk-lib/aws-lambda\';');
            src.line('import { Construct } from \'constructs\';');
        }
        src.line();
        src.line('/**');
        src.line(` * Props for ${constructName}`);
        src.line(' */');
        src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        src.close('}');
        src.line();
        src.line('/**');
        src.line(` * An AWS Lambda function which executes ${convertToPosixPath(basePath)}.`);
        src.line(' */');
        src.open(`export class ${constructName} extends lambda.Function {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open('super(scope, id, {');
        src.line(`description: '${convertToPosixPath(entrypoint)}',`);
        src.line('...props,');
        src.line(`runtime: lambda.Runtime.${runtime.functionRuntime},`);
        src.line('handler: \'index.handler\',');
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${convertToPosixPath(relativeOutfile)}')),`);
        src.close('});');
        src.close('}');
        src.close('}');
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.46.3" };
/**
 * (experimental) The runtime for the AWS Lambda function.
 *
 * @experimental
 */
class LambdaRuntime {
    constructor(
    /**
     * The aws-lambda.Runtime member name to use.
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        /**
         * @experimental
         */
        this.esbuildPlatform = 'node';
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.46.3" };
/**
 * (experimental) Node.js 10.x.
 *
 * @experimental
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime('NODEJS_10_X', 'node10');
/**
 * (experimental) Node.js 12.x.
 *
 * @experimental
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime('NODEJS_12_X', 'node12');
/**
 * (experimental) Node.js 14.x.
 *
 * @experimental
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime('NODEJS_14_X', 'node14');
/**
 * Converts the given path string to posix if it wasn't already.
 */
function convertToPosixPath(p) {
    return p.split(path_1.sep).join(path_1.posix.sep);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLWZ1bmN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2F3c2Nkay9sYW1iZGEtZnVuY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwrQkFBOEU7QUFDOUUsK0JBQThCO0FBQzlCLDRDQUF5QztBQUN6QyxrQ0FBbUM7QUFDbkMsOENBQWlFO0FBRWpFLGdEQUE0QztBQUU1Qyx5Q0FBbUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUEyQm5ELE1BQWEsY0FBZSxTQUFRLHFCQUFTOzs7Ozs7OztJQUUzQyxZQUFZLE9BQWdCLEVBQUUsT0FBOEI7O1FBQzFELEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVmLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUM7UUFDaEMsTUFBTSxPQUFPLEdBQUcsb0JBQU8sQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDcEMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsbUVBQW1FLENBQUMsQ0FBQztTQUN0RjtRQUVELE1BQU0sT0FBTyxTQUFHLE9BQU8sQ0FBQyxPQUFPLG1DQUFJLGFBQWEsQ0FBQyxXQUFXLENBQUM7UUFFN0QsMEVBQTBFO1FBQzFFLGtCQUFrQjtRQUNsQixNQUFNLE1BQU0sR0FBRyxtQkFBTSxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNsQyxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsWUFBWSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUU7UUFFekMsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztRQUV0QyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxnQ0FBcUIsQ0FBQyxFQUFFO1lBQy9DLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxVQUFVLGdCQUFnQixnQ0FBcUIsWUFBWSxDQUFDLENBQUM7U0FDakY7UUFFRCxNQUFNLFFBQVEsR0FBRyxXQUFJLENBQUMsY0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFLGVBQVEsQ0FBQyxVQUFVLEVBQUUsZ0NBQXFCLENBQUMsQ0FBQyxDQUFDO1FBQ3hGLE1BQU0sYUFBYSxTQUFHLE9BQU8sQ0FBQyxhQUFhLG1DQUFJLEdBQUcsUUFBUSxjQUFjLENBQUM7UUFFekUsSUFBSSxjQUFPLENBQUMsYUFBYSxDQUFDLEtBQUssS0FBSyxFQUFFO1lBQ3BDLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLGFBQWEsNkJBQTZCLENBQUMsQ0FBQztTQUNyRjtRQUVELGFBQWE7UUFDYixNQUFNLGFBQWEsU0FBRyxPQUFPLENBQUMsYUFBYSxtQ0FBSSxhQUFNLENBQUMsZUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLEdBQUcsVUFBVSxDQUFDO1FBQ3ZGLE1BQU0sU0FBUyxHQUFHLEdBQUcsYUFBYSxPQUFPLENBQUM7UUFFMUMsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQyxVQUFVLEVBQUU7WUFDM0MsTUFBTSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQzdCLFFBQVEsRUFBRSxPQUFPLENBQUMsZUFBZTtZQUNqQyxTQUFTLEVBQUUsQ0FBQyxTQUFTLENBQUM7WUFDdEIsR0FBRyxPQUFPLENBQUMsZUFBZTtTQUMzQixDQUFDLENBQUM7UUFFSCxtRUFBbUU7UUFDbkUseUVBQXlFO1FBQ3pFLHNEQUFzRDtRQUN0RCxPQUFPO1FBQ1AsOEVBQThFO1FBQzlFLHVFQUF1RTtRQUN2RSxNQUFNLFVBQVUsR0FBRyxXQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDeEQsTUFBTSxZQUFZLEdBQUcsV0FBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDekQsTUFBTSxlQUFlLEdBQUcsZUFBUSxDQUFDLGNBQU8sQ0FBQyxZQUFZLENBQUMsRUFBRSxjQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUU3RSxNQUFNLEdBQUcsR0FBRyxJQUFJLHdCQUFVLENBQUMsT0FBTyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ25ELEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxlQUFRLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQztRQUN6QyxHQUFHLENBQUMsSUFBSSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFFNUMsSUFBSSxPQUFPLENBQUMsZUFBZSxLQUFLLENBQUMsRUFBRTtZQUNqQyxHQUFHLENBQUMsSUFBSSxDQUFDLGtEQUFrRCxDQUFDLENBQUM7WUFDN0QsR0FBRyxDQUFDLElBQUksQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1lBQ3pELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1lBQ2pELE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUM1QzthQUFNO1lBQ0wsR0FBRyxDQUFDLElBQUksQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO1lBQ2hFLEdBQUcsQ0FBQyxJQUFJLENBQUMsMkNBQTJDLENBQUMsQ0FBQztTQUN2RDtRQUVELEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNYLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEIsR0FBRyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsYUFBYSxFQUFFLENBQUMsQ0FBQztRQUMxQyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hCLEdBQUcsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLFNBQVMsbUNBQW1DLENBQUMsQ0FBQztRQUMzRSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2YsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ1gsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQixHQUFHLENBQUMsSUFBSSxDQUFDLDRDQUE0QyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdEYsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQixHQUFHLENBQUMsSUFBSSxDQUFDLGdCQUFnQixhQUFhLDRCQUE0QixDQUFDLENBQUM7UUFDcEUsR0FBRyxDQUFDLElBQUksQ0FBQyxxREFBcUQsU0FBUyxLQUFLLENBQUMsQ0FBQztRQUM5RSxHQUFHLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsa0JBQWtCLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzlELEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDdEIsR0FBRyxDQUFDLElBQUksQ0FBQywyQkFBMkIsT0FBTyxDQUFDLGVBQWUsR0FBRyxDQUFDLENBQUM7UUFDaEUsR0FBRyxDQUFDLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLEdBQUcsQ0FBQyxJQUFJLENBQUMscURBQXFELGtCQUFrQixDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6RyxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2pCLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDZixHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWYsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsUUFBUSxnQkFBZ0IsYUFBYSxzQkFBc0IsYUFBYSxHQUFHLENBQUMsQ0FBQztRQUM1RyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxRQUFRLGtCQUFrQixNQUFNLENBQUMsVUFBVSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7UUFDcEYsSUFBSSxNQUFNLENBQUMsU0FBUyxFQUFFO1lBQ3BCLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLFFBQVEsd0JBQXdCLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQztTQUMxRjtJQUNILENBQUM7O0FBN0ZILHdDQThGQzs7Ozs7Ozs7QUFHRCxNQUFhLGFBQWE7SUFZeEI7SUFDRTs7T0FFRztJQUNhLGVBQXVCO0lBRXZDOztPQUVHO0lBQ2EsYUFBcUI7UUFMckIsb0JBQWUsR0FBZixlQUFlLENBQVE7UUFLdkIsa0JBQWEsR0FBYixhQUFhLENBQVE7Ozs7UUFYdkIsb0JBQWUsR0FBRyxNQUFNLENBQUM7SUFZekMsQ0FBQzs7QUF0Qkgsc0NBdUJDOzs7Ozs7OztBQXJCd0IseUJBQVcsR0FBRyxJQUFJLGFBQWEsQ0FBQyxhQUFhLEVBQUUsUUFBUSxDQUFDLENBQUM7Ozs7OztBQUd6RCx5QkFBVyxHQUFHLElBQUksYUFBYSxDQUFDLGFBQWEsRUFBRSxRQUFRLENBQUMsQ0FBQzs7Ozs7O0FBR3pELHlCQUFXLEdBQUcsSUFBSSxhQUFhLENBQUMsYUFBYSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0FBaUJsRjs7R0FFRztBQUNILFNBQVMsa0JBQWtCLENBQUMsQ0FBUztJQUNuQyxPQUFPLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBRyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUN0QyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgYmFzZW5hbWUsIGRpcm5hbWUsIGV4dG5hbWUsIGpvaW4sIHJlbGF0aXZlLCBzZXAsIHBvc2l4IH0gZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBwYXNjYWwgfSBmcm9tICdjYXNlJztcbmltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gJy4uL2NvbXBvbmVudCc7XG5pbXBvcnQgeyBGaWxlQmFzZSB9IGZyb20gJy4uL2ZpbGUnO1xuaW1wb3J0IHsgQnVuZGxlciwgQnVuZGxpbmdPcHRpb25zLCBFc2xpbnQgfSBmcm9tICcuLi9qYXZhc2NyaXB0JztcbmltcG9ydCB7IFByb2plY3QgfSBmcm9tICcuLi9wcm9qZWN0JztcbmltcG9ydCB7IFNvdXJjZUNvZGUgfSBmcm9tICcuLi9zb3VyY2UtY29kZSc7XG5pbXBvcnQgeyBBd3NDZGtEZXBzIH0gZnJvbSAnLi9hd3NjZGstZGVwcyc7XG5pbXBvcnQgeyBUWVBFU0NSSVBUX0xBTUJEQV9FWFQgfSBmcm9tICcuL2ludGVybmFsJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFGdW5jdGlvbkNvbW1vbk9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcnVudGltZT86IExhbWJkYVJ1bnRpbWU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidW5kbGluZ09wdGlvbnM/OiBCdW5kbGluZ09wdGlvbnM7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhRnVuY3Rpb25PcHRpb25zIGV4dGVuZHMgTGFtYmRhRnVuY3Rpb25Db21tb25PcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbnRyeXBvaW50OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb25zdHJ1Y3RGaWxlPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb25zdHJ1Y3ROYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjZGtEZXBzOiBBd3NDZGtEZXBzO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBMYW1iZGFGdW5jdGlvbiBleHRlbmRzIENvbXBvbmVudCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY29uc3RydWN0b3IocHJvamVjdDogUHJvamVjdCwgb3B0aW9uczogTGFtYmRhRnVuY3Rpb25PcHRpb25zKSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG5cbiAgICBjb25zdCBjZGtEZXBzID0gb3B0aW9ucy5jZGtEZXBzO1xuICAgIGNvbnN0IGJ1bmRsZXIgPSBCdW5kbGVyLm9mKHByb2plY3QpO1xuICAgIGlmICghYnVuZGxlcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdObyBidW5kbGVyIGZvdW5kLiBQbGVhc2UgYWRkIGEgQnVuZGxlciBjb21wb25lbnQgdG8geW91ciBwcm9qZWN0LicpO1xuICAgIH1cblxuICAgIGNvbnN0IHJ1bnRpbWUgPSBvcHRpb25zLnJ1bnRpbWUgPz8gTGFtYmRhUnVudGltZS5OT0RFSlNfMTRfWDtcblxuICAgIC8vIGFsbG93IExhbWJkYSBoYW5kbGVyIGNvZGUgdG8gaW1wb3J0IGRldi1kZXBzIHNpbmNlIHRoZXkgYXJlIG9ubHkgbmVlZGVkXG4gICAgLy8gZHVyaW5nIGJ1bmRsaW5nXG4gICAgY29uc3QgZXNsaW50ID0gRXNsaW50Lm9mKHByb2plY3QpO1xuICAgIGVzbGludD8uYWxsb3dEZXZEZXBzKG9wdGlvbnMuZW50cnlwb2ludCk7XG5cbiAgICBjb25zdCBlbnRyeXBvaW50ID0gb3B0aW9ucy5lbnRyeXBvaW50O1xuXG4gICAgaWYgKCFlbnRyeXBvaW50LmVuZHNXaXRoKFRZUEVTQ1JJUFRfTEFNQkRBX0VYVCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtlbnRyeXBvaW50fSBtdXN0IGhhdmUgYSAke1RZUEVTQ1JJUFRfTEFNQkRBX0VYVH0gZXh0ZW5zaW9uYCk7XG4gICAgfVxuXG4gICAgY29uc3QgYmFzZVBhdGggPSBqb2luKGRpcm5hbWUoZW50cnlwb2ludCksIGJhc2VuYW1lKGVudHJ5cG9pbnQsIFRZUEVTQ1JJUFRfTEFNQkRBX0VYVCkpO1xuICAgIGNvbnN0IGNvbnN0cnVjdEZpbGUgPSBvcHRpb25zLmNvbnN0cnVjdEZpbGUgPz8gYCR7YmFzZVBhdGh9LWZ1bmN0aW9uLnRzYDtcblxuICAgIGlmIChleHRuYW1lKGNvbnN0cnVjdEZpbGUpICE9PSAnLnRzJykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDb25zdHJ1Y3QgZmlsZSBuYW1lIFwiJHtjb25zdHJ1Y3RGaWxlfVwiIG11c3QgaGF2ZSBhIC50cyBleHRlbnNpb25gKTtcbiAgICB9XG5cbiAgICAvLyB0eXBlIG5hbWVzXG4gICAgY29uc3QgY29uc3RydWN0TmFtZSA9IG9wdGlvbnMuY29uc3RydWN0TmFtZSA/PyBwYXNjYWwoYmFzZW5hbWUoYmFzZVBhdGgpKSArICdGdW5jdGlvbic7XG4gICAgY29uc3QgcHJvcHNUeXBlID0gYCR7Y29uc3RydWN0TmFtZX1Qcm9wc2A7XG5cbiAgICBjb25zdCBidW5kbGUgPSBidW5kbGVyLmFkZEJ1bmRsZShlbnRyeXBvaW50LCB7XG4gICAgICB0YXJnZXQ6IHJ1bnRpbWUuZXNidWlsZFRhcmdldCxcbiAgICAgIHBsYXRmb3JtOiBydW50aW1lLmVzYnVpbGRQbGF0Zm9ybSxcbiAgICAgIGV4dGVybmFsczogWydhd3Mtc2RrJ10sXG4gICAgICAuLi5vcHRpb25zLmJ1bmRsaW5nT3B0aW9ucyxcbiAgICB9KTtcblxuICAgIC8vIGNhbGN1bGF0ZSB0aGUgcmVsYXRpdmUgcGF0aCBiZXR3ZWVuIHRoZSBkaXJlY3RvcnkgY29udGFpbmluZyB0aGVcbiAgICAvLyBnZW5lcmF0ZWQgY29uc3RydWN0IHNvdXJjZSBmaWxlIHRvIHRoZSBkaXJlY3RvcnkgY29udGFpbmluZyB0aGUgYnVuZGxlXG4gICAgLy8gaW5kZXguanMgYnkgcmVzb2x2aW5nIHRoZW0gYXMgYWJzb2x1dGUgcGF0aHMgZmlyc3QuXG4gICAgLy8gZS5nOlxuICAgIC8vICAtIG91dGZpbGVBYnMgPT4gYC9wcm9qZWN0LW91dGRpci9hc3NldHMvZm9vL2Jhci9iYXovZm9vLWZ1bmN0aW9uL2luZGV4LmpzYFxuICAgIC8vICAtIGNvbnN0cnVjdEFicyA9PiBgL3Byb2plY3Qtb3V0ZGlyL3NyYy9mb28vYmFyL2Jhei9mb28tZnVuY3Rpb24udHNgXG4gICAgY29uc3Qgb3V0ZmlsZUFicyA9IGpvaW4ocHJvamVjdC5vdXRkaXIsIGJ1bmRsZS5vdXRmaWxlKTtcbiAgICBjb25zdCBjb25zdHJ1Y3RBYnMgPSBqb2luKHByb2plY3Qub3V0ZGlyLCBjb25zdHJ1Y3RGaWxlKTtcbiAgICBjb25zdCByZWxhdGl2ZU91dGZpbGUgPSByZWxhdGl2ZShkaXJuYW1lKGNvbnN0cnVjdEFicyksIGRpcm5hbWUob3V0ZmlsZUFicykpO1xuXG4gICAgY29uc3Qgc3JjID0gbmV3IFNvdXJjZUNvZGUocHJvamVjdCwgY29uc3RydWN0RmlsZSk7XG4gICAgc3JjLmxpbmUoYC8vICR7RmlsZUJhc2UuUFJPSkVOX01BUktFUn1gKTtcbiAgICBzcmMubGluZSgnaW1wb3J0ICogYXMgcGF0aCBmcm9tIFxcJ3BhdGhcXCc7Jyk7XG5cbiAgICBpZiAoY2RrRGVwcy5jZGtNYWpvclZlcnNpb24gPT09IDEpIHtcbiAgICAgIHNyYy5saW5lKCdpbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSBcXCdAYXdzLWNkay9hd3MtbGFtYmRhXFwnOycpO1xuICAgICAgc3JjLmxpbmUoJ2ltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXFwnQGF3cy1jZGsvY29yZVxcJzsnKTtcbiAgICAgIGNka0RlcHMuYWRkVjFEZXBlbmRlbmNpZXMoJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnKTtcbiAgICAgIGNka0RlcHMuYWRkVjFEZXBlbmRlbmNpZXMoJ0Bhd3MtY2RrL2NvcmUnKTtcbiAgICB9IGVsc2Uge1xuICAgICAgc3JjLmxpbmUoJ2ltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFxcJ2F3cy1jZGstbGliL2F3cy1sYW1iZGFcXCc7Jyk7XG4gICAgICBzcmMubGluZSgnaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcXCdjb25zdHJ1Y3RzXFwnOycpO1xuICAgIH1cblxuICAgIHNyYy5saW5lKCk7XG4gICAgc3JjLmxpbmUoJy8qKicpO1xuICAgIHNyYy5saW5lKGAgKiBQcm9wcyBmb3IgJHtjb25zdHJ1Y3ROYW1lfWApO1xuICAgIHNyYy5saW5lKCcgKi8nKTtcbiAgICBzcmMub3BlbihgZXhwb3J0IGludGVyZmFjZSAke3Byb3BzVHlwZX0gZXh0ZW5kcyBsYW1iZGEuRnVuY3Rpb25PcHRpb25zIHtgKTtcbiAgICBzcmMuY2xvc2UoJ30nKTtcbiAgICBzcmMubGluZSgpO1xuICAgIHNyYy5saW5lKCcvKionKTtcbiAgICBzcmMubGluZShgICogQW4gQVdTIExhbWJkYSBmdW5jdGlvbiB3aGljaCBleGVjdXRlcyAke2NvbnZlcnRUb1Bvc2l4UGF0aChiYXNlUGF0aCl9LmApO1xuICAgIHNyYy5saW5lKCcgKi8nKTtcbiAgICBzcmMub3BlbihgZXhwb3J0IGNsYXNzICR7Y29uc3RydWN0TmFtZX0gZXh0ZW5kcyBsYW1iZGEuRnVuY3Rpb24ge2ApO1xuICAgIHNyYy5vcGVuKGBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wcz86ICR7cHJvcHNUeXBlfSkge2ApO1xuICAgIHNyYy5vcGVuKCdzdXBlcihzY29wZSwgaWQsIHsnKTtcbiAgICBzcmMubGluZShgZGVzY3JpcHRpb246ICcke2NvbnZlcnRUb1Bvc2l4UGF0aChlbnRyeXBvaW50KX0nLGApO1xuICAgIHNyYy5saW5lKCcuLi5wcm9wcywnKTtcbiAgICBzcmMubGluZShgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuJHtydW50aW1lLmZ1bmN0aW9uUnVudGltZX0sYCk7XG4gICAgc3JjLmxpbmUoJ2hhbmRsZXI6IFxcJ2luZGV4LmhhbmRsZXJcXCcsJyk7XG4gICAgc3JjLmxpbmUoYGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnJHtjb252ZXJ0VG9Qb3NpeFBhdGgocmVsYXRpdmVPdXRmaWxlKX0nKSksYCk7XG4gICAgc3JjLmNsb3NlKCd9KTsnKTtcbiAgICBzcmMuY2xvc2UoJ30nKTtcbiAgICBzcmMuY2xvc2UoJ30nKTtcblxuICAgIHRoaXMucHJvamVjdC5sb2dnZXIudmVyYm9zZShgJHtiYXNlUGF0aH06IGNvbnN0cnVjdCBcIiR7Y29uc3RydWN0TmFtZX1cIiBnZW5lcmF0ZWQgdW5kZXIgXCIke2NvbnN0cnVjdEZpbGV9XCJgKTtcbiAgICB0aGlzLnByb2plY3QubG9nZ2VyLnZlcmJvc2UoYCR7YmFzZVBhdGh9OiBidW5kbGUgdGFzayBcIiR7YnVuZGxlLmJ1bmRsZVRhc2submFtZX1cImApO1xuICAgIGlmIChidW5kbGUud2F0Y2hUYXNrKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLnZlcmJvc2UoYCR7YmFzZVBhdGh9OiBidW5kbGUgd2F0Y2ggdGFzayBcIiR7YnVuZGxlLndhdGNoVGFzay5uYW1lfVwiYCk7XG4gICAgfVxuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIExhbWJkYVJ1bnRpbWUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBOT0RFSlNfMTBfWCA9IG5ldyBMYW1iZGFSdW50aW1lKCdOT0RFSlNfMTBfWCcsICdub2RlMTAnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBOT0RFSlNfMTJfWCA9IG5ldyBMYW1iZGFSdW50aW1lKCdOT0RFSlNfMTJfWCcsICdub2RlMTInKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBOT0RFSlNfMTRfWCA9IG5ldyBMYW1iZGFSdW50aW1lKCdOT0RFSlNfMTRfWCcsICdub2RlMTQnKTtcblxuICBwdWJsaWMgcmVhZG9ubHkgZXNidWlsZFBsYXRmb3JtID0gJ25vZGUnO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoXG4gICAgLyoqXG4gICAgICogVGhlIGF3cy1sYW1iZGEuUnVudGltZSBtZW1iZXIgbmFtZSB0byB1c2UuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uUnVudGltZTogc3RyaW5nLFxuXG4gICAgLyoqXG4gICAgICogVGhlIGVzYnVpbGQgc2V0dGluZyB0byB1c2UuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGVzYnVpbGRUYXJnZXQ6IHN0cmluZykge1xuICB9XG59XG5cbi8qKlxuICogQ29udmVydHMgdGhlIGdpdmVuIHBhdGggc3RyaW5nIHRvIHBvc2l4IGlmIGl0IHdhc24ndCBhbHJlYWR5LlxuICovXG5mdW5jdGlvbiBjb252ZXJ0VG9Qb3NpeFBhdGgocDogc3RyaW5nKSB7XG4gIHJldHVybiBwLnNwbGl0KHNlcCkuam9pbihwb3NpeC5zZXApO1xufSJdfQ==