"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkDeps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver = require("semver");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * (experimental) Manages dependencies on the AWS CDK.
 *
 * @experimental
 */
class AwsCdkDeps extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d;
        super(project);
        this.cdkDependenciesAsDeps = (_b = options.cdkDependenciesAsDeps) !== null && _b !== void 0 ? _b : true;
        this.dependencyType = options.dependencyType;
        const framework = determineFrameworkVersion(options);
        this.cdkVersion = framework.range;
        this.cdkMajorVersion = framework.major;
        this.cdkMinimumVersion = framework.minimum;
        this.addFrameworkDependency(options);
        // assert/assertions library
        this.addV1AssertionLibraryDependency(options);
        // constructs library
        this.addConstructsDependency(options.constructsVersion);
        // user-defined v1 dependencies (will only fail in CDK v2 if these have values)
        this.addV1Dependencies(...(_c = options.cdkDependencies) !== null && _c !== void 0 ? _c : []);
        this.addV1DevDependencies(...(_d = options.cdkTestDependencies) !== null && _d !== void 0 ? _d : []);
    }
    /**
     * (experimental) Adds dependencies to AWS CDK modules.
     *
     * The type of dependency is determined by the `dependencyType` option.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     * @experimental
     */
    addV1Dependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error('addV1Dependencies() is not supported for CDK 2.x and above, use addDeps() or addPeerDeps() instead');
        }
        // this will add dependencies based on the type requested by the user
        // for libraries, this will be "peer" and for apps it will be "runtime"
        this.addV1DependenciesByType(this.dependencyType, ...deps);
        // add deps as runtime deps if `cdkDepsAsDeps` is true
        if (this.cdkDependenciesAsDeps) {
            this.addV1DependenciesByType(dependencies_1.DependencyType.RUNTIME, ...deps);
        }
    }
    /**
     * (experimental) Adds AWS CDK modules as dev dependencies.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     * @experimental
     */
    addV1DevDependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error('addV1DevDependencies() is not supported for CDK 2.x and above, use addDevDeps() instead');
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.BUILD, ...deps);
    }
    addConstructsDependency(requestedVersion) {
        var _b;
        if (requestedVersion && !semver.parse(requestedVersion)) {
            throw new Error(`"constructsVersion" cannot be parsed as a semver version: ${requestedVersion}`);
        }
        const defaultVersion = this.cdkMajorVersion === 1 ? '3.2.27' : '10.0.5';
        const versionRequirement = `^${requestedVersion !== null && requestedVersion !== void 0 ? requestedVersion : defaultVersion}`;
        const constructsMajorVersion = (_b = semver.minVersion(versionRequirement)) === null || _b === void 0 ? void 0 : _b.major;
        if (!constructsMajorVersion) {
            throw new Error(`Cannot determine major version of constructs version '${versionRequirement}'`);
        }
        switch (this.cdkMajorVersion) {
            case 1:
                if (constructsMajorVersion !== 3) {
                    throw new Error('AWS CDK 1.x requires constructs 3.x');
                }
                break;
            case 2:
                if (constructsMajorVersion !== 10) {
                    throw new Error('AWS CDK 2.x requires constructs 10.x');
                }
                break;
        }
        this.project.deps.addDependency(`constructs@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    /**
     * Adds a dependency on the AWS CDK framework (e.g. `@aws-cdk/core` for V1 or `aws-cdk-lib` for V1).
     */
    addFrameworkDependency(options) {
        switch (this.cdkMajorVersion) {
            case 1:
                this.addV1Dependencies('@aws-cdk/core');
                break;
            case 2:
                if (options.cdkDependencies !== undefined) {
                    throw new Error('cdkDependencies is not used for CDK 2.x. Use "peerDeps" instead');
                }
                if (options.cdkDependenciesAsDeps !== undefined) {
                    throw new Error('cdkDependenciesAsDeps is not used for CDK 2.x');
                }
                if (options.cdkTestDependencies !== undefined) {
                    throw new Error('cdkTestDependencies is not used for CDK 2.x. Use "devDeps" instead');
                }
                this.project.deps.addDependency(`aws-cdk-lib@${this.cdkVersion}`, this.dependencyType);
                break;
            default:
                throw new Error(`Unsupported AWS CDK major version ${this.cdkMajorVersion}.x`);
        }
    }
    addV1AssertionLibraryDependency(options) {
        var _b, _c;
        if (this.cdkMajorVersion !== 1) {
            if (options.cdkAssert !== undefined) {
                throw new Error('cdkAssert is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib');
            }
            if (options.cdkAssertions !== undefined) {
                throw new Error('cdkAssertion is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib');
            }
            return;
        }
        const testDeps = new Array();
        if ((_b = options.cdkAssert) !== null && _b !== void 0 ? _b : true) {
            testDeps.push('@aws-cdk/assert');
        }
        // @aws-cdk/assertions is only available starting v1.111.0
        if (semver.gte(this.cdkMinimumVersion, '1.111.0') && ((_c = options.cdkAssertions) !== null && _c !== void 0 ? _c : true)) {
            testDeps.push('@aws-cdk/assertions');
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.TEST, ...testDeps);
    }
    /**
     * Adds a set of dependencies with the user-specified dependency type.
     * @param deps The set of dependency specifications
     */
    addV1DependenciesByType(type, ...modules) {
        for (const module of modules) {
            this.project.deps.addDependency(`${module}@${this.cdkVersion}`, type);
        }
    }
}
exports.AwsCdkDeps = AwsCdkDeps;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkDeps[_a] = { fqn: "projen.awscdk.AwsCdkDeps", version: "0.46.3" };
function determineFrameworkVersion(options) {
    const ver = semver.parse(options.cdkVersion);
    if (!ver) {
        throw new Error(`"cdkVersion" cannot be parsed as a semver version: ${options.cdkVersion}`);
    }
    return {
        minimum: ver.format(),
        range: options.cdkVersionPinning ? options.cdkVersion : `^${options.cdkVersion}`,
        major: ver.major,
    };
}
//# sourceMappingURL=data:application/json;base64,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