# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['httpmq', 'httpmq.models']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8', 'asyncio>=3.4', 'python-dateutil>=2.8']

setup_kwargs = {
    'name': 'httpmq',
    'version': '0.1.0',
    'description': 'HTTP MQ Python client module',
    'long_description': '# HTTP MQ - Python Client\n\nPython client for [httpmq](https://github.com/alwitt/httpmq)\n\n[![MIT][License-Image]][License-Url] ![CICD workflow](https://github.com/alwitt/httpmq-python/actions/workflows/cicd.yaml/badge.svg) [![PyPI version](https://badge.fury.io/py/httpmq.svg)](https://badge.fury.io/py/httpmq)\n\n[License-Url]: https://mit-license.org/\n[License-Image]: https://img.shields.io/badge/License-MIT-blue.svg\n\n# Table of Content\n\n- [1. Introduction](#1-introduction)\n- [2. Installation](#2-installation)\n- [3. Examples](#3-examples)\n- [4. Local Development](#4-local-development)\n\n---\n\n# [1. Introduction](#table-of-content)\n\nThis is a [async](https://docs.python.org/3/library/asyncio.html) Python client SDK for [httpmq](https://github.com/alwitt/httpmq) built around [aiohttp](https://github.com/aio-libs/aiohttp), and uses the auto-generated data models of [httpmq](https://github.com/alwitt/httpmq)\'s OpenAPI specification. The auto-generated data models are placed under `httpmq/models`. The module provides client objects for interacting with the management and data plane API of [httpmq](https://github.com/alwitt/httpmq).\n\n```python\nimport asyncio\nimport logging\nimport httpmq\n\n\nasync def async_main(log: logging.Logger):\n    try:\n        # Create management plane client\n        mgmt_client = httpmq.ManagementClient(\n            api_client=httpmq.APIClient(base_url="http://127.0.0.1:4100")\n        )\n        # Create data plane client\n        data_client = httpmq.DataClient(\n            api_client=httpmq.APIClient(base_url="http://127.0.0.1:4101")\n        )\n\n        # Check whether management plane is ready\n        await mgmt_client.ready(context=httpmq.RequestContext())\n        log.info("Management API ready")\n\n        # Check whether data plane is ready\n        await data_client.ready(context=httpmq.RequestContext())\n        log.info("Data plane API ready")\n    finally:\n        # Always disconnect\n        await mgmt_client.disconnect()\n        await data_client.disconnect()\n```\n\n# [2. Installation](#table-of-content)\n\n```shell\n$ pip3 install httpmq\n```\n\n# [3. Examples](#table-of-content)\n\n- [Hello World](examples/hello_world.md): basic example showing how to define the client.\n\n- [Manage Streams](examples/manage_streams.md): managing streams through the `management` API.\n\n- [Manage Consumers](examples/manage_consumer.md): managing consumers through the `management` API.\n\n- [Sending and Receiving Messages](examples/sending_messages.md): sending and receiving messages.\n\n# [4. Local Development](#table-of-content)\n\n> **NOTE:** Though the described procedures supports local development, the same `docker-compose.yaml` can be used to create a [httpmq](https://github.com/alwitt/httpmq) test environment independent of this project.\n\nThis project uses [Poetry](https://python-poetry.org/) as the dependency management framework.\n\n```shell\n$ poetry check\nAll set!\n```\n\nA helper Makefile is also included to automate the common development tasks. The available make targets are:\n\n```shell\n$ make help\nlint                           Run python lint\nbuild                          Build module\ntest                           Run unit-tests\none-test                       Run specific unit-tests\ninstall                        Install module\nuninstall                      Uninstall module\nreinstall                      Reinstall module\ncli                            Run venv python CLI\ncompose                        Bring up development environment via docker-compose\nclean                          Clean up the python build artifacts\nhelp                           Display this help screen\n```\n\nFirst, prepare the development environment:\n\n```\nmake\n```\n\nThis will call `Poetry` to setup the virtual environment, install dependencies, and build the module.\n\nNow, start the development environment:\n\n```shell\n$ make compose\ndocker-compose -f docker-compose.yaml down --volume\nRemoving network httpmq-python_httpmq-py-test\nWARNING: Network httpmq-python_httpmq-py-test not found.\ndocker-compose -f docker-compose.yaml up -d\nCreating network "httpmq-python_httpmq-py-test" with driver "bridge"\nCreating httpmq-python_httpmq-data_1 ... done\nCreating httpmq-python_httpmq-mgmt_1 ... done\nCreating httpmq-python_dev-nats_1    ... done\n```\n\nVerify the code passes unit-tests:\n\n```shell\n$ make test\npoetry run pytest --verbose --junitxml=test-reports/test.xml test/\n=============================================== test session starts ===============================================\nplatform linux -- Python 3.10.6, pytest-7.1.2, pluggy-1.0.0 -- /home/harry/Git/HTTP_Message_Broker/httpmq-python/.venv/bin/python\ncachedir: .pytest_cache\nrootdir: /home/harry/Git/HTTP_Message_Broker/httpmq-python\ncollected 11 items\n\ntest/test_client.py::TestAPIClient::test_delete PASSED                                                      [  9%]\ntest/test_client.py::TestAPIClient::test_get PASSED                                                         [ 18%]\ntest/test_client.py::TestAPIClient::test_post PASSED                                                        [ 27%]\ntest/test_client.py::TestAPIClient::test_put PASSED                                                         [ 36%]\ntest/test_client.py::TestAPIClient::test_sse_get PASSED                                                     [ 45%]\ntest/test_dataplane.py::TestDataplane::test_basic_sanity PASSED                                             [ 54%]\ntest/test_dataplane.py::TestDataplane::test_message_splitter PASSED                                         [ 63%]\ntest/test_dataplane.py::TestDataplane::test_push_subscribe PASSED                                           [ 72%]\ntest/test_management.py::TestManagement::test_basic_sanity PASSED                                           [ 81%]\ntest/test_management.py::TestManagement::test_consumer_management PASSED                                    [ 90%]\ntest/test_management.py::TestManagement::test_stream_management PASSED                                      [100%]\n\n----------- generated xml file: /home/harry/Git/HTTP_Message_Broker/httpmq-python/test-reports/test.xml -----------\n=============================================== 11 passed in 2.62s ================================================\n```\n\nA demo application, `scripts/httpmq_test_cli.py`, is also provided. It exercises all the functionalities of the [httpmq](https://github.com/alwitt/httpmq) APIs. These functionalities are separated into subcommands, and the associated usage message explains how to call each functionality.\n\n> **IMPORTANT:** Before starting, perform\n>\n> ```shell\n> $ make install\n> $ poetry shell\n> ```\n>\n> This will install the locally built SDK package into the local virtual environment, and start a new shell within that virtual environment.\n\n```shell\n$ ./scripts/httpmq_test_cli.py --help\nUsage: httpmq_test_cli.py [OPTIONS] COMMAND [ARGS]...\n\n  Demo application for trying out functionalities of httpmq\n\nOptions:\n  --custom-ca-file, --ca TEXT  Custom CA file to use with HTTP client  [env\n                               var: HTTP_CUSTOM_CA_FILE]\n  --access-token, --at TEXT    Bearer access token used for authentication\n                               [env var: HTTP_BEARER_ACCESS_TOKEN]\n  --request-id, --rid TEXT     Request ID to use with this call  [default:\n                               6403e98e-d29a-478b-99e4-49ff07fb210f]\n  -v, --verbose                Verbose logging\n  --help                       Show this message and exit.\n\nCommands:\n  data    Operate the httpmq dataplane API\n  manage  Operate the httpmq management API\n```\n\n```shell\n$ ./scripts/httpmq_test_cli.py manage --help\nUsage: httpmq_test_cli.py manage [OPTIONS] COMMAND [ARGS]...\n\n  Operate the httpmq management API\n\nOptions:\n  -s, --management-server-url TEXT\n                                  Management server URL  [env var:\n                                  MANAGEMENT_SERVER_URL; default:\n                                  http://127.0.0.1:4100]\n  --help                          Show this message and exit.\n\nCommands:\n  consumer  Manages consumers through httpmq management API\n  ready     Verify the management API is ready\n  stream    Manages streams through httpmq management API\n```\n\n```shell\n$ ./scripts/httpmq_test_cli.py manage ready --help\nUsage: httpmq_test_cli.py manage ready [OPTIONS]\n\n  Verify the management API is ready\n\nOptions:\n  --help  Show this message and exit.\n```\n\n```shell\n$ ./scripts/httpmq_test_cli.py manage stream --help\nUsage: httpmq_test_cli.py manage stream [OPTIONS] COMMAND [ARGS]...\n\n  Manages streams through httpmq management API\n\nOptions:\n  --help  Show this message and exit.\n\nCommands:\n  change-retention  Changed a stream\'s message retention policy.\n  change-subject    Changed the target subjects of a stream through management API\n  create            Define a new stream through httpmq management API\n  delete            Delete one stream through management API\n  get               Read information regarding one stream through management API\n  list-all          List all streams through httpmq management API\n```\n\n```shell\n$ ./scripts/httpmq_test_cli.py manage consumer --help\nUsage: httpmq_test_cli.py manage consumer [OPTIONS] COMMAND [ARGS]...\n\n  Manages consumers through httpmq management API\n\nOptions:\n  -s, --target-stream TEXT  Target stream to operate against  [env var:\n                            TARGET_STREAM; required]\n  --help                    Show this message and exit.\n\nCommands:\n  create    Define a new consumer through httpmq management API\n  delete    Delete a consumer through httpmq management API\n  get       Read information regarding one consumer through management API\n  list-all  List all consumers of a stream through httpmq management API\n```\n\n```shell\n$ ./scripts/httpmq_test_cli.py data --help\nUsage: httpmq_test_cli.py data [OPTIONS] COMMAND [ARGS]...\n\n  Operate the httpmq dataplane API\n\nOptions:\n  -s, --dataplane-server-url TEXT\n                                  Dataplane server URL  [env var:\n                                  DATAPLANE_SERVER_URL; default:\n                                  http://127.0.0.1:4101]\n  --help                          Show this message and exit.\n\nCommands:\n  pub    Publish messages on a subject through httpmq dataplane API\n  ready  Verify the dataplane API is ready\n  sub    Subscribe for messages as a consumer on a stream through httpmq dataplane API\n```\n\n```shell\n$ ./scripts/httpmq_test_cli.py data ready --help\nUsage: httpmq_test_cli.py data ready [OPTIONS]\n\n  Verify the dataplane API is ready\n\nOptions:\n  --help  Show this message and exit.\n```\n',
    'author': 'Rhine Cliff',
    'author_email': 'rhinecliff@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/alwitt/httpmq-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.2,<4.0',
}


setup(**setup_kwargs)
