# AUTOGENERATED! DO NOT EDIT! File to edit: nbdev_nbs/spectrum_library/library_base.ipynb (unless otherwise specified).

__all__ = ['SpecLibBase']

# Cell

import pandas as pd
import numpy as np
import typing

import alphabase.peptide.fragment as fragment
import alphabase.peptide.precursor as precursor
from alphabase.io.hdf import HDF_File

# Cell
class SpecLibBase(object):
    def __init__(self,
        # ['b_z1','b_z2','y_z1','y_modloss_z1', ...];
        # 'b_z1': 'b' is the fragment type and
        # 'z1' is the charge state z=1.
        charged_frag_types:typing.List[str] = [
            'b_z1','b_z2','y_z1', 'y_z2'
        ],
        min_precursor_mz = 400, max_precursor_mz = 6000,
        decoy:str = 'pseudo_reverse',
    ):
        self.charged_frag_types = charged_frag_types
        self._precursor_df = pd.DataFrame()
        self._fragment_intensity_df = pd.DataFrame()
        self._fragment_mz_df = pd.DataFrame()
        self.min_precursor_mz = min_precursor_mz
        self.max_precursor_mz = max_precursor_mz

        self.mod_seq_df_columns = [
            'sequence', 'mods', 'mod_sites',
            'nce', 'instrument',
            'protein_idxes',
            'is_prot_nterm', 'is_prot_cterm'
        ]
        self.decoy = decoy

    @property
    def precursor_df(self):
        return self._precursor_df

    @precursor_df.setter
    def precursor_df(self, df):
        self._precursor_df = df
        precursor.refine_precursor_df(self._precursor_df)

    @property
    def fragment_mz_df(self):
        return self._fragment_mz_df

    @property
    def fragment_intensity_df(self):
        return self._fragment_intensity_df

    def refine_df(self):
        precursor.refine_precursor_df(self._precursor_df)

    def append_decoy_sequence(self):
        from alphabase.spectrum_library.decoy_library import (
            decoy_lib_provider
        )
        if self.decoy not in (
            decoy_lib_provider.decoy_dict
        ): return
        decoy_lib = (
            decoy_lib_provider.get_decoy_lib(
                self.decoy, self
            )
        )
        decoy_lib.decoy_sequence()
        self._precursor_df['decoy'] = 0
        decoy_lib._precursor_df['decoy'] = 1
        self._precursor_df = self._precursor_df.append(
            decoy_lib._precursor_df
        )
        self.refine_df()

    def clip_by_precursor_mz_(self):
        '''
        Clip self._precursor_df inplace
        '''
        self._precursor_df.drop(
            self._precursor_df.loc[
                (self._precursor_df['precursor_mz']<self.min_precursor_mz)|
                (self._precursor_df['precursor_mz']>self.max_precursor_mz)
            ].index, inplace=True
        )
        self._precursor_df.reset_index(drop=True, inplace=True)


    def flatten_fragment_data(
        self
    )->typing.Tuple[np.array, np.array]:
        '''
        Create flattened (1-D) np.array for fragment mz and intensity
        dataframes, respectively. The arrays are references to
        original data, that means:
          1. This method is fast;
          2. Changing the array values will change the df values.
        They can be unraveled back using:
          `array.reshape(len(self._fragment_mz_df.columns), -1)`

        Returns:
            np.array: 1-D flattened mz array (a reference to
            original fragment mz df data)
            np.array: 1-D flattened intensity array (a reference to
            original fragment intensity df data)
        '''
        return (
            self._fragment_mz_df.values.reshape(-1),
            self._fragment_intensity_df.values.reshape(-1)
        )

    def calc_precursor_mz(self):
        fragment.update_precursor_mz(self._precursor_df)
        self.clip_by_precursor_mz_()

    def update_precursor_mz(self):
        self.calc_precursor_mz()

    def hash_precursor_df(self):
        precursor.hash_precursor_df(
            self._precursor_df
        )

    def _get_hdf_to_save(self,
        hdf_file,
        delete_existing=False
    ):
        _hdf = HDF_File(
            hdf_file,
            read_only=False,
            truncate=True,
            delete_existing=delete_existing
        )
        return _hdf.library

    def _get_hdf_to_load(self,
        hdf_file,
    ):
        _hdf = HDF_File(
            hdf_file,
        )
        return _hdf.library

    def save_df_to_hdf(self,
        hdf_file:str,
        df_key: str,
        df: pd.DataFrame,
        delete_existing=False
    ):
        self._get_hdf_to_save(
            hdf_file,
            delete_existing=delete_existing
        ).add_group(df_key, df)

    def load_df_from_hdf(self,
        hdf_file:str,
        df_key: str
    ):
        return self._get_hdf_to_load(
            hdf_file
        ).__getattribute__(df_key).values

    def save_hdf(self, hdf_file):
        _hdf = HDF_File(
            hdf_file,
            read_only=False,
            truncate=True,
            delete_existing=True
        )
        if 'mod_seq_charge_hash' not in self._precursor_df.columns:
            self.hash_precursor_df()
        _hdf.library = {
            'precursor_df': self._precursor_df[
                [
                    col for col in self._precursor_df.columns
                    if col not in self.mod_seq_df_columns
                ]
            ],
            'mod_seq_df': self._precursor_df[
                [
                    col for col in self._precursor_df.columns
                    if col in (
                        self.mod_seq_df_columns+[
                        'mod_seq_hash', 'mod_seq_charge_hash']
                    )
                ]
            ],
            'fragment_mz_df': self._fragment_mz_df,
            'fragment_intensity_df': self._fragment_intensity_df,
        }

    def load_hdf(self, hdf_file, load_mod_seq=False):
        _hdf = HDF_File(
            hdf_file,
        )
        self._precursor_df:pd.DataFrame = _hdf.library.precursor_df.values
        if load_mod_seq:
            mod_seq_df = _hdf.library.mod_seq_df.values
            cols = [
                col for col in mod_seq_df.columns
                if col in self.mod_seq_df_columns
            ]
            self._precursor_df[cols] = mod_seq_df[cols]
        self._fragment_mz_df = _hdf.library.fragment_mz_df.values
        self._fragment_intensity_df = _hdf.library.fragment_intensity_df.values