import pytest
import os
import shutil
import tempfile
import textwrap
import pathlib
import yaml


@pytest.fixture
def config_factory():
    filename = tempfile.NamedTemporaryFile(suffix='.yml', delete=False).name

    def get_config(text_or_dict):
        with open(filename, 'w') as conf:
            if isinstance(text_or_dict, str):
                conf.write(textwrap.dedent(text_or_dict))
            elif isinstance(text_or_dict, dict):
                yaml.dump(text_or_dict, conf)
            else:
                raise ValueError(
                    'A text or dictionary is expected for config_factory.')
        return filename

    yield get_config
    os.remove(filename)


@pytest.fixture
def script_factory():
    filename = tempfile.NamedTemporaryFile(suffix='.sos', delete=False).name

    def get_script(text):
        with open(filename, 'w') as conf:
            conf.write(textwrap.dedent(text))
        return filename

    yield get_script
    os.remove(filename)


@pytest.fixture
def temp_factory():

    temp_fds = []
    temp_dirs = []

    def get_tempfiles(*args, **kwargs):
        content = kwargs.get('content', None)
        for names in args:
            if isinstance(names, (str, os.PathLike)):
                names = [names]
            for name in names:
                if content is None:
                    pathlib.Path(name).touch()
                else:
                    with open(name, 'w') as tf:
                        tf.write(content)
                temp_fds.append(name)
        if 'dir' in kwargs:
            if isinstance(kwargs['dir'], (str, os.PathLike)):
                dirs = [kwargs['dir']]
            else:
                dirs = kwargs['dir']
            for dir in dirs:
                if os.path.isdir(dir):
                    shutil.rmtree(dir)
                os.makedirs(dir, exist_ok=True)
                temp_dirs.append(dir)
        return temp_fds, temp_dirs

    yield get_tempfiles

    for temp_file in temp_fds:
        try:
            if os.path.isfile(temp_file):
                os.remove(temp_file)
        except Exception:
            pass

    for temp_dir in temp_dirs:
        try:
            if os.path.isdir(temp_dir):
                shutil.rmtree(temp_dir)
        finally:
            pass


@pytest.fixture
def clear_now_and_after():
    # This fixture takes one or more file names or directory names and will
    # remove them if they exist, and also after the tests are completed.
    temp_fds = []

    def clear_files_and_dirs():
        for temp_fd in temp_fds:
            try:
                if os.path.isfile(temp_fd):
                    os.remove(temp_fd)
                elif os.path.isdir(temp_fd):
                    shutil.rmtree(temp_fd)
            except Exception:
                pass

    def get_names(*args):
        for names in args:
            if isinstance(names, (str, os.PathLike)):
                temp_fds.append(names)
            else:
                temp_fds.extend(names)

        clear_files_and_dirs()
        return temp_fds

    yield get_names

    clear_files_and_dirs()

@pytest.fixture
def sample_workflow():
    with open('sample_workflow.ipynb', 'w') as sn:
        sn.write(r'''{
 "cells": [
  {
   "cell_type": "code",
   "execution_count": 1,
   "metadata": {
    "kernel": "SoS"
   },
   "outputs": [],
   "source": [
    "# this is a test workflow"
   ]
  },
  {
   "cell_type": "markdown",
   "metadata": {
    "kernel": "SoS"
   },
   "source": [
    "This is a markdown cell"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": 2,
   "metadata": {
    "kernel": "R"
   },
   "outputs": [
    {
     "name": "stdout",
     "output_type": "stream",
     "text": [
      "This is a cell with another kernel"
     ]
    }
   ],
   "source": [
    "cat('This is a cell with another kernel')"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": 3,
   "metadata": {
    "kernel": "SoS"
   },
   "outputs": [
    {
     "name": "stdout",
     "output_type": "stream",
     "text": [
      "This is a scratch cell\n"
     ]
    }
   ],
   "source": [
    "print('This is a scratch cell')"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {
    "kernel": "SoS"
   },
   "outputs": [],
   "source": [
    "# this comment will be included but not shown in help message\n",
    "# because it is for the global\n",
    "[global]\n",
    "a = 1\n",
    "# this comment will become the comment for parameter b\n",
    "parameter: b=2\n",
    "parameter: c=3\n",
    "# this comment will become the comment for parameter d\n",
    "parameter: d='d'"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {
    "kernel": "SoS"
   },
   "outputs": [],
   "source": [
    "# this comment will not be included in exported workflow\n",
    "# because it is not immediately before section\n",
    "\n",
    "# this is a section comment, will be displayed\n",
    "[default]\n",
    "print(f'Hello {a}')"
   ]
  }
 ],
 "metadata": {
  "kernelspec": {
   "display_name": "SoS",
   "language": "sos",
   "name": "sos"
  },
  "language_info": {
   "codemirror_mode": "sos",
   "file_extension": ".sos",
   "mimetype": "text/x-sos",
   "name": "sos",
   "nbconvert_exporter": "sos_notebook.converter.SoS_Exporter",
   "pygments_lexer": "sos"
  },
  "sos": {
   "default_kernel": "SoS",
   "kernels": [
    [
     "R",
     "ir",
     "R",
     "#DCDCDA"
    ],
    [
     "SoS",
     "sos",
     "",
     ""
    ]
   ],
   "panel": {
    "displayed": true,
    "height": 0,
    "style": "side"
   },
   "version": "0.9.14.11"
  }
 },
 "nbformat": 4,
 "nbformat_minor": 2
}
''')


@pytest.fixture
def fastq_files():
    if not os.path.isdir('data'):
        os.mkdir('data')
    with open('data/S20_R1.fastq', 'w') as R1:
        R1.write('''\
@ST-J00106:29:H3C5HBBXX:4:1101:1172:1000 1:N:0:TGACCA
CCTCAGTTCCTCAAACACAGTGAGTGAGTACGGACCTCAGGCTGTCTCTTCGCGTCACACCCGCTGTGAAATGGCAGATACAGCATCAGGGCAGACACTC
+
<<AAFKKKKKKKKKFKFKK7FKKKKKFK,7,<AFF<7,AAKKAK,FK7FKA,7A,,,,7,,,<(F<K7,F,,7<A<,,AA,<,<<,A7,F,(A7A7,AAK
@ST-J00106:29:H3C5HBBXX:4:1101:1375:1000 1:N:0:TGACCA
ATATGCATTCATATAAGTTCATGTCATGCAGCTGACAAATAGAATGATATAGCTCAACTCTACTCATGGATGCCGAACGATGTCTGTGGCTTGCACTGTG
+
AAFFAKKKFKKKKFKKKKKFKKKFKAFKKKKKKKKKKKKFKKKKAKKFKKKKF7FF<FKK<KKKFKKKKFKKKKKFAAKF,,AFFKK7,,7F7AFKK,FK
@ST-J00106:29:H3C5HBBXX:4:1101:1395:1000 1:N:0:TGACCA
CTGTTACCATGTTAATAGCCACACCCTTATGGCCAAACCATCCACCTGTATCCACAGAATTGGTCGAGGTGGATGGTTTGGCCATAAGGGTGTTGCTATG
+
AAAFFKKKKKFKKKKKKKKAKKKFFAFAFAKKKFKKKKFKKFKKAAAA7FFFAFKKKKA<AKKKKKFFK,,AAFFF<,,7F,,A,F<<F,F7F,FFKFAF
@ST-J00106:29:H3C5HBBXX:4:1101:1497:1000 1:N:0:TGACCA
GTTCGAGGGCAGGGTGAGGCCGCGGGGCTGAGCACCATGGCCGCTACCGGGACCACCAGGCCGTGCGCGTTTCCACGCTGTCTTTCTAGGATGCTCCC
+
AAFF,AFK<<KFAFKKKKKKFFKKFKKFKKKKFKFKKFKKKKKKKKKKKKKKFKKKKKKKKKKKFKKKKAF7FKKFF7FKKKKKKAAKF7FKA,<FA<
@ST-J00106:29:H3C5HBBXX:4:1101:1598:1000 1:N:0:TGACCA
ATAAAATAAATTTTGCATTCTAAAAATAAAAATGTAAAGAATATACAGCCATAGGCCCCAATTGTTTGCTTCTTCCAGGAACAGCATCTCCTCTTCTGA
+
AAAAAKAKAKAAFKKKK7<FAKKKKKKKKKKK7KFKKKKKFAFFAKKFKKKAK,FF7(<FKK<,<F<7,A<,FKK7FFFKFKFKKKFA,FAFKKKK7FA
@ST-J00106:29:H3C5HBBXX:4:1101:1619:1000 1:N:0:TGACCA
GATCAAGCGAGCTTTTGCCCTTCTGCTCCACGGGAGGTTTCTGTTCTCCTTGATCTCGCCTTAGGACAGCTCGCGGGGAGTCGAAGGACAAAAGCGCGTT
+
AA,FFKKKKKKK,<<AKKKKFFAKKKKAKKK7<FF(<,,,7,A,,7A<A,,,A,7,77,,,,,<F7,,,,,F,,A((,A7,,7,,,,,,AA7<,,,,A(,
@ST-J00106:29:H3C5HBBXX:4:1101:1923:1000 1:N:0:TGACCA
CATGAAAGTGAAGAACAAGGAAGCCCCAACCTACCTTGTTCTTCACTTTCATGCCAGACTACAACTGTACCCCACAAAGAGGTGGGGTACAGTTGTAGTC
+
AAFFFKKFKKKKKKKKKKKFKKKKKKKKKKKKKKKKKKKKKKKKKKKKKFKKKKAKKKKKKKKKKKKKKKAKKKK7FAAKK,7,A,AK7,<A7FKFFK7K
@ST-J00106:29:H3C5HBBXX:4:1101:1964:1000 1:N:0:TGACCA
AGGAAATATAGAGAATACCACAAAGAAACTCCTTGCAAAGAGCAACTCCAAGACCCATAATTGGCAGATTCACCAAGATGGAAATGAACGAAAAAATATT
+
AAFFAKKKKKKKKKKKKKKKKKKKKKKKKKKKAKKKKKKKKKKKKKFKKKKKKKAKKKFKKFK,FF7A,,AKFFFKKKAKKFFA,,FKFKKKKFKKA<AK
@ST-J00106:29:H3C5HBBXX:4:1101:2086:1000 1:N:0:TGACCA
TACCAACAAATGCAAAAACACAAACCAGTGGTAAATATGCAGACCAGCGATACTAGAAAGCAGCCACACAAACAAGACAGCATAATAACCAGCTAAGAAC
+
AAFA<FAKAAFKAKKKKKKF<AAFKKKK,A,A,FFFF,<<FKKF7FKK<A7AAAF,<FKKAF<,,A,A,AFF7A7<FA,7,A,,A,FKF,F,7,AFFKKK
@ST-J00106:29:H3C5HBBXX:4:1101:2187:1000 1:N:0:TGACCA
CCTTTCCCAGTCCACTAAATAAATGTTAATCTCCTTTGGCAACACCCTCACAGACACAACCGGGAAAAACACTTTGGATCCTTCGATCCAAT
+
AAAFFKKKKKFKKKKKKKKKKKKKKAKFFFKAFKKKKKKKKKKKKFKKKKKKKKKKKKKKKKK<FKKKKKKKKKKKKFFKAAAKKKKK<KK<
@ST-J00106:29:H3C5HBBXX:4:1101:2207:1000 1:N:0:TGACCA
GAAGAGCACACGTCTGAACTCCAGTCACTGACCAATCTCGTATGCCGTCTTCTGCTTGAAAAAAAAAAAAAAAAGAAGCAGGGAATGATTCCCTACCGG
+
AAFFFKKKKKKKFKKKKKKFKKKKKKKKFKKKKKK7,FFKKKKKKKKAAKKKKKKFFFKKKKKKKKKKA,,,,,,,,,,,,7,,7,,,,,,,,,,,,,,
@ST-J00106:29:H3C5HBBXX:4:1101:2289:1000 1:N:0:TGACCA
TATTGGATAGAATTCTTAATTCATTCTCTATGTTATCTTGAATTTCTTTGAGCTTCCTTAACACAGCTATTGTGAATTCCGATTCACAACACAAAAAC
+
AAFFFKKAKFKKKKKKKKKKKKKKKKKKAK7FFKA,AKKKAKKFKFA<K,<<FKFFFKKA,FFFFKKFKKKFKFA7FFFFA,<,,,FA,,7<A,,AFA
@ST-J00106:29:H3C5HBBXX:4:1101:2410:1000 1:N:0:TGACCA
TTTCTCTTCTGAGCTCCTAAAGTGCAGGGTGTTTTTCAGACTCAAAGGTCAAGGCCTTCTTAGAATAGGGCTGCATGTGTCTTTGTCCAGTGCATGTTGC
+
AAFF<<FFKKKKFKKKK<KKKKKKKFFKKKA<<A,,FKKKKKKKKKKKKFAFF,7<,,7,,7FF<FKFAKKAFFA,A7F,AFAFKFAKFFAF7F,77,,,
@ST-J00106:29:H3C5HBBXX:4:1101:2512:1000 1:N:0:TGACCA
TCAAGTCTTGAAGCTGGCCCAGGCTTACAGACCAGTTCGCTGAGGTAGCTACCACTCAGTAGAGGGCACTCCCTTTGTCCTGGGAGGCATAGCTGAGTAT
+
<AFFFKKKFKKKKKAKKKAFKKKKKKKKKKKKKFKAFKKKKKKKFFKFK7FAFKFFKKKKKKKFAKKKKKKKK,F,7AAFAKKFK7AAA<FA<7KK,7,<
@ST-J00106:29:H3C5HBBXX:4:1101:2593:1000 1:N:0:TGACCA
TAACTTTGTGTTTGAAGGAGGGGGGCAGGAGATCAGGACCAAGGGTTTGGCCAAGGCCAGATCTAGAAGACTCGGGCCATCCCATAGTTGATACCTT
+
AAFFFKKKKKKKFKKKKKKKKKKKKKKKKKFKAKKKKKKKKKKKKKKKKKKFAKFK<KKKKFFKKKKKKKKKKKKKKFFAFKKKKKKF,AFFFKKKA
@ST-J00106:29:H3C5HBBXX:4:1101:2613:1000 1:N:0:TGACCA
TATTTACAAACCCTGCCATAAAGTGCTCTTGAAAATTATGTCAAGGCCGGGCGCGGTGGCTCACGCCTGTAATCCCAGC
+
AAAFFKKKKKKKKKFFKKKKKKKKKKKKKFFFKKKFKKKKKKKKKKKKKKKKKKKKKKKKKKAFKKKKKFFKKKKKAKA
@ST-J00106:29:H3C5HBBXX:4:1101:2756:1000 1:N:0:TGACCA
CAGCCACGTGGCTTTAAGTACAATCTACATGACAACGACTCCTGTGGTAGCCAGCCTCCAAGATGGTCCCAGTGGCGCCTGCCTCTGTACCAGGGTTGGT
+
AAFFFKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKFKKKKKKKKF<<A<KKKKF<FFF(,FAFFK7<<FF,FAFKFF
@ST-J00106:29:H3C5HBBXX:4:1101:2918:1000 1:N:0:TGACCA
CCAGATGAGAAAATAAAGAAAGGGACATGGAAACAGGATGAGGAAAAACACTTTTTTTCTCATCTGGGCTATTCTTGAGGCAGGAGGAAAAAGAAGAGA
+
AAFFFFKKKKKKKKKKFKKKKKKKKKKKKA7FFKKKAFFKKKKKFKKKKKK7FK7<FAK7FKKKKKKFA7AAAAF,A,<FA7FF<AKK7FFFF7AKFKF
@ST-J00106:29:H3C5HBBXX:4:1101:3202:1000 1:N:0:TGACCA
TTTTTGAGATGGAGTCAGGCTCTGTCGCCCAGGCTGGAGTGCAGTGGTGCGATCTTGGCTCACTGCAAGCTCAGCCTCCCGGGTTCAGGCCA
+
AAFFFFAFKKKKKKKKKKKKFKKKKKKKKKKKKKAKKKKFKFKKKKKAKKKKAKKFKKKKKKKKKKKKKKFKKKKKK,FKK<F,FKKKKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:3344:1000 1:N:0:TGACCA
AAAAACACACCAGAGAGAGACAGAGGTCCTAAAGATGCAGACCCCAGACCCTGAGACAGAGAGACCCTAGACCCACAGAGAAAGATACTTCACCACCCCC
+
AAFFFKKKKKFFKKKKKKKKKKKKKKKKKKKKKKKKKKKFFK7FFKKKKAF7FKKKAKKKKKKFFAFKKKKKKKKKFKKKKKKKKKKF<7AFFKKKKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:3385:1000 1:N:0:TGACCA
TTCTTTCTCTTGCCTGATTGCCCTGGCTAGAACTTCCAATACTATGTTGAATGGGAGTGGTGAGAGAGGGCATCCTTGTTTTGTGCCGGGTTTCAAAGGG
+
AAFFAKKFKKFKKKKKKKFAFFKAFAKAFFKKKFAAAFKKKKFFKKKAFKKAFKKKKKKKFKKKK7F,FA(,F7AKKKKKFKKKK<KFF(FAAKF7AA77
@ST-J00106:29:H3C5HBBXX:4:1101:3425:1000 1:N:0:TGACCA
ACCAGAACCACTTTTTGGCTCCTTCTATAGTTTGAGTATGTCATCTTTATTTCTAAGTCTGTATGATATCAATTTTGCATTAGCACCTCGACTTTGA
+
AAFFFKKKKKKKKKKKKKKAKKKKKKKKKKKKKKKKFKKKKKKKKFKKKKFF7AKKFFKKFKKKKKFKKKKKKKFFKKKK7FAKKFKFAA,FKKKAA
@ST-J00106:29:H3C5HBBXX:4:1101:3446:1000 1:N:0:TGACCA
AGCCTAGACCTTTACCTGCATCTCCCTCAGGGCTAACTTCCCAAGACAGTGGCAGGTAAAGGTCTAGGCTCTCCACTTAGAC
+
AAFFFKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKFKKKKKFFKKKKKKKKKKKKKKKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:3466:1000 1:N:0:TGACCA
AAGGTTGTAGAGTGACCTACAGGTGAACAGCTCTGCTCCTCCCAACTGTTCGACCACTGCTCTCACCTCCACACAATCCCAGTGGCCCCTCGCTCTCCCT
+
A<<FAA,,FFKA<FKKKKKKKAAFKFFKKAKKKFAF7AA,F<AKFFKAAAAKFKF,FFK,7,FFK,,<AA7FKFF<,7AFKKKKF7<((<F<(,A<AA<7
@ST-J00106:29:H3C5HBBXX:4:1101:3507:1000 1:N:0:TGACCA
AGTGAAAATAGCGTTAACTGATGACATTCCACAATTGTGATTTGTTTCTGCCCCACCCTAACTGATCAATGTACTTTGTAATCTCCCCCACCCTTAAGA
+
AAAFFKKKKKKKKKKKKKKKKKKKKKKKFKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKKFKKK7FK<AAKKK,7FAKK,<FAFKKKKKKK<KKKFK
''')
    with open('data/S20_R2.fastq', 'w') as R2:
        R2.write('''\
@ST-J00106:29:H3C5HBBXX:4:1101:1172:1000 2:N:0:TGACCA
AGGATAAGCAGTCCAAGGAGGCATTGACCAAAACATGCAGTGAAATGTTGATGCGGAGCCTAAAGGACGGGGATCTGGGGGAGCCACGCCTGGAGGCG
+
A<,,,,,7A<F7,,,7,,,,,(,7,,,7,,,7<7,,7<7,,,,7,7A,,,,,,7,,,,(,AF,7,A,,,((,,77<<,7(,F7((A,(((7(<,,7A,
@ST-J00106:29:H3C5HBBXX:4:1101:1375:1000 2:N:0:TGACCA
TGAGAATCTATGGGCATGTGTAGTGTTACATAAGTGGGGTCACATACTGGATAGCAAAGTGCAAGACACAGACATCGTTCGGCATCCATGAGTAGAGTTG
+
<AA<FFFFKKKKKKFFKK7KKAKFF7FAFA,KAKKKAK<7AFKF,FAFFKKKKKKKKAFKKKFFKKKF,AFAF7AKFKA,,77AFA7FAFK<,FFAK<AF
@ST-J00106:29:H3C5HBBXX:4:1101:1395:1000 2:N:0:TGACCA
AGTCCTCTTGTCTTCTTCTGTTACCATGTTAATAGCCACACCCTTATGGCCAAACCATCCACCTCGACCAATTCTGTGGATACAGGTGGATGGTTTGGCC
+
AAFAFKKKFKKKKKKKKKKKKKKKKKKKFAKKKKKKKKKKK7F,AKKAKKKKKKKKK,FFKF<KAKKF<<AAF7FK7F<FAK<FFFAFF7<<KKAAFK7,
@ST-J00106:29:H3C5HBBXX:4:1101:1497:1000 2:N:0:TGACCA
AGGCGAATCGATGTGGCTCCCCCAGCCCCTTCCTGGGAGCATCCTAGAAAGACAGCGTGGAAACGCGCACGGCCTGGTGGTCCCGGTAGCGGCCATGGTG
+
AAFFFFAAFKKKKFKKKAKKKKKKKKKKKFKKK,7AAKFKK<KKFAKK,FAAKKAKKFK<FFKKKFF<(7<<F<FFAK7AAFKK<(<F,K7AFKF,AA<F
@ST-J00106:29:H3C5HBBXX:4:1101:1598:1000 2:N:0:TGACCA
AAAACTGAAAGTTCCAGGGGAAGTGGAAAGGATCACCACCTGCAGAGTCAGATATTCACTTCTGGTAACTCTGTGGACCTTTGGCAGGGGGCTTCAGA
+
AAAFFKKFKKKKKKKKKKAFFKKKKKKKFFKKKKKKKKAFKKKKAAFK7<KK7FFFAKKAFA,FKF7K,7FFFKFKKKAFF,AAKKKKFAAAFFAAFK
@ST-J00106:29:H3C5HBBXX:4:1101:1619:1000 2:N:0:TGACCA
TCCTCCTGCTTAAAACCCAAAAGGTCAGAAGGATCGTGAGGCCCCGCTTTCACGGTCTGTATTCGTACTGAAAATCAAGATCAAGCGAGCTTTTGCCCTT
+
,AAFFK<KKKFKKKKKKKKKKKAFFKKKAF7AA,AFKFFKKKF<<AFFKFKKKKKKK,,,FF,FFFKKFKKKKKAAFAK,,,7,,7AAKKA<FFF,FFKK
@ST-J00106:29:H3C5HBBXX:4:1101:1923:1000 2:N:0:TGACCA
TCACTTTCATGCCAGACTACAACTGTACCCCACCTCTTTGTGGGGTACAGTGGTAGTCTGGCATGAAAGTGAAGAACAAGGTAGGTTGGGGCTTCCTTGT
+
<AFFFKKKKKKKFKKKKFKKKKKKKKKFFFKKFF,A,A<<F7FA<,FKF,,,FFFKKKKAFKF7,FKK,7FFFAFKKKFAF7AFAF7FKKFFKKK<FKKK
@ST-J00106:29:H3C5HBBXX:4:1101:1964:1000 2:N:0:TGACCA
CCCTTAATATTTTTTCGTTCATTTCCATCTTGGTTAATCTGACAATTATGGGTCTTGGAGTTGCTCTTTGCAAGGAGTTTCTTTGTGGTATTCTCTATAT
+
<<FAFKKKKKAAKKKKKKKKKKKKFAFKAKKKKK7,FKAKK7<,AFF<KAKAKKKKK7AA<F,AFF7KKF,7A7<,<FKK<FKK7F7AAFFFAF<FKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:2086:1000 2:N:0:TGACCA
TAGTTTTATATATGTGTATTTAATCCTGTCATTGTGTTCTTAGCTGGTTATTATGCTGTCTTGTTTGTTTGGCTGCTTTATAGTATCGCTGGTCTGCAT
+
,AFFFKKKKKKKKKKKKKF<KFKKKKKKKKFKKKKKKK7FKFKKKKK<F<<KKKKKAKKKKK7F7FA7,7FAFKK,FKKK7,FFKAAKF,,AFAKKKKA
@ST-J00106:29:H3C5HBBXX:4:1101:2187:1000 2:N:0:TGACCA
ATTGGATCGAAGGATCCAAAGTGTTTTTCCCGGTTGTGTCTGTGAGGGTGTTGCCAAAGGAGATTAACATTTATTTAGTGGACTGGGAAAGG
+
<AF<FK7FKKKKKKKKKKKKKKKKKKKFKFFFKKFAKKKKKKKFKKK<<KKKAKKKKKKKKKAF,FFK,F<AFAFKKFKFKFAFFAFFFKKK
@ST-J00106:29:H3C5HBBXX:4:1101:2207:1000 2:N:0:TGACCA
GAAGAGCGTCGTGTAGGGAAAGAGTGTAGATCTCGGTGGTCGCCGTATCATTAAAAAAAAAAAACAAGAGTAGTGTACGGAGCGTGGTTGTCTAGAGGTT
+
AAFFFKKKKKK7FKKKKKKKKKKKFKKKKKKKKKKKKKKKKKKKKKKFKKKKKKKKKKKKKKF7,,7,,7,,77,,7,,,,,((A7(,,77,,,,,,77,
@ST-J00106:29:H3C5HBBXX:4:1101:2289:1000 2:N:0:TGACCA
GAGGTACAAATACGCCCAGACTGCAAAGACGAAAAGAAACCCCGAACGCTTCAAAGCCAAGACACAGAGAAACATCTACAAGAATAAAGAAAATCCAGG
+
,AAF,FA,A7,,,,A<FFA,F,AFF,,<FA,F7FA,7FA,,AF<F7A,,777KK,7,,,7A,,A7FAFFF,,<F<,,AF,KA,7,A<FFAAFF,A<7,7
@ST-J00106:29:H3C5HBBXX:4:1101:2410:1000 2:N:0:TGACCA
TTGCAGTGACAGGTGCCCCGAGGAAAGCAGGCAAACATGCACTGGACAAAGACACATGCAGCCCTATTCTAAGAAGGACTTGACCTTTGAGTCTGAAAA
+
,A,FFFFKKK<FKKKKA<7<((<AFF<KAFFKKAFFFK,AFFFK7KKKFFAFKKKKK,FAFFFKFKK7A<FKKAFAFKFAF7F7<,A,AFFFF,FFFKK
@ST-J00106:29:H3C5HBBXX:4:1101:2512:1000 2:N:0:TGACCA
GCAGGTTAGAGAAGTGGGCCTCAGAAGTGCCAGGACATCATGAAGGGAGATAATCAGCTATGCCTCCCAGGACAAAGGGAGTGCCCTCTACTGAGTGGT
+
<AFFF77FFKKKFKFAAA<F(FKKKKKKKKKKK<KAF<<F7KKKFKKFKFKKK<7KKKAAFKAA,A(,,AFKKK77FFKKKKKFA7F<AAFFKA7AFFF
@ST-J00106:29:H3C5HBBXX:4:1101:2593:1000 2:N:0:TGACCA
AAGGTATCAACTATGGGATGGCCCGAGTCTTCTAGATCTGGCCTTGGCCAAACCCTTGGTCCTGATCTCCTGCCCCCCTCCTTCAAACACAAAGTTA
+
AAFFFKKKKKKFKKKFFFFKAFKAFKFFFKFFKKKKKKKKF<F<FAFFKKKKKAFAF,<<A7,7FFA,,,A<A<,<<,F<F7FAFKKKKKKKK7AFK
@ST-J00106:29:H3C5HBBXX:4:1101:2613:1000 2:N:0:TGACCA
GCTGGGATTACAGGCGTGAGCCACCGCGCCCGGCCTTGACATAATTTTCAAGAGCACTTTATGGCAGGGTTTGTAAATA
+
<AFFFKKKKKKKKKKKFFKKKKKKKKKKKFKKKKFFKKKKKKKKFAFKKFKKKKKKKAFFKF7FKKKFKKA<FKKKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:2756:1000 2:N:0:TGACCA
CTTTGTGTTTGAGTGGGCTGAGGGCAGAAGTTATAGTCTCTTATAATCCAGTCTCAGAAGTAACTGAACAGCACTCCTAGCTCTTGTGGGTCACACATAC
+
AAFFFKKKKKKKKKKKKKAAKFKKKKKKKKF<FKAFKKKKKKK<FFKAFKK7AFAKKFKF7,FFAFAFAAKAFAA7FAKF7,AF<<AF,AAFKKAKKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:2918:1000 2:N:0:TGACCA
TGCGCAGCAGCACTTTCTCTTCTTTTTCCTCCTGCCTCAAGAATAGCCCAGATGAGAAAAAAAGTGTTGTTCCTCATCCTGTTTCCATGTCCCTTTCTTT
+
AAFFFKFKKKKKKKKKKKKKKKKKKKKKKKKK<,AFKKKKKFKKKAFKKKKFKKKKKKK<FK,AFFFK,A<FFFFKKFKKKFK,FFFFKKKKKKKKKKKK
@ST-J00106:29:H3C5HBBXX:4:1101:3202:1000 2:N:0:TGACCA
GGCCCGTACCCCTCCGAGCAGACCTTTCCGCGTCCCAACATCGCCCCACCCCACTCCAGCCTGGGCGACAGAGCCTGACTCCATCGCAAAAA
+
,,,,,((,,7A,,,,(,,,,FK,,A,,,,,,(((,(F,,7F7FK,<<FA((<AA7FFF,AFAF7<FKKKKFKKFKAFKKFKK7FF,FKFFFK
@ST-J00106:29:H3C5HBBXX:4:1101:3344:1000 2:N:0:TGACCA
GGGGGGTGGTGAAGTATCTTTCTCTGTGGGTCTAGGGTCTCTCTGTCTCAGGGTCTGGGGTCTGCATCTTTAGGACCTCTGTCTCTCTCTGGTGTGTTTT
+
AAFAFAAAAAAKKKFKA,AA,A<FAF7AAK<FAKKFKFK<FAF7FKF7FKK<FFK7FKFFFF<FKKFA7<FFKKKK7,A,AFAF<FFAA7AKAFKAF7FF
@ST-J00106:29:H3C5HBBXX:4:1101:3385:1000 2:N:0:TGACCA
AAAACTGGAAGCATTCCCTTTGAAAACCGGCACAAAACAAGGATGCCCTCTCTCACCACTCCCATTCAACATAGTATTGGAAGTTCTAGCCAGGGCAATC
+
AAFFFFKFFKAFKF,,F<KFFFKFKKAAA<<A7KAFKKKK77F,7,,A7F7A,FKFKFA,AAFAFAAFKAA7,,FKKKF<FF,7,7AFK7,<7<,AAA7,
@ST-J00106:29:H3C5HBBXX:4:1101:3425:1000 2:N:0:TGACCA
TCAAAGTCGAGGTGCTAATGCAAAATTGATATCATACAGACTTAGAAATAAAGATGACATACTCAAACTATAGAAGGAGCCAAAAAGTGGTTCTGGT
+
,AFFFKKKKKKKKKKKKKKKKKKKKFKKFFFFKKKKKKAKKAKKKKKKFKKKKKKKKKKFFK<KKKKK,KKKKKKKKKKKKKKKKKKKKKAKKFF<A
@ST-J00106:29:H3C5HBBXX:4:1101:3446:1000 2:N:0:TGACCA
GTCTAAGTGGCGATCCTAGTCCTTTACCTGGGACTGTCTTGCCTTTTTAGCCCCCAGGGTGTTGAAGGTAAACGTCTAGGGT
+
A,A<FAA,,<7(,,7A,A,,,,,,AF,,,7,(,,,7,7<,7,,,,,,,,A<AK((,(((77,,,,,<<A77,,,77,,7<7,
@ST-J00106:29:H3C5HBBXX:4:1101:3466:1000 2:N:0:TGACCA
CTCCGCACATGTGAGGGAGAGCGAGGGGCCACTGGGATTGTGTGGAGGTGAGATCAGAGGTCGAACAGTTGGGAGGAGCAGTGCTGTTCACCTGTAGGTC
+
,,A<FFKF7,A7AFKKKKK,AAKKKFF<F,7,7F,F77,,AF,F77FFAA,AFKKAFAFF,<AF,FA7F,FA7<7AFFKKK7FKA,,,<AFAKKFAA7AA
@ST-J00106:29:H3C5HBBXX:4:1101:3507:1000 2:N:0:TGACCA
AGGGTGGGGAGAATTACAAAGAACCTTCTTAAGAGTGGGGGAGATTACAAAGAACATTCTTAAGGGTGAGGGAGATTACAAAGTACATTGATCAGTTAGG
+
<A<FAF7A(<AFFKKA<KKKAFKF7<FKFKKKKK7,7,,(AFAKFKKFKAKKAKAFK7A,,FFA,<F7,,AAFFFK<FFAKKAFKF<,AAAFAFAK7FKK
''')
