'''
A module to process the .per_base_coverage.bed file from
the nCoV pipeline.
'''

import os
import sys
import statistics
import csv


class PerBaseCoverage(object):
    '''
    A class to handle the .per_base_coverage.bed file generated by the nCoV
    analysis pipeline.
    '''

    def __init__(self, file, delimiter='\t'):
        '''
        Initialize the PerBaseCoverage class.
        '''
        if os.path.exists(file):
            self.file = file
        else:
            sys.exit("Invalid or missing file.")
        self.delimiter = delimiter


    def get_coverage_stats(self, delimiter='\t'):
        '''
        A function to calculate the depth of coverage across the genome from the
        bedtools <sample>.per_base_coverage.bed file.

        Arguments:
            * delimiter:    field delimiter for the coverage file (default: '\t')

        Return Value:
            Function returns a dictionary with the following keys:
                * mean_depth
                * median_depth
        '''
        depth = []
        with open(self.file) as file_p:
            cov_reader = csv.DictReader(file_p, delimiter=self.delimiter)
            for data in cov_reader:
                depth.append(int(data['depth']))
        file_p.close()
        mean_depth = round(statistics.mean(depth), 1)
        median_depth = round(statistics.median(depth), 1)
        return {"mean_sequencing_depth" : mean_depth, "median_sequencing_depth" : median_depth}

